#include <stack>
#include <sstream>

#include <expat.h>

#include "Tools.h"
#include "File.h"

#include "XMLElements.h"
#include "XMLVisitors.h"
#include "XMLParser.h"


//============================================================================
// Implementation of XMLException.h
//============================================================================

//----------------------------------------------------------------------------
XMLException::XMLException(const std::string &error)
        : Exception(error)
{
}

//----------------------------------------------------------------------------
XMLException::~XMLException()
{
}



//============================================================================
// Implementation of XMLElements.h
//============================================================================

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const bool value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const int value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const unsigned value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const char *value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const std::string &value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::~XMLProperty()
{
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const bool value)
{
    m_value = value ? "1" : "0";
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const int value)
{
    std::ostringstream s;
    s << value;
    m_value = s.str();
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const unsigned value)
{
    std::ostringstream s;
    s << value;
    m_value = s.str();
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const char *value)
{
    m_value = value;
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const std::string &value)
{
    m_value = value;
}


//----------------------------------------------------------------------------
bool XMLProperty::getAsBool() const throw (XMLException)
{
    const char *prop = getValue().c_str();

    bool value = false;

    if (!strcasecmp(prop, "1") ||
        !strcasecmp(prop, "true") ||
        !strcasecmp(prop, "on"))
    {
        value = true;
    }
    else if (!strcasecmp(prop, "0") ||
             !strcasecmp(prop, "false") ||
             !strcasecmp(prop, "off"))
    {
        value = false;
    }
    else
    {
        throw XMLException(
            std::string("The property '")
            .append(getName()).append("' is not a boolean"));
    }

    return value;
}

//----------------------------------------------------------------------------
int XMLProperty::getAsInt() const throw (XMLException)
{
    const char *prop = getValue().c_str();
    char *endptr;

    int value = strtol(prop, &endptr, 10);
    if (endptr == prop)
    {
        throw XMLException(
            std::string("The property '")
            .append(getName()).append("' is empty"));
    }
    if (*endptr != '\0')
    {
        throw XMLException(
            std::string("The property '")
            .append(getName()).append("' is not an integer"));
    }

    return value;
}

//----------------------------------------------------------------------------
unsigned XMLProperty::getAsUnsigned() const throw (XMLException)
{
    const char *prop = getValue().c_str();
    char *endptr;

    unsigned value = strtoul(prop, &endptr, 10);
    if (endptr == prop)
    {
        throw XMLException(
            std::string("The property '")
            .append(getName()).append("' is empty"));
    }
    if (*endptr != '\0')
    {
        throw XMLException(
            std::string("The property '")
            .append(getName()).append("' is not an integer"));
    }

    return value;
}

//----------------------------------------------------------------------------
DECLARE_XML_VISITOR_API_BODY(XMLProperty);


//----------------------------------------------------------------------------
XMLText::XMLText(const std::string &text)
        : XMLElement(), m_text(text)
{
}

//----------------------------------------------------------------------------
XMLText::~XMLText()
{
}

//----------------------------------------------------------------------------
DECLARE_XML_VISITOR_API_BODY(XMLText);


//----------------------------------------------------------------------------
XMLNode::XMLNode()
{
}

//----------------------------------------------------------------------------
XMLNode::XMLNode(const std::string &name) : XMLElement(name)
{
}

//----------------------------------------------------------------------------
XMLNode::~XMLNode()
{
    clear();
}

//----------------------------------------------------------------------------
void XMLNode::clear()
{
    for (XMLPropertyIter iter = m_properties.begin();
         iter != m_properties.end(); ++iter)
    {
        delete *iter;
    }
    m_properties.clear();

    for (XMLNodeIter iter = m_nodes.begin(); iter != m_nodes.end(); ++iter)
    {
        delete *iter;
    }
    m_nodes.clear();

    for (XMLTextIter iter = m_texts.begin(); iter != m_texts.end(); ++iter)
    {
        delete *iter;
    }
    m_texts.clear();

    m_childs.clear();
}


//----------------------------------------------------------------------------
void XMLNode::addProperty(XMLProperty *p)
{
    m_properties.push_back(p);
}

//----------------------------------------------------------------------------
XMLProperty *XMLNode::getMandatoryProperty(const std::string &name)
    throw (XMLException)
{
    XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory property '")
            .append(name).append("' is missing"));
    }

    return prop;
}

//----------------------------------------------------------------------------
const XMLProperty *XMLNode::getMandatoryProperty(const std::string &name) const
    throw (XMLException)
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory property '")
            .append(name).append("' is missing"));
    }

    return prop;
}

//----------------------------------------------------------------------------
XMLProperty *XMLNode::getProperty(const std::string &name)
{
    for (XMLPropertyCIter iter = m_properties.begin();
         iter != m_properties.end(); ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return *iter;
        }
    }

    return NULL;
}

//----------------------------------------------------------------------------
const XMLProperty *XMLNode::getProperty(const std::string &name) const
{
    for (XMLPropertyCIter iter = m_properties.begin();
         iter != m_properties.end(); ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return *iter;
        }
    }

    return NULL;
}

//----------------------------------------------------------------------------
bool XMLNode::hasProperty(const std::string &name) const
{
    for (XMLPropertyCIter iter = m_properties.begin();
         iter != m_properties.end(); ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return true;
        }
    }

    return false;
}


//----------------------------------------------------------------------------
void XMLNode::addText(XMLText *t)
{
    m_texts.push_back(t);
    m_childs.push_back(t);
}

//----------------------------------------------------------------------------
bool XMLNode::hasText() const
{
    return !m_texts.empty();
}

//----------------------------------------------------------------------------
const std::string &XMLNode::getText() const
    throw (XMLException)
{
    for (XMLTextCIter iter = m_texts.begin(); iter != m_texts.end(); ++iter)
    {
        return (*iter)->getText();
    }

    throw XMLException(std::string("The mandatory text is missing"));
}


//----------------------------------------------------------------------------
void XMLNode::addNode(XMLNode *n)
{
    m_nodes.push_back(n);
    m_childs.push_back(n);
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::getMandatoryNode(const std::string &name)
    throw (XMLException)
{
    XMLNode *node = getNode(name);
    if (!node)
    {
        throw XMLException(
            std::string("The mandatory node '")
            .append(name).append("' is missing"));
    }

    return node;
}

//----------------------------------------------------------------------------
const XMLNode *XMLNode::getMandatoryNode(const std::string &name) const
    throw (XMLException)
{
    const XMLNode *node = getNode(name);
    if (!node)
    {
        throw XMLException(
            std::string("The mandatory node '")
            .append(name).append("' is missing"));
    }

    return node;
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::getNode(const std::string &name)
{
    for (XMLNodeCIter iter = m_nodes.begin(); iter != m_nodes.end(); ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return *iter;
        }
    }

    return NULL;
}

//----------------------------------------------------------------------------
const XMLNode *XMLNode::getNode(const std::string &name) const
{
    for (XMLNodeCIter iter = m_nodes.begin(); iter != m_nodes.end(); ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return *iter;
        }
    }

    return NULL;
}

//----------------------------------------------------------------------------
bool XMLNode::hasNode(const std::string &name) const
{
    for (XMLNodeCIter iter = m_nodes.begin(); iter != m_nodes.end(); ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return true;
        }
    }

    return false;
}

//----------------------------------------------------------------------------
const std::string &XMLNode::getStringProperty(const std::string &name) const
    throw (XMLException)
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory attribute '")
            .append(name).append("' is missing"));
    }

    return prop->getValue();
}

//----------------------------------------------------------------------------
const std::string &XMLNode::getStringProperty(const std::string &name,
                                              const std::string &defaultValue) const
{
    const XMLProperty *prop = getProperty(name);
    return prop ? prop->getValue() : defaultValue;
}

//----------------------------------------------------------------------------
bool XMLNode::getBoolProperty(const std::string &name) const
    throw (XMLException)
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory attribute '")
            .append(name).append("' is missing"));
    }

    return prop->getAsBool();
}

//----------------------------------------------------------------------------
bool XMLNode::getBoolProperty(const std::string &name, bool defaultValue) const
    throw (XMLException)
{
    const XMLProperty *prop = getProperty(name);
    return prop ? prop->getAsBool() : defaultValue;
}

//----------------------------------------------------------------------------
int XMLNode::getIntProperty(const std::string &name) const
    throw (XMLException)
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory attribute '")
            .append(name).append("' is missing"));
    }

    return prop->getAsInt();
}

//----------------------------------------------------------------------------
int XMLNode::getIntProperty(const std::string &name, int defaultValue) const
    throw (XMLException)
{
    const XMLProperty *prop = getProperty(name);
    return prop ? prop->getAsInt() : defaultValue;
}

//----------------------------------------------------------------------------
unsigned XMLNode::getUnsignedProperty(const std::string &name) const
    throw (XMLException)
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory attribute '")
            .append(name).append("' is missing"));
    }

    return prop->getAsUnsigned();
}

//----------------------------------------------------------------------------
unsigned XMLNode::getUnsignedProperty(const std::string &name,
                                      unsigned defaultValue) const
    throw (XMLException)
{
    const XMLProperty *prop = getProperty(name);
    return prop ? prop->getAsUnsigned() : defaultValue;
}


//----------------------------------------------------------------------------
void XMLNode::acceptAllProperties(XMLVisitor &v)
{
    for (XMLPropertyIter iter = m_properties.begin();
         iter != m_properties.end(); ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::acceptAllProperties(XMLConstVisitor &v) const
{
    for (XMLPropertyCIter iter = m_properties.begin();
         iter != m_properties.end(); ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::acceptAllChilds(XMLVisitor &v)
{
    for (XMLChildIter iter = m_childs.begin(); iter != m_childs.end(); ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::acceptAllChilds(XMLConstVisitor &v) const
{
    for (XMLChildCIter iter = m_childs.begin(); iter != m_childs.end(); ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
DECLARE_XML_VISITOR_API_BODY(XMLNode);



//============================================================================
// Implementation of XMLVisitors.h
//============================================================================

//----------------------------------------------------------------------------
XMLWriteToFileVisitor::XMLWriteToFileVisitor(const char *name)
    throw (SyscallException)
        : m_file(name, "w")
{
    m_level = -1;
}

//----------------------------------------------------------------------------
XMLWriteToFileVisitor::~XMLWriteToFileVisitor()
{
}

//----------------------------------------------------------------------------
void XMLWriteToFileVisitor::do_visit(const XMLProperty *p)
{
    m_file.puts(" ");
    m_file.puts(p->getName());
    m_file.puts("=\"");
    m_file.puts(p->getValue());
    m_file.puts("\"");
}

//----------------------------------------------------------------------------
void XMLWriteToFileVisitor::do_visit(const XMLText *t)
{
    m_file.puts(t->getText());
}

//----------------------------------------------------------------------------
void XMLWriteToFileVisitor::do_visit(const XMLNode *n)
{
    if (m_level >= 0)
    {
        writeIndentation();
        m_file.puts("<");
        m_file.puts(n->getName());
        n->acceptAllProperties(*this);
    }

    if (n->getNumberOfNodes() > 0)
    {
        if (m_level >= 0)
        {
            m_file.puts(">\n");
        }

        m_level++;
        n->acceptAllChilds(*this);
        m_level--;

        if (m_level >= 0)
        {
            writeIndentation();
            m_file.puts("</");
            m_file.puts(n->getName());
        }
    }
    else
    {
        if (m_level >= 0)
        {
            m_file.puts("/");
        }
    }

    if (m_level >= 0)
    {
        m_file.puts(">\n");
    }
}

//----------------------------------------------------------------------------
void XMLWriteToFileVisitor::writeIndentation()
{
    for (int i=0; i<m_level; i++)
    {
        m_file.puts("  ");
    }
}



//============================================================================
// Implementation of XMLParser.h
//============================================================================

//----------------------------------------------------------------------------
class XMLParserPImpl
{
    //------------------------------------------------------------------------
    typedef std::stack<XMLNode*> XMLNodeStack;

public:
    //------------------------------------------------------------------------
    XMLParserPImpl();
    ~XMLParserPImpl();

    //------------------------------------------------------------------------
    XMLNode *parse(const char *file) throw (Exception);
    void parse(const char *file, XMLNode &root) throw (Exception);

private:
    //------------------------------------------------------------------------
    void resetParser() throw (Exception);

    //------------------------------------------------------------------------
    void do_parse(const char *file, XMLNode *root) throw (Exception);

    //------------------------------------------------------------------------
    static void startElement(void *userData,
                             const char *name,
                             const char **atts);

    static void endElement(void *userData, const char *name);

    static void textElement(void *userData, const XML_Char *s, int len);

    //------------------------------------------------------------------------
    void startElement(const char *name, const char **atts);
    void endElement(const char *name);
    void textElement(const char *text, int len);

    void handleText();

    //------------------------------------------------------------------------
    XML_Parser m_xmlParser;
    XMLNodeStack m_stack;
    bool m_deleteRootOnError;
    std::string m_text;
};

//----------------------------------------------------------------------------
XMLParserPImpl::XMLParserPImpl()
{
    m_xmlParser = XML_ParserCreate(NULL);
    m_deleteRootOnError = false;
}

//----------------------------------------------------------------------------
XMLParserPImpl::~XMLParserPImpl()
{
    XML_ParserFree(m_xmlParser);
    m_xmlParser = NULL;

    while (m_stack.size() > 1)
    {
        delete m_stack.top();
        m_stack.pop();
    }

    if (m_deleteRootOnError && (m_stack.size() == 1))
    {
        delete m_stack.top();
        m_stack.pop();
    }
}

//----------------------------------------------------------------------------
XMLNode *XMLParserPImpl::parse(const char *file) throw (Exception)
{
    XMLNode *root = new XMLNode();
    m_deleteRootOnError = true;

    do_parse(file, root);
    return root;
}

//----------------------------------------------------------------------------
void XMLParserPImpl::parse(const char *file, XMLNode &root) throw (Exception)
{
    root.clear();
    m_deleteRootOnError = false;

    do_parse(file, &root);
}

//----------------------------------------------------------------------------
void XMLParserPImpl::resetParser() throw (Exception)
{
    if (!XML_ParserReset(m_xmlParser, NULL))
    {
        throw XMLException("Error resetting XML parser");
    }

    XML_SetUserData(m_xmlParser, this);
    XML_SetElementHandler(m_xmlParser, startElement, endElement);
    XML_SetCharacterDataHandler(m_xmlParser, textElement);
}

//----------------------------------------------------------------------------
void XMLParserPImpl::do_parse(const char *file, XMLNode *root)
    throw (Exception)
{
    resetParser();

    m_stack.push(root);

    char buffer[1024];
    File f(file, "r");
    int done;

    do
    {
        size_t len = f.read(buffer, sizeof(buffer));
        done = len < sizeof(buffer);
        if (XML_Parse(m_xmlParser, buffer, len, done) == XML_STATUS_ERROR)
        {
            std::ostringstream s;
            s << "XML parse error of file '" << f.getName()
              << "' at line " << XML_GetCurrentLineNumber(m_xmlParser)
              << ": " << XML_ErrorString(XML_GetErrorCode(m_xmlParser));
            throw XMLException(s.str());
        }
    }
    while (!done);

    m_stack.pop();
}


//----------------------------------------------------------------------------
void XMLParserPImpl::startElement(void *userData,
                                  const char *name,
                                  const char **atts)
{
    static_cast<XMLParserPImpl*>(userData)->startElement(name, atts);
}

//----------------------------------------------------------------------------
void XMLParserPImpl::endElement(void *userData, const char *name)
{
    static_cast<XMLParserPImpl*>(userData)->endElement(name);
}

//----------------------------------------------------------------------------
void XMLParserPImpl::textElement(void *userData, const XML_Char *s, int len)
{
    static_cast<XMLParserPImpl*>(userData)->textElement(s, len);
}

//----------------------------------------------------------------------------
void XMLParserPImpl::startElement(const char *name, const char **atts)
{
    handleText();

    XMLNode *node = new XMLNode(name);
    for (int i=0; atts[i]; i+=2)
    {
        node->addProperty(new XMLProperty(atts[i], atts[i+1]));
    }

    m_stack.push(node);
}

//----------------------------------------------------------------------------
void XMLParserPImpl::endElement(const char *name)
{
    handleText();

    XMLNode *node = m_stack.top();
    m_stack.pop();

    XMLNode *top = m_stack.top();
    top->addNode(node);
}

//----------------------------------------------------------------------------
void XMLParserPImpl::textElement(const char *text, int len)
{
    m_text.append(text, len);
}

//----------------------------------------------------------------------------
void XMLParserPImpl::handleText()
{
    const char *spaces = " \t\r\n";
    size_t index = m_text.find_first_not_of(spaces);
    m_text.erase(0, index);
    index = m_text.find_last_not_of(spaces);
    m_text.erase(index == std::string::npos ? 0 : index+1);

    if (m_text.length() > 0)
    {
        m_stack.top()->addText(new XMLText(m_text));
        m_text.clear();
    }
}


//----------------------------------------------------------------------------
XMLParser::XMLParser()
{
    m_pImpl = new XMLParserPImpl();
}

//----------------------------------------------------------------------------
XMLParser::~XMLParser()
{
    ZAP_POINTER(m_pImpl);
}

//----------------------------------------------------------------------------
XMLNode *XMLParser::parse(const char *file) throw (Exception)
{
    return m_pImpl->parse(file);
}

//----------------------------------------------------------------------------
void XMLParser::parse(const char *file, XMLNode &node) throw (Exception)
{
    m_pImpl->parse(file, node);
}
