#ifndef GAMECONTROLBASE_H
#define GAMECONTROLBASE_H

#include "Tools.h"
#include "SDLTools.h"
#include "XMLElements.h"
#include "Exception.h"

class Platform;
class ProjectileBase;
class GrenadeBase;
class Ship;
class Missile;
class SAMBatteryBase;
class Tank;
class TurretBase;
class MortarBase;


//----------------------------------------------------------------------------
class GameControlBase
{
  public:
    //------------------------------------------------------------------------
    enum ShipControl
    {
        ROT_LEFT,
        ROT_RIGHT,
        ROT_OFF,
        THRUST_ON,
        THRUST_OFF,
        ALIGN_ON,
        ALIGN_OFF,
        FIRE
    };

  protected:

    //------------------------------------------------------------------------
    class GameStateBase;
    friend class GameStateBase;
    class GameStateBase
    {
      protected:
        GameStateBase() {}
      public:
        virtual ~GameStateBase() {}
        
        virtual void updateShipControl(ShipControl control,
                                       GameControlBase *game);
        virtual void updateBonus(GameControlBase *game);

        virtual void onPreUpdate(GameControlBase *game) {}
        virtual void onPostUpdate(GameControlBase *game) {}
        virtual void onShipLanded(Ship *ship, const Platform *platform,
                                  GameControlBase *game) {}
        virtual void onShipTakeoff(const Ship *ship, GameControlBase *game) {}
        virtual void onShipFiring(const Ship *ship, GameControlBase *game) {}
        virtual void onShipOutOfFuel(const Ship *ship, GameControlBase *game);
        virtual bool isFinished(const GameControlBase *game) const
        {
            return false;
        }
    };

    //------------------------------------------------------------------------
    class OutOfFuelDelay;
    friend class OutOfFuelDelay;
    class OutOfFuelDelay : public GameStateBase
    {
        STATE_OBJECT(OutOfFuelDelay);
      public:
        OutOfFuelDelay() {}
        ~OutOfFuelDelay() {}

        void onPreUpdate(GameControlBase *game);
        void onShipOutOfFuel(const Ship *ship, GameControlBase *game) {}
    };

    //------------------------------------------------------------------------
    class ExplosionDelay;
    friend class ExplosionDelay;
    class ExplosionDelay : public GameStateBase
    {
        STATE_OBJECT(ExplosionDelay);
      public:
        ExplosionDelay() {}
        ~ExplosionDelay() {}

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void onPreUpdate(GameControlBase *game);
    };

    //------------------------------------------------------------------------
    class AddFuelToScore;
    friend class AddFuelToScore;
    class AddFuelToScore : public GameStateBase
    {
        STATE_OBJECT(AddFuelToScore);
      public:
        AddFuelToScore() {}
        ~AddFuelToScore() {}

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void updateBonus(GameControlBase *game) {}
        void onPreUpdate(GameControlBase *game);
    };

    //------------------------------------------------------------------------
    class AddBonusToScore;
    friend class AddBonusToScore;
    class AddBonusToScore : public GameStateBase
    {
        STATE_OBJECT(AddBonusToScore);
      public:
        AddBonusToScore() {}
        ~AddBonusToScore() {}

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void updateBonus(GameControlBase *game) {}
        void onPreUpdate(GameControlBase *game);
    };

    //------------------------------------------------------------------------
    class Final;
    friend class Final;
    class Final : public GameStateBase
    {
        STATE_OBJECT(Final);
      public:
        Final() {}
        ~Final() {}

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void updateBonus(GameControlBase *game) {}
        bool isFinished(const GameControlBase *game) const { return true; }
    };

  public:

    virtual ~GameControlBase();

    /// (Re)initialize sm_instance from the given XML file.
    static void init(const char *mission, const char *level)
        throw (Exception);

    /// Destroy the sm_instance GameControl.
    static void destroy();

    static inline GameControlBase *getInstance()
    {
        return sm_instance;
    }

    //------------------------------------------------------------------------
    void initBonus(const XMLNode *bonusNode);

    /// Initializes the game control from the given XML node.
    virtual void initGameControl(const XMLNode *gameControlNode)
        throw (Exception) = 0;


    //------------------------------------------------------------------------
    inline Ship *getPlayerShip()
    {
        return m_player;
    }

    inline const Ship *getPlayerShip() const
    {
        return m_player;
    }

    //------------------------------------------------------------------------
    inline unsigned getPlayerFuel() const
    {
        return m_playerFuel;
    }

    //------------------------------------------------------------------------
    Ship *createPlayerShip() const;
    void resetPlayerShip();

    //------------------------------------------------------------------------
    /// Initiates the out-of-fuel handling.
    void initOutOfFuel();

    /// Initiates the explosion of the player's ship.
    void initExplosion();

    /// Initiates the fuel-to-score adding delay.
    void initAddFuelToScore();

    /// Initiates the bonus-to-score adding.
    void initAddBonusToScore();

    inline unsigned getPlayerFuelDecrement() const
    {
        return m_playerFuelDecrement;
    }

    //------------------------------------------------------------------------
    inline void resetFrameCounter()
    {
        m_frameCounter = 0;
    }

    inline void setFrameCounterLimit(unsigned limit)
    {
        m_frameCounterLimit = limit;
    }

    inline bool hasFrameCounterReachedLimit()
    {
        return (m_frameCounter++ == m_frameCounterLimit);
    }


    //------------------------------------------------------------------------
    void update()
        throw (SDLException);

    //------------------------------------------------------------------------
    inline void setState(GameStateBase *newState)
    {
        m_state = newState;
    }

    //------------------------------------------------------------------------
    inline void updateShipControl(ShipControl control)
    {
        m_state->updateShipControl(control, this);
    }

    inline void updateBonus()
    {
        m_state->updateBonus(this);
    }

    //------------------------------------------------------------------------
    inline void onPreUpdate()
    {
        m_state->onPreUpdate(this);
    }

    inline void onPostUpdate()
    {
        m_state->onPostUpdate(this);
    }

    inline void onShipLanded(Ship *ship, const Platform *platform)
    {
        m_state->onShipLanded(ship, platform, this);
    }

    inline void onShipTakeoff(const Ship *ship)
    {
        m_state->onShipTakeoff(ship, this);
    }

    inline void onShipFiring(const Ship *ship)
    {
        m_state->onShipFiring(ship, this);
    }

    inline void onShipOutOfFuel(const Ship *ship)
    {
        m_state->onShipOutOfFuel(ship, this);
    }

    /**
     * @return true, if the current level is finished (including the
     *         AddFuelToScore handling) and the next level can be started.
     */
    inline bool isFinished() const
    {
        return m_state->isFinished(this);
    }

    //------------------------------------------------------------------------
    /// Called for a Ship-PlayGround collision.
    void onCollision(Ship *ship);
    void onCollision(Ship *ship, GrenadeBase *grenade);
    void onCollision(Ship *ship, Missile *missile);
    void onCollision(Ship *ship, MortarBase *mortar);
    void onCollision(Ship *ship, ProjectileBase *projectile);
    void onCollision(Ship *ship, SAMBatteryBase *sam);
    void onCollision(Ship *ship, Tank *tank);
    void onCollision(Ship *ship, TurretBase *turret);

    /// Called for a Missile-PlayGround collision.
    void onCollision(Missile *missile);
    void onCollision(Missile *missile, GrenadeBase *grenade);
    void onCollision(Missile *missile, MortarBase *mortar);
    void onCollision(Missile *missile, ProjectileBase *projectile);
    void onCollision(Missile *missile, SAMBatteryBase *sam);
    void onCollision(Missile *missile, Tank *tank);
    void onCollision(Missile *missile, TurretBase *turret);

    /// Called for a GrenadeBase-PlayGround collision.
    void onCollision(GrenadeBase *grenade);
    void onCollision(GrenadeBase *grenade, MortarBase *mortar);
    void onCollision(GrenadeBase *grenade, Tank *tank);
    void onCollision(GrenadeBase *grenade, TurretBase *turret);

    /// Called for a ProjectileBase-PlayGround collision.
    void onCollision(ProjectileBase *projectile);
    void onCollision(ProjectileBase *projectile, MortarBase *mortar);
    void onCollision(ProjectileBase *projectile, Tank *tank);
    void onCollision(ProjectileBase *projectile, TurretBase *turret);


  protected:

    //------------------------------------------------------------------------
    GameControlBase();


    //------------------------------------------------------------------------
    /// The player's ship.
    Ship *m_player;

    /// The fuel for the player's ship.
    unsigned m_playerFuel;

    /// The fuel decrement value for each Ship::decFuel() call.
    unsigned m_playerFuelDecrement;

    /// The player's home platform.
    const Platform *m_homePlatform;

    /// The ship's initial x position.
    Sint16 m_initialXPosition;
    
    /// The ship's initial y position.
    Sint16 m_initialYPosition;

    /// True, if the initial position is a platform, else false.
    bool m_initialPositionIsPlatform;

    unsigned m_bonusFrameCounter;
    unsigned m_bonusFrameCounterLimit;

    /// The bonus decrement per second.
    unsigned m_bonusDecrement;

  private:

    //------------------------------------------------------------------------
    void do_updateShipControl(ShipControl control);
    void do_updateBonus();

    //------------------------------------------------------------------------
    void do_explode(GrenadeBase *grenade);
    void do_explode(Missile *missile);
    void do_explode(MortarBase *mortar);
    void do_explode(ProjectileBase *projectile);
    void do_explode(SAMBatteryBase *sam);
    void do_explode(Ship *ship);
    void do_explode(Tank *tank);
    void do_explode(TurretBase *turret);

    //------------------------------------------------------------------------
    virtual GameStateBase *getRunningState() const = 0;

    virtual bool continueAfterExplosion() const = 0;


    //------------------------------------------------------------------------
    /// The current active GameControl instance.
    static GameControlBase *sm_instance;

    /// Used for timeout-specific handling (e.g. ExplosionDelay).
    unsigned m_frameCounter;
    unsigned m_frameCounterLimit;

    GameStateBase *m_state;
};

#endif //GAMECONTROLBASE_H
