#include <cmath>

#include "Configuration.h"
#include "Tiles.h"
#include "Barriers.h"
#include "BlackHole.h"
#include "Magnets.h"
#include "ParticleFountains.h"
#include "Platform.h"
#include "SAMBattery.h"
#include "Turrets.h"
#include "Mortars.h"
#include "Ship.h"
#include "Missile.h"
#include "Crates.h"
#include "Projectile.h"
#include "Grenade.h"
#include "Particle.h"
#include "Grinder.h"
#include "Tank.h"
#include "GameControlBase.h"

#include "PlayGroundVisitors.h"
#include "PlayGround.h"


//----------------------------------------------------------------------------
PlayGround *PlayGround::sm_instance = NULL;


//----------------------------------------------------------------------------
void CreateGravityVisitor::do_visit(const XMLNode *n)
{
    const std::string &name = n->getName();
    PlayGround *playGround = PlayGround::getInstance();

    if (name == "gravity")
    {
        n->acceptAllChilds(*this);
    }
    else if (name == "global")
    {
        int gx = n->getIntProperty("gx", 0);
        int gy = n->getIntProperty("gy", 100);

        int vx = n->getIntProperty("vx", 0);
        int vy = n->getIntProperty("vy", 0);

        unsigned friction = n->getUnsignedProperty("friction", 50);

        playGround->setXGravityAll(gx);
        playGround->setYGravityAll(gy);
        playGround->setXVelocityAll(vx);
        playGround->setYVelocityAll(vy);
        playGround->setFrictionAll(friction);
    }
    else if (name == "absolute" || name == "relative")
    {
        Uint16 x = n->getUnsignedProperty("x");
        Uint16 y = n->getUnsignedProperty("y");
        Uint16 w = n->getUnsignedProperty("w");
        Uint16 h = n->getUnsignedProperty("h");

        int gx = n->getIntProperty("gx", 0);
        int gy = n->getIntProperty("gy", 0);

        int vx = n->getIntProperty("vx", 0);
        int vy = n->getIntProperty("vy", 0);

        unsigned friction = n->getUnsignedProperty("friction", 0);

        if (name == "absolute")
        {
            for (Uint16 yy = y; yy < y + h; yy++)
            {
                for (Uint16 xx = x; xx < x + w; xx++)
                {
                    playGround->setXGravity(xx, yy, gx);
                    playGround->setYGravity(xx, yy, gy);
                    playGround->setXVelocity(xx, yy, vx);
                    playGround->setYVelocity(xx, yy, vy);
                    playGround->setFriction(xx, yy, friction);
                }
            }
        }
        else
        {
            for (Uint16 yy = y; yy < y + h; yy++)
            {
                for (Uint16 xx = x; xx < x + w; xx++)
                {
                    playGround->addXGravity(xx, yy, gx);
                    playGround->addYGravity(xx, yy, gy);
                    playGround->addXVelocity(xx, yy, vx);
                    playGround->addYVelocity(xx, yy, vy);
                    playGround->addFriction(xx, yy, friction);
                }
            }
        }
    }
    else
    {
        throw XMLException(
            std::string("The gravity node '").append(name)
            .append("' is unknown"));
    }
}


//----------------------------------------------------------------------------
void CreateDecorationsVisitor::do_visit(const XMLNode *n)
{
    PlayGround *playGround = PlayGround::getInstance();

    if (n->getName() == "decorations")
    {
        n->acceptAllChilds(*this);
    }
    else
    {
        playGround->addObject(DecorationBase::create(n));
    }
}


//----------------------------------------------------------------------------
PlayGround::PlayGround()
{
    m_collisionSurface = NULL;
    m_mapSurface = NULL;
    m_shadowSurface = NULL;

    memset(&m_boundingBox, 0, sizeof(m_boundingBox));
}

//----------------------------------------------------------------------------
PlayGround::~PlayGround()
{
    ZAP_SURFACE(m_collisionSurface);
    ZAP_SURFACE(m_mapSurface);
    ZAP_SURFACE(m_shadowSurface);

    for (ObjectIter iter = m_objects.begin();
         iter != m_objects.end(); ++iter)
    {
        delete iter->object;
    }
    m_objects.clear();

    for (ParticleIter iter = m_particles.begin();
         iter != m_particles.end(); ++iter)
    {
        delete *iter;
    }
    m_particles.clear();
}

//----------------------------------------------------------------------------
void PlayGround::init(const char *mission, const XMLNode *playGroundNode)
    throw (Exception)
{
    destroy();

    sm_instance = new PlayGround();
    sm_instance->readMapFrom(
        mission, playGroundNode->getStringProperty("map").c_str());

    const XMLNode *gravity = playGroundNode->getNode("gravity");
    if (gravity)
    {
        CreateGravityVisitor v1;
        gravity->accept(v1);
    }

    CreateDecorationsVisitor v2;
    playGroundNode->getMandatoryNode("decorations")->accept(v2);
}

//----------------------------------------------------------------------------
void PlayGround::destroy()
{
    ZAP_POINTER(sm_instance);
}


//----------------------------------------------------------------------------
void PlayGround::addObject(ObjectBase *o)
{
    ObjectEntry e = { o, false, true };
    m_objects.push_back(e);
}

//----------------------------------------------------------------------------
void PlayGround::addParticle(ParticleBase *p)
{
    m_particles.push_back(p);
}

//----------------------------------------------------------------------------
void PlayGround::markObjectToRemove(ObjectBase *o, bool autoDelete)
{
    for (ObjectIter iter = m_objects.begin(); iter != m_objects.end(); ++iter)
    {
        if (iter->object == o)
        {
            iter->toRemove = true;
            iter->autoDelete = autoDelete;
            break;
        }
    }
}

//----------------------------------------------------------------------------
const Platform *PlayGround::getPlatform(unsigned number) const
{
    FindPlatformByNumberConstVisitor v(number);
    v.visit(m_objects);
    return v.getPlatform();
}

//----------------------------------------------------------------------------
unsigned PlayGround::countCratesToRescue() const
{
    CountCratesToRescueConstVisitor v;
    v.visit(m_objects);
    return v.getCratesToRescue();
}

//----------------------------------------------------------------------------
Crate *PlayGround::hasCrateFor(const Ship *ship) const
{
    HasCrateForShipConstVisitor v(ship);
    v.visit(m_objects);
    return const_cast<Crate*>(v.getCrate());
}

//----------------------------------------------------------------------------
void PlayGround::resize(Uint16 xTiles, Uint16 yTiles)
    throw (SDLException)
{
    m_boundingBox.w = xTiles * 16;
    m_boundingBox.h = yTiles * 16;

    m_collisionSurface = SDL_CALLS::CreateRGBSurface(
        SDL_SWSURFACE, m_boundingBox.w, m_boundingBox.h, 32);
    m_mapSurface = SDL_CALLS::CreateRGBSurface(
        SDL_SWSURFACE, m_boundingBox.w, m_boundingBox.h, 32);
    m_shadowSurface = SDL_CALLS::CreateRGBSurface(
        SDL_SWSURFACE, m_boundingBox.w, m_boundingBox.h, 32);

    m_background.resize(xTiles, yTiles, true);
    m_xGravity.resize(xTiles, yTiles, 0);
    m_yGravity.resize(xTiles, yTiles, 100);
    m_xVelocity.resize(xTiles, yTiles, 0);
    m_yVelocity.resize(xTiles, yTiles, 0);
    m_friction.resize(xTiles, yTiles, 50);
}


//----------------------------------------------------------------------------
void PlayGround::readMapFrom(const char *mission, const char *map)
    throw (Exception)
{
    std::string full;
    full.append(Configuration::getInstance()->getDataDir())
        .append("/levels/").append(mission).append("/").append(map);

    File f(full.c_str(), "r");

    if (f.readUint8() != '\x01')
    {
        throw Exception("Map file versions other than 0x01 are unsupported");
    }


    Uint16 xTiles = f.readUint16();
    Uint16 yTiles = f.readUint16();
    resize(xTiles, yTiles);

    Uint8 layers = f.readUint8();
    if (layers == 0)
    {
        throw Exception("There must be at least one layer");
    }

    do
    {
        Uint8 category = f.readUint8();

        for (Uint16 y=0; y<yTiles; y++)
        {
            SDL_Rect r;
            for (Uint16 x=0; x<xTiles; x++)
            {
                r.x = 16*x;
                r.y = 16*y;

                Uint8 tile = f.readUint8();

                SDL_CALLS::BlitSurface(
                    TileCategories::getInstance()[category][tile], 0,
                    m_mapSurface, &r);

                if (category != 0 && category != 0xff && tile != 0)
                {
                    SDL_CALLS::BlitSurface(
                        TileCategories::getInstance()[category][tile], 0,
                        m_collisionSurface, &r);

                    m_background.set(x, y, false);
                }
            }
        }
    }
    while (--layers > 0);

    SDL_CALLS::BlitSurface(m_mapSurface, NULL, m_shadowSurface, NULL);
}


//----------------------------------------------------------------------------
template <class T>
static T getTileAverage(const Matrix<T> &matrix, const SDL_Rect &r,
                        const T oobValue)
{
    // If r lies (partially or fully) outside of the given matrix,
    // we take oobValue for all outside points.

    // Since the matrix value within a tile is always constant,
    // we don't need to sweep through pixels.
    // Instead, we sweep through the tiles.

    T sum = 0;
    Sint16 rxrw = r.x + r.w;
    Sint16 ryrh = r.y + r.h;

    Sint16 w = 0, h = 0;

    Sint16 y = r.y;
    while (y < ryrh)
    {
        Sint16 x = r.x;
        while (x < rxrw)
        {
            Sint16 xTile = x/16;
            Sint16 yTile = y/16;

            w = 16 - x % 16;
            if (w > 16)  w -= 16;
            if (x + w > rxrw)
            {
                w = rxrw - x;
            }

            h = 16 - y % 16;
            if (h > 16)  h -= 16;
            if (y + h > ryrh)
            {
                h = ryrh - y;
            }

            if (xTile < 0 || yTile < 0 ||
                (unsigned)xTile >= matrix.getXSize() ||
                (unsigned)yTile >= matrix.getYSize())
            {
                xTile = 0;
                yTile = 0;
            }

            sum += matrix.get(xTile, yTile) * w * h;

            x += w;
        }

        y += h;
    }

    return r.w == 0 || r.h == 0 ? oobValue : sum / (r.w * r.h);
}

//----------------------------------------------------------------------------
int PlayGround::getXGravity(const SDL_Rect &r) const
{
    return getTileAverage(m_xGravity, r, m_xGravity.get(0, 0));
}

//----------------------------------------------------------------------------
int PlayGround::getYGravity(const SDL_Rect &r) const
{
    return getTileAverage(m_yGravity, r, m_yGravity.get(0, 0));
}

//----------------------------------------------------------------------------
int PlayGround::getXVelocity(const SDL_Rect &r) const
{
    return getTileAverage(m_xVelocity, r, m_xVelocity.get(0, 0));
}

//----------------------------------------------------------------------------
int PlayGround::getYVelocity(const SDL_Rect &r) const
{
    return getTileAverage(m_yVelocity, r, m_yVelocity.get(0, 0));
}

//----------------------------------------------------------------------------
unsigned PlayGround::getFriction(const SDL_Rect &r) const
{
    return getTileAverage(m_friction, r, (unsigned)0);
}


//----------------------------------------------------------------------------
bool PlayGround::collidesWith(const ObjectBase *object) const
{
    const SDL_Rect &objPos = object->getPosition();
    Sint16 objPosX = objPos.x;
    Sint16 objPosY = objPos.y;


    // The most common case will be, that the object does not collide with the
    // playfield. So we check, if the object's surface only overlaps background
    // tiles. If yes, we can return false, without checking the pixel data
    // of the object and the playground.

    Sint16 yMax = (objPosY + objPos.h + 15) / 16;
    Sint16 xMax = (objPosX + objPos.w + 15) / 16;

    for (Sint16 y = objPosY/16; y < yMax; y++)
    {
        for (Sint16 x = objPosX/16; x < xMax; x++)
        {
            if (!isBackgroundTile(x, y))
            {
                // @todo Since the object usually lies on some
                // background tiles, it would be faster to perform
                // a pixel check only for the object tiles,
                // that do not lie on a background tile.
                // However, this shouldn't be much of a bottleneck...

                return SDL_TOOLS::isCollision(
                    object->getSurface(), objPos,
                    getCollisionSurface(), getBoundingBox());
            }
        }
    }

    return false;
}

//----------------------------------------------------------------------------
bool PlayGround::collidesWithParticle(const ParticleBase *particle) const
{
    const SDL_Rect &particlePos = particle->getPosition();

    return SDL_TOOLS::getPixel(
        getCollisionSurface(), particlePos.x, particlePos.y) != 0;
}

//----------------------------------------------------------------------------
const Platform *PlayGround::isInLandingZone(const Ship *ship) const
{
    FindLandingPlatformConstVisitor v(ship);
    v.visit(m_objects);
    return v.getPlatform();
}

//----------------------------------------------------------------------------
bool PlayGround::isBackgroundBetween(Uint16 x1, Uint16 y1,
                                     Uint16 x2, Uint16 y2) const
{
    Sint16 diffX = (Sint16)x1 - (Sint16)x2;
    Sint16 diffY = (Sint16)y1 - (Sint16)y2;

    if (diffX == 0 && diffY == 0)
    {
        return isBackgroundTile(x1, y1);
    }

    if (abs(diffX) > abs(diffY))
    {
        // Iterate through the x axis.

        if (x1 < x2)
        {
            return do_isBackgroundBetweenX(x1, y1, x2, y2);
        }
        else
        {
            return do_isBackgroundBetweenX(x2, y2, x1, y1);
        }
    }
    else
    {
        // Iterate through the y axis.

        if (y1 < y2)
        {
            return do_isBackgroundBetweenY(x1, y1, x2, y2);
        }
        else
        {
            return do_isBackgroundBetweenY(x2, y2, x1, y1);
        }
    }
}

//----------------------------------------------------------------------------
bool PlayGround::do_isBackgroundBetweenX(Uint16 x1, Uint16 y1,
                                         Uint16 x2, Uint16 y2) const
{
    double k = (1.0 * (y2-y1)) / (1.0 * (x2-x1));
    double y = 1.0 * y1;

    for (Uint16 x = x1; x <= x2; x++)
    {
        if (!isBackgroundTile(x, (Uint16)round(y)))
        {
            return false;
        }

        y += k;
    }

    return true;
}

//----------------------------------------------------------------------------
bool PlayGround::do_isBackgroundBetweenY(Uint16 x1, Uint16 y1,
                                         Uint16 x2, Uint16 y2) const
{
    double k = (1.0 * (x2-x1)) / (1.0 * (y2-y1));
    double x = 1.0 * x1;

    for (Uint16 y = y1; y <= y2; y++)
    {
        if (!isBackgroundTile((Uint16)round(x), y))
        {
            return false;
        }

        x += k;
    }

    return true;
}

//----------------------------------------------------------------------------
void PlayGround::update()
    throw (SDLException)
{
    OverdrawObjectsConstVisitor v1(m_mapSurface, m_shadowSurface);
    v1.visit(m_objects);
    v1.visit(m_particles);

    UpdateObjectsVisitor v2;
    v2.visit(m_objects);
    v2.visit(m_particles);

    CollisionDetectorVisitor v3(m_objects);
    v3.visit(m_objects);
    v3.visit(m_particles);

    removeObjectsToRemove();

    BlitObjectsConstVisitor v4(m_shadowSurface);
    v4.visit(m_objects);
    v4.visit(m_particles);
}

//----------------------------------------------------------------------------
void PlayGround::updateForPreview() throw (SDLException)
{
    BlitObjectsConstVisitor v(m_shadowSurface);
    v.visit(m_objects);
}

//----------------------------------------------------------------------------
void PlayGround::removeObjectsToRemove()
{
    ObjectIter iter = m_objects.begin();
    while (iter != m_objects.end())
    {
        if (iter->toRemove)
        {
            if (iter->autoDelete)
            {
                delete iter->object;
            }
            iter = m_objects.erase(iter);
        }
        else
        {
            ++iter;
        }
    }

    ParticleIter iter2 = m_particles.begin();
    while (iter2 != m_particles.end())
    {
        if ((*iter2)->isRemove())
        {
            delete *iter2;
            iter2 = m_particles.erase(iter2);
        }
        else
        {
            ++iter2;
        }
    }
}



//----------------------------------------------------------------------------
void OverdrawObjectsConstVisitor::do_visit(const ParticleBase *p)
{
    // This is an optimized version of do_do_visit(p);

    const SDL_Rect &pos = p->getPosition();

    Uint8 *mapPixel = (Uint8*)m_mapSurface->pixels
        + pos.y * m_mapSurface->pitch + pos.x * 4;
    Uint8 *shadowPixel = (Uint8*)m_shadowSurface->pixels
        + pos.y * m_shadowSurface->pitch + pos.x * 4;

    *(Uint32*)shadowPixel = *(Uint32*)mapPixel;
}

//----------------------------------------------------------------------------
void OverdrawObjectsConstVisitor::do_do_visit(const ObjectBase *o)
{
    const SDL_Rect *r = o->getPositionPointer();
    SDL_CALLS::BlitSurface(m_mapSurface, r, m_shadowSurface, r);
}



//----------------------------------------------------------------------------
void UpdateObjectsVisitor::do_visit(GrenadeBase *g)
{
    do_do_visit(g);

    if (g->isExplode())
    {
        // The GameControl's collision handling does what we want,
        // though we actually have no collision here.
        GameControlBase::getInstance()->onCollision(g);
    }
}

//----------------------------------------------------------------------------
void UpdateObjectsVisitor::do_visit(Missile *m)
{
    do_do_visit(m);

    if (m->isExplode())
    {
        // The GameControl's collision handling does what we want,
        // though we actually have no collision here.
        GameControlBase::getInstance()->onCollision(m);
    }
}

//----------------------------------------------------------------------------
void UpdateObjectsVisitor::do_visit(ParticleBase *p)
{
    do_do_visit(p);
}

//----------------------------------------------------------------------------
void UpdateObjectsVisitor::do_visit(Ship *s)
{
    do_do_visit(s);

    if (s->isExplode())
    {
        // The GameControl's collision handling does what we want,
        // though we actually have no collision here.
        GameControlBase::getInstance()->onCollision(s);
    }
}

//----------------------------------------------------------------------------
void UpdateObjectsVisitor::do_do_visit(ObjectBase *o)
{
    SDL_Rect r = o->getPosition();
    o->update();
    SDL_TOOLS::unite(r, o->getPosition(), r);

    PlayGround::getInstance()->addUpdateRect(r);
}



//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ParticleCollisionDetectorVisitor::do_onCollision(ObjectBase *o)
{
    if (ObjectBase::isCollision(m_particle, o))
    {
        m_particle->setRemove();
        setCollision();
    }
}


//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ShipCollisionDetectorVisitor::do_visit(GrenadeBase *g)
{
    if (ObjectBase::isCollision(m_ship, g))
    {
        GameControlBase::getInstance()->onCollision(m_ship, g);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ShipCollisionDetectorVisitor::do_visit(Missile *m)
{
    if (ObjectBase::isCollision(m_ship, m))
    {
        GameControlBase::getInstance()->onCollision(m_ship, m);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ShipCollisionDetectorVisitor::do_visit(MortarBase *m)
{
    if (ObjectBase::isCollision(m_ship, m))
    {
        GameControlBase::getInstance()->onCollision(m_ship, m);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ShipCollisionDetectorVisitor::do_visit(ProjectileBase *p)
{
    if (ObjectBase::isCollision(m_ship, p))
    {
        GameControlBase::getInstance()->onCollision(m_ship, p);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ShipCollisionDetectorVisitor::do_visit(SAMBatteryBase *s)
{
    if (ObjectBase::isCollision(m_ship, s))
    {
        GameControlBase::getInstance()->onCollision(m_ship, s);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ShipCollisionDetectorVisitor::do_visit(Tank *t)
{
    if (ObjectBase::isCollision(m_ship, t))
    {
        GameControlBase::getInstance()->onCollision(m_ship, t);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ShipCollisionDetectorVisitor::do_visit(TurretBase *t)
{
    if (ObjectBase::isCollision(m_ship, t))
    {
        GameControlBase::getInstance()->onCollision(m_ship, t);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ShipCollisionDetectorVisitor::do_onCollision(ObjectBase *object)
{
    if (ObjectBase::isCollision(m_ship, object))
    {
        GameControlBase::getInstance()->onCollision(m_ship);
    }
}


//----------------------------------------------------------------------------
void CollisionDetectorVisitor::MissileCollisionDetectorVisitor::do_visit(GrenadeBase *g)
{
    if (ObjectBase::isCollision(m_missile, g))
    {
        GameControlBase::getInstance()->onCollision(m_missile, g);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::MissileCollisionDetectorVisitor::do_visit(MortarBase *m)
{
    if (ObjectBase::isCollision(m_missile, m))
    {
        GameControlBase::getInstance()->onCollision(m_missile, m);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::MissileCollisionDetectorVisitor::do_visit(ProjectileBase *p)
{
    if (ObjectBase::isCollision(m_missile, p))
    {
        GameControlBase::getInstance()->onCollision(m_missile, p);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::MissileCollisionDetectorVisitor::do_visit(SAMBatteryBase *s)
{
    if (ObjectBase::isCollision(m_missile, s))
    {
        GameControlBase::getInstance()->onCollision(m_missile, s);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::MissileCollisionDetectorVisitor::do_visit(TurretBase *t)
{
    if (ObjectBase::isCollision(m_missile, t))
    {
        GameControlBase::getInstance()->onCollision(m_missile, t);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::MissileCollisionDetectorVisitor::do_visit(Tank *t)
{
    if (ObjectBase::isCollision(m_missile, t))
    {
        GameControlBase::getInstance()->onCollision(m_missile, t);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::MissileCollisionDetectorVisitor::do_onCollision(ObjectBase *o)
{
    if (ObjectBase::isCollision(m_missile, o))
    {
        GameControlBase::getInstance()->onCollision(m_missile);
    }
}


//----------------------------------------------------------------------------
void CollisionDetectorVisitor::GrenadeCollisionDetectorVisitor::do_visit(MortarBase *m)
{
    if (ObjectBase::isCollision(m_grenade, m))
    {
        GameControlBase::getInstance()->onCollision(m_grenade, m);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::GrenadeCollisionDetectorVisitor::do_visit(Tank *t)
{
    if (ObjectBase::isCollision(m_grenade, t))
    {
        GameControlBase::getInstance()->onCollision(m_grenade, t);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::GrenadeCollisionDetectorVisitor::do_visit(TurretBase *t)
{
    if (ObjectBase::isCollision(m_grenade, t))
    {
        GameControlBase::getInstance()->onCollision(m_grenade, t);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::GrenadeCollisionDetectorVisitor::do_onCollision(ObjectBase *o)
{
    if (ObjectBase::isCollision(m_grenade, o))
    {
        GameControlBase::getInstance()->onCollision(m_grenade);
    }
}


//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ProjectileCollisionDetectorVisitor::do_visit(MortarBase *m)
{
    if (ObjectBase::isCollision(m_projectile, m))
    {
        GameControlBase::getInstance()->onCollision(m_projectile, m);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ProjectileCollisionDetectorVisitor::do_visit(Tank *t)
{
    if (ObjectBase::isCollision(m_projectile, t))
    {
        GameControlBase::getInstance()->onCollision(m_projectile, t);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ProjectileCollisionDetectorVisitor::do_visit(TurretBase *t)
{
    if (ObjectBase::isCollision(m_projectile, t))
    {
        GameControlBase::getInstance()->onCollision(m_projectile, t);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::ProjectileCollisionDetectorVisitor::do_onCollision(ObjectBase *o)
{
    if (ObjectBase::isCollision(m_projectile, o))
    {
        GameControlBase::getInstance()->onCollision(m_projectile);
    }
}


//----------------------------------------------------------------------------
void CollisionDetectorVisitor::do_visit(ParticleBase *p)
{
    PlayGround *playGround = PlayGround::getInstance();

    if (playGround->collidesWithParticle(p))
    {
        p->setRemove();
    }
    else
    {
        ParticleCollisionDetectorVisitor v(p);
        v.visit(m_objects);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::do_visit(Ship *s)
{
    PlayGround *playGround = PlayGround::getInstance();

    if (playGround->collidesWith(s))
    {
        GameControlBase::getInstance()->onCollision(s);
    }
    else
    {
        ShipCollisionDetectorVisitor v(s);
        v.visit(m_objects);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::do_visit(Missile *m)
{
    PlayGround *playGround = PlayGround::getInstance();

    if (playGround->collidesWith(m))
    {
        GameControlBase::getInstance()->onCollision(m);
    }
    else
    {
        MissileCollisionDetectorVisitor v(m);
        v.visit(m_objects);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::do_visit(GrenadeBase *g)
{
    PlayGround *playGround = PlayGround::getInstance();

    if (playGround->collidesWith(g))
    {
        GameControlBase::getInstance()->onCollision(g);
    }
    else
    {
        GrenadeCollisionDetectorVisitor v(g);
        v.visit(m_objects);
    }
}

//----------------------------------------------------------------------------
void CollisionDetectorVisitor::do_visit(ProjectileBase *p)
{
    PlayGround *playGround = PlayGround::getInstance();

    if (playGround->collidesWith(p))
    {
        GameControlBase::getInstance()->onCollision(p);
    }
    else
    {
        ProjectileCollisionDetectorVisitor v(p);
        v.visit(m_objects);
    }
}


//----------------------------------------------------------------------------
void BlitObjectsConstVisitor::do_visit(const ParticleBase *p)
{
    // This is an optimized version of do_do_visit(p);

    const SDL_Rect &pos = p->getPosition();
    const SDL_Surface *surface = p->getSurface();

    Uint8 *shadowPixel = (Uint8*)m_shadowSurface->pixels
        + pos.y * m_shadowSurface->pitch + pos.x * 4;

    *(Uint32*)shadowPixel = *(Uint32*)surface->pixels;
}

//----------------------------------------------------------------------------
void BlitObjectsConstVisitor::do_do_visit(const ObjectBase *o)
{
    o->blitTo(m_shadowSurface);
}
