#include <time.h>
#include <iostream>

#include "SoundInterface.h"

#include "HighscoreMenu.h"
#include "PlayGroundMenu.h"
#include "LevelSelectorMenu.h"
#include "MainMenu.h"

#include "PlayerStatus.h"
#include "SinglePlayerGame.h"

#include "PlayerConfiguration.h"

#include <pglabel.h>


//----------------------------------------------------------------------
#define MAX_TIME_LEN 64

MENU_SINGLETON_INIT(Highscore);

//----------------------------------------------------------------------
HighscoreMenu::HighscoreMenu() throw(GuiMenuException)
{
    getWidget(ID_HIGHSCORE_MENU_DONE)->SetEventObject(
        MSG_BUTTONCLICK, this, (MSG_CALLBACK_OBJ)&HighscoreMenu::onDone);

    getWidget(ID_HIGHSCORE_MENU_DONE)->Show();
    getWidget(ID_HIGHSCORE_MENU_TXT)->Show();
    getWidget(ID_HIGHSCORE_MENU_LABEL)->Show();
    getWidget(ID_HIGHSCORE_MENU_MISSION)->Show();

    int startWidget = ID_HIGHSCORE_MENU_N1;
    for (int i=0; i<HIGHSCORE_TABLE_MAX_SIZE; i++)
    {
        getWidget(startWidget++)->Show();
        getWidget(startWidget++)->Show();
        getWidget(startWidget++)->Show();
        startWidget += 2;
    }
}

//----------------------------------------------------------------------
HighscoreMenu::~HighscoreMenu()
{
}

//----------------------------------------------------------------------
PARAGUI_CALLBACK(HighscoreMenu::onDone)
{
  PlayerStatus::destroy();
  SinglePlayerGame::destroy();
  PlayGroundMenu::destroy();

  MenuManager::getInstance()->changeTo(
      LevelSelectorMenu::getInstance(), true, false);

  return true;
}

//----------------------------------------------------------------------------
void HighscoreMenu::eventShow()
{
    getWidget(ID_HIGHSCORE_MENU)->Show();

    const std::string &mission =
        LevelSelectorMenu::getInstance()->getCurrentlySelectedFull();
    getWidget(ID_HIGHSCORE_MENU_MISSION)->SetText(mission.c_str());

    if (m_insertEntryOnShow)
    {
        do_insertEntryAndShow();
    }
    else
    {
        do_onlyShow();
    }
}

//----------------------------------------------------------------------------
void HighscoreMenu::do_insertEntryAndShow()
{
    PlayerConfiguration::Highscore *highscore =
        PlayerConfiguration::getInstance()->getHighscore();

    const std::string &mission =
        LevelSelectorMenu::getInstance()->getCurrentlySelected();

    unsigned pos = highscore->insertEntry(
        mission,
        PlayerConfiguration::getInstance()->getPlayer()->getName(),
        PlayerStatus::getInstance()->getScore());

    switch (pos)
    {
    case 0:
        SoundInterface::getInstance()->playHighScoreMusic1stEntry();
        break;

    case HIGHSCORE_TABLE_MAX_SIZE:
        SoundInterface::getInstance()->playHighScoreMusicNoEntry();
        break;

    default:
        SoundInterface::getInstance()->playHighScoreMusicOtherEntry();
    }

    fillUpHighscoreEntries(mission, pos);
}

//----------------------------------------------------------------------------
void HighscoreMenu::do_onlyShow()
{
    const std::string &mission =
        LevelSelectorMenu::getInstance()->getCurrentlySelected();

    fillUpHighscoreEntries(mission, HIGHSCORE_TABLE_MAX_SIZE);
}

//----------------------------------------------------------------------------
void HighscoreMenu::fillUpHighscoreEntries(const std::string &mission,
                                           unsigned highlightedEntry)
{
    int startWidget = ID_HIGHSCORE_MENU_N1;
    SDL_Color normal = {255,255,255};
    SDL_Color highlight = {255,255,0};
    unsigned entryCounter = 0;

    PlayerConfiguration::Highscore *highscore =
        PlayerConfiguration::getInstance()->getHighscore();

    if (highscore->hasMission(mission))
    {
        PlayerConfiguration::Highscore::const_iterator iter;
        for (iter = highscore->begin(mission);
             iter != highscore->end(mission); ++iter)
        {
            bool me = (highlightedEntry == entryCounter++);

            PG_Label *nameLab = (PG_Label*)getWidget(startWidget++);
            PG_Label *scoreLab = (PG_Label*)getWidget(startWidget++);
            PG_Label *timeLab = (PG_Label*)getWidget(startWidget++);
            startWidget += 2;

            nameLab->SetText(iter->getName().c_str());
            nameLab->SetFontColor(me ? highlight : normal);
            scoreLab->SetTextFormat("%d", iter->getScore());
            scoreLab->SetFontColor(me ? highlight : normal);

            time_t secs = iter->getTimestamp();
            struct tm tim;
            localtime_r(&secs, &tim);
            char timestr[MAX_TIME_LEN];
            strftime(timestr, sizeof(timestr), "%Y-%m-%d %H:%M", &tim);
            timeLab->SetTextFormat("%s", timestr);
            timeLab->SetFontColor(me ? highlight : normal);
        }
    }
}

//----------------------------------------------------------------------
void HighscoreMenu::eventHide()
{
    getWidget(ID_HIGHSCORE_MENU)->Hide();

    // Clear the mission and the score entry labels.
    // When showing a second highscore table, one will not see
    // the content of the first table, until it is overwritten.

    getWidget(ID_HIGHSCORE_MENU_MISSION)->SetText("");

    int startWidget = ID_HIGHSCORE_MENU_N1;
    for (unsigned i = 0; i < HIGHSCORE_TABLE_MAX_SIZE; i++)
    {
        PG_Label *nameLab = (PG_Label*)getWidget(startWidget++);
        PG_Label *scoreLab = (PG_Label*)getWidget(startWidget++);
        PG_Label *timeLab = (PG_Label*)getWidget(startWidget++);
        startWidget += 2;

        nameLab->SetText("");
        scoreLab->SetText("");
        timeLab->SetText("");
    }
}
