#ifndef STATICDECORATIONBASE_H
#define STATICDECORATIONBASE_H

#include <cassert>

#include "XMLElements.h"
#include "DecorationBase.h"

//----------------------------------------------------------------------------
#define COMPARE_SURFACE_KEY_MEMBER(member) \
   if (k1.member < k2.member)  return true; \
   if (k1.member > k2.member)  return false


//----------------------------------------------------------------------------
/**
 * This is a base class for all decorations, that are not moving.
 */
class StaticDecorationBase : public DecorationBase
{
  public:
    //------------------------------------------------------------------------
    /**
     * Common initialization data for all static decorations.
     */
    class InitializationData
    {
      public:
        InitializationData(const XMLNode *node)
            throw (XMLException);
        virtual ~InitializationData();

        /// The x-tile position of the decoration.
        Sint16 x;

        /// The y-tile position of the decoration.
        Sint16 y;
    };

    //------------------------------------------------------------------------
    virtual ~StaticDecorationBase();

  protected:
    //------------------------------------------------------------------------
    StaticDecorationBase(const InitializationData &init);
};


//----------------------------------------------------------------------------
/**
 * Additional base class for decorations,
 * that will turn something on and off.
 *
 * Decorations implementing this functionality
 * must be derivated from StaticDecorationBase and from this class.
 * The same must be done with the inner InitializationData class.
 */
class AnimatedDecorationBase
{
  public:
    //------------------------------------------------------------------------
    class InitializationData
    {
      public:
        InitializationData(const XMLNode *node)
            throw (XMLException);
        virtual ~InitializationData();

        unsigned on;
        unsigned off;
        unsigned delay;
    };

    //------------------------------------------------------------------------
    virtual ~AnimatedDecorationBase();

    //------------------------------------------------------------------------
    inline unsigned getOnInterval() const
    {
        return m_onInterval;
    }

    inline unsigned getOffInterval() const
    {
        return m_offInterval;
    }

    inline unsigned getInitialDelay() const
    {
        return m_initialDelay;
    }

  protected:
    //------------------------------------------------------------------------
    AnimatedDecorationBase(const InitializationData &init);

  private:
    //------------------------------------------------------------------------
    unsigned m_onInterval;
    unsigned m_offInterval;
    unsigned m_initialDelay;
};


//----------------------------------------------------------------------------
/**
 * Additional base class for decorations,
 * that can be orientated on the bottom, left, top, or right.
 *
 * Decorations implementing this functionality
 * must be derivated from StaticDecorationBase and from this class.
 * The same must be done with the inner InitializationData class.
 */
class OrientatingDecorationBase
{
  public:
    //------------------------------------------------------------------------
    enum Orientation
    {
        O_BOTTOM = 0,
        O_LEFT = 1,
        O_TOP = 2,
        O_RIGHT = 3
    };

    //------------------------------------------------------------------------
    class InitializationData
    {
      public:
        InitializationData(const XMLNode *node)
            throw (XMLException);
        virtual ~InitializationData();

        Orientation orientation;
    };

    //------------------------------------------------------------------------
    virtual ~OrientatingDecorationBase();

    //------------------------------------------------------------------------
    inline Orientation getOrientation() const
    {
        return m_orientation;
    }

  protected:
    //------------------------------------------------------------------------
    OrientatingDecorationBase(const InitializationData &init);

  private:
    //------------------------------------------------------------------------
    Orientation m_orientation;
};


//----------------------------------------------------------------------------
/**
 * Additional base class for decorations,
 * that can be orientated horziontal or vertical.
 *
 * Decorations implementing this functionality
 * must be derivated from StaticDecorationBase and from this class.
 * The same must be done with the inner InitializationData class.
 */
class HVDecorationBase
{
  public:
    //------------------------------------------------------------------------
    enum Orientation
    {
        O_HORIZONTAL = 0,
        O_VERTICAL = 1
    };

    //------------------------------------------------------------------------
    class InitializationData
    {
      public:
        InitializationData(const XMLNode *node)
            throw (XMLException);
        virtual ~InitializationData();

        Orientation orientation;
        unsigned wh;
    };

    //------------------------------------------------------------------------
    virtual ~HVDecorationBase();

    //------------------------------------------------------------------------
    inline Orientation getOrientation() const
    {
        return m_orientation;
    }

    inline unsigned getSize() const
    {
        return m_wh;
    }

  protected:
    //------------------------------------------------------------------------
    HVDecorationBase(const InitializationData &init);

  private:
    //------------------------------------------------------------------------
    Orientation m_orientation;
    unsigned m_wh;
};


//----------------------------------------------------------------------------
class SingleSurfacesBase : public SurfacesBase
{
  public:
    //------------------------------------------------------------------------
    virtual ~SingleSurfacesBase();

    //------------------------------------------------------------------------
    inline const SDL_Surface *getSurface() const
    {
        return m_surface;
    }

  protected:
    SingleSurfacesBase();

    void initSurfaceFrom(const char *name)
        throw (SDLException);

  private:
    SDL_Surface *m_surface;
};


//----------------------------------------------------------------------------
class OrientatingSurfacesBase : public SurfacesBase
{
  public:
    //------------------------------------------------------------------------
    virtual ~OrientatingSurfacesBase();

    //------------------------------------------------------------------------
    inline const SDL_Surface *getSurface(
        OrientatingDecorationBase::Orientation o) const
    {
        assert(o < 4);
        return m_surfaces[o];
    }

  protected:
    //------------------------------------------------------------------------
    OrientatingSurfacesBase();

    void initSurfacesFrom(const char *name)
        throw (SDLException);

  private:
    //------------------------------------------------------------------------
    SDL_Surface *m_surfaces[4];
};


//----------------------------------------------------------------------------
/**
 * The rotation angle for one step in degrees.
 */
#define SURFACES_ROTATION_ANGLE 6

/**
 * The number of steps for a full rotation of 360 degrees.
 * (= 360 / SURFACE_ROTATION_ANGLE)
 */
#define SURFACES_ROTATION_STEPS 60

//----------------------------------------------------------------------------
class RotatingSurfacesBase : public SurfacesBase
{
  public:
    //------------------------------------------------------------------------
    virtual ~RotatingSurfacesBase();

    //------------------------------------------------------------------------
    inline const SDL_Surface *getSurface(unsigned frame) const
    {
        assert(frame < SURFACES_ROTATION_STEPS);
        return m_surfaces[frame];
    }

  protected:
    //------------------------------------------------------------------------
    RotatingSurfacesBase();

    void initSurfacesFrom(const char *name)
        throw (SDLException);

  private:
    //------------------------------------------------------------------------
    SDL_Surface *m_surfaces[SURFACES_ROTATION_STEPS];
};

#endif //STATICDECORATIONBASE_H
