#include "Tools.h"
#include "MathTools.h"
#include "SDLFrameRate.h"

#include "GameInterface.h"
#include "SoundInterface.h"

#include "Crates.h"
#include "DecorationBase.h"
#include "StaticDecorationBase.h"
#include   "Barriers.h"
#include   "BlackHole.h"
#include   "Crates.h"
#include   "Magnets.h"
#include   "Mortars.h"
#include   "ParticleFountains.h" 
#include   "Platform.h"
#include   "SAMBattery.h"
#include   "Turrets.h"
#include "MovingDecorationBase.h"
#include   "Grinder.h"
#include   "Tank.h"

#include "ObjectVisitors.h"
#include "Ship.h"
#include "Particle.h"
#include "Projectile.h"
#include "Grenade.h"


//----------------------------------------------------------------------------
GameInterface *GameInterface::sm_instance = NULL;
NullGameInterface NullGameInterface::sm_instance;


//============================================================================
// Implementation of ObjectBase.h
//============================================================================

//----------------------------------------------------------------------------
ObjectBase::ObjectBase(const ObjectBase *creator)
{
    m_surface = NULL;
    memset(&m_position, 0, sizeof(m_position));
    m_creator = creator;
}

//----------------------------------------------------------------------------
ObjectBase::~ObjectBase()
{
    m_surface = NULL;
    m_creator = NULL;
}



//============================================================================
// Implementation of DecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
DecorationBase::DecorationBase() : ObjectBase(NULL)
{
}

//----------------------------------------------------------------------------
DecorationBase::~DecorationBase()
{
}

//----------------------------------------------------------------------------
DecorationBase *DecorationBase::create(const XMLNode *decorationNode)
    throw (Exception)
{
    const std::string &name = decorationNode->getName();

    if (name == "barrier")
    {
        return Barrier::create(decorationNode);
    }
    else if (name == "blackhole")
    {
        return BlackHole::create(decorationNode);
    }
    else if (name == "crate")
    {
        return Crate::create(decorationNode);
    }
    else if (name == "fountain")
    {
        return ParticleFountainBase::create(decorationNode);
    }
    else if (name == "grinder")
    {
        return Grinder::create(decorationNode);
    }
    else if (name == "magnet")
    {
        return MagnetBase::create(decorationNode);
    }
    else if (name == "mortar")
    {
        return MortarBase::create(decorationNode);
    }
    else if (name == "platform")
    {
        return Platform::create(decorationNode);
    }
    else if (name == "sam")
    {
        return SAMBatteryBase::create(decorationNode);
    }
    else if (name == "tank")
    {
        return Tank::create(decorationNode);
    }
    else if (name == "turret")
    {
        return TurretBase::create(decorationNode);
    }

    throw XMLException(
        std::string("The decoration node '").append(name)
        .append("' is unknown"));
}

//----------------------------------------------------------------------------
bool DecorationBase::canReachPlayerShip() const
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return false;
    }

    const SDL_Rect &position = getPosition();
    Uint16 tileX = position.x / 16;
    Uint16 tileY = position.y / 16;

    Sint16 shipX, shipY;
    SDL_TOOLS::getCentre(player->getPosition(), shipX, shipY);

    return GameInterface::getInstance()->isBackgroundBetween(
        shipX/16, shipY/16, tileX, tileY);
}



//============================================================================
// Implementation of StaticDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
StaticDecorationBase::InitializationData::InitializationData(const XMLNode *node)
    throw (XMLException)
{
    x = node->getIntProperty("x");
    y = node->getIntProperty("y");
}

//----------------------------------------------------------------------------
StaticDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
StaticDecorationBase::StaticDecorationBase(const InitializationData &init)
{
    setTilePosition(init.x, init.y);
}

//----------------------------------------------------------------------------
StaticDecorationBase::~StaticDecorationBase()
{
}



//============================================================================
// Implementation of AnimatedDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
AnimatedDecorationBase::InitializationData::InitializationData(const XMLNode *node)
    throw (XMLException)
{
    on = node->getUnsignedProperty("on", 0);
    off = node->getUnsignedProperty("off", 0);
    delay = node->getUnsignedProperty("delay", 0);
}

//----------------------------------------------------------------------------
AnimatedDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
AnimatedDecorationBase::AnimatedDecorationBase(
    const InitializationData &init)
{
    m_onInterval = init.on * SDLFrameRate::getFrameRate();
    m_offInterval = init.off * SDLFrameRate::getFrameRate();
    m_initialDelay = init.delay * SDLFrameRate::getFrameRate();
}

//----------------------------------------------------------------------------
AnimatedDecorationBase::~AnimatedDecorationBase()
{
}



//============================================================================
// Implementation of OrientatingDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
OrientatingDecorationBase::InitializationData::InitializationData(const XMLNode *node)
    throw (XMLException)
{
    const std::string &prop = node->getStringProperty("orientation");
    if (prop == "top")
    {
        orientation = O_TOP;
    }
    else if (prop == "bottom")
    {
        orientation = O_BOTTOM;
    }
    else if (prop == "left")
    {
        orientation = O_LEFT;
    }
    else if (prop == "right")
    {
        orientation = O_RIGHT;
    }
    else
    {
        throw XMLException(
            "The value of the 'orientation' attribute must be "
            "'top', 'bottom', 'left', or 'right'");
    }
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
OrientatingDecorationBase::OrientatingDecorationBase(
    const InitializationData &init)
{
    m_orientation = init.orientation;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::~OrientatingDecorationBase()
{
}



//============================================================================
// Implementation of HVDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
HVDecorationBase::InitializationData::InitializationData(const XMLNode *node)
    throw (XMLException)
{
    if (node->hasProperty("w"))
    {
        orientation = O_HORIZONTAL;
        wh = node->getUnsignedProperty("w");
    }
    else if (node->hasProperty("h"))
    {
        orientation = O_VERTICAL;
        wh = node->getUnsignedProperty("h");
    }
    else
    {
        throw XMLException("The attribute 'w' or 'h' is mandatory");
    }
}

//----------------------------------------------------------------------------
HVDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
HVDecorationBase::HVDecorationBase(const InitializationData &init)
{
    m_orientation = init.orientation;
    m_wh = init.wh;
}

//----------------------------------------------------------------------------
HVDecorationBase::~HVDecorationBase()
{
}



//============================================================================
// Implementation of MovingDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
MovingDecorationBase::InitializationData::InitializationData(const XMLNode *node)
    throw (XMLException)
{
}

//----------------------------------------------------------------------------
MovingDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
MovingDecorationBase::MovingDecorationBase(const InitializationData &init)
        : DecorationBase()
{
}

//----------------------------------------------------------------------------
MovingDecorationBase::~MovingDecorationBase()
{
}



//============================================================================
// Implementation of Barriers.h
//============================================================================

//----------------------------------------------------------------------------
Barrier::InitializationData::InitializationData(const XMLNode *barrierNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(barrierNode),
          AnimatedDecorationBase::InitializationData(barrierNode),
          HVDecorationBase::InitializationData(barrierNode)
{
}

//----------------------------------------------------------------------------
Barrier::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Barrier::Initial Barrier::Initial::sm_instance;
Barrier::InitialDelay Barrier::InitialDelay::sm_instance;
Barrier::PermanentActivation Barrier::PermanentActivation::sm_instance;
Barrier::Activated Barrier::Activated::sm_instance;
Barrier::Deactivated Barrier::Deactivated::sm_instance;

//----------------------------------------------------------------------------
void Barrier::Initial::update(Barrier *barrier)
{
    UpdateState *state = NULL;

    barrier->setSurface(
        BarrierSurfaces::getInstance()->getSurface(
            barrier->getOrientation(),
            barrier->getInitialDelay() == 0,
            barrier->getSize()));

    if (barrier->getInitialDelay() > 0)
    {
        state = InitialDelay::getInstance();
    }
    else if (barrier->getOnInterval() == 0 || barrier->getOffInterval() == 0)
    {
        state = PermanentActivation::getInstance();
    }
    else
    {
        state = Activated::getInstance();
    }

    barrier->setUpdateState(state);
    state->update(barrier);
}

//----------------------------------------------------------------------------
void Barrier::InitialDelay::update(Barrier *barrier)
{
    if (++barrier->m_frameCounter == barrier->getInitialDelay())
    {
        barrier->m_frameCounter = 0;

        barrier->setSurface(
            BarrierSurfaces::getInstance()->getSurface(
                barrier->getOrientation(), true, barrier->getSize()));

        UpdateState *state = NULL;
        if (barrier->getOnInterval() == 0 || barrier->getOffInterval() == 0)
        {
            state = PermanentActivation::getInstance();
        }
        else
        {
            state = Activated::getInstance();
        }
        barrier->setUpdateState(state);
    }
}

//----------------------------------------------------------------------------
void Barrier::PermanentActivation::update(Barrier *barrier)
{
}

//----------------------------------------------------------------------------
void Barrier::Activated::update(Barrier *barrier)
{
    if (++barrier->m_frameCounter == barrier->getOnInterval())
    {
        barrier->m_frameCounter = 0;

        barrier->setSurface(
            BarrierSurfaces::getInstance()->getSurface(
                barrier->getOrientation(), false, barrier->getSize()));

        barrier->setUpdateState(Deactivated::getInstance());
    }
}

//----------------------------------------------------------------------------
void Barrier::Deactivated::update(Barrier *barrier)
{
    if (++barrier->m_frameCounter == barrier->getOffInterval())
    {
        barrier->m_frameCounter = 0;

        barrier->setSurface(
            BarrierSurfaces::getInstance()->getSurface(
                barrier->getOrientation(), true, barrier->getSize()));

        barrier->setUpdateState(Activated::getInstance());
    }
}


//----------------------------------------------------------------------------
Barrier::Barrier(const InitializationData &init)
        : StaticDecorationBase(init),
          AnimatedDecorationBase(init),
          HVDecorationBase(init)
{
    setSurface(BarrierSurfaces::getInstance()->getSurface(
                   getOrientation(), false, getSize()));

    m_frameCounter = 0;

    setUpdateState(Initial::getInstance());
}

//----------------------------------------------------------------------------
Barrier::~Barrier()
{
    setUpdateState(NULL);
}

//----------------------------------------------------------------------------
Barrier *Barrier::create(const XMLNode *barrierNode)
    throw (Exception)
{
    return new Barrier(InitializationData(barrierNode));
}

//----------------------------------------------------------------------------
void Barrier::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Barrier);



//============================================================================
// Implementation of BlackHole.h
//============================================================================

//----------------------------------------------------------------------------
BlackHole::InitializationData::InitializationData(const XMLNode *blackHoleNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(blackHoleNode)
{
    gravity = blackHoleNode->getIntProperty("gravity", 2000);
}

//----------------------------------------------------------------------------
BlackHole::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
BlackHole::BlackHole(const InitializationData &init)
        : StaticDecorationBase(init)
{
    setSurface(BlackHoleSurfaces::getInstance()->getSurface());

    m_gravity = init.gravity;

    addGravityToPlayGround();
}

//----------------------------------------------------------------------------
BlackHole::~BlackHole()
{
}

//----------------------------------------------------------------------------
BlackHole *BlackHole::create(const XMLNode *blackHoleNode)
    throw (Exception)
{
    return new BlackHole(InitializationData(blackHoleNode));
}

//----------------------------------------------------------------------------
void BlackHole::update()
{
}

//----------------------------------------------------------------------------
void BlackHole::addGravityToPlayGround() const
{
    Uint16 w = GameInterface::getInstance()->getPlayGroundXTiles();
    Uint16 h = GameInterface::getInstance()->getPlayGroundYTiles();

    Sint16 tileX = getPosition().x / 16;
    Sint16 tileY = getPosition().y / 16;

    Sint16 gx, gy;

    for (Uint16 y = 0; y < h; y++)
    {
        for (Uint16 x = 0; x < w; x++)
        {
            int rx = x - tileX;
            int ry = y - tileY;

            int r2 = rx*rx + ry*ry;
            int a = MATH_TOOLS::getAngle(rx, ry);
            if (r2 != 0)
            {
                gx = (Sint16)rint(getGravity() * SinusTable::sin(a) / r2);
                gy = (Sint16)rint(getGravity() * SinusTable::cos(a) / r2);
            }
            else
            {
                gx = 0;
                gy = 0;
            }

            GameInterface::getInstance()->addXGravityToPlayGround(x, y, -gx);
            GameInterface::getInstance()->addYGravityToPlayGround(x, y, -gy);
        }
    }
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(BlackHole);



//============================================================================
// Implementation of Crate.h
//============================================================================

//----------------------------------------------------------------------------
Crate::InitializationData::InitializationData(const XMLNode *crateNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(crateNode)
{
    const std::string &prop = crateNode->getStringProperty("type");
    if (prop == "small")  type = CrateSurfaces::T_SMALL;
    else if (prop == "medium")  type = CrateSurfaces::T_MEDIUM;
    else if (prop == "big")  type = CrateSurfaces::T_BIG;
    else if (prop == "bonus")  type = CrateSurfaces::T_BONUS;
    else if (prop == "fuel")  type = CrateSurfaces::T_FUEL;
    else throw XMLException(
        "The value of the 'type' attribute must be "
        "'small', 'medium', 'big', 'bonus', or 'fuel'");

    score = crateNode->getUnsignedProperty("score", 0);
}

//----------------------------------------------------------------------------
Crate::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Crate::Crate(const InitializationData &init)
        : StaticDecorationBase(init)
{
    m_type = init.type;
    m_score = init.score;

    setSurface(CrateSurfaces::getInstance()->getSurface(m_type));

    // init contains the tile coordinates, where the crate shall lie.
    // Calculate the exact pixel position.

    Uint16 x = init.x * 16;
    Uint16 y = init.y * 16;
    x += (16 - getPosition().w) / 2;
    y += 16 - getPosition().h - 4;  // 4 is the height of the platform.

    setPosition(x, y);
}

//----------------------------------------------------------------------------
Crate::~Crate()
{
}

//----------------------------------------------------------------------------
Crate *Crate::create(const XMLNode *crateNode)
    throw (Exception)
{
    return new Crate(InitializationData(crateNode));
}

//----------------------------------------------------------------------------
void Crate::update()
{
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Crate);



//============================================================================
// Implementation of Magnets.h
//============================================================================

//----------------------------------------------------------------------------
MagnetBase::InitializationData::InitializationData(const XMLNode *magnetNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(magnetNode),
          AnimatedDecorationBase::InitializationData(magnetNode),
          OrientatingDecorationBase::InitializationData(magnetNode)
{
    wh = 0;
    switch (orientation)
    {
    case O_TOP:
    case O_BOTTOM:
        wh = magnetNode->getUnsignedProperty("w");
        break;

    case O_LEFT:
    case O_RIGHT:
        wh = magnetNode->getUnsignedProperty("h");
        break;
    }

    strength = magnetNode->getUnsignedProperty("strength", 100);
    distance = magnetNode->getUnsignedProperty("distance", 5);
}

//----------------------------------------------------------------------------
MagnetBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
MagnetBase::Initial MagnetBase::Initial::sm_instance;
MagnetBase::InitialDelay MagnetBase::InitialDelay::sm_instance;
MagnetBase::PermanentActivation MagnetBase::PermanentActivation::sm_instance;
MagnetBase::Activated MagnetBase::Activated::sm_instance;
MagnetBase::Deactivated MagnetBase::Deactivated::sm_instance;

//----------------------------------------------------------------------------
void MagnetBase::Initial::update(MagnetBase *magnet)
{
    UpdateState *state = NULL;
    if (magnet->getInitialDelay() > 0)
    {
        state = InitialDelay::getInstance();
    }
    else if (magnet->getOnInterval() == 0 || magnet->getOffInterval() == 0)
    {
        magnet->addGravityToPlayGround();
        state = PermanentActivation::getInstance();
    }
    else
    {
        magnet->addGravityToPlayGround();
        state = Activated::getInstance();
    }

    magnet->setUpdateState(state);
    state->update(magnet);
}

//----------------------------------------------------------------------------
void MagnetBase::InitialDelay::update(MagnetBase *magnet)
{
    if (++magnet->m_frameCounter == magnet->getInitialDelay())
    {
        magnet->m_frameCounter = 0;
        magnet->addGravityToPlayGround();
        magnet->setUpdateState(Activated::getInstance());
    }
}

//----------------------------------------------------------------------------
void MagnetBase::PermanentActivation::update(MagnetBase *magnet)
{
    if (++magnet->m_frameDelay == SDLFrameRate::getFrameRate() / 12)
    {
        magnet->m_frameDelay = 0;
        magnet->m_currentFrame = (magnet->m_currentFrame+1) % 4;

        magnet->setSurface(
            MagnetSurfaces::getInstance()->getSurface(
                magnet->getOrientation(),
                magnet->m_currentFrame,
                magnet->getSize()));
    }
}

//----------------------------------------------------------------------------
void MagnetBase::Activated::update(MagnetBase *magnet)
{
    PermanentActivation::getInstance()->update(magnet);

    if (++magnet->m_frameCounter == magnet->getOnInterval())
    {
        magnet->m_frameCounter = 0;
        magnet->subGravityFromPlayGround();
        magnet->setUpdateState(Deactivated::getInstance());
    }
}

//----------------------------------------------------------------------------
void MagnetBase::Deactivated::update(MagnetBase *magnet)
{
    if (++magnet->m_frameCounter == magnet->getOffInterval())
    {
        magnet->m_frameCounter = 0;
        magnet->addGravityToPlayGround();
        magnet->setUpdateState(Activated::getInstance());
    }
}


//----------------------------------------------------------------------------
MagnetBase::MagnetBase(const InitializationData &init)
        : StaticDecorationBase(init),
          AnimatedDecorationBase(init),
          OrientatingDecorationBase(init)
{
    m_size = init.wh * 16;
    m_strength = init.strength;
    m_distance = init.distance;

    setSurface(MagnetSurfaces::getInstance()->getSurface(
                   getOrientation(), 0, getSize()));

    m_frameDelay = 0;
    m_currentFrame = 0;
    m_frameCounter = 0;

    setUpdateState(Initial::getInstance());
}

//----------------------------------------------------------------------------
MagnetBase::~MagnetBase()
{
    setUpdateState(NULL);
}

//----------------------------------------------------------------------------
MagnetBase *MagnetBase::create(const XMLNode *magnetNode)
    throw (Exception)
{
    return create(InitializationData(magnetNode));
}

//----------------------------------------------------------------------------
MagnetBase *MagnetBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_BOTTOM:
        return new BottomMagnet(init);
    case O_TOP:
        return new TopMagnet(init);
    case O_LEFT:
        return new LeftMagnet(init);
    case O_RIGHT:
        return new RightMagnet(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void MagnetBase::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(MagnetBase);


//----------------------------------------------------------------------------
TopMagnet::TopMagnet(const InitializationData &init)
        : MagnetBase(init)
{
}

//----------------------------------------------------------------------------
TopMagnet::~TopMagnet()
{
}

//----------------------------------------------------------------------------
void TopMagnet::addGravityToPlayGround() const
{
    for (Uint16 y=0; y<getDistance(); y++)
    {
        for (Uint16 x=0; x<getPosition().w/16; x++)
        {
            GameInterface::getInstance()->addYGravityToPlayGround(
                getPosition().x/16+x, getPosition().y/16+1+y, -getStrength());
        }
    }
}

//----------------------------------------------------------------------------
void TopMagnet::subGravityFromPlayGround() const
{
    for (Uint16 y=0; y<getDistance(); y++)
    {
        for (Uint16 x=0; x<getPosition().w/16; x++)
        {
            GameInterface::getInstance()->addYGravityToPlayGround(
                getPosition().x/16+x, getPosition().y/16+1+y, getStrength());
        }
    }
}


//----------------------------------------------------------------------------
BottomMagnet::BottomMagnet(const InitializationData &init)
        : MagnetBase(init)
{
}

//----------------------------------------------------------------------------
BottomMagnet::~BottomMagnet()
{
}

//----------------------------------------------------------------------------
void BottomMagnet::addGravityToPlayGround() const
{
    for (Uint16 y=0; y<getDistance(); y++)
    {
        for (Uint16 x=0; x<getPosition().w/16; x++)
        {
            GameInterface::getInstance()->addYGravityToPlayGround(
                getPosition().x/16+x, getPosition().y/16-1-y, getStrength());
        }
    }
}

//----------------------------------------------------------------------------
void BottomMagnet::subGravityFromPlayGround() const
{
    for (Uint16 y=0; y<getDistance(); y++)
    {
        for (Uint16 x=0; x<getPosition().w/16; x++)
        {
            GameInterface::getInstance()->addYGravityToPlayGround(
                getPosition().x/16+x, getPosition().y/16-1-y, -getStrength());
        }
    }
}


//----------------------------------------------------------------------------
LeftMagnet::LeftMagnet(const InitializationData &init)
        : MagnetBase(init)
{
}

//----------------------------------------------------------------------------
LeftMagnet::~LeftMagnet()
{
}

//----------------------------------------------------------------------------
void LeftMagnet::addGravityToPlayGround() const
{
    for (Uint16 y=0; y<getPosition().h/16; y++)
    {
        for (Uint16 x=0; x<getDistance(); x++)
        {
            GameInterface::getInstance()->addXGravityToPlayGround(
                getPosition().x/16+1+x, getPosition().y/16+y, -getStrength());
        }
    }
}

//----------------------------------------------------------------------------
void LeftMagnet::subGravityFromPlayGround() const
{
    for (Uint16 y=0; y<getPosition().h/16; y++)
    {
        for (Uint16 x=0; x<getDistance(); x++)
        {
            GameInterface::getInstance()->addXGravityToPlayGround(
                getPosition().x/16+1+x, getPosition().y/16+y, getStrength());
        }
    }
}


//----------------------------------------------------------------------------
RightMagnet::RightMagnet(const InitializationData &init)
        : MagnetBase(init)
{
}

//----------------------------------------------------------------------------
RightMagnet::~RightMagnet()
{
}

//----------------------------------------------------------------------------
void RightMagnet::addGravityToPlayGround() const
{
    for (Uint16 y=0; y<getPosition().h/16; y++)
    {
        for (Uint16 x=0; x<getDistance(); x++)
        {
            GameInterface::getInstance()->addXGravityToPlayGround(
                getPosition().x/16-1-x, getPosition().y/16+y, getStrength());
        }
    }
}

//----------------------------------------------------------------------------
void RightMagnet::subGravityFromPlayGround() const
{
    for (Uint16 y=0; y<getPosition().h/16; y++)
    {
        for (Uint16 x=0; x<getDistance(); x++)
        {
            GameInterface::getInstance()->addXGravityToPlayGround(
                getPosition().x/16-1-x, getPosition().y/16+y, -getStrength());
        }
    }
}



//============================================================================
// Implementation of ParticleFountains.h
//============================================================================

//----------------------------------------------------------------------------
ParticleFountainBase::InitializationData::InitializationData(const XMLNode *fountainNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(fountainNode),
          AnimatedDecorationBase::InitializationData(fountainNode),
          OrientatingDecorationBase::InitializationData(fountainNode)
{
    speed = fountainNode->getUnsignedProperty("speed", 40);
    lifeTime = fountainNode->getUnsignedProperty("lifetime", 1);
    scatter = fountainNode->getUnsignedProperty("scatter", 20);
}

//----------------------------------------------------------------------------
ParticleFountainBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
ParticleFountainBase::Initial ParticleFountainBase::Initial::sm_instance;
ParticleFountainBase::InitialDelay ParticleFountainBase::InitialDelay::sm_instance;
ParticleFountainBase::PermanentActivation ParticleFountainBase::PermanentActivation::sm_instance;
ParticleFountainBase::Activated ParticleFountainBase::Activated::sm_instance;
ParticleFountainBase::Deactivated ParticleFountainBase::Deactivated::sm_instance;

//----------------------------------------------------------------------------
void ParticleFountainBase::Initial::update(ParticleFountainBase *f)
{
    UpdateState *state = NULL;
    if (f->getInitialDelay() > 0)
    {
        state = InitialDelay::getInstance();
    }
    else if (f->getOnInterval() == 0 || f->getOffInterval() == 0)
    {
        state = PermanentActivation::getInstance();
    }
    else
    {
        state = Activated::getInstance();
    }

    f->setUpdateState(state);
    state->update(f);
}

//----------------------------------------------------------------------------
void ParticleFountainBase::InitialDelay::update(ParticleFountainBase *f)
{
    if (++f->m_frameCounter == f->getInitialDelay())
    {
        f->m_frameCounter = 0;
        f->setUpdateState(Activated::getInstance());
    }
}

//----------------------------------------------------------------------------
void ParticleFountainBase::PermanentActivation::update(ParticleFountainBase *f)
{
    // @todo: Use a configurable value from the PlayerConfiguration.
    if (++f->m_createParticleCounter == 1)
    {
        f->m_createParticleCounter = 0;
        GameInterface::getInstance()->addParticleToPlayGround(
            new FountainParticle(f));
    }
}

//----------------------------------------------------------------------------
void ParticleFountainBase::Activated::update(ParticleFountainBase *f)
{
    PermanentActivation::getInstance()->update(f);

    if (++f->m_frameCounter == f->getOnInterval())
    {
        f->m_frameCounter = 0;
        f->setUpdateState(Deactivated::getInstance());
    }
}

//----------------------------------------------------------------------------
void ParticleFountainBase::Deactivated::update(ParticleFountainBase *f)
{
    if (++f->m_frameCounter == f->getOffInterval())
    {
        f->m_frameCounter = 0;
        f->setUpdateState(Activated::getInstance());
    }
}


//----------------------------------------------------------------------------
ParticleFountainBase::ParticleFountainBase(const InitializationData &init)
        : StaticDecorationBase(init),
          AnimatedDecorationBase(init),
          OrientatingDecorationBase(init)
{
    setSurface(
        ParticleFountainSurfaces::getInstance()->getSurface(getOrientation()));

    m_particleSpeed = init.speed;
    m_particleLifeTime = init.lifeTime;
    m_particleScatter = init.scatter;

    m_frameCounter = 0;
    m_createParticleCounter = 0;

    setUpdateState(Initial::getInstance());
}

//----------------------------------------------------------------------------
ParticleFountainBase::~ParticleFountainBase()
{
    setUpdateState(NULL);
}

//----------------------------------------------------------------------------
ParticleFountainBase *ParticleFountainBase::create(const XMLNode *fountainNode)
    throw (Exception)
{
    return create(InitializationData(fountainNode));
}

//----------------------------------------------------------------------------
ParticleFountainBase *ParticleFountainBase::create(
    const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_BOTTOM:
        return new BottomParticleFountain(init);
    case O_TOP:
        return new TopParticleFountain(init);
    case O_LEFT:
        return new LeftParticleFountain(init);
    case O_RIGHT:
        return new RightParticleFountain(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void ParticleFountainBase::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(ParticleFountainBase);


//----------------------------------------------------------------------------
TopParticleFountain::TopParticleFountain(const InitializationData &init)
        : ParticleFountainBase(init)
{
    getParticleInitialVelocity().set(0.0, 1.0 * getParticleSpeed());
}

//----------------------------------------------------------------------------
TopParticleFountain::~TopParticleFountain()
{
}


//----------------------------------------------------------------------------
BottomParticleFountain::BottomParticleFountain(const InitializationData &init)
        : ParticleFountainBase(init)
{
    getParticleInitialVelocity().set(0.0, -1.0 * getParticleSpeed());
}

//----------------------------------------------------------------------------
BottomParticleFountain::~BottomParticleFountain()
{
}


//----------------------------------------------------------------------------
LeftParticleFountain::LeftParticleFountain(const InitializationData &init)
        : ParticleFountainBase(init)
{
    getParticleInitialVelocity().set(1.0 * getParticleSpeed(), 0.0);
}

//----------------------------------------------------------------------------
LeftParticleFountain::~LeftParticleFountain()
{
}


//----------------------------------------------------------------------------
RightParticleFountain::RightParticleFountain(const InitializationData &init)
        : ParticleFountainBase(init)
{
    getParticleInitialVelocity().set(-1.0 * getParticleSpeed(), 0.0);
}

//----------------------------------------------------------------------------
RightParticleFountain::~RightParticleFountain()
{
}



//============================================================================
// Implementation of Platform.h
//============================================================================

//----------------------------------------------------------------------------
Platform::InitializationData::InitializationData(const XMLNode *platformNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(platformNode)
{
    w = platformNode->getUnsignedProperty("w");
    number = platformNode->getUnsignedProperty("number", 0);

    leftBeacon = getBeaconColor(platformNode, "left");
    rightBeacon = getBeaconColor(platformNode, "right");
}

//----------------------------------------------------------------------------
Platform::InitializationData::~InitializationData()
{
}

//----------------------------------------------------------------------------
PlatformSurfaces::BeaconColor Platform::InitializationData::getBeaconColor(
    const XMLNode *node,
    const char *prop) const
    throw (XMLException)
{
    if (!node->hasProperty(prop))
    {
        return PlatformSurfaces::BC_NO_BEACON;
    }

    const std::string &color = node->getStringProperty(prop);

    if (color == "red")  return PlatformSurfaces::BC_RED;
    if (color == "yellow")  return PlatformSurfaces::BC_YELLOW;
    if (color == "green")  return PlatformSurfaces::BC_GREEN;
    if (color == "cyan")  return PlatformSurfaces::BC_GREEN;
    if (color == "blue")  return PlatformSurfaces::BC_BLUE;
    if (color == "magenta")  return PlatformSurfaces::BC_MAGENTA;

    throw XMLException(
        std::string("The value of the '").append(prop)
        .append("' attribute must be ")
        .append("'red', 'yellow', 'green', 'cyan', 'blue' or 'magenta'"));
}


//----------------------------------------------------------------------------
Platform::Platform(const InitializationData &init)
        : StaticDecorationBase(init)
{
    m_width = init.w * 16;
    m_number = init.number;

    m_leftBeaconColor = init.leftBeacon;
    m_rightBeaconColor = init.rightBeacon;

    setSurface(PlatformSurfaces::getInstance()->getSurface(
                   m_leftBeaconColor, m_rightBeaconColor,
                   m_number, 0, getWidth()));

    initLandingZone();

    m_frameCounter = 0;
    m_currentFrame = 0;
}

//----------------------------------------------------------------------------
Platform::~Platform()
{
}


//----------------------------------------------------------------------------
Platform *Platform::create(const XMLNode *platformNode)
    throw (Exception)
{
    return new Platform(InitializationData(platformNode));
}

//----------------------------------------------------------------------------
void Platform::initLandingZone()
{
    // The ship cannot land on the border tiles.
    m_landingZone.x = getPosition().x + 16;
    m_landingZone.w = getPosition().w - 32;

    Sint16 y;
    for (y = getPosition().h - 1; y >= 0; y--)
    {
        if (!SDL_TOOLS::getPixel(getSurface(), 16, y))
        {
            break;
        }
    }

    m_landingZone.y = getPosition().y + y;
    m_landingZone.h = 1;
}

//----------------------------------------------------------------------------
void Platform::update()
{
    // If there's no beacon on the left and on the right,
    // we don't need to update.
    if (m_leftBeaconColor != PlatformSurfaces::BC_NO_BEACON ||
        m_rightBeaconColor != PlatformSurfaces::BC_NO_BEACON)
    {
        if (++m_frameCounter == SDLFrameRate::getFrameRate())
        {
            m_frameCounter = 0;
            m_currentFrame = (m_currentFrame+1) % 2;

            setSurface(PlatformSurfaces::getInstance()->getSurface(
                           m_leftBeaconColor, m_rightBeaconColor,
                           m_number, m_currentFrame, getWidth()));
        }
    }
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Platform);



//============================================================================
// Implementation of SAMBatteries.h
//============================================================================

//----------------------------------------------------------------------------
SAMBatteryBase::InitializationData::InitializationData(const XMLNode *samNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(samNode),
          OrientatingDecorationBase::InitializationData(samNode)
{
    delay = samNode->getUnsignedProperty("delay", 5);

    missileFuel = samNode->getUnsignedProperty("fuel", 5);

    const std::string &heading = samNode->getStringProperty("heading");
    if (heading == "direct")
        missileHeadingStrategy = Missile::HS_DIRECT;
    else if (heading == "smart")
        missileHeadingStrategy = Missile::HS_SMART;
    else
        throw XMLException(
            "The value of the 'heading' attribute must be "
            "'direct' or 'smart'");

    const std::string &warhead = samNode->getStringProperty("warhead");
    if (warhead == "normal")
        missileWarheadStrategy = Missile::WH_NORMAL;
    else if (warhead == "coneburst")
        missileWarheadStrategy = Missile::WH_CONEBURST;
    else if (warhead == "starburst")
        missileWarheadStrategy = Missile::WH_STARBURST;
    else
        throw XMLException(
            "The value of the 'warhead' attribute must be "
            "'normal', 'coneburst' or 'starburst'");
}

//----------------------------------------------------------------------------
SAMBatteryBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
SAMBatteryBase::SAMBatteryBase(const InitializationData &init)
        : StaticDecorationBase(init),
          OrientatingDecorationBase(init)
{
    setSurface(
        SAMBatterySurfaces::getInstance()->getSurface(getOrientation()));

    m_frameDelay = init.delay * SDLFrameRate::getFrameRate();
    m_frameCounter = 0;

    m_missileFuel = init.missileFuel * SDLFrameRate::getFrameRate();
    m_missileHeadingStrategy = init.missileHeadingStrategy;
    m_missileWarheadStrategy = init.missileWarheadStrategy;
}

//----------------------------------------------------------------------------
SAMBatteryBase::~SAMBatteryBase()
{
}

//----------------------------------------------------------------------------
SAMBatteryBase *SAMBatteryBase::create(const XMLNode *samNode)
    throw (Exception)
{
    return create(InitializationData(samNode));
}

//----------------------------------------------------------------------------
SAMBatteryBase *SAMBatteryBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_TOP:
        return new TopSAMBattery(init);
    case O_BOTTOM:
        return new BottomSAMBattery(init);
    case O_LEFT:
        return new LeftSAMBattery(init);
    case O_RIGHT:
        return new RightSAMBattery(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void SAMBatteryBase::createExplosionParticles()
{
    unsigned numParticles = getPosition().w * getPosition().h / 4;
    for (unsigned i=0; i<numParticles; i++)
    {
        GameInterface::getInstance()->addParticleToPlayGround(
            new ExplosionParticle(this));
    }
}

//----------------------------------------------------------------------------
void SAMBatteryBase::update()
{
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (canReachPlayerShip())
        {
            SoundInterface::getInstance()->onSAMBatteryShoot();
            GameInterface::getInstance()->addObjectToPlayGround(
                createMissile());
        }
    }
}

//----------------------------------------------------------------------------
Missile *SAMBatteryBase::do_createMissile(Sint16 angle) const
{
    Missile *m = new Missile(m_missileHeadingStrategy,
                             m_missileWarheadStrategy,
                             this);
    do_setInitialMissilePosition(m);
    m->setFuel(m_missileFuel);
    m->setAngle(angle);

    return m;
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(SAMBatteryBase);


//----------------------------------------------------------------------------
TopSAMBattery::TopSAMBattery(const InitializationData &init)
        : SAMBatteryBase(init)
{
}

//----------------------------------------------------------------------------
TopSAMBattery::~TopSAMBattery()
{
}

//----------------------------------------------------------------------------
Missile *TopSAMBattery::createMissile() const
{
    return do_createMissile(0);
}

//----------------------------------------------------------------------------
void TopSAMBattery::do_setInitialMissilePosition(Missile *m) const
{
    m->setPosition(
        getPosition().x - (m->getPosition().w - getPosition().w) / 2,
        getPosition().y + 4);
}


//----------------------------------------------------------------------------
BottomSAMBattery::BottomSAMBattery(const InitializationData &init)
        : SAMBatteryBase(init)
{
}

//----------------------------------------------------------------------------
BottomSAMBattery::~BottomSAMBattery()
{
}

//----------------------------------------------------------------------------
Missile *BottomSAMBattery::createMissile() const
{
    return do_createMissile(180);
}

//----------------------------------------------------------------------------
void BottomSAMBattery::do_setInitialMissilePosition(Missile *m) const
{
    m->setPosition(
        getPosition().x - (m->getPosition().w - getPosition().w) / 2,
        getPosition().y - 4);
}


//----------------------------------------------------------------------------
LeftSAMBattery::LeftSAMBattery(const InitializationData &init)
        : SAMBatteryBase(init)
{
}

//----------------------------------------------------------------------------
LeftSAMBattery::~LeftSAMBattery()
{
}

//----------------------------------------------------------------------------
Missile *LeftSAMBattery::createMissile() const
{
    return do_createMissile(90);
}

//----------------------------------------------------------------------------
void LeftSAMBattery::do_setInitialMissilePosition(Missile *m) const
{
    m->setPosition(
        getPosition().x + 4,
        getPosition().y - (m->getPosition().h - getPosition().h) / 2);
}


//----------------------------------------------------------------------------
RightSAMBattery::RightSAMBattery(const InitializationData &init)
        : SAMBatteryBase(init)
{
}

//----------------------------------------------------------------------------
RightSAMBattery::~RightSAMBattery()
{
}

//----------------------------------------------------------------------------
Missile *RightSAMBattery::createMissile() const
{
    return do_createMissile(270);
}

//----------------------------------------------------------------------------
void RightSAMBattery::do_setInitialMissilePosition(Missile *m) const
{
    m->setPosition(
        getPosition().x - 4,
        getPosition().y - (m->getPosition().h - getPosition().h) / 2);
}



//============================================================================
// Implementation of Turrets.h
//============================================================================

//----------------------------------------------------------------------------
TurretBarrel::InitializationData::InitializationData(const XMLNode *barrelNode,
                                                     TurretBase *t)
    throw (XMLException)
{
    turret = t;

    const std::string &type = barrelNode->getStringProperty("type");
    if (type == "fixed")  updateStrategy = US_FIXED;
    else if (type == "sweep")  updateStrategy = US_SWEEP;
    else if (type == "random")  updateStrategy = US_RANDOM;
    else if (type == "smart")  updateStrategy = US_SMART;
    else throw XMLException(
        "The value of the 'type' attribute must be "
        "'fixed', 'sweep', 'random', or 'smart'");

    angle = barrelNode->getIntProperty("angle", 0);
    step = barrelNode->getIntProperty("step", 0);
    speed = barrelNode->getUnsignedProperty("speed", 200);
    delay = barrelNode->getUnsignedProperty("delay", 20);
}

//----------------------------------------------------------------------------
TurretBarrel::InitializationData::~InitializationData()
{
    turret = NULL;
}


//----------------------------------------------------------------------------
TurretBarrel::Fixed TurretBarrel::Fixed::sm_instance;
TurretBarrel::Random TurretBarrel::Random::sm_instance;
TurretBarrel::Sweep TurretBarrel::Sweep::sm_instance;
TurretBarrel::Smart TurretBarrel::Smart::sm_instance;

//----------------------------------------------------------------------------
TurretBarrel::UpdateStrategy *TurretBarrel::UpdateStrategy::get(
    EnumUpdateStrategy strategy)
{
    switch (strategy)
    {
    case US_FIXED:
        return Fixed::getInstance();
    case US_RANDOM:
        return Random::getInstance();
    case US_SWEEP:
        return Sweep::getInstance();
    case US_SMART:
        return Smart::getInstance();
    }

    return NULL;
}


//----------------------------------------------------------------------------
void TurretBarrel::Fixed::update(TurretBase *turret, TurretBarrel *barrel)
{
    GameInterface::getInstance()->addObjectToPlayGround(
        new TurretProjectile(turret, barrel));
}

//----------------------------------------------------------------------------
void TurretBarrel::Random::update(TurretBase *turret, TurretBarrel *barrel)
{
    barrel->setAngle(myRand(180) - 90);
    GameInterface::getInstance()->addObjectToPlayGround(
        new TurretProjectile(turret, barrel));
}

//----------------------------------------------------------------------------
void TurretBarrel::Sweep::update(TurretBase *turret, TurretBarrel *barrel)
{
    GameInterface::getInstance()->addObjectToPlayGround(
        new TurretProjectile(turret, barrel));

    int angle = barrel->getAngle() + barrel->getAngleStep();
    if (!turret->isAngleInRange(angle))
    {
        angle = barrel->getAngleStep() < 0 ? 90 : -90;
    }

    barrel->setAngle(angle);
}

//----------------------------------------------------------------------------
void TurretBarrel::Smart::update(TurretBase *turret, TurretBarrel *barrel)
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return;
    }

    int angle =
        SDL_TOOLS::getAngle(turret->getPosition(), player->getPosition())
        - turret->getCenterAngle();

    if (turret->isAngleInRange(angle))
    {
        barrel->setAngle(angle);
        GameInterface::getInstance()->addObjectToPlayGround(
            new TurretProjectile(turret, barrel));
    }
}


//----------------------------------------------------------------------------
TurretBarrel::TurretBarrel(const InitializationData &init)
{
    m_turret = init.turret;

    m_angle = init.angle;
    m_angleStep = init.step;

    m_projectileSpeed = init.speed;

    m_frameDelay = init.delay * SDLFrameRate::getFrameRate() / 10;
    m_frameCounter = 0;

    m_strategy = UpdateStrategy::get(init.updateStrategy);
}

//----------------------------------------------------------------------------
TurretBarrel::~TurretBarrel()
{
    m_turret = NULL;
    m_strategy = NULL;
}

//----------------------------------------------------------------------------
TurretBarrel *TurretBarrel::create(const XMLNode *barrelNode,
                                   TurretBase *turret)
    throw (Exception)
{
    return new TurretBarrel(InitializationData(barrelNode, turret));
}

//----------------------------------------------------------------------------
void TurretBarrel::update(TurretBase *turret)
{
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (turret->canReachPlayerShip())
        {
            SoundInterface::getInstance()->onTurretShoot();
            m_strategy->update(turret, this);
        }
    }
}


//----------------------------------------------------------------------------
void TurretBase::BarrelVisitor::do_visit(const XMLProperty *a)
{
}

//----------------------------------------------------------------------------
void TurretBase::BarrelVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "barrel")
    {
        m_turret->m_barrels.push_back(TurretBarrel::create(n, m_turret));
    }
    else
    {
        n->acceptAllChilds(*this);
    }
}


//----------------------------------------------------------------------------
TurretBase::InitializationData::InitializationData(const XMLNode *turretNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(turretNode),
          OrientatingDecorationBase::InitializationData(turretNode)
{
    const std::string &prop = turretNode->getStringProperty("type");
    if (prop == "tube")
        type = TurretSurfaces::T_TUBE;
    else if (prop == "dome")
        type = TurretSurfaces::T_DOME;
    else
        throw XMLException(
            "The value of the 'type' attribute must be 'tube' or 'dome'");

    hitPoints = turretNode->getUnsignedProperty("hitpoints", 10);
}

//----------------------------------------------------------------------------
TurretBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
TurretBase::TurretBase(const InitializationData &init)
        : StaticDecorationBase(init),
          OrientatingDecorationBase(init)
{
    setSurface(TurretSurfaces::getInstance()->getSurface(
                   getOrientation(), init.type));

    m_tipOffset = TurretSurfaces::getInstance()->getTipOffset(init.type);

    m_hitPoints = init.hitPoints;
}

//----------------------------------------------------------------------------
TurretBase::~TurretBase()
{
    while (!m_barrels.empty())
    {
        delete m_barrels.front();
        m_barrels.pop_front();
    }
}

//----------------------------------------------------------------------------
TurretBase *TurretBase::create(const XMLNode *turretNode)
    throw (Exception)
{
    TurretBase *turret = create(InitializationData(turretNode));

    BarrelVisitor v(turret);
    turretNode->accept(v);

    return turret;
}

//----------------------------------------------------------------------------
TurretBase *TurretBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_TOP:
        return new TopTurret(init);
    case O_BOTTOM:
        return new BottomTurret(init);
    case O_LEFT:
        return new LeftTurret(init);
    case O_RIGHT:
        return new RightTurret(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void TurretBase::createExplosionParticles()
{
    unsigned numParticles = getPosition().w * getPosition().h / 4;
    for (unsigned i=0; i<numParticles; i++)
    {
        GameInterface::getInstance()->addParticleToPlayGround(
            new ExplosionParticle(this));
    }
}

//----------------------------------------------------------------------------
void TurretBase::update()
{
    for (TurretBarrelListIter iter = m_barrels.begin();
         iter != m_barrels.end(); ++iter)
    {
        (*iter)->update(this);
    }
}

//----------------------------------------------------------------------------
bool TurretBase::isAngleInRange(int angle) const
{
    angle = SinusTable::normalize(angle);
    return (angle >= 270 && angle < 360) || (angle >= 0 && angle <= 90);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(TurretBase);


//----------------------------------------------------------------------------
TopTurret::TopTurret(const InitializationData &init)
        : TurretBase(init)
{
    
}

//----------------------------------------------------------------------------
TopTurret::~TopTurret()
{
}

//----------------------------------------------------------------------------
int TopTurret::getCenterAngle() const
{
    return 0;
}

//----------------------------------------------------------------------------
void TopTurret::getTipPosition(Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w/2;
    y = getPosition().y + getPosition().h - m_tipOffset;
}


//----------------------------------------------------------------------------
BottomTurret::BottomTurret(const InitializationData &init)
        : TurretBase(init)
{
}

//----------------------------------------------------------------------------
BottomTurret::~BottomTurret()
{
}

//----------------------------------------------------------------------------
int BottomTurret::getCenterAngle() const
{
    return 180;
}

//----------------------------------------------------------------------------
void BottomTurret::getTipPosition(Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w/2;
    y = getPosition().y + m_tipOffset - 1;
}


//----------------------------------------------------------------------------
LeftTurret::LeftTurret(const InitializationData &init)
        : TurretBase(init)
{
}

//----------------------------------------------------------------------------
LeftTurret::~LeftTurret()
{
}

//----------------------------------------------------------------------------
int LeftTurret::getCenterAngle() const
{
    return 90;
}

//----------------------------------------------------------------------------
void LeftTurret::getTipPosition(Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w - m_tipOffset;
    y = getPosition().y + getPosition().h/2;
}


//----------------------------------------------------------------------------
RightTurret::RightTurret(const InitializationData &init)
        : TurretBase(init)
{
}

//----------------------------------------------------------------------------
RightTurret::~RightTurret()
{
}

//----------------------------------------------------------------------------
int RightTurret::getCenterAngle() const
{
    return 270;
}

//----------------------------------------------------------------------------
void RightTurret::getTipPosition(Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + m_tipOffset - 1;
    y = getPosition().y + getPosition().h/2;
}



//============================================================================
// Implementation of Mortars.h
//============================================================================

//----------------------------------------------------------------------------
MortarBarrel::InitializationData::InitializationData(const XMLNode *barrelNode,
                                                     MortarBase *m)
    throw (XMLException)
{
    mortar = m;

    const std::string &type = barrelNode->getStringProperty("type");
    if (type == "fixed")  updateStrategy = US_FIXED;
    else if (type == "random")  updateStrategy = US_RANDOM;
    else if (type == "smart")  updateStrategy = US_SMART;
    else throw XMLException(
        "The value of the 'type' attribute must be "
        "'fixed', 'random', or 'smart'");

    angle = barrelNode->getIntProperty("angle", 0);
    speed = barrelNode->getUnsignedProperty("speed", 200);
    delay = barrelNode->getUnsignedProperty("delay", 20);
    exploderDelay = barrelNode->getUnsignedProperty("exploderdelay", 10);

    const std::string &warhead = barrelNode->getStringProperty("warhead");
    if (warhead == "none")
        grenadeWarheadStrategy = GrenadeBase::WH_NONE;
    else if (warhead == "starburst")
        grenadeWarheadStrategy = GrenadeBase::WH_STARBURST;
    else
        throw XMLException(
            "The value of the 'warhead' attribute must be "
            "'none' or 'starburst'");
}

//----------------------------------------------------------------------------
MortarBarrel::InitializationData::~InitializationData()
{
    mortar = NULL;
}


//----------------------------------------------------------------------------
MortarBarrel::Fixed MortarBarrel::Fixed::sm_instance;
MortarBarrel::Random MortarBarrel::Random::sm_instance;
MortarBarrel::Smart MortarBarrel::Smart::sm_instance;

//----------------------------------------------------------------------------
MortarBarrel::UpdateStrategy *MortarBarrel::UpdateStrategy::get(EnumUpdateStrategy strategy)
{
    switch (strategy)
    {
    case US_FIXED:
        return Fixed::getInstance();
    case US_RANDOM:
        return Random::getInstance();
    case US_SMART:
        return Smart::getInstance();
    }

    return NULL;
}


//----------------------------------------------------------------------------
void MortarBarrel::Fixed::update(MortarBase *mortar, MortarBarrel *barrel)
{
    GameInterface::getInstance()->addObjectToPlayGround(
        new MortarGrenade(mortar, barrel));
}

//----------------------------------------------------------------------------
void MortarBarrel::Random::update(MortarBase *mortar, MortarBarrel *barrel)
{
    barrel->setAngle(myRand(180) - 90);
    GameInterface::getInstance()->addObjectToPlayGround(
        new MortarGrenade(mortar, barrel));
}

//----------------------------------------------------------------------------
void MortarBarrel::Smart::update(MortarBase *mortar, MortarBarrel *barrel)
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return;
    }

    int angle =
        SDL_TOOLS::getAngle(mortar->getPosition(), player->getPosition())
        - mortar->getCenterAngle();

    if (mortar->isAngleInRange(angle))
    {
        barrel->setAngle(angle);
        GameInterface::getInstance()->addObjectToPlayGround(
            new MortarGrenade(mortar, barrel));
    }
}


//----------------------------------------------------------------------------
MortarBarrel::MortarBarrel(const InitializationData &init)
{
    m_mortar = init.mortar;
    m_angle = init.angle;
    m_grenadeSpeed = init.speed;
    m_grenadeExploderDelay = init.exploderDelay;
    m_grenadeWarheadStrategy = init.grenadeWarheadStrategy;

    m_frameDelay = init.delay * SDLFrameRate::getFrameRate() / 10;
    m_frameCounter = 0;

    m_strategy = UpdateStrategy::get(init.updateStrategy);
}

//----------------------------------------------------------------------------
MortarBarrel::~MortarBarrel()
{
    m_mortar = NULL;
    m_strategy = NULL;
}

//----------------------------------------------------------------------------
MortarBarrel *MortarBarrel::create(const XMLNode *barrelNode,
                                   MortarBase *mortar)
    throw (Exception)
{
    return new MortarBarrel(InitializationData(barrelNode, mortar));
}

//----------------------------------------------------------------------------
void MortarBarrel::update(MortarBase *mortar)
{
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (mortar->canReachPlayerShip())
        {
            SoundInterface::getInstance()->onMortarShoot();
            m_strategy->update(mortar, this);
        }
    }
}


//----------------------------------------------------------------------------
void MortarBase::BarrelVisitor::do_visit(const XMLProperty *a)
{
}

//----------------------------------------------------------------------------
void MortarBase::BarrelVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "barrel")
    {
        m_mortar->m_barrels.push_back(MortarBarrel::create(n, m_mortar));
    }
    else
    {
        n->acceptAllChilds(*this);
    }
}


//----------------------------------------------------------------------------
MortarBase::InitializationData::InitializationData(const XMLNode *mortarNode)
    throw (XMLException)
        : StaticDecorationBase::InitializationData(mortarNode),
          OrientatingDecorationBase::InitializationData(mortarNode)
{
    hitPoints = mortarNode->getUnsignedProperty("hitpoints", 10);
}

//----------------------------------------------------------------------------
MortarBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
MortarBase::MortarBase(const InitializationData &init)
        : StaticDecorationBase(init),
          OrientatingDecorationBase(init)
{
    setSurface(TurretSurfaces::getInstance()->getSurface(
                   getOrientation(), TurretSurfaces::T_BIGTUBE));

    m_tipOffset =
        TurretSurfaces::getInstance()->getTipOffset(TurretSurfaces::T_BIGTUBE);

    m_hitPoints = init.hitPoints;
}

//----------------------------------------------------------------------------
MortarBase::~MortarBase()
{
    while (!m_barrels.empty())
    {
        delete m_barrels.front();
        m_barrels.pop_front();
    }
}

//----------------------------------------------------------------------------
MortarBase *MortarBase::create(const XMLNode *mortarNode)
    throw (Exception)
{
    MortarBase *mortar = create(InitializationData(mortarNode));

    BarrelVisitor v(mortar);
    mortarNode->accept(v);

    return mortar;
}

//----------------------------------------------------------------------------
MortarBase *MortarBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_TOP:
        return new TopMortar(init);
    case O_BOTTOM:
        return new BottomMortar(init);
    case O_LEFT:
        return new LeftMortar(init);
    case O_RIGHT:
        return new RightMortar(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void MortarBase::createExplosionParticles()
{
    unsigned numParticles = getPosition().w * getPosition().h / 4;
    for (unsigned i=0; i<numParticles; i++)
    {
        GameInterface::getInstance()->addParticleToPlayGround(
            new ExplosionParticle(this));
    }
}

//----------------------------------------------------------------------------
void MortarBase::update()
{
    for (MortarBarrelListIter iter = m_barrels.begin();
         iter != m_barrels.end(); ++iter)
    {
        (*iter)->update(this);
    }
}

//----------------------------------------------------------------------------
bool MortarBase::isAngleInRange(int angle) const
{
    angle = SinusTable::normalize(angle);
    return (angle >= 270 && angle < 360) || (angle >= 0 && angle <= 90);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(MortarBase);


//----------------------------------------------------------------------------
TopMortar::TopMortar(const InitializationData &init)
        : MortarBase(init)
{
}

//----------------------------------------------------------------------------
TopMortar::~TopMortar()
{
}

//----------------------------------------------------------------------------
int TopMortar::getCenterAngle() const
{
    return 0;
}

//----------------------------------------------------------------------------
void TopMortar::getTipPosition(Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w/2;
    y = getPosition().y + getPosition().h - m_tipOffset;
}


//----------------------------------------------------------------------------
BottomMortar::BottomMortar(const InitializationData &init)
        : MortarBase(init)
{
}

//----------------------------------------------------------------------------
BottomMortar::~BottomMortar()
{
}

//----------------------------------------------------------------------------
int BottomMortar::getCenterAngle() const
{
    return 180;
}

//----------------------------------------------------------------------------
void BottomMortar::getTipPosition(Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w/2;
    y = getPosition().y + m_tipOffset - 1;
}


//----------------------------------------------------------------------------
LeftMortar::LeftMortar(const InitializationData &init)
        : MortarBase(init)
{
}

//----------------------------------------------------------------------------
LeftMortar::~LeftMortar()
{
}

//----------------------------------------------------------------------------
int LeftMortar::getCenterAngle() const
{
    return 90;
}

//----------------------------------------------------------------------------
void LeftMortar::getTipPosition(Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w - m_tipOffset;
    y = getPosition().y + getPosition().h/2;
}


//----------------------------------------------------------------------------
RightMortar::RightMortar(const InitializationData &init)
        : MortarBase(init)
{
}

//----------------------------------------------------------------------------
RightMortar::~RightMortar()
{
}

//----------------------------------------------------------------------------
int RightMortar::getCenterAngle() const
{
    return 270;
}

//----------------------------------------------------------------------------
void RightMortar::getTipPosition(Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + m_tipOffset - 1;
    y = getPosition().y + getPosition().h/2;
}



//============================================================================
// Implementation of Grinder.h
//============================================================================

//----------------------------------------------------------------------------
Grinder::WayPoint::InitializationData::InitializationData(const XMLNode *wayPointNode)
    throw (XMLException)
{
    x = wayPointNode->getIntProperty("x");
    y = wayPointNode->getIntProperty("y");
}

//----------------------------------------------------------------------------
Grinder::WayPoint::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Grinder::WayPoint::WayPoint(const InitializationData &init)
{
    m_x = init.x;
    m_y = init.y;
}

//----------------------------------------------------------------------------
Grinder::WayPoint::~WayPoint()
{
}

//----------------------------------------------------------------------------
Grinder::WayPoint *Grinder::WayPoint::create(const XMLNode *wayPointNode)
    throw (Exception)
{
    return new WayPoint(InitializationData(wayPointNode));
}


//----------------------------------------------------------------------------
void Grinder::WayPointVisitor::do_visit(const XMLProperty *a)
{
}

//----------------------------------------------------------------------------
void Grinder::WayPointVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "waypoint")
    {
        m_grinder->m_wayPoints.push_back(WayPoint::create(n));
    }
    else
    {
        n->acceptAllChilds(*this);
    }
}


//----------------------------------------------------------------------------
Grinder::InitializationData::InitializationData(const XMLNode *grinderNode)
    throw (XMLException)
        : MovingDecorationBase::InitializationData(grinderNode)
{
    velocity = grinderNode->getUnsignedProperty(
        "velocity", SDLFrameRate::getFrameRate());
}

//----------------------------------------------------------------------------
Grinder::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Grinder::Grinder(const InitializationData &init) : MovingDecorationBase(init)
{
    m_frameDelay = 1;
    m_currentFrame = 0;
    m_frameCounter = 0;

    setSurface(GrinderSurfaces::getInstance()->getSurface(m_currentFrame));

    m_velocity = init.velocity;
}

//----------------------------------------------------------------------------
Grinder::~Grinder()
{
    for (WayPointIter iter = m_wayPoints.begin();
         iter != m_wayPoints.end(); ++iter)
    {
        delete *iter;
    }
    m_wayPoints.clear();
}

//----------------------------------------------------------------------------
void Grinder::initVelocity()
{
    int angle = MATH_TOOLS::getAngle(
        getPosition().x, getPosition().y,
        (*m_wayPointIter)->getX(), (*m_wayPointIter)->getY());

    getFineVelocity().setExponential(m_velocity, angle);
}

//----------------------------------------------------------------------------
Grinder *Grinder::create(const XMLNode *grinderNode)
    throw (Exception)
{
    Grinder *grinder = create(InitializationData(grinderNode));

    WayPointVisitor v(grinder);
    grinderNode->accept(v);

    if (grinder->m_wayPoints.size() < 2)
    {
        throw Exception("A grinder must have at least two waypoints");
    }

    grinder->m_wayPointIter = grinder->m_wayPoints.begin();
    grinder->setPosition((*grinder->m_wayPointIter)->getX(),
                         (*grinder->m_wayPointIter)->getY());
    ++grinder->m_wayPointIter;
    grinder->initVelocity();

    return grinder;
}

//----------------------------------------------------------------------------
Grinder *Grinder::create(const InitializationData &init)
{
    return new Grinder(init);
}


//----------------------------------------------------------------------------
void Grinder::update()
{
    // Update the position.

    const WayPoint *wayPoint = *m_wayPointIter;

    MATH_TOOLS::Vector relPosition = getFineVelocity();
    relPosition /= SDLFrameRate::getFrameRate();
    getFinePosition() += relPosition;
    updatePosition();

    if ((getPosition().x >= wayPoint->getX()-2) &&
        (getPosition().x <= wayPoint->getX()+2) &&
        (getPosition().y >= wayPoint->getY()-2) &&
        (getPosition().y <= wayPoint->getY()+2))
    {
        ++m_wayPointIter;
        if (m_wayPointIter == m_wayPoints.end())
        {
            m_wayPointIter = m_wayPoints.begin();
        }

        initVelocity();
    }

    // Update the frame.
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (m_currentFrame == 0)
        {
            m_currentFrame = SURFACES_ROTATION_STEPS;
        }

        setSurface(
            GrinderSurfaces::getInstance()->getSurface(--m_currentFrame));
    }
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Grinder);



//============================================================================
// Implementation of Tank.h
//============================================================================

//----------------------------------------------------------------------------
Tank::Random Tank::Random::sm_instance;
Tank::Smart Tank::Smart::sm_instance;

//----------------------------------------------------------------------------
Tank::AimStrategy *Tank::AimStrategy::get(Tank::EnumAimStrategy strategy)
{
    switch (strategy)
    {
    case AS_RANDOM:
        return Random::getInstance();
    case AS_SMART:
        return Smart::getInstance();
    }

    return NULL;
}

//----------------------------------------------------------------------------
void Tank::Random::setAngle(Tank *tank)
{
    tank->m_angle = myRand(180) + 90;
}

//----------------------------------------------------------------------------
void Tank::Smart::setAngle(Tank *tank)
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return;
    }

    tank->m_angle =
        SDL_TOOLS::getAngle(tank->getPosition(), player->getPosition());
}


//----------------------------------------------------------------------------
Tank::Moving Tank::Moving::sm_instance;
Tank::WaitForShooting Tank::WaitForShooting::sm_instance;
Tank::WaitForMoving Tank::WaitForMoving::sm_instance;


//----------------------------------------------------------------------------
void Tank::Moving::update(Tank *tank)
{
    if (++tank->m_frameCounter == tank->m_frameDelay)
    {
        if (tank->getPosition().x < tank->m_nextOffset)
        {
            tank->m_frameCounter = 0;
            tank->setPosition(tank->getPosition().x + 1,
                              tank->getPosition().y);
        }
        else if (tank->getPosition().x > tank->m_nextOffset)
        {
            tank->m_frameCounter = 0;
            tank->setPosition(tank->getPosition().x - 1,
                              tank->getPosition().y);
        }
        else
        {
            tank->initWaitForShooting();
        }
    }
}

//----------------------------------------------------------------------------
void Tank::WaitForShooting::update(Tank *tank)
{
    if (++tank->m_frameCounter == tank->m_frameDelay)
    {
        tank->initWaitForMoving();
    }
}

//----------------------------------------------------------------------------
void Tank::WaitForMoving::update(Tank *tank)
{
    if (++tank->m_frameCounter == tank->m_frameDelay)
    {
        tank->initMoving();
    }
}


//----------------------------------------------------------------------------
Tank::InitializationData::InitializationData(const XMLNode *tankNode)
    throw (XMLException)
        : MovingDecorationBase::InitializationData(tankNode)
{
    x = tankNode->getUnsignedProperty("x");
    y = tankNode->getUnsignedProperty("y");
    w = tankNode->getUnsignedProperty("w");

    hitPoints = tankNode->getUnsignedProperty("hitpoints", 10);

    speed = tankNode->getUnsignedProperty("speed", 200);

    const std::string &type = tankNode->getStringProperty("type");
    if (type == "random")  aimStrategy = AS_RANDOM;
    else if (type == "smart")  aimStrategy = AS_SMART;
    else throw XMLException(
        "The value of the 'type' attribute must be 'random' or 'smart'");
}

//----------------------------------------------------------------------------
Tank::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Tank::Tank(const InitializationData &init) : MovingDecorationBase(init)
{
    setSurface(TankSurfaces::getInstance()->getSurface());

    m_x = init.x;
    m_y = init.y;
    m_w = init.w;

    setTilePosition(m_x, m_y);

    m_hitPoints = init.hitPoints;
    m_projectileSpeed = init.speed;
    m_aimStrategy = AimStrategy::get(init.aimStrategy);

    initMoving();
}

//----------------------------------------------------------------------------
Tank::~Tank()
{
    m_updateState = NULL;
}

//----------------------------------------------------------------------------
Tank *Tank::create(const XMLNode *tankNode)
    throw (Exception)
{
    return create(InitializationData(tankNode));
}

//----------------------------------------------------------------------------
Tank *Tank::create(const InitializationData &init)
{
    return new Tank(init);
}

//----------------------------------------------------------------------------
void Tank::createExplosionParticles()
{
    unsigned numParticles = getPosition().w * getPosition().h / 4;
    for (unsigned i=0; i<numParticles; i++)
    {
        GameInterface::getInstance()->addParticleToPlayGround(
            new ExplosionParticle(this));
    }
}

//----------------------------------------------------------------------------
void Tank::initMoving()
{
    m_nextOffset = m_x*16 + myRand((m_w*16 - getPosition().w));

    m_frameDelay = 2;
    m_frameCounter = 0;

    setUpdateState(Moving::getInstance());
}

//----------------------------------------------------------------------------
void Tank::initWaitForShooting()
{
    m_frameDelay = SDLFrameRate::getFrameRate() / 2;
    m_frameCounter = 0;

    setUpdateState(WaitForShooting::getInstance());
}

//----------------------------------------------------------------------------
void Tank::initWaitForMoving()
{
    m_frameDelay = SDLFrameRate::getFrameRate() / 2;
    m_frameCounter = 0;

    if (canReachPlayerShip())
    {
        m_aimStrategy->setAngle(this);

        SoundInterface::getInstance()->onTankShoot();
        GameInterface::getInstance()->addObjectToPlayGround(
            new TankProjectile(this));
    }

    setUpdateState(WaitForMoving::getInstance());
}


//----------------------------------------------------------------------------
void Tank::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Tank);
