#include "Tools.h"
#include "SinusTable.h"
#include "MathTools.h"
#include "SDLFrameRate.h"

#include "GameInterface.h"
#include "SoundInterface.h"

#include "MovingObjectBase.h"
#include   "ControllableObjectBase.h"
#include     "Grinder.h"
#include     "Missile.h"
#include     "Ship.h"
#include   "Particle.h"
#include   "Projectile.h"
#include   "Grenade.h"

#include "ObjectVisitors.h"
#include "Crates.h"
#include "Mortars.h"
#include "ParticleFountains.h"
#include "SAMBattery.h"
#include "Turrets.h"
#include "Tank.h"

#include "ObjectVisitors.h"


//============================================================================
// Implementation of MovingObjectBase.h
//============================================================================

//----------------------------------------------------------------------------
MovingObjectBase::MovingObjectBase(const ObjectBase *creator)
        : ObjectBase(creator)
{
}

//----------------------------------------------------------------------------
MovingObjectBase::~MovingObjectBase()
{
}



//============================================================================
// Implementation of ControllableObjectBase.h
//============================================================================

//----------------------------------------------------------------------------
ControllableObjectBase::ControllableObjectBase(const ObjectBase *creator)
        : MovingObjectBase(creator)
{
    m_angle = SURFACES_ROTATION_STEPS / 2;
    m_fineAngle = 180.0;
    m_rotation = ROT_NONE;
    m_fuel = 0;
    m_thrust = false;
    m_explode = false;

    setPosition(0, 0);
}

//----------------------------------------------------------------------------
ControllableObjectBase::~ControllableObjectBase()
{
}

//----------------------------------------------------------------------------
void ControllableObjectBase::initBottomPixels()
{
    Sint16 x = 0;
    Sint16 y;

    for (y = getSurface()->h-1; y >= 0; y--)
    {
        for (x = 0; x < getSurface()->w; x++)
        {
            if (SDL_TOOLS::getPixel(getSurface(), x, y))
            {
                goto out;
            }
        }
    }

  out:

    m_bottomPixels.x = x;
    m_bottomPixels.y = y;

    for (x = getSurface()->w-1; x >= 0; x--)
    {
        if (SDL_TOOLS::getPixel(getSurface(), x, y))
        {
            break;
        }
    }

    m_bottomPixels.w = x - m_bottomPixels.x + 1;
    m_bottomPixels.h = 1;
}

//----------------------------------------------------------------------------
void ControllableObjectBase::setThrust(bool enable)
{
    if (enable && hasFuel())
    {
        SoundInterface::getInstance()->onShipThrustOn();
        m_thrust = true;
    }
    else
    {
        SoundInterface::getInstance()->onShipThrustOff();
        m_thrust = false;
    }
}

//----------------------------------------------------------------------------
void ControllableObjectBase::createExplosionParticles()
{
    unsigned numParticles = getPosition().w * getPosition().h / 4;
    for (unsigned i=0; i<numParticles; i++)
    {
        GameInterface::getInstance()->addParticleToPlayGround(
            new ExplosionParticle(this));
    }
}

//----------------------------------------------------------------------------
void ControllableObjectBase::updateRotation()
{
    double angleStep =
        1.0 * getRotationDegreesPerSecond() / SDLFrameRate::getFrameRate();

    switch (m_rotation)
    {
    case ROT_NONE:
        break;

    case ROT_LEFT:
        m_fineAngle += angleStep;
        break;

    case ROT_RIGHT:
        m_fineAngle -= angleStep;
        break;

    case ROT_ALIGN:
        if (m_fineAngle < 180.0)
        {
            m_fineAngle += angleStep;
        }
        else if (m_fineAngle > 180.0)
        {
            m_fineAngle -= angleStep;
        }
        break;
    }

    // m_fineAngle must always be inside of [0 ... 360).
    if (m_fineAngle < 0.0)
    {
        m_fineAngle += 360.0;
    }
    else if (m_fineAngle >= 360.0)
    {
        m_fineAngle -= 360.0;
    }
    assert(m_fineAngle >= 0.0 && m_fineAngle < 360.0);

    m_angle = (unsigned)rint(m_fineAngle) / SURFACES_ROTATION_ANGLE;
    setSurface(getRotationSurface(m_angle));

    if (m_rotation == ROT_ALIGN && m_angle == SURFACES_ROTATION_STEPS / 2)
    {
        m_rotation = ROT_NONE;
    }
}

//----------------------------------------------------------------------------
void ControllableObjectBase::updateVelocityAndPosition()
{
    GameInterface *gameInterface = GameInterface::getInstance();

    // The old velocity values are needed.
    MATH_TOOLS::Vector oldFineVelocity = getFineVelocity();

    MATH_TOOLS::Vector fineAccel;

    // Angle-dependend acceleration due thrusting.
    if (isThrust() && hasFuel())
    {
        --m_fuel;

        fineAccel.setExponential(
            1.0 * getThrustForce() / getMass(), 1.0 * getAngle());

        gameInterface->addParticleToPlayGround(new ThrustParticle(this));
    }

    // Apply the playground's gravity force.
    fineAccel.add(1.0 * gameInterface->getXGravity(getPosition()),
                  1.0 * gameInterface->getYGravity(getPosition()));

    // Friction will reduce the acceleration of this object.
    unsigned frictionScale = 32*32 / (getPosition().w * getPosition().h);
    MATH_TOOLS::Vector frictionSub(
        gameInterface->getXVelocity(getPosition()),
        gameInterface->getYVelocity(getPosition()));
    frictionSub -= oldFineVelocity;
    frictionSub *= 1.0 * gameInterface->getFriction(getPosition());
    frictionSub /= 1.0 * frictionScale * getMass();
    fineAccel += frictionSub;

    // Calculate and add the velocity increment.
    fineAccel /= 1.0 * SDLFrameRate::getFrameRate();
    getFineVelocity() += fineAccel;

    // To be exact, the average of the old and the new velocity values
    // must be used for the division with the frame rate.
    oldFineVelocity += getFineVelocity();
    oldFineVelocity /= 2.0 * SDLFrameRate::getFrameRate();
    getFinePosition() += oldFineVelocity;

    updatePosition();
}



//============================================================================
// Implementation of Missile.h
//============================================================================

//----------------------------------------------------------------------------
Missile::DirectHeadingStrategy Missile::DirectHeadingStrategy::sm_instance;
Missile::SmartHeadingStrategy Missile::SmartHeadingStrategy::sm_instance;


//----------------------------------------------------------------------------
Missile::HeadingStrategy *Missile::HeadingStrategy::getStrategy(
    EnumHeadingStrategy strategy)
{
    switch (strategy)
    {
    case HS_DIRECT:
        return DirectHeadingStrategy::getInstance();
    case HS_SMART:
        return SmartHeadingStrategy::getInstance();
    }

    return NULL;
}


//----------------------------------------------------------------------------
Sint16 Missile::DirectHeadingStrategy::calculateHeadingAngle(
    Sint16 a, Sint16 angle, Sint16 gx, Sint16 gy) const
{
    // Always head directly to the player's ship.

    return angle;
}


//----------------------------------------------------------------------------
Sint16 Missile::SmartHeadingStrategy::calculateHeadingAngle(
    Sint16 a, Sint16 angle, Sint16 gx, Sint16 gy) const
{
    // Calculate the rotation such way, that it flies directly
    // to the player's ship, while compensating gravity.

    double ax = a * SinusTable::sin(angle);
    double ay = a * SinusTable::cos(angle);

    double aa = 1.0 * a * a;
    double axgy_m_aygx = ax*gy - ay*gx;
    double sqrtTerm = 1.0 - axgy_m_aygx * axgy_m_aygx / (aa*aa);

    if (sqrtTerm <= 0.0)
    {
        // If no or only one solution exists,
        // let the missile fly against gravity.
        return MATH_TOOLS::getAngle(-gx, -gy);
    }
    else
    {
        double term1 = (ax*gx + ay*gy) / aa;
        double term2 = sqrt(sqrtTerm);

        // There are two solutions for t: term1+term2 or term1-term2.
        // However, term1+term2 is always the right solution.
        // (Proof by lack of test cases ;-)
        double t = term1 + term2;
        int bx = -gx + (int)round(t*ax);
        int by = -gy + (int)round(t*ay);

        return MATH_TOOLS::getAngle(bx, by);
    }
}



//----------------------------------------------------------------------------
Missile::NormalWarheadStrategy Missile::NormalWarheadStrategy::sm_instance;
Missile::ConeBurstWarheadStrategy Missile::ConeBurstWarheadStrategy::sm_instance;
Missile::StarBurstWarheadStrategy Missile::StarBurstWarheadStrategy::sm_instance;


//----------------------------------------------------------------------------
Missile::WarheadStrategy *Missile::WarheadStrategy::getStrategy(
    EnumWarheadStrategy strategy)
{
    switch (strategy)
    {
    case WH_NORMAL:
        return NormalWarheadStrategy::getInstance();
    case WH_CONEBURST:
        return ConeBurstWarheadStrategy::getInstance();
    case WH_STARBURST:
        return StarBurstWarheadStrategy::getInstance();
    }

    return NULL;
}

//----------------------------------------------------------------------------
void Missile::WarheadStrategy::do_updateWarhead(Missile *missile)
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return;
    }

    Sint16 mx, my;
    SDL_TOOLS::getCentre(missile->getPosition(), mx, my);

    Sint16 sx, sy;
    SDL_TOOLS::getCentre(player->getPosition(), sx, sy);

    // Calculate the square of the distance between the missile and the ship.
    int dx = mx - sx;
    int dy = my - sy;
    int d2 = dx*dx + dy*dy;

    if (d2 < 96*96)
    {
        // Signal the PlayGround, that this missile shall explode.
        missile->setExplode();
    }
}


//----------------------------------------------------------------------------
void Missile::NormalWarheadStrategy::updateWarhead(Missile *missile)
{
}

//----------------------------------------------------------------------------
void Missile::NormalWarheadStrategy::onCreateExplosionParticles(Missile *missile)
{
}


//----------------------------------------------------------------------------
void Missile::ConeBurstWarheadStrategy::updateWarhead(Missile *missile)
{
    do_updateWarhead(missile);
}

//----------------------------------------------------------------------------
void Missile::ConeBurstWarheadStrategy::onCreateExplosionParticles(Missile *missile)
{
    // If the missile crashed or was shot down,
    // no projectiles shall be created.

    if (missile->isExplode())
    {
        int angle = (int)missile->getAngle() - 45;
        for (int i=0; i<=90; i+=15)
        {
            GameInterface::getInstance()->addObjectToPlayGround(
                new MissileProjectile(missile, angle+i));
        }
    }
}


//----------------------------------------------------------------------------
void Missile::StarBurstWarheadStrategy::updateWarhead(Missile *missile)
{
    do_updateWarhead(missile);
}

//----------------------------------------------------------------------------
void Missile::StarBurstWarheadStrategy::onCreateExplosionParticles(Missile *missile)
{
    // If the missile crashed or was shot down,
    // no projectiles shall be created.

    if (missile->isExplode())
    {
        for (int angle=0; angle<360; angle+=45)
        {
            GameInterface::getInstance()->addObjectToPlayGround(
                new MissileProjectile(missile, angle));
        }
    }
}



//----------------------------------------------------------------------------
Missile::Missile(EnumHeadingStrategy heading,
                 EnumWarheadStrategy warhead,
                 const ObjectBase *creator)
        : ControllableObjectBase(creator)
{
    m_mass = 20;
    m_thrust = true;

    setSurface(getRotationSurface(SURFACES_ROTATION_STEPS / 2));

    initBottomPixels();

    m_headingStrategy = HeadingStrategy::getStrategy(heading);
    m_warheadStrategy = WarheadStrategy::getStrategy(warhead);
}

//----------------------------------------------------------------------------
Missile::~Missile()
{
    m_headingStrategy = NULL;
    m_warheadStrategy = NULL;
}

//----------------------------------------------------------------------------
void Missile::createExplosionParticles()
{
    ControllableObjectBase::createExplosionParticles();

    m_warheadStrategy->onCreateExplosionParticles(this);
}

//----------------------------------------------------------------------------
void Missile::updateHeading()
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        setRotation(ROT_NONE);
        return;
    }

    Sint16 a = getThrustForce() / getMass();

    int angle = SDL_TOOLS::getAngle(getPosition(), player->getPosition());

    Sint16 gx = GameInterface::getInstance()->getXGravity(getPosition());
    Sint16 gy = GameInterface::getInstance()->getYGravity(getPosition());

    angle = m_headingStrategy->calculateHeadingAngle(a, angle, gx, gy);
    angle -= getAngle();

    // Let the relative angle lie between [-180..180) degrees.
    while (angle >= 180)  angle -= 360;
    while (angle < -180)  angle += 360;

    setRotation(angle > 0 ? ROT_LEFT :
                angle < 0 ? ROT_RIGHT : ROT_NONE);
}

//----------------------------------------------------------------------------
void Missile::updateWarhead()
{
    m_warheadStrategy->updateWarhead(this);
}

//----------------------------------------------------------------------------
void Missile::update()
{
    if (hasFuel())
    {
        updateHeading();
        updateRotation();
    }

    updateVelocityAndPosition();

    updateWarhead();
}

//----------------------------------------------------------------------------
const SDL_Surface *Missile::getRotationSurface(unsigned frame) const
{
    return MissileSurfaces::getInstance()->getSurface(frame);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Missile);



//============================================================================
// Implementation of Ship.h
//============================================================================

//----------------------------------------------------------------------------
/**
 * If the ship's y-velocity is bigger than SHIP_MIN_LANDING_VELOCITY,
 * the ship will be reflected by the landing platform.
 */
#define SHIP_MIN_LANDING_VELOCITY  20.0

/**
 * If the ship's y-velocity is bigger than SHIP_MAX_LANDING_VELOCITY,
 * the ship will be destroyed.
 */
#define SHIP_MAX_LANDING_VELOCITY  60.0

//----------------------------------------------------------------------------
/**
 * The maximum crate mass, the ship is able to carry.
 */
#define SHIP_MAX_CRATES_MASS  60


//----------------------------------------------------------------------------
Ship::Landed Ship::Landed::sm_instance;
Ship::Flying Ship::Flying::sm_instance;


//----------------------------------------------------------------------------
void Ship::Landed::update(Ship *s)
{
    s->updateFire();

    if (s->isThrust() && s->hasFuel())
    {
        --s->m_fuel;

        // If the thrusters are enabled, raise the ship by one pixel,
        // so the next update won't detect a landing again.

        s->getFinePosition().subY(1.0);
        s->updatePosition();
        s->setUpdateState(Flying::getInstance());

        GameInterface::getInstance()->onShipTakeoff(s);
    }
}

//----------------------------------------------------------------------------
void Ship::Flying::update(Ship *s)
{
    assert(s->m_mass > 0);

    s->updateRotation();
    s->updateVelocityAndPosition();
    s->updateFire();

    if (s->isThrust() && !s->hasFuel())
    {
        s->setThrust(false);
        GameInterface::getInstance()->onShipOutOfFuel(s);
    }

    // Check if we are landed.
    const Platform *platform =
        GameInterface::getInstance()->isInLandingZone(s);
    if (platform && s->getFineVelocity().getY() > 0.0)
    {
        if (s->getFineVelocity().getY() < SHIP_MIN_LANDING_VELOCITY)
        {
            // We are landing.
            s->getFineVelocity().set(0.0, 0.0);
            s->setUpdateState(Landed::getInstance());

            SoundInterface::getInstance()->onShipPlatformBump();
            GameInterface::getInstance()->onShipLanded(s, platform);
        }
        else if (s->getFineVelocity().getY() < SHIP_MAX_LANDING_VELOCITY)
        {
            // We are reflecting from the platform.
            s->getFineVelocity().set(
                3.0 * s->getFineVelocity().getX() / 4.0,
                - s->getFineVelocity().getY() / 2.0);

            SoundInterface::getInstance()->onShipPlatformBump();
        }

        // If the ship's y velocity is faster than
        // SHIP_MAX_LANDING_VELOCITY, we do nothing.
        // This will lead to a background collision in the next update.
    }
}



//----------------------------------------------------------------------------
Ship::Ship(ShipSurfaces::Ship shipType) : ControllableObjectBase(NULL)
{
    m_shipType = shipType;
    m_mass = 100;
    m_fire = false;

    setSurface(getRotationSurface(SURFACES_ROTATION_STEPS / 2));
    m_tipOffset = ShipSurfaces::getInstance()->getTipOffset(m_shipType);

    initBottomPixels();

    setUpdateState(Flying::getInstance());
}

//----------------------------------------------------------------------------
Ship::~Ship()
{
    setUpdateState(NULL);
}


//----------------------------------------------------------------------------
bool Ship::decFuel(unsigned decrement)
{
    if (m_fuel < decrement)
    {
        m_fuel = 0;
        return true;
    }

    m_fuel -= decrement;
    return false;
}

//----------------------------------------------------------------------------
bool Ship::hasCapacity(const Crate *crate) const
{
    unsigned sum = 0;
    for (CrateCIter iter = m_crates.begin(); iter != m_crates.end(); ++iter)
    {
        sum += (*iter)->getMass();
    }

    return sum + crate->getMass() <= SHIP_MAX_CRATES_MASS;
}

//----------------------------------------------------------------------------
void Ship::pushCrate(Crate *crate)
{
    m_crates.push_back(crate);
    m_mass += crate->getMass();
}

//----------------------------------------------------------------------------
Crate *Ship::popCrate()
{
    assert(!m_crates.empty());

    Crate *crate = m_crates.front();
    m_crates.pop_front();

    m_mass -= crate->getMass();

    return crate;
}

//----------------------------------------------------------------------------
void Ship::removeAllCrates()
{
    while (!m_crates.empty())
    {
        delete popCrate();
    }
}



//----------------------------------------------------------------------------
bool Ship::isInLandingZone(const SDL_Rect &r) const
{
    if (m_angle == SURFACES_ROTATION_STEPS / 2)
    {
        SDL_Rect bottom = m_bottomPixels;
        bottom.x += getPosition().x;
        bottom.y += getPosition().y;

        SDL_Rect i;
        if (SDL_TOOLS::intersect(r, bottom, i))
        {
            if (i.w == bottom.w)
            {
                return true;
            }
        }
    }

    return false;
}


//----------------------------------------------------------------------------
void Ship::resetPosition(Sint16 x, Sint16 y, bool landed)
{
    setPosition(x, y);
    getFineVelocity().set(0.0, 0.0);

    if (landed)
    {
        setUpdateState(Landed::getInstance());
    }
    else
    {
        setUpdateState(Flying::getInstance());
    }

    // @todo The following reset stuff should not be in this method.

    setAngle(180);
    setRotation(ROT_NONE);
    updateRotation();

    setThrust(false);
}



//----------------------------------------------------------------------------
void Ship::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
const SDL_Surface *Ship::getRotationSurface(unsigned frame) const
{
    return ShipSurfaces::getInstance()->getSurface(m_shipType, frame);
}


//----------------------------------------------------------------------------
void Ship::updateFire()
{
    if (m_fire)
    {
        m_fire = false;
        SoundInterface::getInstance()->onShipShoot();
        GameInterface::getInstance()->addObjectToPlayGround(
            new ShipProjectile(this));
        GameInterface::getInstance()->onShipFiring(this);
    }
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Ship);



//============================================================================
// Implementation of Particles.h
//============================================================================

//----------------------------------------------------------------------------
ParticleBase::ParticleBase(const ObjectBase *creator)
        : MovingObjectBase(creator)
{
    m_lifeCount = 0;
    m_remove = false;
    m_gravity = false;

    m_gradient = false;
    m_gradientFineIndex = 0;
    m_gradientFineIndexIncrement = 0;

    updateSurface();
}

//----------------------------------------------------------------------------
ParticleBase::~ParticleBase()
{
}

//----------------------------------------------------------------------------
void ParticleBase::setGradient(unsigned fineSweepSpeed)
{
    m_gradient = true;

    m_gradientFineIndex =
        myRand(ParticleSurfaces::getInstance()->getGradientSize() / 16);
    m_gradientFineIndexIncrement = fineSweepSpeed/2 + (myRand(fineSweepSpeed));

    updateSurface();
}

//----------------------------------------------------------------------------
void ParticleBase::update()
{
    GameInterface *gameInterface = GameInterface::getInstance();

    // The old velocity values are needed.
    MATH_TOOLS::Vector oldFineVelocity = getFineVelocity();

    MATH_TOOLS::Vector fineAccel;

    if (m_gravity)
    {
        fineAccel.set(
            1.0 * gameInterface->getXGravity(getPosition()),
            1.0 * gameInterface->getYGravity(getPosition()));
    }

    // No friction is calculated for particles.
    fineAccel /= 1.0 * SDLFrameRate::getFrameRate();
    getFineVelocity() += fineAccel;

    // Use the average value of the old and new velocity.
    oldFineVelocity += getFineVelocity();
    oldFineVelocity /= 2.0 * SDLFrameRate::getFrameRate();
    getFinePosition() += oldFineVelocity;

    updatePosition();

    if (m_gradient)
    {
        updateSurface();
        m_gradientFineIndex += m_gradientFineIndexIncrement;

        if (m_gradientFineIndex >= 100*PARTICLES_GRADIENT_SIZE)
        {
            setRemove();
        }
    }

    if (--m_lifeCount == 0)
    {
        setRemove();
    }
}

//----------------------------------------------------------------------------
void ParticleBase::updateSurface()
{
    setSurface(
        ParticleSurfaces::getInstance()->getSurface(m_gradientFineIndex/100));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(ParticleBase);


//----------------------------------------------------------------------------
ThrustParticle::ThrustParticle(const ControllableObjectBase *object)
        : ParticleBase(object)
{
    setLifeCount(20 + myRand(5));
    setGradient(200);

    init();
}

//----------------------------------------------------------------------------
ThrustParticle::~ThrustParticle()
{
}

//----------------------------------------------------------------------------
void ThrustParticle::init()
{
    const ControllableObjectBase *object =
        static_cast<const ControllableObjectBase*>(getCreator());

    // Initialize the position.

    const SDL_Rect &objPosition = object->getPosition();
    Sint16 xm, ym;
    SDL_TOOLS::getCentre(objPosition, xm, ym);

    // Add 2 to the radius to prevent immediate collisions
    // between the object and this particle.
    Uint16 r = objPosition.y + object->getBottomPixels().y - ym + 2;

    setPosition((Uint16)trunc(xm - r * SinusTable::sin(object->getAngle())),
                (Uint16)(ym - r * SinusTable::cos(object->getAngle())));


    // Initialize the velocity.

    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.setExponential(
        1.0 * PARTICLE_VELOCITY, 1.0 * object->getAngle());
    relativeVelocity.add(1.0 * (20 - myRand(41)),
                         1.0 * (20 - myRand(41)));

    getFineVelocity() = object->getFineVelocity() - relativeVelocity;
}


//----------------------------------------------------------------------------
ExplosionParticle::ExplosionParticle(const ObjectBase *object)
        : ParticleBase(object)
{
    InitializeExplosionParticleConstVisitor v(this);
    object->accept(&v);
}

//----------------------------------------------------------------------------
ExplosionParticle::~ExplosionParticle()
{
}


//----------------------------------------------------------------------------
FountainParticle::FountainParticle(const ParticleFountainBase *f)
        : ParticleBase(f)
{
    setLifeCount(
        f->getParticleLifeTime()*SDLFrameRate::getFrameRate() + myRand(5));
    setGradient(100 / f->getParticleLifeTime());
    setGravity();

    init();
}

//----------------------------------------------------------------------------
FountainParticle::~FountainParticle()
{
}

//----------------------------------------------------------------------------
void FountainParticle::init()
{
    const ParticleFountainBase *f =
        static_cast<const ParticleFountainBase*>(getCreator());

    // Initialize the position.

    SDL_Rect position = f->getPosition();
    position.x += 6;
    position.y += 6;
    position.w = 4;
    position.h = 4;

    setPosition(position.x + myRand(position.w),
                position.y + myRand(position.h));


    // Initialize the velocity.

    Sint32 scatter = f->getParticleScatter();
    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.set(
        1.0 * (scatter - myRand(2*scatter+1)),
        1.0 * (scatter - myRand(2*scatter+1)));

    getFineVelocity() = f->getParticleInitialVelocity() + relativeVelocity;
}



//============================================================================
// Implementation of Projectiles.h
//============================================================================

//----------------------------------------------------------------------------
ProjectileBase::ProjectileBase(const ObjectBase *creator)
        : MovingObjectBase(creator)
{
    setSurface(ProjectileSurfaces::getInstance()->getSurface());
}

//----------------------------------------------------------------------------
ProjectileBase::~ProjectileBase()
{
}

//----------------------------------------------------------------------------
void ProjectileBase::update()
{
    MATH_TOOLS::Vector relPosition = getFineVelocity();
    relPosition /= 1.0 * SDLFrameRate::getFrameRate();

    getFinePosition() += relPosition;

    updatePosition();
}

//----------------------------------------------------------------------------
void ProjectileBase::createExplosionParticles()
{
    unsigned numParticles = 5;
    for (unsigned i=0; i<numParticles; i++)
    {
        GameInterface::getInstance()->addParticleToPlayGround(
            new ExplosionParticle(this));
    }
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(ProjectileBase);


//----------------------------------------------------------------------------
ShipProjectile::ShipProjectile(Ship *shooter) 
        : ProjectileBase(shooter)
{
    // Calculate the ship's rotation point.
    const SDL_Rect &shipPosition = shooter->getPosition();
    Sint16 xm, ym;
    SDL_TOOLS::getCentre(shipPosition, xm, ym);

    // When rotating, the tip of the ship moves in a circle with a radius,
    // we are calculating here.
    // To keep sure, the projectile won't touch the ship initially,
    // we add the height of the projectile's surface.
    Uint16 r =
        ym - (shipPosition.y + shooter->getTipOffset()) + getPosition().h;

    setPosition(
        (Uint16)(xm - getPosition().w/2 + r*SinusTable::sin(shooter->getAngle())),
        (Uint16)(ym - getPosition().h/2 + r*SinusTable::cos(shooter->getAngle())));

    // Calculate the projectile's velocity.
    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.setExponential(
        1.0 * PROJECTILE_VELOCITY, 1.0 * shooter->getAngle());

    // Add the ship's current velocity to the projectile's velocity.
    getFineVelocity() = shooter->getFineVelocity() + relativeVelocity;
}

//----------------------------------------------------------------------------
ShipProjectile::~ShipProjectile()
{
}


//----------------------------------------------------------------------------
TurretProjectile::TurretProjectile(TurretBase *turret,
                                   const TurretBarrel *barrel)
        : ProjectileBase(turret)
{
    // Set the initial position.
    Sint16 x, y;
    turret->getTipPosition(x, y);
    x -= getPosition().w / 2;
    y -= getPosition().h / 2;

    setPosition(x, y);

    // Set the initial velocity.
    getFineVelocity().setExponential(
        1.0 * barrel->getProjectileSpeed(),
        1.0 * (turret->getCenterAngle() + barrel->getAngle()));
}

//----------------------------------------------------------------------------
TurretProjectile::~TurretProjectile()
{
}


//----------------------------------------------------------------------------
MissileProjectile::MissileProjectile(Missile *missile, int angle)
        : ProjectileBase(missile->getCreator())
{
    // Set the initial position.
    Sint16 x, y;
    SDL_TOOLS::getCentre(missile->getPosition(), x, y);
    x -= getPosition().w / 2;
    y -= getPosition().h / 2;

    setPosition(x, y);

    // Calculate the projectile's velocity.
    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.setExponential(
        0.25 * PROJECTILE_VELOCITY, 1.0 * angle);

    // Add the missile's current velocity to the projectile's velocity.
    getFineVelocity() = missile->getFineVelocity() + relativeVelocity;
}

//----------------------------------------------------------------------------
MissileProjectile::~MissileProjectile()
{
}


//----------------------------------------------------------------------------
TankProjectile::TankProjectile(Tank *tank)
        : ProjectileBase(tank)
{
    // Set the initial position.
    Sint16 x, y;
    SDL_TOOLS::getCentre(tank->getPosition(), x, y);
    x -= getPosition().w / 2;
    y -= getPosition().h / 2 + 2;

    x += (Sint16)trunc(9 * SinusTable::sin(tank->getAngle()));
    y += (Sint16)trunc(5 * SinusTable::cos(tank->getAngle()));

    setPosition(x, y);

    // Set the initial velocity.
    getFineVelocity().setExponential(
        1.0 * tank->getProjectileSpeed(), 1.0 * tank->getAngle());
}

//----------------------------------------------------------------------------
TankProjectile::~TankProjectile()
{
}


//----------------------------------------------------------------------------
GrenadeProjectile::GrenadeProjectile(GrenadeBase *grenade, int angle)
        : ProjectileBase(grenade->getCreator())
{
    // Set the initial position.
    Sint16 x, y;
    SDL_TOOLS::getCentre(grenade->getPosition(), x, y);
    x -= getPosition().w / 2;
    y -= getPosition().h / 2;

    setPosition(x, y);

    // Calculate the projectile's velocity.
    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.setExponential(
        0.25 * PROJECTILE_VELOCITY, 1.0 * angle);

    // Add the grenade's current velocity to the projectile's velocity.
    getFineVelocity() = grenade->getFineVelocity() + relativeVelocity;
}

//----------------------------------------------------------------------------
GrenadeProjectile::~GrenadeProjectile()
{
}



//============================================================================
// Implementation of Grenade.h
//============================================================================

//----------------------------------------------------------------------------
GrenadeBase::NoWarheadStrategy GrenadeBase::NoWarheadStrategy::sm_instance;
GrenadeBase::StarBurstWarheadStrategy GrenadeBase::StarBurstWarheadStrategy::sm_instance;

//----------------------------------------------------------------------------
GrenadeBase::WarheadStrategy *GrenadeBase::WarheadStrategy::getStrategy(
    EnumWarheadStrategy strategy)
{
    switch (strategy)
    {
    case WH_NONE:
        return NoWarheadStrategy::getInstance();
    case WH_STARBURST:
        return StarBurstWarheadStrategy::getInstance();
    }

    return NULL;
}

//----------------------------------------------------------------------------
void GrenadeBase::NoWarheadStrategy::onCreateExplosionParticles(GrenadeBase *missile)
{
}

//----------------------------------------------------------------------------
void GrenadeBase::StarBurstWarheadStrategy::onCreateExplosionParticles(GrenadeBase *grenade)
{
    // If the grenade crashed, no projectiles shall be created.

    if (grenade->isExplode())
    {
        for (int angle=0; angle<360; angle+=45)
        {
            GameInterface::getInstance()->addObjectToPlayGround(
                new GrenadeProjectile(grenade, angle));
        }
    }
}


//----------------------------------------------------------------------------
GrenadeBase::GrenadeBase(EnumWarheadStrategy warhead,
                         const ObjectBase *creator)
        : MovingObjectBase(creator)
{
    m_lifeCount = 0;
    m_explode = false;

    setSurface(GrenadeSurfaces::getInstance()->getSurface());

    m_warheadStrategy = WarheadStrategy::getStrategy(warhead);
}

//----------------------------------------------------------------------------
GrenadeBase::~GrenadeBase()
{
    m_warheadStrategy = NULL;
}

//----------------------------------------------------------------------------
void GrenadeBase::update()
{
    GameInterface *gameInterface = GameInterface::getInstance();

    // The old velocity values are needed.
    MATH_TOOLS::Vector oldFineVelocity = getFineVelocity();

    MATH_TOOLS::Vector fineAccel(
        1.0 * gameInterface->getXGravity(getPosition()),
        1.0 * gameInterface->getYGravity(getPosition()));

    // No friction is calculated for particles.
    fineAccel /= 1.0 * SDLFrameRate::getFrameRate();
    getFineVelocity() += fineAccel;

    // Use the average value of the old and new velocity.
    oldFineVelocity += getFineVelocity();
    oldFineVelocity /= 2.0 * SDLFrameRate::getFrameRate();
    getFinePosition() += oldFineVelocity;

    updatePosition();

    if (--m_lifeCount == 0)
    {
        m_explode = true;
    }
}

//----------------------------------------------------------------------------
void GrenadeBase::createExplosionParticles()
{
    unsigned numParticles = 15;
    for (unsigned i=0; i<numParticles; i++)
    {
        GameInterface::getInstance()->addParticleToPlayGround(
            new ExplosionParticle(this));
    }

    m_warheadStrategy->onCreateExplosionParticles(this);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(GrenadeBase);


//----------------------------------------------------------------------------
MortarGrenade::MortarGrenade(MortarBase *mortar, const MortarBarrel *barrel)
        : GrenadeBase(barrel->getGrenadeWarheadStrategy(), mortar)
{
    // Set the initial position.
    Sint16 x, y;
    mortar->getTipPosition(x, y);
    x -= getPosition().w / 2;
    y -= getPosition().h / 2;

    setPosition(x, y);

    // Set the initial velocity.
    getFineVelocity().setExponential(
        1.0 * barrel->getGrenadeSpeed(),
        1.0 * (mortar->getCenterAngle() + barrel->getAngle()));

    setLifeCount(
        barrel->getGrenadeExploderDelay() * SDLFrameRate::getFrameRate() / 10);
}

//----------------------------------------------------------------------------
MortarGrenade::~MortarGrenade()
{
}
