#ifndef OBJECTREPOSITORY_H
#define OBJECTREPOSITORY_H

#include <list>
#include <map>

#include "Tools.h"
#include "ObjectVisitors.h"

//----------------------------------------------------------------------------
/**
 * This class is used to store all objects of a level
 * using their object ids as key.
 * An object id may have the following value:
 * <ul>
 *   <li>0: Reserved as "uninitialized" value.
 *          No fully initialized object must have a zero object id.</li>
 *   <li>1-1023: Reserved manually assigned values for objects defined
 *               in the level file (e.g. crates, turrets, platforms).</li>
 *   <li>1024-: Reserved dynamically assigned values for objects
 *              defined in the level file with omitted object id
 *              (e.g. black holes, platforms not relevant for the
 *              game flow) or objects that are created in-game
 *              (e.g. projectiles, particles).</li>
 * </ul>
 */
class ObjectRepository
{
    SINGLETON_OBJECT(ObjectRepository);

    typedef std::map<unsigned, ObjectBase*> Container;

    //------------------------------------------------------------------------
    /**
     * This visitor is used to insert an object into
     * m_objectList or m_particleList, dependend on its type.
     */
    class ShowObjectVisitor : public ObjectVisitor
    {
      public:
        //--------------------------------------------------------------------
        ShowObjectVisitor(ObjectRepository *r) : m_objectRepository(r) {}
        ~ShowObjectVisitor() { m_objectRepository = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(Barrier *b);
        void do_visit(BlackHole *b);
        void do_visit(Crate *c);
        void do_visit(GrenadeBase *g);
        void do_visit(Grinder *g);
        void do_visit(MagnetBase *m);
        void do_visit(Missile *m);
        void do_visit(MortarBase *m);
        void do_visit(ParticleBase *p);
        void do_visit(ParticleFountainBase *f);
        void do_visit(Platform *p);
        void do_visit(ProjectileBase *p);
        void do_visit(SAMBatteryBase *s);
        void do_visit(Ship *s);
        void do_visit(SwitchBase *s);
        void do_visit(Tank *t);
        void do_visit(TurretBase *t);

        //--------------------------------------------------------------------
        void do_do_visit(ObjectBase *o);
        
        //--------------------------------------------------------------------
        ObjectRepository *m_objectRepository;
    };

  public:
    //------------------------------------------------------------------------
    struct ObjectEntry
    {
        /// The object itself.
        ObjectBase *object;

        /// Set to true, if the object shall be deleted when removing.
        bool autoDelete;
    };

    typedef std::list<ObjectEntry> ObjectList;

    typedef std::list<ParticleBase*> ParticleList;


    //------------------------------------------------------------------------
    ObjectRepository();
    ~ObjectRepository();

    //------------------------------------------------------------------------
    unsigned getDynamicObjectId() const;

    //------------------------------------------------------------------------
    inline size_t size() const
    {
        return m_id2Object.size();
    }

    //------------------------------------------------------------------------
    /**
     * @param o A pointer to the object to add.
     */
    void addObject(ObjectBase *o);

    /**
     * @param o A pointer to the object (existing in m_id2object) to show.
     */
    void showObject(ObjectBase *o);

    /**
     * @param id The object id to find the object in m_id2object to show.
     */
    void showObjectId(const unsigned id);

    //------------------------------------------------------------------------
    /**
     * Sets the toRemove and autoDelete flag of the matching object
     * in m_objectList.
     *
     * @param o A pointer to the object to mark for removal.
     * @param autoDelete true, if the object shall be deleted, else false.
     */
    void markObjectToRemove(ObjectBase *o, bool autoDelete = true);

    /**
     * Removes all objects that were marked for removal.
     * Further they will be deleted, if their autoDelete flag is set.
     */
    void removeObjectsToRemove();

    //------------------------------------------------------------------------
    void clear();

    //------------------------------------------------------------------------
    bool doesExist(const unsigned id) const;

    //------------------------------------------------------------------------
    ObjectBase *getObject(const unsigned id);
    const ObjectBase *getObject(const unsigned id) const;

    //------------------------------------------------------------------------
    void acceptParticles(ObjectVisitor &v);
    void acceptParticles(ObjectConstVisitor &v) const;
    
    //------------------------------------------------------------------------
    void acceptObjects(ObjectVisitor &v);
    void acceptObjects(ObjectConstVisitor &v) const;

    //------------------------------------------------------------------------
    inline void acceptAll(ObjectVisitor &v)
    {
        acceptObjects(v);
        acceptParticles(v);
    }

    inline void acceptAll(ObjectConstVisitor &v) const
    {
        acceptObjects(v);
        acceptParticles(v);
    }

  private:

    //------------------------------------------------------------------------
    void do_showObject(ObjectBase *o);


    //------------------------------------------------------------------------
    mutable unsigned m_dynamicObjectIdCounter;

    /**
     * m_id2Object[id] is a pointer to the ObjectBase with the given id.
     */
    Container m_id2Object;

    /**
     * A list of all objects (except for particles) visible on the playground.
     */
    ObjectList m_objectList;

    /**
     * A list of all particles visible on the playground.
     * They are stored in an own list for a better performance
     * in the CollisionDetectionVisitor.
     */
    ParticleList m_particleList;
};

#endif //OBJECTREPOSITORY_H
