#include <cassert>
#include <sstream>

#include "Exception.h"

#include "AllObjects.h"
#include "ObjectRepository.h"


//----------------------------------------------------------------------------
ObjectRepository ObjectRepository::sm_instance;


//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(Barrier *b)
{
    do_do_visit(b);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(BlackHole *b)
{
    do_do_visit(b);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(Crate *c)
{
    do_do_visit(c);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(GrenadeBase *g)
{
    do_do_visit(g);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(Grinder *g)
{
    do_do_visit(g);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(MagnetBase *m)
{
    do_do_visit(m);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(Missile *m)
{
    do_do_visit(m);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(MortarBase *m)
{
    do_do_visit(m);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(ParticleBase *p)
{
    m_objectRepository->m_particleList.push_back(p);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(ParticleFountainBase *f)
{
    do_do_visit(f);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(Platform *p)
{
    do_do_visit(p);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(ProjectileBase *p)
{
    do_do_visit(p);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(SAMBatteryBase *s)
{
    do_do_visit(s);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(Ship *s)
{
    do_do_visit(s);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(SwitchBase *s)
{
    do_do_visit(s);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(Tank *t)
{
    do_do_visit(t);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_visit(TurretBase *t)
{
    do_do_visit(t);
}

//----------------------------------------------------------------------------
void ObjectRepository::ShowObjectVisitor::do_do_visit(ObjectBase *o)
{
    ObjectEntry e = { o, true };
    m_objectRepository->m_objectList.push_back(e);
}


//----------------------------------------------------------------------------
ObjectRepository::ObjectRepository()
{
    clear();
}

//----------------------------------------------------------------------------
ObjectRepository::~ObjectRepository()
{
    clear();
}

//----------------------------------------------------------------------------
unsigned ObjectRepository::getDynamicObjectId() const
{
    return m_dynamicObjectIdCounter++;
}

//----------------------------------------------------------------------------
void ObjectRepository::addObject(ObjectBase *o)
{
    assert(o != NULL);
    assert(o->getId() != 0);

    if (doesExist(o->getId()))
    {
        // Not an assert(), because a duplicate id might be given
        // via the XML level file.

        std::ostringstream s;
        s << "An object with the id '" << o->getId()
          << "' already exists in the object repository";
        throw Exception(s.str());
    }

    m_id2Object[o->getId()] = o;

    if (!o->isHidden())
    {
        do_showObject(o);
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::showObject(ObjectBase *o)
{
    assert(o != NULL);
    assert(doesExist(o->getId()));

    if (o->isHidden())
    {
        o->setHidden(false);
        do_showObject(o);
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::showObjectId(const unsigned id)
{
    Container::const_iterator iter = m_id2Object.find(id);
    assert(iter != m_id2Object.end());

    showObject(iter->second);
}

//----------------------------------------------------------------------------
void ObjectRepository::do_showObject(ObjectBase *o)
{
    ShowObjectVisitor v(this);
    o->accept(v);
}


//----------------------------------------------------------------------------
void ObjectRepository::markObjectToRemove(ObjectBase *o, bool autoDelete)
{
    for (ObjectList::iterator
             iter = m_objectList.begin(),
             end = m_objectList.end();
         iter != end; ++iter)
    {
        if (iter->object == o)
        {
            o->setHidden(true);
            iter->autoDelete = autoDelete;
            break;
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::removeObjectsToRemove()
{
    do
    {
        ObjectList::iterator iter = m_objectList.begin();
        ObjectList::iterator end = m_objectList.end();
        while (iter != end)
        {
            if (iter->object->isHidden())
            {
                if (iter->autoDelete)
                {
                    m_id2Object.erase(iter->object->getId());
                    delete iter->object;
                }
                iter = m_objectList.erase(iter);
            }
            else
            {
                ++iter;
            }
        }
    }
    while (0);

    do
    {
        ParticleList::iterator iter = m_particleList.begin();
        ParticleList::iterator end = m_particleList.end();
        while (iter != end)
        {
            if ((*iter)->isHidden())
            {
                m_id2Object.erase((*iter)->getId());
                delete *iter;
                iter = m_particleList.erase(iter);
            }
            else
            {
                ++iter;
            }
        }
    }
    while (0);
}

//----------------------------------------------------------------------------
void ObjectRepository::clear()
{
    m_objectList.clear();
    m_particleList.clear();

    for (Container::iterator
             iter = m_id2Object.begin(),
             end = m_id2Object.end();
         iter != end; ++iter)
    {
        delete iter->second;
    }
    m_id2Object.clear();

    m_dynamicObjectIdCounter = 1024;
}

//----------------------------------------------------------------------------
bool ObjectRepository::doesExist(const unsigned id) const
{
    return m_id2Object.find(id) != m_id2Object.end();
}

//----------------------------------------------------------------------------
ObjectBase *ObjectRepository::getObject(const unsigned id)
{
    Container::iterator iter = m_id2Object.find(id);
    return iter != m_id2Object.end() ? iter->second : NULL;
}

//----------------------------------------------------------------------------
const ObjectBase *ObjectRepository::getObject(const unsigned id) const
{
    Container::const_iterator iter = m_id2Object.find(id);
    return iter != m_id2Object.end() ? iter->second : NULL;
}

//----------------------------------------------------------------------------
void ObjectRepository::acceptParticles(ObjectVisitor &v)
{
    for (ParticleList::iterator
             iter = m_particleList.begin(),
             end = m_particleList.end();
         iter != end; ++iter)
    {
        // Only visit particles, that are not to be removed!
        if (!(*iter)->isHidden())
        {
            (*iter)->accept(v);
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::acceptParticles(ObjectConstVisitor &v) const
{
    for (ParticleList::const_iterator
             iter = m_particleList.begin(),
             end = m_particleList.end();
         iter != end; ++iter)
    {
        // Only visit particles, that are not to be removed!
        if (!(*iter)->isHidden())
        {
            (*iter)->accept(v);
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::acceptObjects(ObjectVisitor &v)
{
    for (ObjectList::iterator
             iter = m_objectList.begin(),
             end = m_objectList.end();
         iter != end && v.continueForLoopInVisitObjectList(); ++iter)
    {
        // Only visit objects, that are not to be removed!
        if (!iter->object->isHidden())
        {
            iter->object->accept(v);
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::acceptObjects(ObjectConstVisitor &v) const
{
    for (ObjectList::const_iterator
             iter = m_objectList.begin(),
             end = m_objectList.end();
         iter != end && v.continueForLoopInVisitObjectList(); ++iter)
    {
        // Only visit objects, that are not to be removed!
        if (!iter->object->isHidden())
        {
            iter->object->accept(v);
        }
    }
}
