/*******************************************************************************
 * Copyright (c) MOBAC developers
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package mobac.program.tiledatawriter;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.OutputStream;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageOutputStream;

import org.apache.log4j.Logger;

import mobac.program.interfaces.TileImageDataWriterBuilder;
import mobac.program.model.TileImageType;

public class TileImageJpegDataWriterBuilder implements TileImageDataWriterBuilder {

	protected static final Logger log = Logger.getLogger(TileImageJpegDataWriterBuilder.class);

	protected ImageWriter jpegImageWriter = null;

	protected ImageWriteParam iwp = null;

	protected float jpegCompressionLevel;

	/**
	 * 
	 * @param jpegCompressionLevel
	 *            a float between 0 and 1; 1 specifies minimum compression and maximum quality
	 */
	public TileImageJpegDataWriterBuilder(double jpegCompressionLevel) {
		this((float) jpegCompressionLevel);
	}

	public TileImageJpegDataWriterBuilder(float jpegCompressionLevel) {
		this.jpegCompressionLevel = (float) jpegCompressionLevel;
	}

	public TileImageJpegDataWriterBuilder(TileImageJpegDataWriterBuilder jpegWriter) {
		this(jpegWriter.getJpegCompressionLevel());
	}

	public TileImageJpegDataWriter build() {
		return new TileImageJpegDataWriter(jpegCompressionLevel);
	}

	public void setJpegCompressionLevel(float jpegCompressionLevel) {
		this.jpegCompressionLevel = jpegCompressionLevel;
		iwp.setCompressionQuality(jpegCompressionLevel);
	}

	public float getJpegCompressionLevel() {
		return jpegCompressionLevel;
	}

	public void processImage(BufferedImage image, OutputStream out) throws IOException {

		if (image.getColorModel().hasAlpha()) {
			// Javas JPEG writes has a bug when the image has alpha transparency
			// see http://stackoverflow.com/questions/4386446/problem-using-imageio-write-jpg-file

			BufferedImage imageRGB = new BufferedImage(image.getWidth(), image.getHeight(), BufferedImage.TYPE_INT_RGB);
			Graphics2D g = imageRGB.createGraphics();
			g.drawImage(image, null, 0, 0);
			g.dispose();
			image = imageRGB;
		}

		ImageOutputStream imageOut = ImageIO.createImageOutputStream(out);
		jpegImageWriter.setOutput(imageOut);
		IIOImage ioImage = new IIOImage(image, null, null);
		jpegImageWriter.write(null, ioImage, iwp);
	}

	public void dispose() {
		jpegImageWriter.dispose();
		jpegImageWriter = null;
	}

	public TileImageType getType() {
		return TileImageType.JPG;
	}

}
