/*
 * This plugin based on the example dummy plugin from the Maemo
 * maemo-af-desktop repository.  Original copyright declaration:
 * Lucas Rocha <lucas.rocha@nokia.com>
 * Copyright 2006 Nokia Corporation.
 *
 * Modifcations transform the example to an embedded mozilla home applet:
 * Bob Spencer <bob.spencer@intel.com>
 * Copyright 2007 Intel Corporation
 *
 * Contributors:
 * Michael Frey <michael.frey@pepper.com>
 * Rusty Lynch <rusty.lynch@intel.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <errno.h>
#include <sys/resource.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <glib-object.h>

#include "mobile-basic-home-plugin.h"
#include "gtkmozembed.h"
#include "dom/nsIDOMKeyEvent.h"
#include "nsIDOMMouseEvent.h"
#include "nsIDOMUIEvent.h"
#include "prenv.h"

#include <libhildondesktop/libhildondesktop.h>
#include <libhildonwm/hd-wm.h>

#define LOG_FILE
#ifdef LOG_FILE
#include <stdlib.h>
#include <stdio.h>
#define DBG_LOG trace
#else
#define DBG_LOG 
#endif

#define PROFILE_DIR "/usr/share/mobile-basic-flash"
#define HOME_HTML   "home.html"

#define PROFILE_NAME "mozapplet"
#define MARQUEE_HEIGHT 52
#define DESKTOP_FILE_SUFFIX ".desktop"
#define DESKTOP_ENTRY_GROUP "Mobile Entry"

#define DESKTOP_DIR    "/usr/share/mobile-basic-flash/applications"
//TBD: use theme pattern to get fonts
#define ICON_DIR       "/usr/share/mobile-basic-flash/icons"

//TBD: use gconf header to replace all gconf defines
#define BKGD_PATH             "/desktop/gnome/background"
#define BKGD_FILE             BKGD_PATH "/picture_filename"
#define BKGD_PIC_OPTIONS      BKGD_PATH "/picture_options"
#define BKGD_PRIMARY_COLOR    BKGD_PATH "/primary_color"

#define INTERFACE_PATH        "/desktop/gnome/interface"
#define GTK_THEME             INTERFACE_PATH "/gtk_theme"
#define ICON_THEME            INTERFACE_PATH "/icon_theme"
#define FONT_NAME             INTERFACE_PATH "/font_name"



HD_DEFINE_PLUGIN (MobileBasicHomePlugin, mobile_basic_home_plugin, HILDON_DESKTOP_TYPE_HOME_ITEM);

typedef struct {
	int index;
	gchar *name;
	gchar *icon;
	gchar *exec;
	gchar *service;
	gchar *filename;
} application_entry_t;

typedef struct {
	GtkWidget *container;
	GList *app_list;
	GConfClient *client;
} plugin_context_t;


#ifdef LOG_FILE
static void trace (const gchar * msg) {
  FILE *fp;
  if (fp = fopen("/tmp/mobile_trace.log", "a")) {
	  fprintf (fp, "%s\n", msg);
  }
  fclose (fp);
}
#endif

static void update_background(plugin_context_t *context) {
	gchar *pic, *opt, *color;

	DBG_LOG ("update_background");

	pic = gconf_client_get_string (context->client,BKGD_FILE,NULL);
	opt = gconf_client_get_string (context->client,BKGD_PIC_OPTIONS,NULL);
	color = gconf_client_get_string (context->client,BKGD_PRIMARY_COLOR,NULL);

	//if a picture is selected (not "no Wallpaper"), make sure it is valid
	if ((pic != NULL) &&
	    !g_utf8_validate(pic, -1, NULL) || !g_file_test(pic, G_FILE_TEST_EXISTS)) {
		pic = g_filename_from_utf8 (pic, -1, NULL,NULL, NULL);
	}

	gchar *url = g_strdup_printf("javascript:setBackground(['%s', '%s', '%s'])", 
				     pic, opt, color);
	gtk_moz_embed_load_url(GTK_MOZ_EMBED(context->container), url);
	g_free(url);

	if (pic) {
		g_free (pic);
	}
       	if (opt) {
		g_free (opt);
	}
	if (color) {
		g_free (color);
	}
}


static void update_theme(plugin_context_t *context) {

	DBG_LOG ("interface_changed");

	gchar *theme, *icon_theme, *font;
	theme =      gconf_client_get_string (context->client, GTK_THEME, NULL);
	icon_theme = gconf_client_get_string (context->client, ICON_THEME, NULL);
	font =       gconf_client_get_string (context->client, FONT_NAME, NULL);

	//a little hack -- Human is default now, but it doesn't exist. 
	if (!theme || (!strcmp (theme, "Human"))) {
		theme = g_strdup ("mobilebasic");
	}
	if (!icon_theme) {
		icon_theme = g_strdup ("hicolor");
        }
	gchar *url = g_strdup_printf("javascript:setThemeValues(['%s', '%s', '%s'])", 
				     theme, icon_theme, font);
	gtk_moz_embed_load_url(GTK_MOZ_EMBED(context->container), url);
	g_free(url);

	if (theme) {
		g_free (theme);
	}
       	if (icon_theme) {
		g_free (icon_theme);
	}
	if (font) {
		g_free (font);
	}
}


static void dump_application(gpointer data, gpointer ignore)
{
	application_entry_t *item = (application_entry_t *)data;
	g_print("app[%i] = %s\n", item->index, item->exec);
}

static void add_application(gpointer data, gpointer payload)
{
	application_entry_t *item = (application_entry_t *)data;
	plugin_context_t *c = (plugin_context_t *)payload;

	gchar *url = g_strdup_printf("javascript:addApp([%i, '%s', '%s'])",
				     item->index, item->name, item->icon);
	gtk_moz_embed_load_url(GTK_MOZ_EMBED(c->container), url);
	g_free(url);
}

static gboolean init_program(plugin_context_t *c)
{
	gchar *url;
	
	update_background(c);
	update_theme(c);
	url = g_strdup_printf("javascript:launchDesktop()"); 
	gtk_moz_embed_load_url(GTK_MOZ_EMBED(c->container), url);
	g_free(url);

	return FALSE;
}

static void start_app(const gchar *app)
{
	gchar *program = NULL;
	GError *error = NULL;
	gint argc;
	gchar **argv;
	GPid child_pid;
	gchar *space;

	space = strchr(app, ' ');
	if (space) {
		gchar *cmd = g_strdup (app);
		cmd[space - app] = 0;
		
		gchar *exc = g_find_program_in_path (cmd);
		
		program = g_strconcat (exc, space, NULL);
		
		g_free (exc);
		g_free (cmd);
	} else {
		program = g_find_program_in_path (app);
	}

	if (!program) {
		g_warning("Attempt to exec invalid entry: %s", app);
		return;
	}

	if (g_shell_parse_argv (program, &argc, &argv, &error)) {
		g_spawn_async (
			/* Child's current working directory,
			   or NULL to inherit parent's */
			NULL,
			/* Child's argument vector. [0] is the path of
			   the program to execute */
			argv,
			/* Child's environment, or NULL to inherit
			   parent's */
			NULL,
			/* Flags from GSpawnFlags */
			(GSpawnFlags)0,
			/* Function to run in the child just before
			   exec() */
			NULL,
			/* User data for child_setup */
			NULL,
			/* Return location for child process ID 
			   or NULL */
			&child_pid,
			/* Return location for error */
			&error);
	}
      
	if (error) {
		g_warning ("Others_menu_activate_app: failed to execute %s: %s.",
			   program, error->message);
		g_clear_error (&error);

	} else {
		int priority;
		errno = 0;
		gchar *oom_filename;
		int fd;
		
		/* If the child process inherited desktop's high 
		 * priority, give child default priority */
		priority = getpriority (PRIO_PROCESS, child_pid);
		
		if (!errno && priority < 0) {
			setpriority (PRIO_PROCESS, child_pid, 0);
		}
		
		/* Unprotect from OOM */
		oom_filename = g_strdup_printf ("/proc/%i/oom_adj",
						child_pid);
		fd = open (oom_filename, O_WRONLY);
		g_free (oom_filename);
		
		if (fd >= 0) {
			write (fd, "0", sizeof (char));
			close (fd);
		}
	}
}

static void start_app_from_index(plugin_context_t *c, int index) 
{
	GList *l;
	application_entry_t *item = NULL;

	g_print("Starting app %i\n", index);
	
	l = g_list_nth(c->app_list, index);
	if (!l) {
		g_warning("start_app::Invalid application index");
		g_list_foreach(c->app_list, dump_application, c->container);
		return;
	}
	item = (application_entry_t *)l->data;

	if (item->service) {
		g_print("Kicking the %s service\n", item->service);
		hd_wm_top_service (item->service);
		return;
	}
	start_app(item->exec);
}

/////////////////////////////////////////////////////////
// After the container HTML file has completed loading,
// loop through each application in the app_list and
// register the application appropriate app data with 
// a JavaScript funciton running in the HTML content.
//
// Once all application data has been registered, then
// trigger the flash content to initialize the desktop
void net_stop_cb (GtkMozEmbed *embed, gpointer data)
{
	plugin_context_t *context = (plugin_context_t *)data;
	g_list_foreach(context->app_list, add_application, data);
	g_idle_add((GSourceFunc)init_program, data);
}

/////////////////////////////////////////////////////////
// The Flash content (via the HTML container) will trigger
// actions in the native container by updating the HTML
// status field.  To do this we implement a very primitive
// text based protocol consisting of:
// "command:arg1:arg2:..."
//
// Current commands:
// run:index    Start the application associated with index
// log:msg      Write a string to the log
void js_status_cb (GtkMozEmbed *embed, gpointer data)
{
	char *message;
	gchar** tokens;
	plugin_context_t *c = (plugin_context_t *)data;

	message = gtk_moz_embed_get_js_status(embed);
	if (message) {
		tokens = g_strsplit(message, ":", 2); //(examples: "run_id:5" "run_app:myspace /mydocs/photos")
		if (tokens[0] == NULL || tokens[1] == NULL) {
			return;
		}
    		if (!g_ascii_strncasecmp(tokens[0],"run", 3) ||   //run: deprecated, use run_id or run_app
		    !g_ascii_strncasecmp(tokens[0],"run_id", 6)) {
			start_app_from_index(c, atoi(tokens[1]));
    		} else if (!g_ascii_strncasecmp(tokens[0],"run_app", 6)) {
			start_app(tokens[1]);
		} else if (!g_ascii_strncasecmp(tokens[0],"log", 3)) {
			g_print("LOG: %s\n", tokens[1]);
		}
		g_strfreev(tokens);
		g_free(message);
	}
}


gint compare_items(gconstpointer a, gconstpointer b)
{
	gchar *f1 = ((application_entry_t *)a)->filename;
	gchar *f2 = ((application_entry_t *)b)->filename;

	return g_strcasecmp(f1, f2);
}

/////////////////////////////////////////////////////////
// Read the system desktop entries and build a list of
// applications that the Flash content should expose in
// its interface.


//TBD:  Get list from /usr/share/applications/<pkg>  MobileId
static void build_app_list(plugin_context_t *c)
{
	DIR *dir_handle = NULL;
	struct dirent *d = NULL;
	struct stat buf;
	gchar *current_path = NULL;
	GKeyFile *key_file = NULL;
	GError *error = NULL;
	gchar *directory = DESKTOP_DIR;

	if ((dir_handle = opendir(directory)) == NULL) {
		g_warning("Error reading file '%s'\n", directory );
		return;
	}

	while (d = readdir(dir_handle)) {
		application_entry_t *i = NULL;

		current_path = g_build_filename(directory, d->d_name, NULL);

		if ((stat(current_path, &buf) == 0) &&
		    S_ISREG(buf.st_mode) &&
		    g_str_has_suffix(current_path, DESKTOP_FILE_SUFFIX)) {
			key_file = g_key_file_new();

			if ( g_key_file_load_from_file( key_file, current_path,
					G_KEY_FILE_NONE, &error ) == FALSE ) {

				g_warning("Error reading '%s': %s\n",
						current_path, error->message);
				g_error_free(error);
				error = NULL;
	                	g_free(current_path);
				continue;
			}

			i = (application_entry_t *)
				g_malloc0(sizeof(application_entry_t));

			i->name = g_key_file_get_string(key_file,
							DESKTOP_ENTRY_GROUP,
							DESKTOP_ENTRY_NAME_FIELD,
							NULL);
			if (!i->name) {
				g_key_file_free(key_file);
				g_free(current_path);
				g_free(i);
				g_warning("Missing name entry");
				continue;
			}

			gchar *t = g_key_file_get_string(key_file,
							 DESKTOP_ENTRY_GROUP,
							 DESKTOP_ENTRY_ICON_FIELD,
							 NULL);
			if (!t) {
				g_key_file_free(key_file);
				g_free(current_path);
				g_free(i->name);
				g_free(i);
				g_warning("Missing icon entry");
				continue;
			}
			i->icon = g_strdup_printf("%s/%s.png", ICON_DIR, t);
			g_free(t);

			i->service = g_key_file_get_string(key_file,
							   DESKTOP_ENTRY_GROUP,
							   DESKTOP_ENTRY_SERVICE_FIELD,
							   NULL);
			i->exec = g_key_file_get_string(key_file,
							DESKTOP_ENTRY_GROUP,
							DESKTOP_ENTRY_EXEC_FIELD,
							NULL);
			if (!i->exec && !i->service) {
				g_key_file_free(key_file);
				g_free(current_path);
				g_free(i->name);
				g_free(i->icon);
				g_free(i);
				g_warning("Missing both exec and service");
				continue;
			}				

			i->filename = g_strdup(d->d_name);
			c->app_list = g_list_append(c->app_list, i);
			g_key_file_free(key_file);
		}

		if (current_path)
			g_free(current_path);
	}

	c->app_list = g_list_sort(c->app_list, compare_items);
	for (int i=0; i<g_list_length(c->app_list); i++) {
		GList *l = g_list_nth(c->app_list, i);
		application_entry_t *item = (application_entry_t *)l->data;
		item->index = i;
	}
	
	closedir(dir_handle);
	g_free(d);
	return;
}


//gconf notification callback (registered in init below)
static void background_changed (GConfClient *client, guint id,
				GConfEntry *entry,
				plugin_context_t *c) {
	//ignoring input: update all background properites
	update_background(c);
}

//gconf notification callback (for theme and font)
static void gconf_interface_changed (GConfClient *client, guint id,
				GConfEntry *entry,
				plugin_context_t *c) {
	//ignoring input:  update relevent theme and interface properties
	update_theme(c);
}



static void mobile_basic_home_plugin_init (MobileBasicHomePlugin *home_plugin)
{
	DBG_LOG ("init");

	plugin_context_t *context = (plugin_context_t *)g_malloc0(
		sizeof(plugin_context_t));

	context->container = gtk_moz_embed_new();
	
	//setup context object
	context->client = gconf_client_get_default ();

	//create application list
	context->app_list = NULL;
	build_app_list(context);

	gtk_moz_embed_set_profile_path(PROFILE_DIR, PROFILE_NAME);
  
	//auto-size plugin to fill screen
	int scn_width = 800;
	int scn_height = 480;
	GdkScreen *screen = gtk_widget_get_screen ((GtkWidget*)home_plugin);
	if (screen != NULL) {
		scn_width = gdk_screen_get_width(screen);
		scn_height = gdk_screen_get_height(screen);
	}

	gtk_widget_set_size_request ((GtkWidget*)home_plugin, scn_width, 
				     scn_height-MARQUEE_HEIGHT);

	// load html file which auto-loads correct .swf file for screen 
	// and shows error if no flash plugin
	gchar *url = g_strdup_printf("%s/%s", PROFILE_DIR, HOME_HTML);
	gtk_moz_embed_load_url(GTK_MOZ_EMBED(context->container), url);
	g_free(url);

	DBG_LOG ("loaded main URL");

	gtk_widget_show_all(context->container);
	gtk_container_add (GTK_CONTAINER (home_plugin), context->container);
	gtk_signal_connect(GTK_OBJECT(context->container), "js_status",
			   GTK_SIGNAL_FUNC(js_status_cb), (gpointer)context);
	gtk_signal_connect(GTK_OBJECT(context->container), "net_stop",
			   GTK_SIGNAL_FUNC(net_stop_cb), (gpointer)context);


	gconf_client_add_dir (context->client, BKGD_PATH,
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_notify_add (context->client,
				 BKGD_FILE,
				 (GConfClientNotifyFunc) background_changed,
				 context, NULL, NULL);

	//notification when theme changes (send up)
	gconf_client_add_dir (context->client, INTERFACE_PATH,
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_notify_add (context->client,
				 INTERFACE_PATH,
				 (GConfClientNotifyFunc) gconf_interface_changed,
				 context, NULL, NULL);

	DBG_LOG ("init finished"); 
}

static void
mobile_basic_home_plugin_class_init (MobileBasicHomePluginClass *klass)
{
}

/* WTF? */
}}
