/*
 * Copyright (C) 2007 Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <ctype.h>
#include <glib.h>
#include <glib/gi18n.h>

#include <glibtop/fsusage.h>
#include <glibtop/mountlist.h>
#include <glibtop/mem.h>
#include <glibtop/sysinfo.h>

#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <math.h>
#include <errno.h>

#include <sys/utsname.h>
#include <gtk/gtk.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>

#include <string.h>
#include <sys/vfs.h>
#include <gdk/gdkx.h>
#include <X11/extensions/Xrandr.h>
#include <moko-finger-scroll.h>

#define X_PAD  5
#define Y_PAD  12
#define LOGO_W 92
#define LOGO_H 351
#define RADIUS 5

gchar*
byte_size_format (guint64 size)
{
    guint64 factor;
    const char* format;

        if (size < 1024) {
                return g_strdup_printf (dngettext(GETTEXT_PACKAGE, "%u byte", "%u bytes",(guint) size), (guint) size);
        } else {
                if (size < (1024*1024)) {
                  factor = 1024;
                  format = N_("%.1f KiB");
                } else if (size < (1024*1024*1024)) {
                  factor = (1024*1024);
                  format = N_("%.1f MiB");
                } else {
                  factor = (1024*1024*1024);
                  format = N_("%.1f GiB");
                }

                return g_strdup_printf(_(format), size / (double)factor);
        }
}

static gboolean
sysinfo_logo_expose (GtkWidget *widget,
		     GdkEventExpose *event,
		     gpointer data_ptr)
{
  cairo_t *cr;
  cairo_pattern_t *cp;

  cr = gdk_cairo_create(widget->window);

  cairo_translate(cr, event->area.x, event->area.y);

  cairo_move_to(cr, X_PAD + RADIUS, Y_PAD);
  cairo_line_to(cr, X_PAD + LOGO_W - RADIUS, Y_PAD);
  cairo_arc(cr, X_PAD + LOGO_W - RADIUS, Y_PAD + RADIUS, RADIUS, -0.5 * M_PI, 0);
  cairo_line_to(cr, X_PAD + LOGO_W, Y_PAD + LOGO_H - RADIUS);
  cairo_arc(cr, X_PAD + LOGO_W - RADIUS, Y_PAD + LOGO_H - RADIUS, RADIUS, 0, 0.5 * M_PI);
  cairo_line_to(cr, X_PAD + RADIUS, Y_PAD + LOGO_H);
  cairo_arc(cr, X_PAD + RADIUS, Y_PAD + LOGO_H - RADIUS, RADIUS, 0.5 * M_PI, -1.0 * M_PI);
  cairo_line_to(cr, X_PAD, Y_PAD + RADIUS);
  cairo_arc(cr,  X_PAD + RADIUS, Y_PAD + RADIUS, RADIUS, -1.0 * M_PI, -0.5 * M_PI);

  cp = cairo_pattern_create_linear(0, Y_PAD, 0, Y_PAD + LOGO_H);
  cairo_pattern_add_color_stop_rgba(cp, 0.0,
				    widget->style->base[GTK_STATE_SELECTED].red / 65535.0,
				    widget->style->base[GTK_STATE_SELECTED].green / 65535.0,
				    widget->style->base[GTK_STATE_SELECTED].blue / 65535.0,
				    1.0);
  cairo_pattern_add_color_stop_rgba(cp, 1.0,
				    widget->style->base[GTK_STATE_SELECTED].red / 65535.0,
				    widget->style->base[GTK_STATE_SELECTED].green / 65535.0,
				    widget->style->base[GTK_STATE_SELECTED].blue / 65535.0,
				    0.0);
  cairo_set_source(cr, cp);
  cairo_fill(cr);

  cairo_pattern_destroy(cp);
  cairo_destroy(cr);

  return FALSE;
}

static void
dialog_response (GtkWidget *widget, gint response_id)
{

	switch (response_id) {
	case GTK_RESPONSE_CLOSE:
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
	    gtk_widget_destroy(widget);
	    gtk_main_quit();
	    break;
	}
}

int main(int argc, char *argv[], char *env[])
{
    GtkWidget *dialog;
    GtkWidget *moko;
    GtkWidget *hbox, *vbox, *logo, *distro_frame;
    GtkWidget *distro_release_label, *hardware_frame;
    GtkWidget *hardware_table, *memory_label, *header, *alignment;
    GtkWidget *processor_label, *disk_space_frame;
    GtkWidget *disk_space_table, *disk_space_label;
    gchar *markup, *distro_name=NULL, *distro_codename=NULL;
    gchar *distro_release=NULL, *model = 0;
    char *p, buf[1000];
    guint i, j, tmp;
    guint64 free_space_bytes;
    const glibtop_sysinfo *info = glibtop_get_sysinfo();
    const char * const keys[] = { "model name", "cpu" };
    struct utsname name;
    glibtop_mem mem;
    glibtop_fsusage usage;
    glibtop_mountentry *entries;
    glibtop_mountlist mountlist;
    FILE *fp;
    GdkScreen *screen;
    GdkDisplay *display;
    int height, width;

    gtk_init(&argc, &argv);

    hbox = gtk_hbox_new(FALSE, 12);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 6);

    /* left-side logo */

    logo = gtk_image_new_from_file(MA_PIXMAPS_DIR "/side.png");
    gtk_misc_set_alignment(GTK_MISC(logo), 0.5, 0.0);
    gtk_misc_set_padding(GTK_MISC(logo), 5, 12);
    gtk_box_pack_start(GTK_BOX(hbox), logo, FALSE, FALSE, 0);

    g_signal_connect(G_OBJECT(logo), "expose-event",
		   G_CALLBACK(sysinfo_logo_expose), NULL);

    vbox = gtk_vbox_new(FALSE, 12);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

    // hostname

    uname(&name);

    markup = g_strdup_printf("<big><big><b><u>%s</u></b></big></big>",
	name.nodename);
    GtkWidget *hostname_frame = gtk_frame_new(markup);
    g_free(markup);
    gtk_frame_set_label_align(GTK_FRAME(hostname_frame), 0.0, 0.5);
    gtk_label_set_use_markup(
			   GTK_LABEL(gtk_frame_get_label_widget(GTK_FRAME(hostname_frame))),
			   TRUE
			   );
    gtk_frame_set_shadow_type(GTK_FRAME(hostname_frame), GTK_SHADOW_NONE);
    gtk_box_pack_start(GTK_BOX(vbox), hostname_frame, FALSE, FALSE, 0);

    /* distro section */

    distro_name = name.sysname;
    distro_codename = "";
    distro_release = name.release;
    if ((p = g_find_program_in_path("lsb_release")) != NULL) {
	g_free(p);
	if((fp = popen("lsb_release -irc", "r")) == NULL) {
	    perror("lsb_release -irc");
	} else {
	    for(i = 0; fgets(buf, 1000, fp) != NULL; i++)
	    {
		buf[strlen(buf)-1] = '\0';
		for(j=strlen(buf)-1; (j >= 0)&&(!isspace(buf[j])); j--);
		switch(i) {
		case 0:
		    distro_name = strdup(&buf[j+1]);
		    break;
		case 1:
		    distro_release = strdup(&buf[j+1]);
		    break;
		case 2:
		    distro_codename = strdup(&buf[j+1]);
		    break;
		}
	    }
	    pclose(fp);
	}
    }

    markup = g_strdup_printf("<big><big><b>%s</b></big></big>",
	distro_name);
    distro_frame = gtk_frame_new(markup);
    g_free(markup);
    gtk_frame_set_label_align(GTK_FRAME(distro_frame), 0.0, 0.5);
    gtk_label_set_use_markup(
			   GTK_LABEL(gtk_frame_get_label_widget(GTK_FRAME(distro_frame))),
			   TRUE
			   );
    gtk_frame_set_shadow_type(GTK_FRAME(distro_frame), GTK_SHADOW_NONE);
    gtk_box_pack_start(GTK_BOX(vbox), distro_frame, FALSE, FALSE, 0);

    alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 0, 0, 12, 0);
    gtk_container_add(GTK_CONTAINER(distro_frame), alignment);

    if (distro_codename != NULL)
	markup = g_strdup_printf(
			     _("Release %s (%s)"),
			     distro_release,
			     distro_codename);
    else
	markup = g_strdup_printf(
			     _("Release %s"),
			     distro_release);
    distro_release_label = gtk_label_new(markup);
    g_free(markup);
    gtk_misc_set_alignment(GTK_MISC(distro_release_label), 0.0, 0.5);
    gtk_misc_set_padding(GTK_MISC(distro_release_label), 6, 6);
    gtk_container_add(GTK_CONTAINER(alignment), distro_release_label);

    /* hardware section */

    glibtop_get_mem(&mem);

    markup = g_strdup_printf(_("<b>Hardware</b>"));
    hardware_frame = gtk_frame_new(markup);
    g_free(markup);
    gtk_frame_set_label_align(GTK_FRAME(hardware_frame), 0.0, 0.5);
    gtk_label_set_use_markup(
			   GTK_LABEL(gtk_frame_get_label_widget(GTK_FRAME(hardware_frame))),
			   TRUE
			   );
    gtk_frame_set_shadow_type(GTK_FRAME(hardware_frame), GTK_SHADOW_NONE);
    gtk_box_pack_start(GTK_BOX(vbox), hardware_frame, FALSE, FALSE, 0);

    alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 0, 0, 12, 0);
    gtk_container_add(GTK_CONTAINER(hardware_frame), alignment);

    hardware_table = gtk_table_new(1, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(hardware_table), 6);
    gtk_table_set_col_spacings(GTK_TABLE(hardware_table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(hardware_table), 6);
    gtk_container_add(GTK_CONTAINER(alignment), hardware_table);

    header = gtk_label_new(_("Memory:"));
    gtk_misc_set_alignment(GTK_MISC(header), 0.0, 0.5);
    gtk_table_attach(
		   GTK_TABLE(hardware_table), header,
		   0, 1, 0, 1,
		   GTK_FILL, GTK_FILL, 0, 0
		   );

    markup = byte_size_format(mem.total);
    memory_label = gtk_label_new(markup);
    g_free(markup);
    gtk_misc_set_alignment(GTK_MISC(memory_label), 0.0, 0.5);
    gtk_table_attach(
		   GTK_TABLE(hardware_table), memory_label,
		   1, 2, 0, 1,
		   GTK_FILL, GTK_FILL, 0, 0
		   );

    for (i = 0; i < info->ncpu; ++i) {
	if (info->ncpu > 1) {
	    markup = g_strdup_printf(_("Processor %d:"), i);
	    header = gtk_label_new(markup);
	    g_free(markup);
	}
	else
	    header = gtk_label_new(_("Processor:"));

	for (j = 0; !model && j != G_N_ELEMENTS(keys); ++j)
	    model = (gchar *)g_hash_table_lookup(info->cpuinfo[i].values,
		keys[j]);
	if (!model)
	    model = _("Unknown CPU model");

	gtk_misc_set_alignment(GTK_MISC(header), 0.0, 0.5);
	gtk_table_attach(
		     GTK_TABLE(hardware_table), header,
		     0, 1, 1 + i, 2 + i,
		     GTK_FILL, GTK_FILL, 0, 0
		     );

	processor_label = gtk_label_new(model);
	gtk_misc_set_alignment(GTK_MISC(processor_label), 0.0, 0.5);
	gtk_table_attach(
		     GTK_TABLE(hardware_table), processor_label,
		     1, 2, 1 + i, 2 + i,
		     GTK_FILL, GTK_FILL, 0, 0
		     );
    }

    /* disk space section */

    entries = glibtop_get_mountlist(&mountlist, 0);
    free_space_bytes = 0;
    for (i = 0; i != mountlist.number; ++i) {
	if (strstr(entries[i].devname, "/dev/") == NULL)
	  continue;
	if (strstr(entries[i].mountdir, "/media/") != NULL)
	  continue;
	glibtop_get_fsusage(&usage, entries[i].mountdir);
	free_space_bytes += usage.bavail * usage.block_size;
    }
    g_free(entries);
    if(free_space_bytes == 0) {
	if((fp = popen("df -l", "r")) == NULL) {
	    perror("df");
	} else {
	    for(i = 0; fgets(buf, 1000, fp) != NULL; i++)
	    {
		for(j = 0, p = strtok(buf, " \t\n"); p != NULL; p = strtok(NULL, " \t\n"), j++)
		{
		    if(j == 3) {
			if(sscanf(p, "%u", &tmp) == 1)
				free_space_bytes += (tmp*1024);
			break;
		    }
		}
	    }
	    pclose(fp);
	}
    }

    markup = g_strdup_printf(_("<b>System Status</b>"));
    disk_space_frame = gtk_frame_new(markup);
    g_free(markup);
    gtk_frame_set_label_align(GTK_FRAME(disk_space_frame), 0.0, 0.5);
    gtk_label_set_use_markup(
			   GTK_LABEL(gtk_frame_get_label_widget(GTK_FRAME(disk_space_frame))),
			   TRUE
			   );
    gtk_frame_set_shadow_type(GTK_FRAME(disk_space_frame), GTK_SHADOW_NONE);
    gtk_box_pack_start(GTK_BOX(vbox), disk_space_frame, FALSE, FALSE, 0);

    header = gtk_label_new(_("Available disk space:"));
    alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 0, 0, 12, 0);
    gtk_container_add(GTK_CONTAINER(disk_space_frame), alignment);

    disk_space_table = gtk_table_new(1, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(disk_space_table), 6);
    gtk_table_set_col_spacings(GTK_TABLE(disk_space_table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(disk_space_table), 6);
    gtk_container_add(GTK_CONTAINER(alignment), disk_space_table);

    gtk_misc_set_alignment(GTK_MISC(header), 0.0, 0.5);
    gtk_table_attach(
		   GTK_TABLE(disk_space_table), header,
		   0, 1, 0, 1,
		   GTK_FILL, GTK_FILL, 0, 0
		   );

    markup = byte_size_format(free_space_bytes);
    disk_space_label = gtk_label_new(markup);
    g_free(markup);
    gtk_misc_set_alignment(GTK_MISC(disk_space_label), 0.0, 0.5);
    gtk_table_attach(
		   GTK_TABLE(disk_space_table), disk_space_label,
		   1, 2, 0, 1,
		   GTK_FILL, GTK_FILL, 0, 0
		   );

    display = gdk_display_get_default ();
    screen = gdk_display_get_screen (display, 0);
    width = gdk_screen_get_width (screen);
    height = gdk_screen_get_height (screen);

    dialog=gtk_dialog_new_with_buttons("About Device",
				NULL,
				GTK_DIALOG_NO_SEPARATOR,
                                GTK_STOCK_CLOSE,
                                GTK_RESPONSE_CLOSE,
				NULL);
    moko = moko_finger_scroll_new();
    moko_finger_scroll_add_with_viewport(MOKO_FINGER_SCROLL(moko), hbox);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
	moko, TRUE, TRUE, 0);
    gtk_widget_show_all(moko);

    g_signal_connect (GTK_WIDGET(dialog), "response",
             G_CALLBACK (dialog_response), NULL);

    gtk_widget_show_all(hbox);
    gtk_widget_set_size_request(GTK_WIDGET(dialog), (width*3)/4, (height*3)/4);
    gtk_widget_show(GTK_WIDGET(dialog));

    gtk_main();
    return 0;
}
