#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtkbox.h>
#include <gdk/gdkkeysyms.h>
#include <glib-object.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <dbus/dbus-glib.h>
#include <libhildondesktop/hildon-status-bar-item.h>
#include "moblin.h"

enum {
	AC_ADAPTER=0,
	BATTERY,
	SUPPORTED_NAMESPACES
};

static char *namespace[SUPPORTED_NAMESPACES] = {
	"ac_adapter",
	"battery"
};

enum {
	ICON_CHARGING_0,
	ICON_CHARGING_25,
	ICON_CHARGING_50,
	ICON_CHARGING_75,
	ICON_CHARGING_100,
	ICON_DISCHARGING_0,
	ICON_DISCHARGING_25,
	ICON_DISCHARGING_50,
	ICON_DISCHARGING_75,
	ICON_DISCHARGING_100,
	ICON_NOBATTERY,
	ICON_COUNT
};

enum {
	POWER_UNIT_MAH=0,
	POWER_UNIT_MWH,
	SUPPORTED_POWER_UNITS
};

static const char *supported_power_units[SUPPORTED_POWER_UNITS] = {
	"mAh",
	"mWh",
};

static const gchar *pix_filenames[ICON_COUNT] = {
    "moblin-ac_full0",
    "moblin-ac_full25",
    "moblin-ac_full50",
    "moblin-ac_full75",
    "moblin-ac_full100",
    "moblin-battery_full0",
    "moblin-battery_full25",
    "moblin-battery_full50",
    "moblin-battery_full75",
    "moblin-battery_full100",
    "moblin-nobattery"
};

typedef struct {
        GtkButton parent;
	DBusGConnection *connection;
	DBusGProxy *mproxy;
	DBusGProxy *proxy[SUPPORTED_NAMESPACES];
	gchar *udi[SUPPORTED_NAMESPACES];
	GType struct_array_type;
	gboolean ac_present;
	gboolean battery_present;
	char *charge_unit;
	gint charge_max;
	gint charge_curr;
	gint rate;
	GtkImage *image;
	GdkPixbuf *pix[ICON_COUNT];
	gint current_icon;
	gint panel_size;
	GtkWidget *menu;
	GtkWidget *menu_status;
	gboolean popped;
} MoblinBatteryApplet;

typedef struct {
	GtkButtonClass  parent_class;
} MoblinBatteryAppletClass;

typedef struct _MoblinBattery {
  HildonStatusBarItem* item;
  GtkWidget* button;
  MoblinBatteryApplet* applet;
} MoblinBattery;

GType            moblin_battery_applet_get_type   (void);
static void      moblin_battery_applet_class_init (MoblinBatteryAppletClass *klass);
static void      moblin_battery_applet_init       (MoblinBatteryApplet *applet);
gboolean         battery_dbus_connect             (MoblinBatteryApplet *bc);

G_DEFINE_TYPE (MoblinBatteryApplet, moblin_battery_applet, GTK_TYPE_BUTTON)

#define HAL_DBUS_SERVICE                        "org.freedesktop.Hal"
#define HAL_DBUS_PATH_MANAGER                   "/org/freedesktop/Hal/Manager"
#define HAL_DBUS_INTERFACE_MANAGER              "org.freedesktop.Hal.Manager"
#define HAL_DBUS_INTERFACE_DEVICE               "org.freedesktop.Hal.Device"
#define g_marshal_value_peek_int(v)      g_value_get_int (v)
#define g_marshal_value_peek_string(v)   g_value_get_string (v)
#define g_marshal_value_peek_boxed(v)    g_value_get_boxed (v)

static void
moblin_battery_applet_class_init (MoblinBatteryAppletClass *class)
{
        /* nothing to do here */
}

static void
init_pixbufs (MoblinBatteryApplet *bc)
{
    int i;

    for (i = 0; i < ICON_COUNT; i++) {
	if (bc->pix[i])
	    g_object_unref (bc->pix[i]);

	bc->pix[i] = gtk_icon_theme_load_icon (moblin_icon_theme_get_default (),
	    pix_filenames[i], bc->panel_size - 2, 0, NULL);
	if (gtk_widget_get_default_direction () == GTK_TEXT_DIR_RTL) {
	    GdkPixbuf *temp;
	    temp = gdk_pixbuf_flip (bc->pix[i], TRUE);
	    g_object_unref (G_OBJECT (bc->pix[i]));
	    bc->pix[i] = temp;
	}
    }
}

static gboolean
GetPropertyBoolean(DBusGProxy *proxy, char *prop)
{
	GError  *error = NULL;
	gboolean value;
	gboolean ret;

     	ret = dbus_g_proxy_call (proxy, "GetPropertyBoolean", &error,
				 G_TYPE_STRING, prop,
				 G_TYPE_INVALID,
				 G_TYPE_BOOLEAN, &value,
				 G_TYPE_INVALID);

	if (error) {
		g_error_free (error);
		return FALSE;
	}

	return value;
}

static gint
GetPropertyInteger(DBusGProxy *proxy, char *prop)
{
	GError  *error = NULL;
	gint value = -1;
	gboolean ret;

     	ret = dbus_g_proxy_call (proxy, "GetPropertyInteger", &error,
				 G_TYPE_STRING, prop,
				 G_TYPE_INVALID,
				 G_TYPE_INT, &value,
				 G_TYPE_INVALID);
	if (error) {
		g_error_free (error);
		return -1;
	}

	return value;
}

static gchar *
GetPropertyString(DBusGProxy *proxy, char *prop)
{
	GError  *error = NULL;
	gchar *value = NULL;
	gboolean ret;

     	ret = dbus_g_proxy_call (proxy, "GetPropertyString", &error,
				 G_TYPE_STRING, prop,
				 G_TYPE_INVALID,
				 G_TYPE_STRING, &value,
				 G_TYPE_INVALID);

	if (error) {
		g_error_free (error);
		return NULL;
	}

	return value;
}

gboolean
refresh_battery_info (MoblinBatteryApplet *bc)
{
int n;

	if (bc->proxy[AC_ADAPTER] != NULL) 
	{
	    bc->ac_present = GetPropertyBoolean(bc->proxy[AC_ADAPTER], "ac_adapter.present");
	    g_debug("ac_present = %s", (bc->ac_present)?"TRUE":"FALSE");
	}

	if (bc->proxy[BATTERY] != NULL) 
	{
	    bc->battery_present = GetPropertyBoolean(bc->proxy[BATTERY], "battery.present");
	    g_debug("battery_present = %s", (bc->battery_present)?"TRUE":"FALSE");
	    bc->charge_unit = GetPropertyString(bc->proxy[BATTERY], "battery.charge_level.unit");
	    if(bc->charge_unit) g_debug("charge_unit = %s", bc->charge_unit);
	    bc->charge_max = GetPropertyInteger(bc->proxy[BATTERY], "battery.charge_level.last_full");
	    if(bc->charge_max != -1) g_debug("charge_max = %d", bc->charge_max);
	    bc->charge_curr = GetPropertyInteger(bc->proxy[BATTERY], "battery.charge_level.current");
	    if(bc->charge_curr != -1) g_debug("charge_curr = %d", bc->charge_curr);
	    bc->rate = GetPropertyInteger(bc->proxy[BATTERY], "battery.charge_level.rate");
	    if(bc->rate != -1) g_debug("rate = %d", bc->rate);
	}

	if(!bc->battery_present) {
		bc->current_icon = ICON_NOBATTERY;
	} else if(bc->ac_present) {
		bc->current_icon = ICON_CHARGING_0 + (4*(bc->charge_curr))/(bc->charge_max);
		n = (100*(bc->charge_curr))/(bc->charge_max);
		if(n >= 95) bc->current_icon = ICON_CHARGING_100;
	} else {
		bc->current_icon = ICON_DISCHARGING_0 + (4*(bc->charge_curr))/(bc->charge_max);
		n = (100*(bc->charge_curr))/(bc->charge_max);
		if(n >= 95) bc->current_icon = ICON_DISCHARGING_100;
	}

	gtk_image_set_from_pixbuf (bc->image, bc->pix[bc->current_icon]);

	return TRUE;
}

/**
 * moblin_applet_size_allocate:
 * @applet: Brightness applet instance
 *
 * check if panel size has changed and applet adapt size
 **/
static void
moblin_applet_size_allocate (MoblinBatteryApplet *applet,
        GtkAllocation *allocation)
{
	if (applet->panel_size != allocation->height)
	{
		applet->panel_size = allocation->height;
		init_pixbufs (applet);
		refresh_battery_info(applet);
	}
}

char *
get_device_udi(DBusGProxy *proxy, char *cpb)
{
	GError *error = NULL;
	gchar **names = NULL;
	char *udi;
	int i;

	dbus_g_proxy_call (proxy, "FindDeviceByCapability", &error,
	  G_TYPE_STRING, cpb, G_TYPE_INVALID,
	  G_TYPE_STRV, &names, G_TYPE_INVALID);
	if (error) {
	    g_warning ("%s", error->message);
	    g_error_free (error);
	    return NULL;
	}
	if (names == NULL || names[0] == NULL) {
	    g_warning ("No devices of capability %s", cpb);
	    if(names) g_free(names);
	    return NULL;
	}
	udi = g_strdup (names[0]);
	for (i=0; names[i]; i++) {
	  g_free (names[i]);
	}
	g_free (names);
	return udi;
}

static void
property_modified_cb(DBusGProxy *proxy,
	gint type, GPtrArray  *properties, MoblinBatteryApplet *bc)
{
	refresh_battery_info(bc);
}

static void
device_removed_cb(DBusGProxy *proxy,
	const gchar *udi, MoblinBatteryApplet *bc)
{
	g_debug("DeviceRemoved: %s", udi);
	battery_dbus_connect(bc);
	refresh_battery_info(bc);
}

static void
device_added_cb(DBusGProxy *proxy,
	const gchar *udi, MoblinBatteryApplet *bc)
{
	g_debug("DeviceAdded: %s", udi);
	battery_dbus_connect(bc);
	refresh_battery_info(bc);
}

void
libhal_marshal_VOID__INT_BOXED (GClosure     *closure,
                                GValue       *return_value,
                                guint         n_param_values,
                                const GValue *param_values,
                                gpointer      invocation_hint,
                                gpointer      marshal_data)
{
  typedef void (*GMarshalFunc_VOID__INT_BOXED) (gpointer     data1,
                                                gint         arg_1,
                                                gpointer     arg_2,
                                                gpointer     data2);
  register GMarshalFunc_VOID__INT_BOXED callback;
  register GCClosure *cc = (GCClosure*) closure;
  register gpointer data1, data2;

  g_return_if_fail (n_param_values == 3);

  if (G_CCLOSURE_SWAP_DATA (closure))
    {
      data1 = closure->data;
      data2 = g_value_peek_pointer (param_values + 0);
    }
  else
    {
      data1 = g_value_peek_pointer (param_values + 0);
      data2 = closure->data;
    }
  callback = (GMarshalFunc_VOID__INT_BOXED) (marshal_data ? marshal_data : cc->callback);

  callback (data1,
            g_marshal_value_peek_int (param_values + 1),
            g_marshal_value_peek_boxed (param_values + 2),
            data2);
}

gboolean
battery_dbus_connect (MoblinBatteryApplet *bc)
{
GError *error = NULL;
int i;
gboolean res=TRUE;

    if(bc->connection == NULL)
    {
	GType struct_type;
	bc->connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	if(bc->connection == NULL)
	{
	    if(error) {
		g_warning ("Could not connect to DBUS daemon: %s", 
		    error->message);
		g_error_free (error);
	    } else {
		g_warning ("Could not connect to DBUS daemon");
	    }
	    return FALSE;
	}
	bc->mproxy = dbus_g_proxy_new_for_name (bc->connection,
	    HAL_DBUS_SERVICE, HAL_DBUS_PATH_MANAGER, 
	    HAL_DBUS_INTERFACE_MANAGER);
	if(bc->mproxy == NULL)
	{
	    g_warning("Failed to get proxy for %s", 
		HAL_DBUS_INTERFACE_MANAGER);
	    bc->connection = NULL;
	    return FALSE;
	}

	dbus_g_proxy_add_signal (bc->mproxy, "DeviceRemoved",
	    G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal (bc->mproxy, "DeviceRemoved",
	    G_CALLBACK (device_removed_cb), bc, NULL);
	dbus_g_proxy_add_signal (bc->mproxy, "DeviceAdded",
	    G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal (bc->mproxy, "DeviceAdded",
	    G_CALLBACK (device_added_cb), bc, NULL);
	struct_type = dbus_g_type_get_struct ("GValueArray",
	    G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_INVALID);
	bc->struct_array_type = 
	    dbus_g_type_get_collection ("GPtrArray", struct_type);
	dbus_g_object_register_marshaller (libhal_marshal_VOID__INT_BOXED,
	    G_TYPE_NONE, G_TYPE_INT, bc->struct_array_type, G_TYPE_INVALID);
    }

    for(i = 0; i < SUPPORTED_NAMESPACES; i++)
    {
	if(bc->udi[i] == NULL)
	{
	    bc->udi[i] = get_device_udi(bc->mproxy, namespace[i]);
	    if(bc->udi[i] == NULL)
	    {
		res = FALSE;
		continue;
	    }
	}
	if (bc->proxy[i] == NULL)
	{
	    bc->proxy[i] = dbus_g_proxy_new_for_name(bc->connection,
		HAL_DBUS_SERVICE, bc->udi[i], HAL_DBUS_INTERFACE_DEVICE);
	    if (bc->proxy[i] == NULL)
	    {
		res = FALSE;
		continue;
	    }
	    dbus_g_proxy_add_signal (bc->proxy[i], "PropertyModified",
		G_TYPE_INT, bc->struct_array_type, G_TYPE_INVALID);
	    dbus_g_proxy_connect_signal (bc->proxy[i], "PropertyModified",
		G_CALLBACK (property_modified_cb), bc, NULL);
	}
    }
    return res;
}

gboolean
battery_dbus_disconnect (MoblinBatteryApplet *bc)
{
int i;

	for(i = 0; i < SUPPORTED_NAMESPACES; i++)
	{
	    if (bc->proxy[i] != NULL) {
		dbus_g_proxy_disconnect_signal (bc->proxy[i], "PropertyModified",
			G_CALLBACK (property_modified_cb), bc);
		g_object_unref (bc->proxy[i]);
		bc->proxy[i] = NULL;
	    }
	}
	return TRUE;
}

static void 
menu_position_func (GtkMenu *menu, int *ox, int *oy, 
	gboolean *push_in, gpointer user_data)
{
	int x, y;
        MoblinBatteryApplet *applet = (MoblinBatteryApplet *)user_data;

        /* retrieve geometry parameters and move window appropriately */
        gdk_window_get_origin (GTK_WIDGET(applet)->window, &x, &y);
        x += GTK_WIDGET(applet)->allocation.x;
        y += GTK_WIDGET(applet)->allocation.y
                + GTK_WIDGET(applet)->allocation.height;
	*ox = x;
	*oy = y;
        *push_in = TRUE;
}

static gint
get_charge_time(MoblinBatteryApplet *applet)
{
	gint timeleft, i;

	if(applet->rate <= 0) return applet->rate;

	if(applet->charge_unit) {
	    for(i = 0; (i < SUPPORTED_POWER_UNITS)&&
		strcasecmp(supported_power_units[i], applet->charge_unit); 
		i++);
	} else {
	    i = 0;
	}

	switch(i) {
	case POWER_UNIT_MAH:
	case POWER_UNIT_MWH:
	default:
	    if(applet->ac_present) {
		timeleft = ((applet->charge_max - applet->charge_curr)*3600)/applet->rate;
	    } else {
		timeleft = ((applet->charge_curr)*3600)/applet->rate;
	    }		
	}
	return timeleft;
}

static void
create_battstat_string(MoblinBatteryApplet *applet, char *buf)
{
	gint percent, timeleft;

	if(applet->ac_present && !(applet->battery_present)) {
	    /* no battery attached, on AC power only */
	    sprintf(buf, "AC Power Only");
	} else {
	    percent = (applet->charge_curr * 100) / applet->charge_max;
	    if((applet->rate < 0)||((timeleft = get_charge_time(applet)) <= 0)) {
		/* battery attached, rate not supported */
		sprintf(buf, "%d%% Charged", percent);
	    } else {
		/* battery */
		sprintf(buf, "%d%% Charged, %dh:%02dm %s time", 
		    percent, timeleft/3600, (timeleft%3600)/60,
		    (applet->ac_present)?"charge":"battery");
	    }
	}
}

static gboolean
moblin_applet_popup_cb (MoblinBatteryApplet *applet, GdkEventButton *event)
{
char buf[1000];

	if(applet->popped) {
		gtk_menu_popdown (GTK_MENU(applet->menu));
		gtk_widget_destroy (applet->menu_status);
		gtk_widget_destroy (applet->menu);
	} else {
		refresh_battery_info(applet);
		applet->menu = gtk_menu_new ();
		create_battstat_string(applet, buf);
		applet->menu_status = gtk_menu_item_new_with_label (buf);
		gtk_menu_append (GTK_MENU (applet->menu), applet->menu_status);
		gtk_widget_show (applet->menu_status);

		gtk_menu_popup (GTK_MENU(applet->menu), NULL, NULL, menu_position_func, 
			applet, event->button, event->time);
	}

	return TRUE;
}

static void
moblin_battery_applet_init(MoblinBatteryApplet *applet)
{
    GtkWidget *image;
    int i;

	/* initialize the fields */
	applet->connection = NULL;
	applet->ac_present = FALSE;
	applet->battery_present = FALSE;
	applet->struct_array_type = G_TYPE_INVALID;
	applet->charge_unit = NULL;
	applet->charge_max = 0;
	applet->charge_curr = 0;
	applet->image = NULL;
	applet->current_icon = 0;
	applet->panel_size = 50;
	applet->menu = NULL;
	applet->menu_status = NULL;
	applet->popped = FALSE;
	applet->mproxy = NULL;
	for(i = 0; i < SUPPORTED_NAMESPACES; i++)
	{
	    applet->proxy[i] = NULL;
	    applet->udi[i] = NULL;
	}
	for(i = 0; i < ICON_COUNT; i++)
	{
	    applet->pix[i] = NULL;
	}

	init_pixbufs(applet);

	image = gtk_image_new ();
	applet->image = GTK_IMAGE (image);

   	gtk_container_add (GTK_CONTAINER (applet), image);
	gtk_widget_show (image);

	battery_dbus_connect(applet);

	g_signal_connect (G_OBJECT(applet), "button-press-event",
			G_CALLBACK(moblin_applet_popup_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "size-allocate",
                          G_CALLBACK(moblin_applet_size_allocate), NULL);
}

MoblinBattery*
moblin_battery_new()
{
MoblinBattery *obj;

  if(!moblin_battery_applet_get_type())
  {
    g_warning("Failed to register type MoblinBatteryApplet\n");
    return NULL;
  }
  obj = g_new0(MoblinBattery, 1);
  obj->applet = (MoblinBatteryApplet*)g_object_new(moblin_battery_applet_get_type(), NULL);

  return obj;
}

void *battery_initialize(HildonStatusBarItem *item, GtkWidget **button)
{
    MoblinBattery *obj = moblin_battery_new();

    if(!obj)
    {
	g_warning("Failed to initialize the battery applet\n");
	return NULL;
    }

    obj->item = item;
    *button = obj->button = GTK_WIDGET(obj->applet);

    refresh_battery_info(obj->applet);
    gtk_widget_show (GTK_WIDGET (obj->applet));

    return obj;
}

void battery_update(void *data, gint value1, gint value2, const gchar *str)
{

}

void battery_destroy(void *data)
{

}

gint battery_get_priority(void *data)
{
	return 1;
}

void battery_entry (HildonStatusBarPluginFn_st *fn)
{
	if (fn == NULL)
	{
		g_warning("bad setup call\n");
		return;
	}

	fn->initialize = battery_initialize;
	fn->destroy = battery_destroy;
	fn->update = battery_update;
	fn->get_priority = battery_get_priority;
}
