/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: datatype.c,v 1.28 2004/08/27 18:32:55 ostborn Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <zlib.h>

#include "config.h"
#include "mdatatypes.h"
#include "datatype.h"
#include "mlist.h"
#include "misc.h"
#include "md5_global.h"
#include "md5.h"
#include "ctype.h"

/* begin of Visit */
extern size_t mem_mdata_type_count[M_DATA_TYPE_IPPLWATCH];

/*
 *
 * strlen(s) == 32 and all characters are alpha-numeric
 *
 * @returns 0 (false) no MD5 string, 1 (true) for MD4 strings
 */
static int is_md5(char *s) {
	int i;

	if (!s) return 0;
	if (s[32] != '\0') return 0;

	for (i = 0; i < 32; i++) {
		if (!isxdigit(s[i])) return 0;
	}

	return 1;
}

int mdata_Visit_to_xml(gzFile *fd, mdata *data) {
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.visit->x, #x);
#define WRS(x,f) \
	if (data->data.visit->x != NULL) { \
		char *s = url_encode(data->data.visit->x); \
		gzprintf(fd, "<%s>", #x); \
		gzwrite(fd, s, strlen(s)); \
		gzprintf(fd, "</%s>", #x); \
		free(s); \
	} else { \
		gzprintf(fd, "<%s />", #x); \
	}

	WR(count, d);
	WR(type, d);
	WR(timestamp, ld);
	WR(timediff, ld);
	WRS(useragent, s);

	gzprintf(fd, "<hitlist>\n");
	mlist_write(fd, data->data.visit->hits);
	gzprintf(fd, "</hitlist>\n");
#undef WRS
#undef WR
	return 0;
}

int mdata_Visit_free(mdata *data) {
	if (!data) return -1;
	if (data->type != M_DATA_TYPE_VISIT) return -1;

	if (data->data.visit->hits) mlist_free(data->data.visit->hits);
	
	free(data->data.visit);

	return 0;
}

int mdata_Visit_setdata(mdata *data, const char *str,
			const char *useragent,
			int count, time_t timestamp,
			time_t timediff, int type) {

	data->key = str;
	assert(data->key);

	if (data->type == M_DATA_TYPE_UNSET)
		data->type		= M_DATA_TYPE_VISIT;

	data->data.visit->count		= count;
	data->data.visit->type		= type;
	data->data.visit->timestamp	= timestamp;
	data->data.visit->timediff	= timediff;

	data->data.visit->useragent     = useragent;
	mem_mdata_type_count[data->type]++;

/*	data->data.visit->hits = mlist_init();*/
#if 0
	fprintf(stderr, "++ %p: %s.%d.%s\n", data->data.visit->hits, __FILE__, __LINE__, __FUNCTION__);
#endif
	return 0;
}

int mdata_Visit_from_xml(void *user_data, int tagtype, const xmlChar *value, const xmlChar **attrs) {
	int i;

	/* add your definitions here */
	const mdata_values data_values[] = {
		{ "count", M_DATA_FIELDTYPE_LONG  },
		{ "type", M_DATA_FIELDTYPE_LONG },
		{ "timestamp", M_DATA_FIELDTYPE_LONG },
		{ "timediff", M_DATA_FIELDTYPE_LONG },
		{ "useragent", M_DATA_FIELDTYPE_STRING },
		{ "lasturl", M_DATA_FIELDTYPE_STRING }, /* unused */
		{ "hitlist", M_DATA_FIELDTYPE_LIST },

		/* 0.7.x */
		{ "hits", M_DATA_FIELDTYPE_LIST },

		{ NULL, M_DATA_FIELDTYPE_UNSET }
	};

	mstate_stack *m = user_data;

//	M_WP();

	switch(tagtype) {
	case M_TAG_BEGIN:
		for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
			;

		if (data_values[i].string == NULL) {
			M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "unknown tag '%s'\n",
				 value);
			return -1;
		}

#define SET_DATA(x) \
	m->st[m->st_depth].data = &(((mdata *)(m->st[m->st_depth-1].data))->x);

		switch(i) {
		case 0: SET_DATA(data.visit->count); break;
		case 1: SET_DATA(data.visit->type); break;
		case 2: SET_DATA(data.visit->timestamp); break;
		case 3: SET_DATA(data.visit->timediff); break;
		case 4: SET_DATA(data.visit->useragent); break;
		case 7: /* 0.7.x */
		case 6: m->st[m->st_depth].data = (((mdata *)(m->st[m->st_depth-1].data))->data.visit->hits) = mlist_init();
			break;
		default:
			return -1;
		}
#undef SET_DATA
		/* the same for all 4 values */
		m->st[m->st_depth].function = mdata_insert_value;
		m->st[m->st_depth].type = data_values[i].type;

		break;
	case M_TAG_END: {
		mdata *data = ((mdata *)(m->st[m->st_depth-1].data));

		data->type = M_DATA_TYPE_VISIT;

		if (!is_md5(data->key)) {
			char md5str[33];
			MD5_CTX context;
			unsigned char digest[16];
			char *r;
			int j;
			/* update 0.7.0-0.7.16 automaticly
			 * from
			 *   key = "hostname"
			 * to
			 *   key = md5("hostname" + "useragent")
			 */


			md5str[0] = '\0';
			MD5Init(&context);

			MD5Update(&context, data->key, strlen(data->key));

			if (data->data.visit->useragent) {
				MD5Update(&context, data->data.visit->useragent, strlen(data->data.visit->useragent));
			}

			MD5Final(digest, &context);
			for (j = 0, r = md5str; j < 16; j++, r += 2) {
				sprintf(r, "%02x", digest[j]);
			}
			*r = '\0';

			fprintf(stderr, "updating <visit key=\"%s\"> to modlogan 0.7.17 and above: %s\n",
				data->key, md5str);

			free(data->key);
			data->key = strdup(md5str);
		}

		switch(m->st[m->st_depth-2].type) {
		case M_DATA_FIELDTYPE_HASH: {
			mhash *h = m->st[m->st_depth-2].data;

			mhash_insert_sorted(h, data);
			break;
		}
		default:
			M_WP();
		}

		break;
	}
	case M_TAG_TEXT:
		M_WP();
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't handle tagtype '%d'\n",
			 tagtype);
		return -1;
	}

	return 0;
}

int mdata_Visit_append(mdata *dst, mdata *src) {
	return 0;
}

mdata *mdata_Visit_init() {
	mdata *data = mdata_init();
	assert(data);

	data->key		= NULL;
	data->type		= M_DATA_TYPE_VISIT;
	
	
	data->data.visit        = malloc(sizeof(data_Visit));
	memset(data->data.visit, 0, sizeof(data_Visit));

#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: got %ld (%d)\n", __FILE__, __LINE__, data->id, data->type);
#endif
/* init specifics */
	data->data.visit->count		= 0;
	data->data.visit->type		= M_DATA_VISIT_USER;
	data->data.visit->timestamp	= 0;
	data->data.visit->timediff	= 0;
	data->data.visit->useragent	= NULL;
	data->data.visit->hits          = mlist_init();

	return data;
}

mdata *mdata_Visit_copy(mdata *src) {
	mdata *data = mdata_Visit_init();
	assert(data);
	mdata_Visit_setdata(data, src->key,
		src->data.visit->useragent,
		src->data.visit->count,
		src->data.visit->timestamp,
		src->data.visit->timediff,
		src->data.visit->type
	);

	return data;
}

mdata *mdata_Visit_create(const char *str,
			  const char *useragent,
			  int count, time_t timestamp,
			  time_t timediff, int type) {

	mdata *data = mdata_Visit_init();
	assert(data);

	mdata_Visit_setdata(data, str,
		useragent, count, timestamp,
		timediff, type);

	return data;
}

/* end of Visit */
