
/*
 * Example Interface for libmodplug without using XMMS
 * This is code written by:
 * gurkan@linuks.mine.nu www.linuks.mine.nu
 * 
 * TODO
 * - getopt
 *  -l --loop
 *  -m --mono
 *  -s --stereo
 *  -4 --44100
 *  -2 --22050
 *  -1 --11025
 *  -8 --8bit
 *  -h --help
 *  -q --quiet
 *  -i --info    don't play, only show module info
 *  getopt(argc,argv,"lms4218hqi"))
 *  
 *  warp doesn't like getopt
 *  i haven't yet used it ever
 *  
 * - interactivity
 *  p pause
 *
 * - maybe show an oscilloscope
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <modplug.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <getopt.h>
#include <termios.h>

#define VERSION 0.0.1

#define BUF_SIZE 4096
#define DEVICE_NAME "/dev/dsp"

/* state variable */
int audio_fd;
unsigned char audio_buffer[BUF_SIZE];
static struct termios stored_settings;

int getEndianType() 
{
#define sz sizeof(int)
    int ival;
    int format;
    char s[sz];
    char t[sz];
    int i, lit, big;

    for (i=0; i<sz; i++) s[i] = i;
    ival = *(int *)s;
    big = lit = 0;

    for (i=0; i<sz; i++) {
        char c = ival&0xff;
        ival >>= 8;
        if (s[i] == c) lit++;
        if (s[sz-i-1] == c) big++;
        t[i] = c;
    }
    if (lit == sz && big == 0) {
        /*printf("little endian\n");*/
	format=AFMT_S16_LE;
    } else if (big == sz && lit == 0) {
        /*printf("big endian\n");*/
	format=AFMT_S16_BE;
    } else {
        printf("unknown\n");
    }
    return(format);
}

char *getFileData(char *filename, long *size) 
{
    FILE *f;
    char *data;

    f = fopen(filename, "rb");
    fseek(f, 0L, SEEK_END);
    (*size) = ftell(f); 
    rewind(f);
    data = (char*)malloc(*size);
    fread(data, *size, sizeof(char), f);
    fclose(f);

    return(data);
}

void set_keypress(void)
{
    struct termios new_settings;
    tcgetattr(0,&stored_settings);
    new_settings=stored_settings;
    new_settings.c_lflag &= (~ICANON);
    new_settings.c_lflag &= (~ECHO);
    new_settings.c_cc[VTIME] = 0;
    new_settings.c_cc[VMIN] = 1;
    tcsetattr(0,TCSANOW,&new_settings);
    return;
}

void reset_keypress(void)
{
    tcsetattr(0,TCSANOW,&stored_settings);
    return;
}

int main(int argc, char** argv)
{
    long size;
    char *d;
    ModPlugFile *f2;
    int mlen, nModLength;
    struct timeval tvstart;
    struct timeval tv;
    char status[81];

    int format;
    int channels = 2;
    int speed = 44100;

    char buffer[128];
    int result, nread;
    struct pollfd pollfds;
    int timeout = 1;            /* Timeout in msec. */

    ModPlug_Settings settings;
    ModPlug_GetSettings(&settings);

    format = getEndianType();
		        
    /* Initialize pollfds; we're looking at input, stdin */
    pollfds.fd = 0;             /* stdin */
    pollfds.events = POLLIN;    /* Wait for input */

    if (argc == 1) {
	printf("%s: too few arguments\n",argv[0]);
	exit(1);
    }

    if ((audio_fd=open(DEVICE_NAME, O_WRONLY, 0)) == -1) {
	perror(DEVICE_NAME);
	exit(1);
    } else {
	printf("opened %s for playing ",DEVICE_NAME);
    }
    printf("%s\n",argv[1]);

    d = getFileData(argv[1], &size);

    if (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &format) == -1) {
	perror("SND_CTL_DSP_SETFMT");
	exit(1);
    }
    /*if (format!=AFMT_S16_LE) {
	printf("format AFMT_S16_LE not supported");
	exit(1);
    }*/

    if (ioctl(audio_fd, SNDCTL_DSP_CHANNELS, &channels) == -1) {
	perror("SNDCTL_DSP_CHANNELS");
	exit(1);
    }
    /* int mChannels; */   /* Number of channels - 1=mono or 2=stereo */
    /* int mBits; */       /* Bits per sample - 8, 16, or 32 */
    /* int mFrequency; */  /* Sampling rate - 11025, 22050, or 44100 */

    if (ioctl(audio_fd, SNDCTL_DSP_SPEED, &speed) == -1) {
	perror("SNDCTL_DSP_SPEED");
	exit(1);
    }

    f2 = ModPlug_Load(d, size);

    settings.mResamplingMode = MODPLUG_RESAMPLE_FIR;
    settings.mChannels = 2;
    settings.mBits = 16;
    settings.mFrequency = 44100;
    /* insert more setting changes here */
    ModPlug_SetSettings(&settings);

    set_keypress();
    nModLength = ModPlug_GetLength(f2); /* Store Length of MOD */
    sprintf(status, "[1GPlaying %s (%%d.%%d/%d\")", ModPlug_GetName(f2), nModLength/1000);
    gettimeofday(&tvstart, NULL);
    
    while(mlen != 0) {

	gettimeofday(&tv, NULL);
	printf(status, tv.tv_sec-tvstart.tv_sec, tv.tv_usec/100000);
	fflush(stdout);

	mlen = ModPlug_Read(f2, audio_buffer, BUF_SIZE);

	if ((write(audio_fd, audio_buffer, mlen)) == -1) {
	    perror("audio write");
	    exit(1);
	}

        result = poll(&pollfds, 1, timeout);
        switch (result) {
        case 0:
            /*printf(".");*/
            break;
        case -1:
            perror("select");
            exit(1);

        default:
            if (pollfds.revents && POLLIN) {
                nread = read(0, buffer, nread);
                if (nread == 0) {
                    printf("keyboard done\n");
                    exit(0);
                }
		buffer[nread] = 0; /* set end of buffer to null */
		/* printf("%s", buffer); */

                switch (buffer[0]) { /* ignore excessive characters? */
		case 'q': mlen = 0; /* quit */
			break;
		case 'f': 
		    {
			if ((tv.tv_sec-tvstart.tv_sec+10) < nModLength/1000) {
			    ModPlug_Seek(f2,(tv.tv_sec-tvstart.tv_sec+1)*1000);
			    tvstart.tv_sec -= 10;
			}
		    } /* forward 10" */
		break;

		case 'b':
		    {
			if ((tv.tv_sec-tvstart.tv_sec-10) > 0) {
			    ModPlug_Seek(f2,(tv.tv_sec-tvstart.tv_sec-1)*1000);
			    tvstart.tv_sec += 10;
			}
		    } /* backward 10" */
		 break;

		    /*
		    m	mono
		    s	stereo
		    1	11025
		    2	22050
		    4	44100
		    6	16bit
		    8	8bit
		    l	loop/unloop
		    p	pause/unpause
		    */
                }
            }
        }

    }
    printf("\n");

    reset_keypress();
    ModPlug_Unload(f2);

    return 0;
}

