/* libmondo-devices.c                 Subroutines for handling devices


08/25
- modified am_I_in_disaster_recovery_mode() to allow for stupid distros
- interactively_obtain_...() - pause & ask user to remove CD/floppy

07/11
- in which_boot_loader(), also search /dev/cciss for boot loader

07/02
- modified calls to popup_and_get_string()

06/06
- fixed bug in where_is_root_mounted()

05/18
- fixed call to inject_device() --- Andre L.

05/15
- fixed support for subdir-within-NFS-mount

05/06
- back-ported insist_on_this_cd_number() from 1.7x

05/05
- mount_CDROM_here() --- mkdir -p mountpt

04/26
- made which_boot_loader() friendlier towards Red Hat 9.0
- find_and_mount_actual_cd() retracts all CD trays

04/24
- removed unused var and assoc'd unlink from find_cdrw_device()

04/03
- fixing nfs support
- changed /mnt/isodir to /tmp/isodir
- cleaned up insist_on_this_cd_number() and mount_CDROM_here()
- fixed am_i_in_disaster_recovery_mode()

03/31
- more work on find_cdrom_device()

03/29
- cleaner find_cdrom_device(), to find DVD instead of CD-ROM as last resort
- fix insist_on_this_cd_number()

03/15
- if root is /dev/root then assume not a ramdisk

02/10
- cleaned up 'where are the ISOs?' question (Stan Benoit)
- only eject if bkpinfo->please_dont_eject_when_restoring is false

01/16
- if root is at /dev/root/root then LVM (not ramdisk)

01/07
- fix am_I_in_disaster_recovery_mode() to recognize that /dev/rd/cp*
  entries are not ramdisks but RAID disks
- find_cdrw_device() no longer complains to stderr if no cdrecord

01/02/2003
- sensibly_set_tmpdir_and_scratchdir() --- exclude smb and smbfs

12/01/2002
- don't ask for isodir path if in disaster recovery mode

11/29
- added g_restoring_live_from_cd;

11/13
- handle_incoming_parameters() will sensibly set tmpdir and scratchdir
- properly wipe spurious scratchdir and tmpdir

11/09
- modified sensibly_...() to use tmp.mondo.* and mondo.scratch.*
  instead of mondo.tmpdir.* and mondo.scratchdir.*

11/08
- line 861,912: set media_size[0] to 1999 as well as 1 thru N
- chmod 700, not 770, in make_fifo()

11/02
- sensibly_set_tmpdir_and_scratchdir() will now prefix a '/' to the
  strings if it is missing

10/01 - 10/31
- changed find_cdrom_device() to ignore DVD drives in the hd[a-d] search;
  consequently, the subroutine will try scd0, sr0, etc. afterwards
- commented code
- fixed obscure bug in find_and_mount_actual_cd()
- code which sensibly configures tmpdir,scratchdir is now sep. sub.
- changed sort -s to sort -n <-- sensibly...()

09/01 - 09/30
- added list_of_NFS_devices_and_mounts()
- try /dev/st0 if no tape specified at all
- change '64k' to TAPE_BLOCK_SIZE
- improved find_cdrom_device()
- if cdrecord not found then find_cdrom_device() returns 1
- if disaster recovery mode then don't ask for CD-ROM's /dev entry;
  assume /dev/cdrom
- if restoring data then ask for CD-ROM's /dev entry (not its SCSI node)
- run_program_and_log_output() now takes boolean operator to specify
  whether it will log its activities in the event of _success_
- changed sort -n +4 to sort -s +4 (Troff)
- ask if user's kernel is sane; if 'no' then use FAILSAFE
- ask user to confirm the tape/CD device name
- if restoring (not archiving) then don't try to find CD-ROM's SCSI
  node; try to find its /dev entry instead
- better detection of biggest partition

08/01 - 08/31
- fixed verify bug --- CD#1 was being verified again & again & ...
- detect boot loader + device; if not detectible then _ask_
- if run interactively, assume tape size is irrelevant & don't ask
  for it; Mondo should handle end-of-tape gracefully now
- worked on handling unknowng media size
- added interactively_obtain_media_parameters_from_user()
- when trying to mount CD-ROM, find_and_mount_actual_cd() will
  now call find_cdrom_device() to try to find the /dev entry first
- fixed serious bug in zero_out_a_device()
- cleaned up some log_it() calls
- fixed minor bug in find_cdrom_device()
- better at distinguishing between live filesystem and ramdisk

07/24
- created
*/


#include "my-stuff.h"
#include "mondostructures.h"
#include "libmondo-devices.h"
#include "lib-common-externs.h"
#include "libmondo-string-EXT.h"
#include "libmondo-tools-EXT.h"
#include "libmondo-gui-EXT.h"
#include "libmondo-files-EXT.h"
#include "libmondo-fork-EXT.h"
#include "libmondo-stream-EXT.h"



extern int g_current_media_number;
extern bool g_ISO_restore_mode;
bool g_restoring_live_from_cd=FALSE;




void retract_CD_tray_and_defeat_autorun(void)
{
  log_it("rctada: Retracting all CD trays", __LINE__);
  run_program_and_log_output("for i in `ls /dev/cdr* /dev/dvd*` ; do eject -t $i ; done", TRUE);
//  log_it("rctada: killing autorun");
//  run_program_and_log_output("killall autorun", TRUE);
  if (!run_program_and_log_output("ps | grep autorun | grep -v grep", TRUE))
    {
      log_it("autorun detected; sleeping for 2 seconds");
      sleep(2);
    }
  log_it("rctada: Unmounting all CD drives", __LINE__);
  run_program_and_log_output("umount /dev/cdr* /dev/dvd*", TRUE);
}




/*************************************************************************
 * am_I_in_disaster_recovery_mode() -- Hugo Rabson                       *
 *                                                                       *
 * Purpose:  Find out if I am booting from a ramdisk or not              *
 * Called by:...                                                         *
 * Params:   none                                                        *
 * Returns:  TRUE (yes, I am) or FALSE (no, I'm not)                     *
 *************************************************************************/
bool
am_I_in_disaster_recovery_mode (void)
{
  char tmp[MAX_STR_LEN], comment[MAX_STR_LEN];
  bool is_this_a_ramdisk;

  strcpy(tmp, where_is_root_mounted());
  sprintf(comment, "root is mounted at %s\n", tmp);
  log_to_screen(comment);
  if (strstr(tmp, "/dev/ram") || (strstr(tmp, "/dev/rd")&&!strstr(tmp, "/dev/rd/cd")) || strstr(tmp, "rootfs")
    || (strstr(tmp, "/dev/root")&&!strstr(tmp, "/dev/root/root")))
    { is_this_a_ramdisk = TRUE; }
  else
    { is_this_a_ramdisk = FALSE; }

  if (is_this_a_ramdisk)
    {
      if (!does_file_exist("/THIS-IS-A-RAMDISK") && !does_file_exist("/tmp/mountlist.txt.sample"))
        { 
	  log_it("Using /dev/root is stupid of you but I'll forgive you.");
	  is_this_a_ramdisk = FALSE;
	}
    }
  if (does_file_exist("/THIS-IS-A-RAMDISK"))
    {
      is_this_a_ramdisk = TRUE;
    }

  if (is_this_a_ramdisk)
    {
      log_to_screen("I believe I am running on a ramdisk.");
      return(TRUE);
    }
  else
    {
      log_to_screen("I believe I am running on a live filesystem."); 
      return(FALSE);
    }
}



int inject_device(char*dev)
{
  char command[MAX_STR_LEN];

#ifdef __FreeBSD__
  if (strstr (dev, "acd")) {
      sprintf (command, "cdcontrol -f %s close");
  } else {
      sprintf (command, "camcontrol load `echo %s | sed 's|/dev/||'`");
  }
#else
  sprintf(command, "eject -t %s", dev);
#endif
  return(run_program_and_log_output(command, FALSE));
}




/*************************************************************************
 * am_I_in_disaster_recovery_mode() -- Hugo Rabson                       *
 *                                                                       *
 * Purpose:  Find out if I am booting from a ramdisk or not              *
 * Called by:...                                                         *
 * Params:   none                                                        *
 * Returns:  TRUE (yes, I am) or FALSE (no, I'm not)                     *
 *************************************************************************/
static char *bkptype_to_string(t_bkptype bt)
{
  static char output[MAX_STR_LEN];
  switch(bt)
    {
	case none: strcpy(output, "none");
		break;
	case iso:  strcpy(output, "iso");
		break;
	case cdr:  strcpy(output, "cdr");
		break;
	case cdrw: strcpy(output, "cdrw");
		break;
	case cdstream: strcpy(output, "cdstream");
		break;
	case nfs:  strcpy(output, "nfs");
		break;
	case tape: strcpy(output, "tape");
		break;
	case udev: strcpy(output, "udev");
		break;
	default: strcpy(output, "default");
    }
  return(output);
}





int eject_device(char*dev)
{
  char command[MAX_STR_LEN];
  sprintf(command, "eject %s", dev);
  return(run_program_and_log_output(command, FALSE));
}



/*************************************************************************
 * does_device_exist() -- Hugo Rabson                                    *
 *                                                                       *
 * Purpose:  Find out if specified device (e.g. /dev/fd0u1722) exists.   *
 * Called by:                                                            *
 * Params:   device               device string (e.g. /dev/fd0u1722)     *
 * Returns:  TRUE (yes, it does) or FALSE (no, it doesn't)               *
 *************************************************************************/
bool
does_device_exist (char *device)
{

	/** buffers ************************************************************/
  char tmp[MAX_STR_LEN];


  sprintf (tmp, "ls %s > /dev/null 2> /dev/null", device);
  if (system (tmp))
    {
      return (FALSE);
    }
  else
    {
      return (TRUE);
    }
}





/*************************************************************************
 * does_partition_exist() -- Hugo Rabson                                 *
 *                                                                       *
 * Purpose:   Can given partition (string) be found in fdisk's list of   *
 *            known partitions (i.e. "fdisk -l | grep $partition")       *
 * Called by: ...                                                        *
 * Params:    drive                     drive, e.g. /dev/hda             *
 *            partno                    e.g. 4 ---> /dev/hda4            *
 * Returns:   0=success; nonzero=failure                                 *
 *************************************************************************/
int
does_partition_exist (char *drive, int partno)
{
	/** buffers *****************************************************/
  char program[MAX_STR_LEN];
  char incoming[MAX_STR_LEN];
  char searchstr[MAX_STR_LEN];

	/** ints ********************************************************/
  int res = 0;

	/** pointers ****************************************************/
  FILE *fin;


	/****************************************************************/
  sprintf (program, "fdisk -l %s 2> /dev/null", drive);
  fin = popen (program, "r");
  if (!fin)
    {
      return (0);
    }
  build_partition_name (searchstr, drive, partno);
  strcat(searchstr, " ");
  for (res = 0; !res && fgets (incoming, MAX_STR_LEN - 1, fin); )
    {
      if (strstr (incoming, searchstr))
	{
	  res = 1;
	}
    }
  pclose (fin);
  return (res);
}






/*************************************************************************
 * does_string_exist_in_boot_block() -- Hugo Rabson                      *
 *                                                                       *
 * Purpose:   Is given string present in boot block of devide $dev?      *
 * Called by: ...                                                        *
 * Params:    dev                       device, e.g. /dev/hda            *
 *            str                       string to look for, e.g. "GRUB"  *
 * Returns:   TRUE=success; FALSE=failure                                *
 *************************************************************************/
bool
does_string_exist_in_boot_block (char *dev, char *str)
{
	/** buffers *****************************************************/
  char command[MAX_STR_LEN*2];

	/****************************************************************/

  sprintf (command,
	   "dd if=%s bs=446 count=1 2> /dev/null | strings | grep \"%s\" > /dev/null 2> /dev/null",
	   dev, str);
  if (system (command))
    {
      return (FALSE);
    }
  else
    {
      return (TRUE);
    }
}




/*************************************************************************
 * find_and_mount_actual_cd() -- Hugo Rabson                             *
 *                                                                       *
 * Purpose:   Locate and mount CD-ROM drive and the CD in it.            *
 * Called by: ...                                                        *
 * Params:    bkpinfo               backup information structure         *
 *            mountpoint            where to mount the CD-ROM drive      *
 * Returns:   0=success; nonzero=failure                                 *
 * NB:        If the initial attempt to mount bkpinfo->media_device at   *
 *            $mountpoint fails then it will popup and ask for an alter- *
 *            nate mountpoint, try that, and _then_ fail if that fails.  *
 *************************************************************************/
int
find_and_mount_actual_cd (struct s_bkpinfo *bkpinfo, char *mountpoint)
{
	/** buffers ******************************************************/
//  char tmp[MAX_STR_LEN];

	/** int's  *******************************************************/
  int res;

	/** end vars *****************************************************/

  retract_CD_tray_and_defeat_autorun();

  if ((res=mount_CDROM_here (bkpinfo->media_device, mountpoint)))
    {
      log_it("(find_and_mount_actual_cd) Unable to mount CD-ROM at %s", bkpinfo->media_device);
      log_it("Calling find_cdrom_device() to reinitialize bkpinfo->media_device");
      res = find_cdrom_device (bkpinfo->media_device);
      if (res)
        {
          log_it("Warning - find_cdrom_device() returned an error. Assuming /dev/cdrom.");
          strcpy(bkpinfo->media_device, "/dev/cdrom");
        }
      log_it("Retrying with media_device==%s", bkpinfo->media_device);
      res=mount_CDROM_here (bkpinfo->media_device, mountpoint);
    }
  if (res)
    {
      if (!popup_and_get_string
			("CD-ROM device", "Please enter your CD-ROM's /dev device",
			bkpinfo->media_device, MAX_STR_LEN/4))
	{ res = 1; }
      else
	{ res=mount_CDROM_here (bkpinfo->media_device, mountpoint); }
    }
  if (res) { log_it("(find_and_mount_actual_cd() --- failed"); }
  else { log_it("(find_and_mount_actual_cd() --- succeeded with %s", bkpinfo->media_device); }
  return (res);
}






/*************************************************************************
 * find_cdrw_device() -- Hugo Rabson                                     *
 *                                                                       *
 * Purpose:  Find CD-R[W] drive's SCSI node by calling cdrecord          *
 * Called by:...                                                         *
 * Params:   output              [returned] CD-R[W} drive's SCSI node    *
 * Returns:  0=success; nonzero=failure                                  *
 *************************************************************************/
int
find_cdrw_device (char *cdrw_device)
{
	/** buffers *************************/
  /*char scratchfile[MAX_STR_LEN]; removed 19 apr 02 stan benoit */
  char comment[MAX_STR_LEN];
  char tmp[MAX_STR_LEN];
  char cdr_exe[MAX_STR_LEN];
  char command[MAX_STR_LEN*2];

  run_program_and_log_output("insmod ide-scsi", FALSE);
  if (!run_program_and_log_output("which dvdrecord", TRUE))
    { strcpy(cdr_exe, "dvdrecord"); }
  else
    { strcpy(cdr_exe, "cdrecord"); }
  tmp[0]='\0';
  if (find_home_of_exe(cdr_exe))
    {
      sprintf(command, "%s -scanbus 2> /dev/null | tr -s '\t' ' ' | grep \"[0-9]*,[0-9]*,[0-9]*\" | grep -v \"[0-9]*) \\*\" | grep CD | cut -d' ' -f2 | head -n1", cdr_exe);
      strcpy (tmp, call_program_and_get_last_line_of_output(command));
    }
  if (strlen (tmp) < 2)
    {
//      log_to_screen ("Could not find CDRW device");
      return 1;
    }
  else
    {
      strcpy (cdrw_device, tmp);
      sprintf (comment, "Found CDRW device - %s", cdrw_device);
      log_it (comment);
    }
  return (0);

}




/*************************************************************************
 * find_cdrom_device() -- Hugo Rabson                                    *
 *                                                                       *
 * Purpose:  Find CD-R[W] drive's /dev entry by calling cdrecord -scanbus*
 *           and dmesg                                                   *
 * Called by:...                                                         *
 * Params:   output              [returned] CD-R[W} drive's /dev entry   *
 * Returns:  0=success; nonzero=failure                                  *
 * NB:       Only works if CD is in drive!                               *
 *************************************************************************/
int
find_cdrom_device (char *output)
{
	/** pointers *****************************************************/
  FILE *fin;
  char *p;
  char *q;
  char *r;

	/** bool's *******************************************************/
  bool found_it = FALSE;

	/** buffers ******************************************************/
  char tmp[MAX_STR_LEN];
  char phrase_one[MAX_STR_LEN];
  char phrase_two[MAX_STR_LEN];
  char command[MAX_STR_LEN*2];
  char dvd_last_resort[MAX_STR_LEN];

	/** intialize ****************************************************/
  output[0] = '\0';
  phrase_one[0] = '\0';
  phrase_two[0] = '\0';
  dvd_last_resort[0] = '\0';

	/** end vars *****************************************************/

  if (!find_home_of_exe("cdrecord")) 
    {
      strcpy(output, "/dev/cdrom");
      log_it("Can't find cdrecord; assuming %s", output);
      if (!does_device_exist(output))
	{
	  log_it("That didn't work. Sorry.");
	  return(1);
	}
      else
	{
	  return(0);
	}
    }
  fin = popen ("cdrecord -scanbus 2> /dev/null", "r");
  if (!fin) { log_it("Can't read cdrecord -scanbus"); return (1); }
  for (fgets (tmp, MAX_STR_LEN, fin); !feof (fin);
       fgets (tmp, MAX_STR_LEN, fin))
    {
      p = strchr (tmp, '\'');
      if (p)
	{
	  q = strchr (++p, '\'');
	  if (q)
	    {
	      for (r = q; *(r - 1) == ' '; r--);
	      *r = '\0';
	      strcpy (phrase_one, p);
	      p = strchr (++q, '\'');
	      if (p)
		{
		  q = strchr (++p, '\'');
		  if (q)
		    {
		      while (*(q - 1) == ' ')
			{
			  q--;
			}
		      *q = '\0';
		      strcpy (phrase_two, p);
		    }
		}
	    }
	}
    }
  pclose (fin);
  sprintf (command, "dmesg | grep \"%s\"", phrase_two);
  log_it ("'%s' is the 2nd command", command);
  fin = popen (command, "r");
  if (fin)
    {
      for (fgets (tmp, MAX_STR_LEN, fin); !feof (fin);
	   fgets (tmp, MAX_STR_LEN, fin))
	{
          log_it("--> '%s'", tmp);
	  if (tmp[0] != ' ' && tmp[1] != ' ')
	    {
	      p = strchr (tmp, ':');
	      if (p)
		{
		  *p = '\0';
                  if (strstr(tmp, "DVD"))
                    {
                      sprintf(dvd_last_resort, "/dev/%s", tmp);
                      log_it("Ignoring '%s' because it's a DVD drive", tmp);
                    }
                  else
                    {
		      sprintf (output, "/dev/%s", tmp);
		      found_it = TRUE;
                    }
		}
	    }
	}
      pclose (fin);
    }
  if (!found_it && strlen(dvd_last_resort)>0)
    {
      log_it("Well, I'll use the DVD - %s - as a last resort", dvd_last_resort);
      strcpy(output, dvd_last_resort);
      found_it = TRUE;
    }
  if (found_it)
    {
      sprintf(tmp, "grep \"%s=ide-scsi\" /proc/cmdline", strrchr(output,'/')+1);
      if (run_program_and_log_output(tmp, TRUE) == 0)
	{
	  log_it("%s is not right. It's being SCSI-emulated. Continuing.", output);
	  found_it = FALSE;
	  output[0] = '\0';
	}
    }
  if (found_it)
    {
      log_it("(find_cdrom_device) --> '%s'", output);
      if (!does_device_exist(output)) { found_it=FALSE; log_it("OK, I was wrong, I haven't found it... yet."); }
    }
  if (!found_it)
      { log_it("OK, approach 2"); if (!(found_it = set_dev_to_this_if_rx_OK(output, "/dev/scd0")))
        { if (!(found_it = set_dev_to_this_if_rx_OK(output, "/dev/sr0")))
          { if (!(found_it = set_dev_to_this_if_rx_OK(output, "/dev/cdrom")))
            { if (!(found_it = set_dev_to_this_if_rx_OK(output, "/dev/cdrom0")))
              { if (!(found_it = set_dev_to_this_if_rx_OK(output, "/dev/cdrom1")))
                { if (!(found_it = set_dev_to_this_if_rx_OK(output, "/dev/sr1")))
                  { return(1); }
      } } } } } }
  if (found_it)
    {
      log_it("(find_cdrom_device) --> '%s'", output);
      if (!does_device_exist(output)) { log_it("I still haven't found it."); return(1); }
      return (0);
    }
  else
    {
      return (1);
    }
}




/*************************************************************************
 * get_phys_size_of_drive() -- Hugo Rabson                               *
 *                                                                       *
 * Purpose:  Return the physical size of the specified drive, in MBytes  *
 * Called by:...                                                         *
 * Params:   drive        device (e.g. /dev/hda)                         *
 * Returns:  long         size of drive in megabytes                     *
 *************************************************************************/
long
get_phys_size_of_drive (char *drive)
{
	/** pointers *****************************************************/
  FILE *fin;
  char *p;
  char *q;
  char *r;

	/** buffers ******************************************************/
  char tmp[MAX_STR_LEN];
  char command[MAX_STR_LEN*2];

	/** long *********************************************************/
  long outL;
  long tempLa;
  long tempLb;
  long tempLc;

	/** end vars *****************************************************/

  sprintf (command,
	   "fdisk -l %s | head -n4 | tr -s '\n' '\t' | tr -s ' ' '\t' | cut -f8,14,16",
	   drive);
  strcpy (tmp, call_program_and_get_last_line_of_output (command));
  if (tmp[0])
    {
      p = tmp;
      q = strchr (p, ' ');
      if (q)
	{
	  *(q++) = '\0';
	  r = strchr (q, ' ');
	  if (r)
	    {
	      *(r++) = '\0';
	      tempLa = atol (p);
	      tempLb = atol (q);
	      tempLc = atol (r);
	      outL = tempLa * tempLb / 1024 * tempLc / 1024;
	      if (outL > 100)
		{
		  return (outL);
		}
	    }
	}
    }
  /* else, do it the old-fashioned way */
  p = strrchr (drive, (int) '/');
  if (p)
    {
      strcpy (tmp, p + 1);
    }
  else
    {
      return (-1);
    }
  sprintf (command, "dmesg | grep %s 2> /dev/null", tmp);
  fin = popen (command, "r");
  fgets (tmp, MAX_STR_LEN - 1, fin);
  while (!feof (fin) && !strstr (tmp, "GB") && !strstr (tmp, "MB"))
    {
      fgets (tmp, MAX_STR_LEN - 1, fin);
    }
  pclose (fin);
  if (!(p = strstr (tmp, "GB")) && !(p = strstr (tmp, "MB")))
    {
      log_it ("Cannot find %s's size: dmesg isn't helping either.",
	       drive);
      return (-1);
    }
  for (; !isdigit (*(p - 1)); p--);
  *p = '\0';
  for (p--; isdigit (*(p - 1)); p--);
  outL = atol (p);
  if (outL <= 0)
    {
      return (-1);
    }
  if (strstr (tmp, "GB"))
    {
      outL = outL * 1024;
    }
  return (outL * 20 / 19);
}







/*************************************************************************
 * is_this_a_valid_disk_format() -- Hugo Rabson                          *
 *                                                                       *
 * Purpose:  See if user's kernel supports the specified format type     *
 *           (e.g. vfat, ext2, xfs, jfs, ...)                            *
 * Called by:...                                                         *
 * Params:   format           format to check up on                      *
 * Returns:  TRUE=ok, FALSE=not supported                                *
 *************************************************************************/
bool
is_this_a_valid_disk_format (char *format)
{
  char good_formats[MAX_STR_LEN];
	char command[MAX_STR_LEN*2];
	char format_sz[MAX_STR_LEN];

  FILE *pin;

  sprintf (format_sz, "%s ", format);
  sprintf (command,
	   "cat /proc/filesystems | grep -v nodev | tr -s '\t' ' ' | cut -d' ' -f2 | tr -s '\n' ' '");
  pin = popen (command, "r");
  if (pin)
    {
      strcpy (good_formats, " ");
      fgets (good_formats + 1, MAX_STR_LEN, pin);
      pclose (pin);
    }
  else
    {
      log_to_screen ("Unable to read good formats");
      return (FALSE);
    }
  strip_spaces (good_formats);
  strcat (good_formats, " swap lvm raid ");
  if (strstr (good_formats, format_sz))
    {
      return (TRUE);
    }
  else
    {
      return (FALSE);
    }
}



/*************************************************************************
 * is_this_device_mounted() -- Hugo Rabson                               *
 *                                                                       *
 * Purpose:   Examine /proc/mounts to see if specified device is mounted *
 * Called by: ...                                                        *
 * Params:    device_raw          e.g. /dev/hda5                         *
 * Returns:   TRUE if it is mounted; FALSE if it is not mounted          *
 *************************************************************************/
bool is_this_device_mounted (char *device_raw)
{

	/** pointers *****************************************************/
  FILE *fin;

	/** buffers ******************************************************/
  char incoming[MAX_STR_LEN];
  char device_with_tab[MAX_STR_LEN];
  char device_with_space[MAX_STR_LEN];
  char tmp[MAX_STR_LEN];

	/** end vars *****************************************************/

  sprintf (device_with_tab, "%s\t", device_raw);
  sprintf (device_with_space, "%s ", device_raw);
  fin = fopen ("/proc/mounts", "r");
  for (fgets (incoming, MAX_STR_LEN - 1, fin); !feof (fin);
       fgets (incoming, MAX_STR_LEN - 1, fin))
    {
      if (strstr (incoming, device_with_space)
	  || strstr (incoming, device_with_tab))
	{
	  fclose (fin);
	  return (1);
	}
    }
  fclose (fin);
  sprintf (tmp, "cat /proc/swaps | grep \"%s\" > /dev/null 2> /dev/null",
	   device_with_space);
  if (!system (tmp))
    {
      return (1);
    }
  return (0);
}






/*************************************************************************
 * mount_CDROM_here() -- Hugo Rabson                                     *
 *                                                                       *
 * Purpose:   mount specified device at specified mountpoint as iso9660  *
 * Called by: ...                                                        *
 * Params:    device                               e.g. /dev/cdrom :)    *
 *            mountpoint                           e.g. /mnt/cdrom       *
 * Returns:   0=success; nonzero=failure                                 *
 *************************************************************************/
int
mount_CDROM_here (char *device, char *mountpoint)
{
	/** buffer *******************************************************/
  char command[MAX_STR_LEN];
  char dev[MAX_STR_LEN];
  char options[MAX_STR_LEN];

  sprintf(command, "mkdir -p %s", mountpoint);
  run_program_and_log_output(command, TRUE);
  strcpy(options, "ro");
  if (isdigit(device[0]))
    { find_cdrom_device(device); }
  else
    { strcpy(dev, device); }
  if (g_ISO_restore_mode)
    { strcat(options,",loop"); }
  log_it("(mount_CDROM_here --- device=%s, mountpoint=%s", device, mountpoint);
	/** end vars ****************************************************/
  sprintf (command, "mount %s -o %s -t iso9660 %s 2> /dev/null", device, options,
	   mountpoint);
  log_it (command);
  return (system (command));
}







/*************************************************************************
 * insist_on_this_cd_number() -- Hugo Rabson       							         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * returns:                                                              *
 *************************************************************************/
void
insist_on_this_cd_number (struct s_bkpinfo* bkpinfo, int cd_number_i_want)
{

	/** int **************************************************************/
  int res = 0;


	/** buffers **********************************************************/
  char tmp[MAX_STR_LEN];
  char request[MAX_STR_LEN];

  assert(bkpinfo!=NULL);
  assert(cd_number_i_want > 0);

//  log_it("Insisting on CD number %d", cd_number_i_want);

  if (IS_THIS_A_STREAMING_BACKUP(bkpinfo->backup_media_type)) 
    {
      log_it("No need to insist_on_this_cd_number when the backup type isn't CD-R(W) or NFS or ISO");
      return;
    }
  if (g_ISO_restore_mode || bkpinfo->backup_media_type == iso || bkpinfo->backup_media_type == nfs)
    {
          if (is_this_device_mounted("/mnt/cdrom")) { run_program_and_log_output("umount /mnt/cdrom", TRUE); }
	  run_program_and_log_output("mkdir -p /tmp/isodir", TRUE);
	  sprintf(tmp, "%s/%d.iso -o loop", bkpinfo->isodir, cd_number_i_want);
          log_it("Mounting /mnt/cdrom at %s", tmp);
	  if (mount_CDROM_here(tmp, "/mnt/cdrom")) { fatal_error("Mommy!"); }
	  g_current_media_number = cd_number_i_want;
	  return;
    }
  if ((res=what_number_cd_is_this (bkpinfo)) != cd_number_i_want)
    {
      log_it("Currently, we hold %d but we want %d", res, cd_number_i_want);
      sprintf (tmp, "Insisting on CD #%d", cd_number_i_want);
      sprintf (request, "Please insert CD #%d and press Enter.",
	       cd_number_i_want);
      log_it (tmp);
      while (what_number_cd_is_this (bkpinfo) != cd_number_i_want)
	{
	  system ("sync");
	  if (is_this_device_mounted ("/mnt/cdrom"))
	    {
	      res = run_program_and_log_output ("umount /mnt/cdrom", FALSE);
	    }
	  else
	    {
	      res = 0;
	    }
	  if (res)
	    {
	      log_to_screen ("WARNING - failed to unmount CD-ROM drive");
	    }
          if (!bkpinfo->please_dont_eject_when_restoring)
            { res = eject_device(bkpinfo->media_device); }
          else
            { res = 0; }
	  if (res)
	    {
              log_to_screen ("WARNING - failed to eject CD-ROM disk");
	    }
          popup_and_OK (request);
          if (!bkpinfo->please_dont_eject_when_restoring)
            {
              inject_device(bkpinfo->media_device);
            }
	  system ("sync");
	}
      log_it ("Thankyou. Proceeding...");
      g_current_media_number = cd_number_i_want;
    }
}







/*************************************************************************
 * interactively_obtain_media_parameters_from_user() -- Hugo Rabson      *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * returns:                                                              *
 *************************************************************************/
int interactively_obtain_media_parameters_from_user(struct s_bkpinfo *bkpinfo, bool archiving_to_media)
// archiving_to_media is TRUE if I'm being called by mondoarchive
// archiving_to_media is FALSE if I'm being called by mondorestore
{
  char tmp[MAX_STR_LEN];
  char size_sz[MAX_STR_LEN];
  char command[MAX_STR_LEN*2];
  int i;

  assert(bkpinfo!=NULL);
  size_sz[0] = '\0';
  bkpinfo->nonbootable_backup = FALSE;

// Tape, CD, NFS, ...?
  srandom(getpid());
  bkpinfo->backup_media_type = (g_restoring_live_from_cd) ? cdr : which_backup_media_type(bkpinfo->restore_data);
  if (bkpinfo->backup_media_type == none)
    { log_to_screen("User has chosen not to backup the PC"); finish(1); }
  if (bkpinfo->backup_media_type == tape)
    { popup_and_OK("Please remove CD and/or floppy from drive(s)"); }
  log_it("media type = %s", bkptype_to_string(bkpinfo->backup_media_type));
  if (archiving_to_media)
    {
      sensibly_set_tmpdir_and_scratchdir(bkpinfo);
    }
  bkpinfo->cdrw_speed = (bkpinfo->backup_media_type == cdstream) ? 2 : 4;
  bkpinfo->compression_level = (bkpinfo->backup_media_type == cdstream) ? 1 : 5;
  bkpinfo->use_lzo =  (bkpinfo->backup_media_type == cdstream) ? TRUE : FALSE;

  mvaddstr_and_log_it(2, 0, " ");

// Find device's /dev (or SCSI) entry
  switch(bkpinfo->backup_media_type)
    {
      case cdr:
      case cdrw:
	if (archiving_to_media)
          {
            if (ask_me_yes_or_no("Is your computer a laptop, or does the CD writer incorporate BurnProof technology?"))
              { bkpinfo->manual_cd_tray = TRUE; }
            if ((bkpinfo->compression_level = which_compression_level()) == -1)
              { log_to_screen("User has chosen not to backup the PC"); finish(1); }
	    strcpy(tmp, "4");
	    if (!popup_and_get_string("Speed", "What speed is your CD (re)writer?", tmp, MAX_STR_LEN))
	      { log_to_screen("User has chosen not to backup the PC"); finish(1); }
	    bkpinfo->cdrw_speed = atoi(tmp);
          }
      case cdstream:
	if (bkpinfo->disaster_recovery)
	  {
            strcpy(bkpinfo->media_device, "/dev/cdrom");
	    log_it("CD-ROM device assumed to be at %s", bkpinfo->media_device);
	  }
	else if (bkpinfo->restore_data)
	  {
            strcpy(bkpinfo->media_device, "/dev/scd0");
#ifdef __FreeBSD__
            if (TRUE)
#else
	    if (find_cdrom_device (bkpinfo->media_device))
#endif
	      {
	        if (!popup_and_get_string("CD-ROM device", "Please enter your CD-ROM's /dev device", bkpinfo->media_device, MAX_STR_LEN/4))
	          { log_to_screen("User has chosen not to backup the PC"); finish(1); }
	      }
	    log_it("CD-ROM device found at %s", bkpinfo->media_device);
	  }
	else
	  {
	    if (find_cdrw_device(bkpinfo->media_device)) { bkpinfo->media_device[0]='\0'; }
	    if (bkpinfo->media_device[0])
	      {
	        sprintf(tmp, "I think I've found your CD burner at SCSI node %s; am I right on the money?", bkpinfo->media_device);
	        if (!ask_me_yes_or_no(tmp)) { bkpinfo->media_device[0]='\0'; }
	      }
           if (!bkpinfo->media_device[0])
	      {
	        if (!popup_and_get_string("Device node?", "What is the SCSI node of your CD (re)writer, please?", bkpinfo->media_device, MAX_STR_LEN/4))
	          { log_to_screen("User has chosen not to backup the PC"); finish(1); }
	      }
	  }
	for(i=0; i<=MAX_NOOF_MEDIA; i++) { bkpinfo->media_size[i] = 650; }
	break;
      case udev:
	if (!ask_me_yes_or_no("This option is for advanced users only. Are you sure?"))
	  { log_to_screen("User has chosen not to backup the PC"); finish(1); }
      case tape:
	if (find_tape_device_and_size(bkpinfo->media_device, size_sz)) { bkpinfo->media_device[0]='\0'; }
        if (!bkpinfo->media_device[0] && does_device_exist("/dev/st0")) { strcpy(bkpinfo->media_device, "/dev/st0"); }
	if (bkpinfo->media_device[0])
	  {
	    sprintf(tmp, "I think I've found your tape streamer at %s; am I right on the money?", bkpinfo->media_device);
	    if (!ask_me_yes_or_no(tmp)) { bkpinfo->media_device[0]='\0'; }
 	  }
        if (!bkpinfo->media_device[0])
	  {
	    if (!popup_and_get_string("Device name?", "What is the /dev entry of your tape streamer?", bkpinfo->media_device, MAX_STR_LEN/4))
	      { log_to_screen("User has chosen not to backup the PC"); finish(1); }
	  }
	sprintf(tmp, "ls -l %s", bkpinfo->media_device);
	if (run_program_and_log_output(tmp, FALSE))
	      { log_to_screen("User has not specified a valid /dev entry"); finish(1); }
	log_it("size_sz = %s", size_sz);
        size_sz[0] = '\0';
/*
	if ((size_sz[0]=='\0' || atol(size_sz)==0) && archiving_to_media)
	  {
	    if (!popup_and_get_string("Tape size", "How much COMPRESSED data will one of your tape cartridges hold? (e.g. 4GB for 4 gigabytes)", size_sz, MAX_STR_LEN))
	      { log_to_screen("User has chosen not to backup the PC"); finish(1); }
	  }
*/
	if (size_sz[0]=='\0')
	  { bkpinfo->media_size[0] = 0; }
        else
	  { bkpinfo->media_size[0] = friendly_sizestr_to_sizelong (size_sz)/2 - 50; }
	log_it("media_size[0] = %ld", bkpinfo->media_size[0]);
        if (bkpinfo->media_size[0] <= 0) { bkpinfo->media_size[0] = 0; }
	for(i=1; i<=MAX_NOOF_MEDIA; i++) { bkpinfo->media_size[i] = bkpinfo->media_size[0]; }
        if (archiving_to_media)
          {
            if ((bkpinfo->compression_level = which_compression_level()) == -1)
              { log_to_screen("User has chosen not to backup the PC"); finish(1); }
          }
        break;
      case nfs:
	if (!bkpinfo->nfs_mount[0]) { strcpy(bkpinfo->nfs_mount, call_program_and_get_last_line_of_output("cat /proc/mounts | grep \":\" | cut -d' ' -f1 | head -n1")); }
#ifdef __FreeBSD__
        if (TRUE)
#else
	if (!bkpinfo->disaster_recovery )
#endif
	  {
	    if (!popup_and_get_string("NFS dir.", "Please enter path and directory where archives are stored remotely. (Mondo has taken a guess at the correct value. If it is incorrect, delete it and type the correct one.)", bkpinfo->nfs_mount, MAX_STR_LEN))
	      {
		log_to_screen("User has chosen not to backup the PC");
		finish(1); 
	      }
          if (!bkpinfo->restore_data)
            {
              if ((bkpinfo->compression_level = which_compression_level()) == -1)
                {
                  log_to_screen("User has chosen not to backup the PC");
                  finish(1);
                }
            }
	  }
	if (!is_this_device_mounted(bkpinfo->nfs_mount))
	  {
            sprintf(bkpinfo->isodir, "/tmp/isodir");
	    run_program_and_log_output("mkdir -p /tmp/isodir", TRUE);
	    sprintf(tmp, "mount %s -t nfs /tmp/isodir", bkpinfo->nfs_mount);
	    run_program_and_log_output(tmp, TRUE);
	    if (!is_this_device_mounted(bkpinfo->nfs_mount))
	      {
		popup_and_OK("Please mount that partition before you try to backup to or restore from it."); finish(1); 
	      }
	  }
        else
          {
            sprintf(command, "mount | grep %s | cut -d' ' -f3", bkpinfo->nfs_mount);
            strcpy(bkpinfo->isodir, call_program_and_get_last_line_of_output(command));
          }
        strcpy(tmp, bkpinfo->isodir);
        if (!popup_and_get_string("Directory", "Which directory within that mountpoint?", tmp + strlen(tmp), MAX_STR_LEN-strlen(tmp)))
          {
            log_to_screen("User has chosen not to backup the PC");
             finish(1); 
          }
        strcpy(bkpinfo->isodir, tmp);
	for(i=0; i<=MAX_NOOF_MEDIA; i++) { bkpinfo->media_size[i] = 650; }
        log_it("Just set isodir to %s", bkpinfo->isodir);
	break;
      case iso:
        if (!bkpinfo->disaster_recovery)
          {
             if (!popup_and_get_string("Storage dir.", "Please enter the full path that contains your ISO images.  Example: /mnt/raid0_0", bkpinfo->isodir, MAX_STR_LEN/4))
	      { log_to_screen("User has chosen not to backup the PC"); finish(1); }
             if (archiving_to_media) {
                 if ((bkpinfo->compression_level = which_compression_level()) == -1)
                     { log_to_screen("User has chosen not to backup the PC"); finish(1); }
                 if (!popup_and_get_string ("ISO size.", "Please enter how big you want each ISO image to be (in megabytes). This should be less than or equal to the size of the CD-R[W]'s you plan to backup to.", size_sz, MAX_STR_LEN))
                     { log_to_screen ("User has chosen not to backup the PC"); finish(1); }
                 for(i=0; i<=MAX_NOOF_MEDIA; i++) { bkpinfo->media_size[i] = atoi (size_sz); }
             } else {
                 for (i=0;i<=MAX_NOOF_MEDIA; i++) { bkpinfo->media_size[i] = 650;}
             }
          }
	break;
      default:
	fatal_error("I, Mojo Jojo, shall defeat those pesky Powerpuff Girls!");
    }

  if (archiving_to_media)
    {
      i = which_boot_loader(bkpinfo->boot_device);
      if (i=='U') // unknown
        {

#ifdef __FreeBSD__
          if (!popup_and_get_string("Boot device", "What is your boot device? (e.g. /dev/ad0)", bkpinfo->boot_device, MAX_STR_LEN/4))
            { log_to_screen("User has chosen not to backup the PC"); finish(1); }
          i = which_boot_loader (bkpinfo->boot_device);
#else
          if (!popup_and_get_string("Boot device", "What is your boot device? (e.g. /dev/hda)", bkpinfo->boot_device, MAX_STR_LEN/4))
            { log_to_screen("User has chosen not to backup the PC"); finish(1); }
          if (does_string_exist_in_boot_block(bkpinfo->boot_device, "LILO")) { i='L'; }
          else if (does_string_exist_in_boot_block(bkpinfo->boot_device, "GRUB")) { i='G'; }
          else { i = 'U'; }
#endif

          if (i == 'U')
            {
	      if (ask_me_yes_or_no("Unidentified boot loader. Shall I restore it byte-for-byte at restore time and hope for the best?"))
		{
		  i = 'R'; // raw
		}
	      else
		{
		  log_to_screen("I cannot find your boot loader. Please run mondoarchive with parameters."); finish(1); 
		}
	    }
	}
      bkpinfo->boot_loader = i;
      strcpy(bkpinfo->include_paths, "/");
      if (!popup_and_get_string("Backup paths", "Please enter paths which you want me to backup. The default is '/' (i.e. everything).", bkpinfo->include_paths, MAX_STR_LEN))
        { log_to_screen("User has chosen not to backup the PC"); finish(1); }
      if (!popup_and_get_string("Exclude paths", "Please enter paths which you do NOT want to backup. Separate them with spaces. NB: /tmp and /proc are always excluded. :-) Just hit 'Enter' if you want to do a full system backup.", bkpinfo->exclude_paths, MAX_STR_LEN))
        { log_to_screen("User has chosen not to backup the PC"); finish(1); }
      bkpinfo->make_cd_use_lilo = FALSE;
      bkpinfo->backup_data = TRUE;
      bkpinfo->verify_data = ask_me_yes_or_no("Will you want to verify your backups after Mondo has created them?");
      if (!ask_me_yes_or_no("Are you confident that your kernel is a sane, sensible, standard Linux kernel? (Say 'no' if you are using Gentoo or Debian.)"))
        { strcpy(bkpinfo->kernel_path, "FAILSAFE"); }
      if (!ask_me_yes_or_no("Are you sure you want to proceed? Hit 'no' to abort."))
        { log_to_screen("User has chosen not to backup the PC"); finish(1); }
    }
  else
    {
      bkpinfo->restore_data = TRUE; // probably...
    }

  if (bkpinfo->backup_media_type == iso || bkpinfo->backup_media_type == nfs)
    {
      g_ISO_restore_mode = TRUE;
    }

#ifdef __FreeSD__
// skip
#else
  if (bkpinfo->backup_media_type == nfs)
    {
      sprintf(tmp, "mount | grep \"%s\" | cut -d' ' -f3", bkpinfo->nfs_mount);
//      strcpy(bkpinfo->isodir, call_program_and_get_last_line_of_output(tmp));
      log_it("I think the NFS mount is mounted at %s", bkpinfo->isodir);
    }
  log_it("isodir = %s", bkpinfo->isodir);
  log_it("nfs_mount = '%s'", bkpinfo->nfs_mount);
#endif

  log_it("media device = %s", bkpinfo->media_device);
  log_it("media size = %ld", bkpinfo->media_size[1]);
  log_it("media type = %s", bkptype_to_string(bkpinfo->backup_media_type));
  log_it("compression = %ld", bkpinfo->compression_level);
  log_it("include_paths = '%s'", bkpinfo->include_paths);
  log_it("exclude_paths = '%s'", bkpinfo->exclude_paths);
  log_it("scratchdir = '%s'", bkpinfo->scratchdir);
  log_it("tmpdir = '%s'", bkpinfo->tmpdir);
  log_it("boot_device = '%s' (loader=%c)", bkpinfo->boot_device, bkpinfo->boot_loader);
  if (bkpinfo->media_size[0] < 0) { fatal_error("Media size is less than zero."); }
  return(0);
}









/*************************************************************************
 * list_of_NFS_devices_and_mounts() -- Hugo Rabson       							         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * returns:                                                              *
 *************************************************************************/

char *list_of_NFS_devices_and_mounts(void)
{
  char exclude_these_devices[MAX_STR_LEN];
  char exclude_these_directories[MAX_STR_LEN];
  static char result_sz[MAX_STR_LEN];

  strcpy(exclude_these_directories, call_program_and_get_last_line_of_output("mount | tr -s '\t' ' ' | fgrep -w nfs | cut -d' ' -f3 | tr -s '\n' ' ' | awk '{print $0;}'"));
  strcpy(exclude_these_devices, call_program_and_get_last_line_of_output("cat /etc/fstab | grep -w nfs | tr -s '\t' ' ' | cut -d' ' -f1 | tr -s '\n' ' ' | awk '{print $0;}'"));
  printf("ex_these_dirs = %s\n", exclude_these_directories);
  sprintf(result_sz, "%s %s", exclude_these_directories, exclude_these_devices);
  return(result_sz);
}






/*************************************************************************
 * make_fifo() -- Hugo Rabson                                            *
 *                                                                       *
 * Purpose:   Make a randomly-named FIFO; return its name                *
 * Called by: ...                                                        *
 * Params:    store_name_here        (returned) name of FIFO             *
 *            stub                   root of FIFO's new name, e.g.       *
 *                                   "/tmp/random-device."               *
 * Returns:   none (but store_name_here's contents will be returned)     *
 * NB:        If stub=="tmp/random-device." then store_name_here==       *
 *            "/tmp/random-device.NNNNN", NNNN being a random number     *
 *************************************************************************/
void make_fifo(char*store_name_here, char*stub)
{
  char tmp[MAX_STR_LEN];

  sprintf(store_name_here, "%s%d%d", stub, (int)(random()%32768), (int)(random()%32768));
  make_hole_for_file(store_name_here);
  mkfifo(store_name_here, 777); // FIXME - make 700
  sprintf(tmp, "chmod 700 %s", store_name_here);
  system(tmp);
//  log_it("Made yummy fifo %s", store_name_here);
}







/*************************************************************************
 * sensibly_set_tmpdir_and_scratchdir() -- Hugo Rabson       							         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * returns:                                                              *
 *************************************************************************/
void sensibly_set_tmpdir_and_scratchdir(struct s_bkpinfo *bkpinfo)
{
  char tmp[MAX_STR_LEN], command[MAX_STR_LEN], sz[MAX_STR_LEN];

      strcpy(tmp, call_program_and_get_last_line_of_output("df -m -x nfs -x vfat -x smbfs -x smb -x ntfs | tr -s '\t' ' ' | grep -v none | grep -v Filesystem | awk '{printf \"%s %s\\n\", $4, $6;}' | sort -n | tail -n1 | awk '{print $NF;}'"));
      if (tmp[0]!='/') { strcpy(sz, tmp); strcpy(tmp, "/"); strcat(tmp, sz); }
      sprintf(bkpinfo->tmpdir, "%s/tmp.mondo.%d", tmp, (int)(random()%32768));
      sprintf(bkpinfo->scratchdir, "%s/mondo.scratch.%d", tmp, (int)(random()%32768));
      log_it("bkpinfo->tmpdir is being set to %s", bkpinfo->tmpdir);
      log_it("bkpinfo->scratchdir is being set to %s", bkpinfo->scratchdir);
      sprintf(command, "rm -Rf %s/tmp.mondo.* %s/mondo.scratch.*", tmp, tmp);
      system(command);
}








/*************************************************************************
 * set_dev_to_this_if_rx_OK() -- Hugo Rabson       							         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * returns:                                                              *
 *************************************************************************/

bool set_dev_to_this_if_rx_OK(char *output, char*dev)
{
  char command[MAX_STR_LEN];
  if (!does_device_exist(dev)) { return(FALSE); }
  sprintf(command, "dd bs=%ld count=1 if=%s of=/dev/null 2> /dev/null", TAPE_BLOCK_SIZE, dev);
  if (!run_program_and_log_output(command, FALSE) && !run_program_and_log_output(command, FALSE))
    { strcpy(output, dev); log_it("Found it - %s", dev); return(TRUE); }
  else
    { output[0]='\0'; log_it("It's not %s",dev); return(FALSE); }  
}






/*************************************************************************
 * what_number_cd_is_this() -- Hugo Rabson       							         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * returns:                                                              *
 *************************************************************************/

int what_number_cd_is_this(struct s_bkpinfo *bkpinfo)
{
  int cd_number=-1;
  char mountpt[MAX_STR_LEN];

  if (g_ISO_restore_mode)
    {
      log_it("what_number_cd_is_this() -- returning g_current_media_number");
      return(g_current_media_number);
    }

  strcpy(mountpt, bkpinfo->media_device);
  if (!mountpt[0])
    {
      log_it("(what_number_cd_is_this) Warning - media_device unknown. Finding out..."); 
      find_cdrom_device(bkpinfo->media_device);
    }
  if (! is_this_device_mounted("/mnt/cdrom"))
    { mount_CDROM_here(mountpt, "/mnt/cdrom"); }
  cd_number = atoi(last_line_of_file("/mnt/cdrom/archives/THIS-CD-NUMBER"));
  return(cd_number);
}







/*************************************************************************
 * where_is_root_mounted() -- Hugo Rabson                                *
 *                                                                       *
 * Purpose:  Find where root ('/') is mounted. If it is /ram or /dev/root*
 *           then I am probably running on a ramdisk. Just FYI.          *
 * Called by:...                                                         *
 * Returns:  static string pointing to the root device                   *
 *************************************************************************/
char *
where_is_root_mounted ()
{
	/** buffers *****************/
  static char tmp[MAX_STR_LEN];



  strcpy (tmp,
	  call_program_and_get_last_line_of_output
	  ("mount | grep \" on / \" | cut -d' ' -f1 | sed s/[0-9]// | sed s/[0-9]//"));
  if (strstr(tmp, "/dev/md"))
    {
      strcpy(tmp,
	     call_program_and_get_last_line_of_output
	     ("mount | grep \" on / \" | cut -d' ' -f1"));
    }
  return (tmp);
}



/*************************************************************************
 * which_boot_loader() -- Hugo Rabson                                    *
 *                                                                       *
 * Purpose:   Examine specified boot device to see if LILO, GRUB or some *
 *            other boot loader is being used by said device             *
 * Called by: ...                                                        *
 * Params:    which_device [returned]         boot device, e.g. /dev/hda *
 * Returns:   'G' if GRUB; 'L' if LILO; 'U' if unknown                   *
 *************************************************************************/
char
which_boot_loader (char *which_device)
{
	/** buffer ******************************************************/
  char list_drives_cmd[MAX_STR_LEN];
  char current_drive[MAX_STR_LEN];

	/** pointers ****************************************************/
  FILE *pdrives;

	/** int *********************************************************/
  int count_lilos = 0;
  int count_grubs = 0;

	/** end vars ****************************************************/

  log_it("%ld: looking for boot drive", __LINE__);

  //  sprintf (list_drives_cmd,
  //	   "fdisk -l | grep /dev | grep cyl | tr ':' ' ' | cut -d' ' -f2");

  sprintf (list_drives_cmd,
	   "fdisk -l | grep \"/dev/.*:\" | tr -s ':' ' ' | tr -s ' ' '\n' | grep /dev/; echo /dev/cciss");
  log_it("list_drives_cmd = %s", list_drives_cmd);

  if (!(pdrives = popen (list_drives_cmd, "r")))
    {
      log_to_screen ("Unable to open list of drives");
      return ('\0');
    }
  for (fgets (current_drive, MAX_STR_LEN, pdrives); !feof (pdrives);
       fgets (current_drive, MAX_STR_LEN, pdrives))
    {
      strip_spaces (current_drive);
      log_it ("%ld: looking at drive %s's MBR", __LINE__, current_drive);
      if (does_string_exist_in_boot_block (current_drive, "GRUB"))
	{
	  log_it ("%ld: found GRUB in %s", __LINE__, current_drive);
	  count_grubs++;
	  strcpy (which_device, current_drive);
	  break;
	}
      if (does_string_exist_in_boot_block (current_drive, "LILO"))
	{
	  log_it ("%ld: found LILO in %s", __LINE__, current_drive);
	  count_lilos++;
	  strcpy (which_device, current_drive);
	  break;
	}
    }
  pclose (pdrives);
  log_it ("%d grubs and %d lilos\n", count_grubs, count_lilos);
  if (count_grubs && !count_lilos)
    {
      return ('G');
    }
  else if (count_lilos && !count_grubs)
    {
      return ('L');
    }
  else if (count_grubs == 1 && count_lilos == 1)
    {
      log_it ("I'll bet you used to use LILO but switched to GRUB...");
      return ('G');
    }
  else
    {
      log_it ("Unknown boot loader");
      return ('U');
    }
}





/*************************************************************************
 * zero_out_a_device() -- Hugo Rabson                                    *
 *                                                                       *
 * Purpose:  Write zeroes over the first 16K of a device                 *
 * Called by:...                                                         *
 * Params:   device        e.g. /dev/hda                                 *
 * Returns:  0=success; nonzero=failure                                  *
 *************************************************************************/
int
zero_out_a_device (char *device)
{
  FILE *fout;
  int i;

  log_it ("Zeroing drive %s", device);
  if (!(fout=fopen (device, "w")))
    {
      log_it ("Unable to open/write to device");
      return (1);
    }
  for (i = 0; i < 16384; i++)
    {
      fputc ('\0', fout);
    }
  fclose (fout);
  log_it ("Device successfully zeroed.");
  return (0);
}










