# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: config.tcl,v 2.90 2004/02/03 22:27:39 jfontain Exp $


namespace eval configuration {

if {$global::withGUI} {

    variable container
    variable interface
    variable hierarchy
    variable configure                                                              ;# whether folder should appear when configuring
    if {[string equal $::tcl_platform(platform) unix]} {
        set hierarchy {
            application application.size application.colors application.fonts application.printing application.pages
                application.database application.trace
            viewers viewers.colors viewers.graphs viewers.pies viewers.tables viewers.cells
            thresholds thresholds.email thresholds.trace
            daemon
        }
        set configure {1 1 1 0 0 0 0 0 1 1 1 1 1 1 0 0 0 0}
    } else {                                                  ;# printing configuration is done directly at printing time on windows
        set hierarchy {
            application application.size application.colors application.pages application.database application.trace
            viewers viewers.colors viewers.graphs viewers.pies viewers.tables viewers.cells
            thresholds thresholds.email thresholds.trace
        }
        set configure {1 1 1 0 0 0 1 1 1 1 1 1 0 0 0}
    }

    variable helpMessage
    set helpMessage(preferences) "Preferences for the user: $::tcl_platform(user)"
    set helpMessage(configuration) {Configuration for the current view.}

    variable closedIcon [image create photo -data {
        R0lGODlhEAANAPIAAAAAAH9/f7+/v///AP///wAAAAAAAAAAACH+JEZpbGUgd3JpdHRlbiBieSBBZG9iZSBQaG90b3Nob3CoIDUuMAAsAAAAABAADQAAAzNI
        Gsz6kAQxqAjxzcpvc1KWBUDYnRQZWmilYi37EmztlrAt43R8mzrO60P8lAiApHK5TAAAOw==
    }]
    variable openedIcon [image create photo -data {
        R0lGODlhEAANAPIAAAAAAH9/f7+/v///////AAAAAAAAAAAAACH+JEZpbGUgd3JpdHRlbiBieSBBZG9iZSBQaG90b3Nob3CoIDUuMAAsAAAAABAADQAAAzk4
        Gsz6cIQ44xqCZCGbk4MmclAAgNs4ml7rEaxVAkKc3gTAnBO+sbyQT6M7gVQpk9HlAhgHzqhUmgAAOw==
    }]
    variable leafIcon [image create photo -data {
        R0lGODlhDAANAIQAALi8uJiYmPgA+PDw8LC0sGhoaPj4+Pj8+FhYWPD08ODg4IiIiOjs6NDQ0Ojo6Njc2ODk4NjY2NDU0LCwsMjMyKisqMDAwLi4uKioqKCg
        oGhsaAAAAAAAAAAAAAAAAAAAACH5BAEAAAIALAAAAAAMAA0AAAVUIBCM5CicwaCuBFGggyEbB1G/6UzbNZLPh0Bh6IsBEwrCoihLOBmKBqHoTAwYDsUDQLUy
        IIqIZJryZsWNCfUKeUgalIovTLEALoQKJoPQIP6AgQghADs=
    }]
    variable minusIcon [image create photo -data {
        R0lGODlhCQAJAKEAAL+/v////wAAAP///ywAAAAACQAJAAACEYSPoRvG614DQVg7ZZbxoQ8UADs=
    }]
    variable plusIcon [image create photo -data {
        R0lGODlhCQAJAKEAAL+/v////wAAAP///ywAAAAACQAJAAACFISPoRu2spyCyol7G3hxz850CFIAADs=
    }]

}                                                                  ;# else avoid Tk commands when running under tclsh in daemon mode

    proc load {arrayList} {
        foreach {name value} $arrayList {
            set ::global::$name $value
        }
    }

if {$global::withGUI} {

    proc edit {preferencesMode} {                                                                        ;# preferences mode boolean
        variable hierarchy
        variable configure
        variable container
        variable interface
        variable tree
        variable preferences
        variable helpMessage
        variable dialog
        variable entryIcons
        variable leafIcon
        variable minusIcon
        variable plusIcon

        set preferences $preferencesMode                                                                ;# store mode for sub sheets

        set objects {}                                                                  ;# to delete upon destruction of this folder

        set title {moodss: }
        if {$preferences} {
            append title Preferences
        } else {
            append title Configuration
        }
        # do not let the Enter and Return keys close the dialog box so that the thresholds email listEntry widget becomes useful:
        set dialog [new dialogBox .grabber\
            -buttons hoc -default o -title $title -x [winfo pointerx .] -y [winfo pointery .] -enterreturn 0\
            -command configuration::done -helpcommand configuration::help -deletecommand {grab release .grabber} -die 0\
        ]
        grab .grabber                                                      ;# grab siblings such as help window so that it is usable
        lappend objects [linkedHelpWidgetTip $composite::($dialog,help,path)]

        set frame [frame $widget::($dialog,path).frame]

        set tree [Tree $frame.tree\
            -dragenabled 0 -dropenabled 0 -linestipple gray50 -deltay [expr {[font metrics $font::(mediumBold) -linespace] + 4}]\
            -background $widget::option(listbox,background) -selectbackground $widget::option(listbox,selectbackground)\
            -closecmd {configuration::stateChange 0} -opencmd {configuration::stateChange 1} -selectcommand configuration::open\
            -crossopenimage $minusIcon -crosscloseimage $plusIcon\
        ]                                                                                  ;# to detect instance and cell selections
        $tree bindText <Control-Button-1> {}; $tree bindImage <Control-Button-1> {}                   ;# prevent multiple selections
        set container [frame $frame.container -borderwidth 1 -relief sunken]

        set message [createMessage $container.message]
        if {$preferences} {
            $message configure -text $helpMessage(preferences)
        } else {
            $message configure -text $helpMessage(configuration)
        }
        pack $message -fill both -expand 1                                                   ;# initially display help message above
        catch {unset interface(current)}                                                               ;# currently opened interface

        foreach entry $hierarchy specific $configure {
            if {!$preferences && !$specific} continue
            foreach {parent child} [split $entry .] {}
            if {[string length $child] == 0} {                                                            ;# no child implies folder
                set node [$tree insert end root #auto -text $parent -font $font::(mediumBold) -image $configuration::closedIcon]
                set parentNode $node
            } else {
                set node [$tree insert end $parentNode #auto -text $child -font $font::(mediumBold) -image $configuration::leafIcon]
            }
            regsub -all {\.} $entry :: interface($node,class)                            ;# use generated tree index as unique index
            $interface($node,class)::initialize
        }
        pack $tree -side left -fill y -padx 2
        pack $container -fill both -expand 1 -padx 2

        dialogBox::display $dialog $frame

        wm geometry $widget::($dialog,path) 500x300                                                        ;# maintain constant size

        bind $frame <Destroy> "delete $objects"                                      ;# delete objects not managed by the dialog box
    }

    proc open {tree node} {
        variable container
        variable interface

        if {[info exists interface(current)]} {
            if {$node == $interface(current)} return                                                                    ;# no change
            if {![$interface($interface(current),class)::check]} {                 ;# check validity before moving to next interface
                $tree selection set $interface(current)
                bell
                return
            }
        }
        eval destroy [winfo children $container]                                              ;# eventually remove current interface
        set frame [frame $container.frame]             ;# use a separate frame so that interfaces widget management cannot interfere
        pack $frame -fill both -expand 1
        $interface($node,class)::edit $frame
        set interface(current) $node
    }

    proc done {} {
        variable interface
        variable preferences
        variable variables
        variable dialog

        # check validity before closing
        if {[info exists interface(current)] && ![$interface($interface(current),class)::check]} return
        foreach name [array names interface *,class] {
            $interface($name)::apply                                                                             ;# apply new values
        }
        if {$preferences} {
            preferences::save $variables(1)
        }
        delete $dialog
    }

    proc help {} {
        variable interface
        variable preferences

        if {[info exists interface(current)]} {
            $interface($interface(current),class)::help
        } elseif {$preferences} {
            generalHelpWindow #core.preferences
        } else {
            generalHelpWindow #core.configuration
        }
    }

    proc createMessage {path args} {                  ;# create a generic message widget with eventual option / value pairs argument
        message $path -width [winfo screenwidth .] -font $font::(mediumNormal) -justify center
        eval $path configure $args
        return $path
    }

    proc initialize {name} {                  ;# for use by folder classes, in order to initialize local variables depending on mode
        variable preferences

        if {$preferences} {
            if {![info exists ::preferences::$name]} {                       ;# initialize from global value if rc file was not read
                set ::preferences::$name [set ::global::$name]
            }
            return [set ::preferences::$name]
        } else {
            return [set ::global::$name]
        }
    }

    proc apply {name value} {                    ;# for use by folder classes, in order to update global variables depending on mode
        variable preferences

        set namespaces ::global                                                                       ;# always update global values
        if {$preferences} {
            lappend namespaces ::preferences                                                           ;# and eventually preferences
        }
        foreach namespace $namespaces {
            if {![info exists ${namespace}::$name] || ($value != [set ${namespace}::$name])} {
                # update only when necessary in order not to activate a trace needlessly
                set ${namespace}::$name $value
            }
        }
    }

    proc variables {preferences} {                      ;# never access configuration variables directly: use this procedure instead
        variable variables

        return $variables($preferences)
    }

    proc stateChange {opened node} {
        variable tree
        variable closedIcon
        variable openedIcon

        if {$opened} {
            $tree itemconfigure $node -image $openedIcon
        } else {
            $tree itemconfigure $node -image $closedIcon
        }
    }


    namespace eval application {

        proc initialize {} {}

        proc edit {parentPath} {
            set message [configuration::createMessage $parentPath.message -text {Application configuration}]
            pack $message -fill both -expand 1                                               ;# initially display help message above
        }

        proc check {} {
            return 1                                                                                         ;# no data in this page
        }

        proc apply {} {}

        proc variables {} {return {}}

        proc help {} {
            generalHelpWindow #configuration.application
        }

        namespace eval size {                                                                           ;# start of application size

            proc variables {} {                                       ;# list of unqualified global variables handled by this folder
                return {canvasHeight canvasWidth}
            }

            proc initialize {} {
                variable height [configuration::initialize canvasHeight]
                variable width [configuration::initialize canvasWidth]
                variable automatic [expr {($width == 0) && ($height == 0)}]
            }

            proc edit {parentPath} {
                variable height
                variable width
                variable message
                variable automatic
                variable entries

                set message [configuration::createMessage $parentPath.message -text {Enter canvas size (in pixels):}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                    ;# occupy whole width
                grid rowconfigure $parentPath 0 -weight 1
                grid columnconfigure $parentPath 0 -weight 1
                set button [checkbutton $parentPath.automatic\
                    -text {automatic scaling} -command configuration::application::size::update\
                    -variable configuration::application::size::automatic\
                ]
                grid $button -row 1 -column 0 -columnspan 100 -pady 10                                                     ;# center
                set values {640 800 1024 1280 1600}
                if {[package vcompare $::tcl_version 8.4] < 0} {
                    set widthEntry [new spinEntry $parentPath -width 4 -list $values]
                    spinEntry::set $widthEntry $width
                    grid $widget::($widthEntry,path) -row 2 -column 2
                    set path $composite::($widthEntry,entry,path)
                    set entries $widthEntry
                } else {                                                                            ;# use native widget if possible
                    set path [spinbox $parentPath.widthEntry -width 4 -values $values]
                    $path set $width
                    grid $path -row 2 -column 2
                    set entries $path
                }
                $path configure -textvariable configuration::application::size::width
                # filter on positive integers and limit entry length
                setupEntryValidation $path {{checkMaximumLength 4 %P} {checkUnsignedInteger %P}}
                grid [label $parentPath.width -text width:] -row 2 -column 1 -padx 2
                grid columnconfigure $parentPath 3 -weight 1
                set values {400 480 600 768 1024 1280}
                if {[package vcompare $::tcl_version 8.4] < 0} {
                    set heightEntry [new spinEntry $parentPath -width 4 -list $values]
                    spinEntry::set $heightEntry $height
                    grid $widget::($heightEntry,path) -row 2 -column 5
                    set path $composite::($heightEntry,entry,path)
                    lappend entries $heightEntry
                } else {                                                                            ;# use native widget if possible
                    set path [spinbox $parentPath.heightEntry -width 4 -values $values]
                    $path set $height
                    grid $path -row 2 -column 5
                    lappend entries $path
                }
                $path configure -textvariable configuration::application::size::height
                # filter on positive integers and limit entry length
                setupEntryValidation $path {{checkMaximumLength 4 %P} {checkUnsignedInteger %P}}
                grid [label $parentPath.height -text height:] -row 2 -column 4 -padx 2
                grid columnconfigure $parentPath 6 -weight 1
                grid [button $parentPath.apply -text Apply -command configuration::application::size::apply]\
                    -row 3 -column 0 -columnspan 100
                grid rowconfigure $parentPath 3 -weight 1
                if {[package vcompare $::tcl_version 8.4] < 0} {
                    bind $message <Destroy> "delete $widthEntry $heightEntry"               ;# delete inner objects upon destruction
                }
                update
            }

            proc update {} {
                variable automatic
                variable entries

                if {$automatic} {set state disabled} else {set state normal}
                if {[package vcompare $::tcl_version 8.4] < 0} {
                    foreach entry $entries {
                        composite::configure $entry -state $state
                    }
                } else {
                    foreach entry $entries {
                        $entry configure -state $state
                    }
                }
            }

            proc check {} {
                variable height
                variable width
                variable message
                variable automatic

                if {!$automatic} {
                    foreach item {width height} {
                        if {[string length [set $item]] == 0} {
                            set text "please set $item."
                            break
                        }
                    }
                }
                if {[info exists text]} {
                    $message configure -font $font::(mediumBold) -text $text
                    return 0
                } else {
                    return 1
                }
            }

            proc apply {} {
                variable height
                variable width
                variable automatic

                if {![check]} return
                if {$automatic} {
                    set width 0; set height 0
                }
                configuration::apply canvasHeight $height
                configuration::apply canvasWidth $width
            }

            proc help {} {
                generalHelpWindow #configuration.application.size
            }

        }                                                                                                 ;# end of application size

        namespace eval colors {                                                                       ;# start of application colors

            proc variables {} {
                return canvasBackground
            }

            proc initialize {} {
                variable background [configuration::initialize canvasBackground]
            }

            proc edit {parentPath} {
                variable background
                variable colorViewer

                set message [configuration::createMessage $parentPath.message -text {Canvas background color:}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                    ;# occupy whole width
                grid rowconfigure $parentPath 0 -weight 1

                grid columnconfigure $parentPath 0 -weight 1

                set colorViewer\
                    [button $parentPath.choose -text Choose... -command "configuration::application::colors::choose $parentPath"]
                updateColorViewer
                grid $colorViewer -row 1 -column 1
                grid [button $parentPath.apply -text Apply -command configuration::application::colors::apply] -row 1 -column 2

                grid columnconfigure $parentPath 1 -weight 1
                grid columnconfigure $parentPath 2 -weight 1
                grid columnconfigure $parentPath 3 -weight 1

                grid rowconfigure $parentPath 2 -weight 1
            }

            proc check {} {
                return 1                                                                             ;# chosen color is always valid
            }

            proc apply {} {
                variable background

                if {![check]} return
                configuration::apply canvasBackground $background
            }

            proc updateColorViewer {} {
                variable colorViewer
                variable background

                foreach {red green blue} [winfo rgb $colorViewer $background] {}
                if {($red + $green + $blue) >= (32768 * 3)} {                                                    ;# light background
                    $colorViewer configure -foreground black
                } else {                                                                                          ;# dark background
                    $colorViewer configure -foreground white
                }
                $colorViewer configure -background $background
            }

            proc choose {parentPath} {
                variable background

                set choice [tk_chooseColor -initialcolor $background -title {Choose color:} -parent $parentPath]
                if {[string length $choice] > 0} {
                    set background $choice
                    updateColorViewer
                }
            }

            proc help {} {
                generalHelpWindow #configuration.application.colors
            }

        }                                                                                               ;# end of application colors

        namespace eval fonts {                                                                         ;# start of application fonts

            proc variables {} {
                return {fontFamily fontSize}
            }

            proc initialize {} {
                variable family [configuration::initialize fontFamily]
                variable size [configuration::initialize fontSize]
            }

            proc edit {parentPath} {
                variable family
                variable size
                variable label

                grid rowconfigure $parentPath 0 -weight 1
                grid rowconfigure $parentPath 3 -weight 1
                grid columnconfigure $parentPath 0 -weight 1
                grid columnconfigure $parentPath 5 -weight 1
                set message [configuration::createMessage $parentPath.message\
                    -text "Global font:\n(restart application for changes to take effect)"
                ]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100

                grid [label $parentPath.family -text Family:] -row 1 -column 1 -padx 2
                set entry [new comboEntry $parentPath -font $widget::option(entry,font) -editable 0\
                    -list [lsort -dictionary [font families]] -command configuration::application::fonts::family\
                ]
                lappend objects $entry
                composite::configure $entry entry -textvariable configuration::application::fonts::family
                composite::configure $entry button -listheight 10
                grid $widget::($entry,path) -row 1 -column 2
                set entry [new comboEntry $parentPath -font $widget::option(entry,font) -width 2 -editable 0\
                    -command configuration::application::fonts::size\
                    -list {0 2 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 24 25 26 32 33 34}\
                ]                                                                           ;# note: pixel sizes taken from xfontsel
                lappend objects $entry
                composite::configure $entry entry -textvariable configuration::application::fonts::size
                composite::configure $entry button -listheight 10
                grid $widget::($entry,path) -row 1 -column 3
                grid [label $parentPath.pixels -text pixels] -row 1 -column 4 -padx 2

                set label [label $parentPath.label -background $widget::option(entry,background) -relief sunken\
                    -borderwidth 1 -pady 5 -text ABCDEFGHIJKLMNOPQRSTUVWXYZ\nabcdefghijklmnopqrstuvwxyz
                ]
                grid $label -sticky ew -row 2 -column 0 -columnspan 100 -padx 10 -pady 10

                bind $message <Destroy> "delete $objects"                                   ;# delete inner objects upon destruction
                update
            }

            proc check {} {
                return 1                                                                             ;# chosen color is always valid
            }

            proc apply {} {
                variable family
                variable size

                if {![check]} return
                configuration::apply fontFamily $family
                configuration::apply fontSize $size
            }

            proc help {} {
                generalHelpWindow #preferences.application.fonts
            }

            proc family {name} {
                variable family $name
                update
            }

            proc size {value} {
                variable size $value
                update
            }

            proc update {} {
                variable family
                variable size
                variable label

                $label configure -font -*-$family-medium-r-*-*-$size-*
            }

        }                                                                                                ;# end of application fonts

        namespace eval printing {                                                                   ;# start of application printing

            proc variables {} {
                return {printToFile fileToPrintTo printCommand printOrientation printPalette printPaperSize}
            }

            proc initialize {} {
                variable toFile [configuration::initialize printToFile]
                variable printFile [configuration::initialize fileToPrintTo]
                variable command [configuration::initialize printCommand]
                variable orientation [configuration::initialize printOrientation]
                variable palette [configuration::initialize printPalette]
                variable size [configuration::initialize printPaperSize]
            }

            proc edit {parentPath} {
                variable toFile
                variable printFile
                variable command
                variable orientation
                variable palette
                variable size

                set objects {}                                                          ;# to delete upon destruction of this folder

                set row 0
                set message [configuration::createMessage $parentPath.message -text {Printing setup:}]
                grid $message -sticky nsew -row $row -column 0 -columnspan 3                                   ;# occupy whole width
                grid rowconfigure $parentPath $row -weight 1

                incr row
                radiobutton $parentPath.toCommand -variable configuration::application::printing::toFile -value 0 -text Command:
                grid $parentPath.toCommand -row $row -column 0 -sticky w -padx 2
                entry $parentPath.command -textvariable configuration::application::printing::command
                grid $parentPath.command -row $row -column 1 -columnspan 2 -sticky ew -padx 2

                incr row
                radiobutton $parentPath.toFile -variable configuration::application::printing::toFile -value 1 -text {to File:}
                grid $parentPath.toFile -row $row -column 0 -sticky w -padx 2
                entry $parentPath.file -textvariable configuration::application::printing::printFile
                grid $parentPath.file -row $row -column 1 -sticky ew -padx 2
                button $parentPath.browse\
                    -text Browse... -command "configuration::application::printing::inquirePrintFile $parentPath"
                grid $parentPath.browse -row $row -column 2 -sticky ew -padx 2
                if {$toFile} {
                    $parentPath.toFile invoke
                } else {
                    $parentPath.toCommand invoke
                }

                incr row
                grid [label $parentPath.orientation -text Orientation:] -row $row -column 0 -sticky w -padx 2
                set entry\
                    [new comboEntry $parentPath -font $widget::option(entry,font) -list $global::printOrientations -editable 0]
                lappend objects $entry
                composite::configure $entry entry -textvariable configuration::application::printing::orientation
                composite::configure $entry button -listheight [llength $global::printOrientations]
                composite::configure $entry button scroll -selectmode single
                grid $widget::($entry,path) -row $row -column 1 -columnspan 2 -sticky ew -padx 2

                incr row
                grid [label $parentPath.palette -text Palette:] -row $row -column 0 -sticky w -padx 2
                set entry [new comboEntry $parentPath -font $widget::option(entry,font) -list $global::printPalettes -editable 0]
                lappend objects $entry
                composite::configure $entry entry -textvariable configuration::application::printing::palette
                composite::configure $entry button -listheight [llength $global::printPalettes]
                composite::configure $entry button scroll -selectmode single
                grid $widget::($entry,path) -row $row -column 1 -columnspan 2 -sticky ew -padx 2

                incr row
                grid [label $parentPath.size -text {Paper size:}] -row $row -column 0 -sticky w -padx 2
                set entry [new comboEntry $parentPath -font $widget::option(entry,font) -list $global::printPaperSizes -editable 0]
                lappend objects $entry
                composite::configure $entry entry -textvariable configuration::application::printing::size
                composite::configure $entry button -listheight [llength $global::printPaperSizes]
                composite::configure $entry button scroll -selectmode single
                grid $widget::($entry,path) -row $row -column 1 -columnspan 2 -sticky ew -padx 2

                grid rowconfigure $parentPath [incr row] -weight 1
                grid columnconfigure $parentPath 1 -weight 1

                bind $message <Destroy> "delete $objects"                                   ;# delete inner objects upon destruction
            }

            proc inquirePrintFile {parentPath} {
                variable printFile

                set file [tk_getSaveFile\
                    -title {moodss: File to print to} -parent $parentPath -initialdir [file dirname $printFile]\
                    -defaultextension .ps -filetypes {{Postscript .ps} {{All files} *}} -initialfile [file tail $printFile]\
                ]
                if {[string length $file] > 0} {                                                                     ;# not canceled
                    set printFile $file
                }
            }

            proc check {} {
                return 1                                                                          ;# chosen values cannot be invalid
            }

            proc apply {} {
                variable toFile
                variable printFile
                variable command
                variable orientation
                variable palette
                variable size

                configuration::apply printToFile $toFile
                configuration::apply fileToPrintTo [file join [pwd] $printFile]                                ;# save absolute path
                configuration::apply printCommand $command
                configuration::apply printOrientation $orientation
                configuration::apply printPalette $palette
                configuration::apply printPaperSize $size
            }

            proc help {} {
                generalHelpWindow #preferences.application.printing
            }

        }                                                                                             ;# end of application printing

        namespace eval pages {                                                                         ;# start of application pages

            proc variables {} {
                return pagesTabPosition
            }

            proc initialize {} {
                variable position [configuration::initialize pagesTabPosition]
            }

            proc edit {parentPath} {
                set message [configuration::createMessage $parentPath.message -text {Pages tab position:}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                    ;# occupy whole width
                grid rowconfigure $parentPath 0 -weight 1
                grid columnconfigure $parentPath 0 -weight 1

                grid [\
                    radiobutton $parentPath.top -variable configuration::application::pages::position -value top -text top\
                ] -row 1 -column 1
                grid [\
                    radiobutton $parentPath.bottom -variable configuration::application::pages::position -value bottom -text bottom\
                ] -row 1 -column 2
                grid columnconfigure $parentPath 3 -weight 1
                grid rowconfigure $parentPath 2 -weight 1
            }

            proc check {} {
                return 1                                                                          ;# chosen values cannot be invalid
            }

            proc apply {} {
                variable position

                configuration::apply pagesTabPosition $position
                pages::labelsSide $position
            }

            proc help {} {
                generalHelpWindow #preferences.application.pages
            }

        }                                                                                                ;# end of application pages

        namespace eval database {                                                                   ;# start of application database

            proc variables {} {
                return databaseOptions                     ;# a list of switch, value, switch, ... usable as a Tcl array initializer
            }

            proc initialize {} {
                variable data
                variable password                                                                           ;# used for confirmation
                variable type

                set data(-file) {}                                   ;# for backward compatibility when SQLite was not yet supported
                array set data [configuration::initialize databaseOptions]
                if {[string length $data(-dsn)] > 0} {
                    set type odbc
                } elseif {[string length $data(-host)] > 0} {
                    set type mysql
                    if {![info exists data(-database)] || ([string length $data(-database)] == 0)} {
                        set data(-database) moodss                                                                     ;# by default
                    }
                } else {                                                                                                  ;# default
                    set type sqlite
                    if {[string length $data(-file)] == 0} {
                        set data(-file) moodss.dat                                                  ;# by default in local directory
                    }
                }
                catch {set password $data(-password)}                                                           ;# may not yet exist
                if {![info exists data(-debuglevel)]} {
                    set data(-debuglevel) 0                     ;# for backward compatibility when debug level was not yet supported
                }
                if {![string equal $::tcl_platform(platform) unix]} {set data(-debuglevel) 0}
            }

            proc edit {parentPath} {
                variable data
                variable message
                variable label
                variable radioButton
                variable checkButton
                variable entry
                variable password                                                                           ;# used for confirmation
                variable type

                # parameters: file, host, dsn, user, password, port, MySQL database
                set row 0
                set text {Database setup:}
                if {$global::database != 0} {
                    append text "\n(please disconnect from database first)"
                }
                set message [configuration::createMessage $parentPath.message -text $text]
                grid $message -sticky nsew -row $row -column 0 -columnspan 100                                 ;# occupy whole width
                grid rowconfigure $parentPath $row -weight 1
                incr row
                set radioButton(file) [radiobutton $parentPath.fileChoice\
                    -variable configuration::application::database::type -value sqlite -text {SQLite file:}\
                    -command configuration::application::database::update\
                ]
                grid $radioButton(file) -row $row -column 0 -sticky w -padx 2
                set entry(file) [entry $parentPath.file -textvariable configuration::application::database::data(-file)]
                grid $entry(file) -row $row -column 1 -columnspan 3 -sticky ew -padx 2
                set entry(choose) [button $parentPath.chooseFile\
                    -text Choose... -command "configuration::application::database::inquireSQLiteFile $parentPath"\
                ]
                grid $entry(choose) -row $row -column 4 -sticky e -padx 2
                incr row
                set radioButton(dsn) [radiobutton $parentPath.dsnChoice\
                    -variable configuration::application::database::type -value odbc -text {ODBC DSN:}\
                    -command configuration::application::database::update\
                ]
                grid $radioButton(dsn) -row $row -column 0 -sticky w -padx 2
                set entry(dsn) [entry $parentPath.dsn -textvariable configuration::application::database::data(-dsn)]
                grid $entry(dsn) -row $row -column 1 -columnspan 100 -sticky ew -padx 2
                incr row
                set radioButton(host) [radiobutton $parentPath.hostChoice\
                    -variable configuration::application::database::type -value mysql -text {MySQL host:}\
                    -command configuration::application::database::update\
                ]
                grid $radioButton(host) -row $row -column 0 -sticky w -padx 2
                set entry(host) [entry $parentPath.host -textvariable configuration::application::database::data(-host)]
                grid $entry(host) -row $row -column 1 -columnspan 100 -sticky ew -padx 2
                incr row
                set label(user) [label $parentPath.userLabel -text user:]
                grid $label(user) -row $row -column 0 -sticky w -padx 2
                set entry(user) [entry $parentPath.user -textvariable configuration::application::database::data(-user)]
                grid $entry(user) -row $row -column 1 -columnspan 100 -sticky ew -padx 2
                incr row
                set label(password) [label $parentPath.passwordLabel -text password:]
                grid $label(password) -row $row -column 0 -sticky w -padx 2
                set entry(password) [entry $parentPath.password\
                    -textvariable configuration::application::database::data(-password) -width 8 -show *\
                ]
                grid $entry(password) -row $row -column 1 -sticky ew -padx 2
                set label(confirm) [label $parentPath.confirmLabel -text confirm:]
                grid $label(confirm) -row $row -column 2 -padx 2
                set entry(confirm)\
                    [entry $parentPath.confirm -textvariable configuration::application::database::password -width 8 -show *]
                grid $entry(confirm) -row $row -column 3 -sticky ew -padx 2 -columnspan 2
                incr row
                set label(port) [label $parentPath.portLabel -text port:]
                grid $label(port) -row $row -column 0 -sticky w -padx 2
                set entry(port) [entry $parentPath.port -textvariable configuration::application::database::data(-port)]
                setupEntryValidation $entry(port) {{checkUnsignedInteger %P}}                         ;# filter on positive integers
                grid $parentPath.port -row $row -column 1 -columnspan 100 -sticky ew -padx 2
                incr row
                set label(database) [label $parentPath.databaseLabel -text database:]
                grid $label(database) -row $row -column 0 -sticky w -padx 2
                set entry(database) [entry $parentPath.database -textvariable configuration::application::database::data(-database)]
                grid $parentPath.database -row $row -column 1 -columnspan 100 -sticky ew -padx 2
                incr row
                set checkButton(trace) [checkbutton $parentPath.trace\
                    -variable configuration::application::database::data(-debuglevel) -text {Trace SQL statements and queries}\
                ]
                if {![string equal $::tcl_platform(platform) unix]} {$checkButton(trace) configure -state disabled}
                grid $checkButton(trace) -row $row -column 0 -columnspan 100 -sticky w -padx 2
                grid rowconfigure $parentPath [incr row] -weight 1
                grid columnconfigure $parentPath 1 -weight 1
                grid columnconfigure $parentPath 3 -weight 1
                update
            }

            proc update {} {
                variable data
                variable type
                variable label
                variable radioButton
                variable checkButton
                variable entry

                if {$global::database != 0} {                                                       ;# connected: disable everything
                    foreach name {file dsn host} {$radioButton($name) configure -state disabled}
                    foreach name {user password confirm port database} {$label($name) configure -state disabled}
                    foreach name {file choose dsn host user password confirm port database} {
                        $entry($name) configure -state disabled
                    }
                    $checkButton(trace) configure -state disabled
                    return
                }
                switch $type {
                    sqlite {
                        foreach name {file choose} {$entry($name) configure -state normal}
                        foreach name {user password confirm port database} {$label($name) configure -state disabled}
                        foreach name {dsn host user password confirm port database} {$entry($name) configure -state disabled}
                        focus $entry(file)
                    }
                    odbc {
                        foreach name {user password confirm} {$label($name) configure -state normal}
                        foreach name {dsn user password confirm} {$entry($name) configure -state normal}
                        foreach name {port database} {$label($name) configure -state disabled}
                        foreach name {file host choose port database} {$entry($name) configure -state disabled}
                        focus $entry(dsn)
                    }
                    mysql {
                        foreach name {user password confirm port database} {$label($name) configure -state normal}
                        foreach name {host user password confirm port database} {$entry($name) configure -state normal}
                        foreach name {file dsn choose} {$entry($name) configure -state disabled}
                        focus $entry(host)
                    }
                }
            }

            proc inquireSQLiteFile {parentPath} {
                variable data

                set file [tk_getSaveFile\
                    -title {moodss: SQLite File} -parent $parentPath\
                    -initialdir [file dirname $data(-file)] -initialfile [file tail $data(-file)]\
                ]
                if {[string length $file] > 0} {                                                                     ;# not canceled
                    set data(-file) $file
                }
            }

            proc check {} {
                variable data
                variable message
                variable type
                variable entry
                variable password

                if {![string equal $type sqlite] && ![string equal $data(-password) $password]} {
                    $message configure -font $font::(mediumBold) -text {passwords do not match:}
                    focus $entry(password)
                    return 0
                }
                switch $type {
                    sqlite {
                        if {[string length $data(-file)] == 0} {
                            $message configure -font $font::(mediumBold) -text {a file name is needed:}
                            focus $entry(file)
                            return 0
                        }
                        foreach name {host dsn user password port database} {set data(-$name) {}}
                    }
                    odbc {
                        if {[string length $data(-dsn)] == 0} {
                            $message configure -font $font::(mediumBold) -text {a DSN is needed:}
                            focus $entry(dsn)
                            return 0
                        }
                        foreach name {file host database} {set data(-$name) {}}
                    }
                    mysql {
                        if {[string length $data(-host)] == 0} {
                            $message configure -font $font::(mediumBold) -text {a host is needed:}
                            focus $entry(host)
                            return 0
                        }
                        if {[string equal $data(-host) localhost] && ([string length $data(-port)] > 0)} {
                            $message configure -font $font::(mediumBold) -text {port useless with local socket connection:}
                            focus $entry(port)
                            return 0
                        }
                        if {[string length $data(-database)] == 0} {
                            $message configure -font $font::(mediumBold) -text {a database name is needed:}
                            focus $entry(database)
                            return 0
                        }
                        foreach name {file dsn} {set data(-$name) {}}
                    }
                }
                return 1
            }

            proc apply {} {
                variable data

                if {![check]} return
                if {[string length $data(-file)] > 0} {
                    set data(-file) [file join [pwd] $data(-file)]                                             ;# save absolute path
                }
                configuration::apply databaseOptions [array get data]
            }

            proc help {} {
                generalHelpWindow #preferences.application.database
            }

        }                                                                                             ;# end of application database

        namespace eval trace {                                                                         ;# start of application trace

            proc variables {} {
                return traceNumberOfRows
            }

            proc initialize {} {
                variable numberOfRows [configuration::initialize traceNumberOfRows]
            }

            proc edit {parentPath} {
                variable numberOfRows
                variable message

                grid rowconfigure $parentPath 0 -weight 1
                grid rowconfigure $parentPath 2 -weight 1
                grid columnconfigure $parentPath 0 -weight 1
                grid columnconfigure $parentPath 3 -weight 1
                set message [configuration::createMessage $parentPath.message -text {Trace window configuration:}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100
                grid [label $parentPath.rows -text {number of rows:}] -row 1 -column 1 -padx 2 -sticky w
                set values {5 10 15 20 30 50 100}
                if {[package vcompare $::tcl_version 8.4] < 0} {
                    set entry [new spinEntry $parentPath -width 4 -list $values]
                    bind $message <Destroy> "delete $entry"                                         ;# delete entry upon destruction
                    spinEntry::set $entry $numberOfRows
                    grid $widget::($entry,path) -row 1 -column 2 -sticky e
                    set path $composite::($entry,entry,path)
                } else {                                                                            ;# use native widget if possible
                    set path [spinbox $parentPath.entry -width 4 -values $values]
                    $path set $numberOfRows
                    grid $path -row 1 -column 2 -sticky e
                }
                $path configure -textvariable configuration::application::trace::numberOfRows
                # filter on positive integers and limit entry length:
                setupEntryValidation $path {{checkMaximumLength 4 %P} {checkUnsignedInteger %P}}
            }

            proc check {} {
                variable numberOfRows
                variable message

                set valid 1
                if {[string length $numberOfRows] == 0} {
                    set text {please set number of rows.}
                    set valid 0
                } elseif {$numberOfRows == 0} {                                     ;# cannot be negative because of input filtering
                    set text {number of rows cannot be set to 0.}
                    set valid 0
                }
                if {!$valid} {
                    $message configure -font $font::(mediumBold) -text $text
                }
                return $valid
            }

            proc apply {} {
                variable numberOfRows

                if {![check]} return
                configuration::apply traceNumberOfRows $numberOfRows
            }

            proc help {} {
                generalHelpWindow #preferences.application.trace
            }

        }                                                                                                ;# end of application trace

    }                                                                                                          ;# end of application


    namespace eval viewers {

        proc initialize {} {}

        proc edit {parentPath} {
            set message [configuration::createMessage $parentPath.message -text {Viewers configuration}]
            pack $message -fill both -expand 1                                               ;# initially display help message above
        }

        proc check {} {
            return 1                                                                                         ;# no data in this page
        }

        proc apply {} {}

        proc variables {} {return {}}

        proc help {} {
            generalHelpWindow #configuration.viewers
        }

        namespace eval colors {                                                                           ;# start of viewers colors

            proc variables {} {
                return viewerColors
            }

            proc initialize {} {
                variable colors [configuration::initialize viewerColors]
            }

            proc edit {parentPath} {
                variable colorsFrame

                set message [configuration::createMessage $parentPath.message -text {Change colors:}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100
                grid rowconfigure $parentPath 0 -weight 1

                grid columnconfigure $parentPath 0 -weight 1

                set colorsFrame [frame $parentPath.colors -borderwidth 1 -background black]
                refresh
                grid $colorsFrame -row 1 -column 0

                grid rowconfigure $parentPath 2 -weight 1
            }

            proc refresh {} {
                variable colors
                variable colorsFrame

                eval destroy [winfo children $colorsFrame]                                     ;# eventually remove existing buttons
                set index 0
                foreach color $colors {
                    set button [button $colorsFrame.$index -background $color -activebackground $color -borderwidth 1]
                    $button configure -command "configuration::viewers::colors::choose $index"
                    pack $button -side left
                    incr index
                }
            }

            proc choose {index} {
                variable colors
                variable colorsFrame

                set button $colorsFrame.$index
                set background [tk_chooseColor -initialcolor [$button cget -background] -title {Choose color:} -parent $button]
                if {[string length $background] > 0} {
                    $button configure -background $background
                    set colors [lreplace $colors $index $index $background]
                }
            }

            proc check {} {
                return 1
            }

            proc apply {} {
                variable colors
                variable colorsFrame

                if {![check]} return
                if {![info exists colorsFrame]} return                                 ;# nothing to do as folder was never accessed
                configuration::apply viewerColors $colors
            }

            proc help {} {
                generalHelpWindow #configuration.viewers.colors
            }

        }                                                                                                   ;# end of viewers colors

        namespace eval graphs {                                                                           ;# start of viewers graphs

            proc variables {} {
                return {graphNumberOfIntervals graphMinimumY graphXAxisLabelsRotation}
            }

            proc initialize {} {
                variable numberOfSamples [configuration::initialize graphNumberOfIntervals]
                variable zeroBasedOrdinate [string equal [configuration::initialize graphMinimumY] 0]
                variable degrees [configuration::initialize graphXAxisLabelsRotation]
            }

            proc edit {parentPath} {
                variable numberOfSamples
                variable degrees
                variable message

                grid rowconfigure $parentPath 0 -weight 1
                grid rowconfigure $parentPath 4 -weight 1
                grid columnconfigure $parentPath 0 -weight 1
                grid columnconfigure $parentPath 4 -weight 1

                set message [configuration::createMessage $parentPath.message -text {Data graphs settings:}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100
                if {[info exists databaseInstances::singleton]} {                                           ;# database history mode
                    set state disabled
                } else {
                    set state normal
                }
                grid [label $parentPath.samplesLabel -text {X axis:} -state $state] -row 1 -column 1 -padx 2 -sticky e
                grid [set frame [frame $parentPath.samples]] -row 1 -column 2 -columnspan 100 -sticky w
                set values {20 50 100 150 200 300 500 1000}
                if {[package vcompare $::tcl_version 8.4] < 0} {
                    set entry [new spinEntry $frame -width 4 -side right -list $values -state $state]
                    spinEntry::set $entry $numberOfSamples
                    pack $widget::($entry,path) -side left
                    set path $composite::($entry,entry,path)
                } else {                                                                            ;# use native widget if possible
                    set path [spinbox $frame.entry -width 4 -values $values -state $state]
                    $path set $numberOfSamples
                    pack $path -side left
                }
                $path configure -textvariable configuration::viewers::graphs::numberOfSamples
                # filter on positive integers and limit entry length:
                setupEntryValidation $path {{checkMaximumLength 4 %P} {checkUnsignedInteger %P}}
                pack [label $frame.samples -text samples -state $state] -side left

                grid [label $parentPath.yAxis -text {Y axis:}] -row 2 -column 1 -padx 2 -sticky e
                set button [radiobutton $parentPath.zero\
                    -variable ::configuration::viewers::graphs::zeroBasedOrdinate -value 1 -text {zero based}\
                ]
                grid $button -row 2 -column 2
                set button [radiobutton $parentPath.scale\
                    -variable ::configuration::viewers::graphs::zeroBasedOrdinate -value 0 -text {auto scale}\
                ]
                grid $button -row 2 -column 3
                grid [label $parentPath.rotationLabel -text {X axis labels rotation:}] -row 3 -column 1 -padx 2 -sticky e
                grid [set frame [frame $parentPath.rotation]] -row 3 -column 2 -columnspan 100 -sticky w
                if {[package vcompare $::tcl_version 8.4] < 0} {
                    set entry [new spinEntry $frame -width 2 -side right -editable 0 -range {45 90 5}]
                    spinEntry::set $entry $degrees
                    pack $widget::($entry,path) -side left
                    set path $composite::($entry,entry,path)
                } else {                                                                            ;# use native widget if possible
                    set path [spinbox $frame.entry -width 2 -state readonly -from 45 -to 90 -increment 5]
                    $path set $degrees
                    pack $path -side left
                }
                $path configure -textvariable configuration::viewers::graphs::degrees
                pack [label $frame.degrees -text degrees] -side left
                grid [button $parentPath.apply -text Apply -command configuration::viewers::graphs::apply]\
                    -row 4 -column 0 -columnspan 100

                if {[package vcompare $::tcl_version 8.4] < 0} {
                    bind $message <Destroy> "delete $entry"                                 ;# delete inner objects upon destruction
                }
            }

            proc check {} {
                variable numberOfSamples
                variable message

                set valid 1
                if {[string length $numberOfSamples] == 0} {
                    set text {please set number of samples.}
                    set valid 0
                } elseif {$numberOfSamples == 0} {                                  ;# cannot be negative because of input filtering
                    set text {number of samples cannot be set to 0.}
                    set valid 0
                }
                if {!$valid} {
                    $message configure -font $font::(mediumBold) -text $text
                }
                return $valid
            }

            proc apply {} {
                variable numberOfSamples
                variable zeroBasedOrdinate
                variable degrees

                if {![check]} return
                configuration::apply graphNumberOfIntervals $numberOfSamples
                if {$zeroBasedOrdinate} {set minimum 0} else {set minimum {}}
                configuration::apply graphMinimumY $minimum
                configuration::apply graphXAxisLabelsRotation $degrees
                foreach graph $bltGraph::(graphs) {
                    composite::configure $graph -samples $numberOfSamples -xlabelsrotation $degrees
                    catch {composite::configure $graph -yminimum $minimum}                     ;# stacked graphs have no such option
                }
                if {[info exists databaseInstances::singleton]} {
                    composite::configure $databaseInstances::singleton -xlabelsrotation $degrees
                }
            }

            proc help {} {
                generalHelpWindow #configuration.viewers.graphs
            }

        }                                                                                                   ;# end of viewers graphs

        namespace eval pies {                                                                               ;# start of viewers pies

            proc variables {} {
                return pieLabeler
            }

            proc initialize {} {
                variable labeler [configuration::initialize pieLabeler]
            }

            proc edit {parentPath} {
                set message [configuration::createMessage $parentPath.message -text {Choose labeler type for data pies:}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100
                grid rowconfigure $parentPath 0 -weight 1

                grid columnconfigure $parentPath 0 -weight 1

                set button [radiobutton $parentPath.box -variable ::configuration::viewers::pies::labeler -value box -text box]
                grid $button -row 1 -column 1
                set button [radiobutton $parentPath.peripheral\
                    -variable ::configuration::viewers::pies::labeler -value peripheral -text peripheral\
                ]
                grid $button -row 1 -column 2

                grid columnconfigure $parentPath 3 -weight 1

                grid rowconfigure $parentPath 2 -weight 1
            }

            proc check {} {
                return 1                                                                            ;# simple choice so always valid
            }

            proc apply {} {
                variable labeler

                if {![check]} return
                configuration::apply pieLabeler $labeler
            }

            proc help {} {
                generalHelpWindow #configuration.viewers.pies
            }

        }                                                                                                     ;# end of viewers pies

        namespace eval tables {                                                                           ;# start of viewers tables

            proc variables {} {
                return currentValueTableRows
            }

            proc initialize {} {
                variable numberOfRows [configuration::initialize currentValueTableRows]
            }

            proc edit {parentPath} {
                variable numberOfRows
                variable message

                grid rowconfigure $parentPath 0 -weight 1
                grid rowconfigure $parentPath 2 -weight 1
                grid columnconfigure $parentPath 0 -weight 1
                grid columnconfigure $parentPath 3 -weight 1
                set message\
                    [configuration::createMessage $parentPath.message -text {Values table settings (in database history mode):}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100
                grid [label $parentPath.rows -text {maximum number of rows:}] -row 1 -column 1 -padx 2 -sticky w
                set values {10 20 50 100 200 500 1000 2000 5000 10000}
                if {[package vcompare $::tcl_version 8.4] < 0} {
                    set entry [new spinEntry $parentPath -width 6 -list $values]
                    bind $message <Destroy> "delete $entry"                                         ;# delete entry upon destruction
                    spinEntry::set $entry $numberOfRows
                    grid $widget::($entry,path) -row 1 -column 2 -sticky e
                    set path $composite::($entry,entry,path)
                } else {                                                                            ;# use native widget if possible
                    set path [spinbox $parentPath.entry -width 6 -values $values]
                    $path set $numberOfRows
                    grid $path -row 1 -column 2 -sticky e
                }
                $path configure -textvariable configuration::viewers::tables::numberOfRows
                # filter on positive integers and limit entry length:
                setupEntryValidation $path {{checkMaximumLength 6 %P} {checkUnsignedInteger %P}}
            }

            proc check {} {
                variable numberOfRows
                variable message

                set valid 1
                if {[string length $numberOfRows] == 0} {
                    set text {please set number of rows.}
                    set valid 0
                } elseif {$numberOfRows == 0} {                                     ;# cannot be negative because of input filtering
                    set text {number of rows cannot be set to 0.}
                    set valid 0
                }
                if {!$valid} {
                    $message configure -font $font::(mediumBold) -text $text
                }
                return $valid
            }

            proc apply {} {
                variable numberOfRows

                if {![check]} return
                configuration::apply currentValueTableRows $numberOfRows
            }

            proc help {} {
                generalHelpWindow #configuration.viewers.tables
            }

        }                                                                                                   ;# end of viewers tables

        namespace eval cells {                                                                             ;# start of viewers cells

            proc variables {} {
                return cellsLabelModuleHeader
            }

            proc initialize {} {
                variable identify [configuration::initialize cellsLabelModuleHeader]
            }

            proc edit {parentPath} {
                set message [configuration::createMessage $parentPath.message\
                    -text "Whether module identifier\nis included in data cells labels:"\
                ]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                    ;# occupy whole width
                grid rowconfigure $parentPath 0 -weight 1
                grid columnconfigure $parentPath 0 -weight 1
                grid [radiobutton $parentPath.top -variable configuration::viewers::cells::identify -value 1 -text yes]\
                    -row 1 -column 1
                grid [radiobutton $parentPath.bottom -variable configuration::viewers::cells::identify -value 0 -text no]\
                    -row 1 -column 2
                grid columnconfigure $parentPath 3 -weight 1
                grid rowconfigure $parentPath 2 -weight 1
            }

            proc check {} {
                return 1                                                                            ;# simple choice so always valid
            }

            proc apply {} {
                variable identify

                if {![check]} return
                configuration::apply cellsLabelModuleHeader $identify
                foreach viewer $viewer::(list) {
                    viewer::updateLabels $viewer
                }
            }

            proc help {} {
                generalHelpWindow #configuration.viewers.cells
            }

        }                                                                                                    ;# end of viewers cells

    }                                                                                                              ;# end of viewers


    namespace eval thresholds {

        proc initialize {} {}

        proc edit {parentPath} {
            set message [configuration::createMessage $parentPath.message -text {Thresholds configuration}]
            pack $message -fill both -expand 1
        }

        proc check {} {
            return 1                                                                                         ;# no data in this page
        }

        proc apply {} {}

        proc variables {} {return {}}

        proc help {} {
            generalHelpWindow #preferences.thresholds
        }

        namespace eval email {                                                                          ;# start of thresholds email

            proc variables {} {
                return {fromAddress smtpServers mailSubject mailBody}
            }

            proc initialize {} {
                variable from [configuration::initialize fromAddress]
                variable servers [configuration::initialize smtpServers]
                variable subject [configuration::initialize mailSubject]
                variable body [configuration::initialize mailBody]
            }

            proc edit {parentPath} {
                variable servers
                variable list
                variable body
                variable text
                variable parent $parentPath

                set row 0
                set message [configuration::createMessage $parentPath.message -text {Mail settings:}]
                grid $message -row $row -column 0 -columnspan 3 -pady 5                                        ;# occupy whole width
                incr row
                set label [label $parentPath.from -text {From address:}]
                grid $label -row $row -column 0 -columnspan 2 -sticky w -padx 2
                set entry [entry $parentPath.address -textvariable configuration::thresholds::email::from]
                grid $entry -row $row -column 2 -sticky ew -padx 2
                incr row
                set label [label $parentPath.out -justify left -text "Outgoing mail\nSMTP servers:"]
                grid $label -row $row -column 0 -columnspan 2 -sticky nw -padx 2
                set list [new listEntry $parentPath]
                listEntry::set $list $servers
                grid $widget::($list,path) -row $row -column 2 -sticky nsew -padx 2
                incr row
                set label [label $parentPath.subjectLabel -text Subject:]
                grid $label -row $row -column 0 -sticky w -padx 2
                set font $font::(fixedNormal)
                set entry [entry $parentPath.subject -font $font -textvariable configuration::thresholds::email::subject]
                grid $entry -row $row -column 1 -columnspan 2 -sticky ew -padx 2
                incr row
                set label [label $parentPath.bodyLabel -text Body:]
                grid $label -row $row -column 0 -sticky nw -padx 2
                set text [text $parentPath.body -height 1 -background white -font $font]
                $text insert end $body
                setupTextBindings $text
                grid $text -row $row -column 1 -rowspan 2 -columnspan 2 -sticky nsew -padx 2
                incr row
                set button [button $parentPath.default -text Default -command configuration::thresholds::email::default -padx 2]
                set tip [new widgetTip -path $button -text {reset email message subject and body to default values}]
                bind $button <Destroy> "delete $tip"
                grid $button -row $row -column 0 -sticky s
                grid [frame $parentPath.filler -height [font metrics $font -ascent]]
                grid rowconfigure $parentPath $row -weight 1
                grid columnconfigure $parentPath 2 -weight 1
                # delete inner objects upon destruction:
                bind $message <Destroy> "delete $list; unset configuration::thresholds::email::list"
            }

            proc default {} {
                variable subject
                variable body
                variable text

                set subject $global::mail(subject,default)
                set body $global::mail(body,default)
                $text delete 1.0 end
                $text insert end $body
            }

            proc check {} {                                       ;# only possible once this interface has been opened at least once
                variable from
                variable parent

                if {[string length [emailAddressError $from]] > 0} {
                    tk_messageBox -parent $parent -title {moodss: Email error} -type ok -icon error\
                        -message "$from: [emailAddressError $from]"
                    return 0
                }
                return 1
            }

            proc apply {} {
                variable from
                variable servers
                variable subject
                variable body
                variable text
                variable list

                configuration::apply fromAddress $from
                if {[info exists list]} {                                                                 ;# if interface is visible
                    set servers [listEntry::get $list]                                                               ;# update value
                    set body [$text get 1.0 end]
                }
                configuration::apply smtpServers $servers
                configuration::apply mailSubject [string trim $subject]
                configuration::apply mailBody [string trim $body]
            }

            proc help {} {
                generalHelpWindow #preferences.thresholds.email
            }

        }                                                                                                 ;# end of thresholds email

        namespace eval trace {                                                                          ;# start of thresholds trace

            proc variables {} {
                return traceThresholds
            }

            proc initialize {} {
                variable trace [configuration::initialize traceThresholds]
            }

            proc edit {parentPath} {
                set message [configuration::createMessage $parentPath.message\
                    -text "Whether thresholds messages\nare sent to the trace module:"]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100
                grid rowconfigure $parentPath 0 -weight 1

                grid columnconfigure $parentPath 0 -weight 1

                set button [radiobutton $parentPath.yes -variable ::configuration::thresholds::trace::trace -value 1 -text yes]
                grid $button -row 1 -column 1
                set button [radiobutton $parentPath.no -variable ::configuration::thresholds::trace::trace -value 0 -text no]
                grid $button -row 1 -column 2

                grid columnconfigure $parentPath 3 -weight 1

                grid rowconfigure $parentPath 2 -weight 1
            }

            proc check {} {
                return 1                                                                            ;# simple choice so always valid
            }

            proc apply {} {
                variable trace

                if {![check]} return
                configuration::apply traceThresholds $trace
            }

            proc help {} {
                generalHelpWindow #preferences.thresholds.trace
            }

        }                                                                                                 ;# end of thresholds trace

    }


    namespace eval daemon {

        proc variables {} {
            return moompsResourceFile
        }

        proc initialize {} {
            variable file [configuration::initialize moompsResourceFile]
            variable current $file
        }

        proc edit {parentPath} {
            variable file
            variable message

            set message [configuration::createMessage $parentPath.message]
            resetMessage $message
            grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                        ;# occupy whole width
            grid rowconfigure $parentPath 0 -weight 1
            grid [label $parentPath.label -text {preferences file:}] -row 1 -column 0 -sticky w -padx 2
            entry $parentPath.file -textvariable configuration::daemon::file -width 32
            grid $parentPath.file -row 2 -column 0 -sticky ew -padx 2
            grid columnconfigure $parentPath 0 -weight 1
            button $parentPath.browse -text Browse... -command "configuration::daemon::inquireFile $parentPath"
            grid $parentPath.browse -row 2 -column 1 -sticky e -padx 2
            grid rowconfigure $parentPath 3 -weight 1
        }

        proc resetMessage {message} {
            $message configure -font $font::(mediumNormal) -text {moomps daemon configuration:}
        }

        proc inquireFile {parentPath} {
            variable file

            set value [tk_getSaveFile\
                -title {moodss: daemon preferences file} -parent $parentPath\
                -initialdir [file dirname $file] -initialfile [file tail $file]\
            ]
            if {[string length $value] > 0} {                                                                        ;# not canceled
                set file $value
            }
        }

        proc check {} {
            variable file
            variable message

            resetMessage $message
            set user $::tcl_platform(user)
            if {[file exists $file]} {                                                                        ;# file already exists
                if {[file isdirectory $file]} {
                    set error {file cannot be a directory}
                } elseif {![file writable $file]} {
                    set error "file not writable by user: $user"
                } elseif {![catch {set channel [open $file]} error]} {
                    unset error
                    gets $channel
                    set line [string trim [gets $channel]]                                                   ;# retrieve second line
                    if {![string equal $line {<!DOCTYPE moompsPreferences>}]} {
                        set error {not a moomps reference file}
                    }
                    close $channel
                }
            } elseif {![file writable [file dirname $file]]} {                                    ;# file must eventually be created
                set error "directory: [file dirname $file]\nnot writable by user: $user"
            }
            if {[info exists error]} {                                                                        ;# there was a problem
                $message configure -font $font::(mediumBold) -text $error
                return 0
            } else {
                return 1
            }
        }

        proc apply {} {
            variable file
            variable current

            if {[string equal $file $current]} return         ;# no change: avoid checking every time the user validates preferences
            if {![check]} return
            set current $file
            configuration::apply moompsResourceFile [file join [pwd] $file]                                    ;# save absolute path
        }

        proc help {} {
            generalHelpWindow #preferences.moomps
        }

    }

    # contruct configuration variables list which can be used at any time (i.e. when saving configuration to file)
    variable variables
    set variables(0) {}                                                                                         ;# for configuration
    set variables(1) {}                                                                                           ;# for preferences
    foreach entry $hierarchy specific $configure {
        regsub -all {\.} $entry :: class
        if {$specific} {
            set variables(0) [concat $variables(0) [${class}::variables]]
        }
        set variables(1) [concat $variables(1) [${class}::variables]]
    }

}

}
