# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: kernmods.tcl,v 1.21 2004/01/01 11:39:06 jfontain Exp $


package provide kernmods [lindex {$Revision: 1.21 $} 1]
package require network 1

namespace eval kernmods {

    variable flag                                                                                                ;# module flags map
    array set flag {autoclean C deleted D uninitialized I unused U}

    array set data {
        updates 0
        0,label name 0,type ascii 0,message {Linux kernel module name}
        1,label size 1,type integer 1,message {size in bytes}
        2,label used 2,type integer 2,message {use count}
        3,label flags 3,type ascii 3,message {C: autoclean, D: deleted, I: uninitialized, U: unused}
        4,label referring 4,type ascii 4,message {list of referring modules} 4,anchor left
        sort {0 increasing}
        switches {-r 1 --remote 1}
    }
    set file [open kernmods.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar 1 $optionsName options
        variable remote
        variable data
        variable modulesFile

        if {![catch {set locator $options(--remote)}] || ![catch {set locator $options(-r)}]} {                 ;# remote monitoring
            set data(pollTimes) {60 10 20 30 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(protocol) $remote(user) $remote(host)
            set data(identifier) kernmods($remote(host))
            if {[string equal $::tcl_platform(platform) unix]} {
                set remote(command) "$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/version"
            } else {                                                                                                      ;# windows
                set remote(command) "plink -batch $remote(host) cat /proc/version"                 ;# host is rather a putty session
            }
            set file [open "| $remote(command)"]
            fileevent $file readable {set ::kernmods::remote(busy) 0}
            vwait ::kernmods::remote(busy)                                                             ;# do not hang user interface
        } else {
            set data(pollTimes) {30 5 10 20 60 120 300 600}
            set file [open /proc/version]
            set modulesFile [open /proc/modules]
        }
        regexp {^[\d\.]+} [lindex [gets $file] 2] version                          ;# ignore extra characters, such as in 2.2.0-pre1
        if {[info exists remote]} {
            # detect errors early (but avoid write on pipe with no readers errors by reading whole data)
            if {[catch {read $file} message] || [catch {close $file} message]} {
                error "on remote host $remote(host) as user $remote(user): $message"
            }
            # now set command for updates:
            if {[string equal $::tcl_platform(platform) unix]} {
                set remote(command) "$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/modules"
            } else {                                                                                                      ;# windows
                set remote(command) "plink -batch $remote(host) cat /proc/modules"
            }
        } else {
            close $file
        }
        if {[package vcompare $version 2] < 0} {                                                             ;# check kernel version
            error {at least a version 2 kernel is needed}
        }
    }

    set nextIndex 0

    proc update {} {
        variable remote

        if {[info exists remote]} {
            if {!$remote(busy)} {
                remoteUpdate
            }
        } else {
            localUpdate
        }
    }

    proc parse {line} {
        variable flag

        if {[regexp {^([^ ]+) +(\d+) +(\d+) *(.*)$} $line dummy name size count comments]} {
            set flags {}
            set modules {}
            regsub {\[} $comments \{ comments                             ;# replace brackets around modules in order to make a list
            regsub {\]} $comments \} comments                        ;# as comments format is: (flag) ... (flag) [module ... module]
            foreach item $comments {
                if {[scan $item {(%[^)]} literal]} {                                                                         ;# flag
                    append flags $flag($literal)
                } else {
                    set modules $item                                                                      ;# referring modules list
                }
            }
            return [list $name $size $count $flags $modules]
        } else {
            return {}
        }
    }

    proc localUpdate {} {
        variable data
        variable modulesFile

        seek $modulesFile 0                                                                         ;# rewind before retrieving data
        while {[gets $modulesFile line] >= 0} {
            if {[llength [set list [parse $line]]] == 0} continue
            foreach {name size count flags modules} $list {}
            updateEntryData $name $size $count $flags $modules
            set current($name) {}
        }
        cleanupEntriesData current
        incr data(updates)
    }

    proc remoteUpdate {} {
        variable remote

        set remote(busy) 1
        set file [open "| $remote(command)"]
        fileevent $file readable "kernmods::remoteUpdated $file"             ;# do not hang user interface and other modules updates
    }

    proc remoteUpdated {file} {
        variable remote
        variable data

        while {[gets $file line] >= 0} {
            lappend lines $line
        }
        read $file                                           ;# avoid write on pipe with no readers errors by reading remaining data
        if {[catch {close $file} message]} {
            flashMessage "error: $message"
            set lines {}                                                                       ;# consider data corrupted as a whole
        }
        foreach line $lines {
            if {[llength [set list [parse $line]]] == 0} continue
            foreach {name size count flags modules} $list {}
            updateEntryData $name $size $count $flags $modules
            set current($name) {}
        }
        cleanupEntriesData current
        set remote(busy) 0
        incr data(updates)
    }

    proc updateEntryData {name size count flags modules} {
        variable index
        variable nextIndex
        variable data

        if {[catch {set row $index($name)}]} {                                                                          ;# new entry
            set row [set index($name) $nextIndex]
            incr nextIndex
            set data($row,0) $name                                                                         ;# initialize static data
            set data($row,1) $size
        }
        set data($row,2) $count
        set data($row,3) $flags
        set data($row,4) $modules
    }

    proc cleanupEntriesData {currentNames} {
        upvar 1 $currentNames current
        variable index
        variable data

        foreach {name row} [array get index] {                                                        ;# cleanup disappeared entries
            if {[info exists current($name)]} continue
            unset index($name) data($row,0) data($row,1) data($row,2) data($row,3)
        }
    }

}
