# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: pages.tcl,v 1.48 2004/01/01 11:39:06 jfontain Exp $


# Display the unique global canvas in notebook pages, each displaying a disjoint area of the canvas (via the scroll region canvas
# option).
# Displayed objects (tables, viewers, ...) have no knowledge of which page they reside in, except perhaps by looking at their
# coordinates (but that would be cheating and would require knowing the page class implementation).
# In effect, pages are totally disconnected from displayed objects and that is a good thing.


class page {

    # use composite as base class instead of switched for compatibility with other viewers:
    proc page {this parentPath args} composite {[new frame $parentPath] $args} viewer {} {    ;# base widget frame remains invisible
        set book [pages::book]                                                                              ;# a single book is used
        set first [expr {[llength [$book pages]] == 0}]
        set ($this,page) [$book insert end $this -raisecmd "page::raised $this"]
        set ($this,book) $book
        set ($this,canvas) $pages::(canvas)                                              ;# only notebook canvas is useful to a page
        set ($this,drop) [new dropSite\
            -path $book -regioncommand "page::dropRegion $this" -formats HANDLES -command "pages::transferHandles $this"\
        ]
        composite::complete $this
        composite::configure $this -deletecommand "pages::deleted $this"
        if {$first} {   ;# the first page must be raised otherwise canvas is hidden when page is created dynamically (notebook bug?)
            $book raise $this
            pages::monitorActiveCells                                              ;# only for first page since others will be empty
        }
        # now that first page is artificially raised, keep track of user initiated raised state
        $book itemconfigure $this\
            -raisecmd "page::raised $this; composite::configure $this -raised 1" -leavecmd "composite::configure $this -raised 0"
    }

    proc ~page {this} {
        variable index
        variable ${this}monitored

        unset index($this)
        catch {unset ${this}monitored}
        delete $($this,drop)
        if {[info exists ($this,thresholds)]} {delete $($this,thresholds)}
        if {[info exists ($this,sequencer)]} {delete $($this,sequencer)}
        if {[info exists ($this,tip)]} {delete $($this,tip)}
        $($this,book) delete $this
        if {[string length $composite::($this,-deletecommand)] > 0} {
            uplevel #0 $composite::($this,-deletecommand)                                   ;# always invoke command at global level
        }
    }

    proc options {this} {    ;# force initialization of index and label. -raised option is only used internally and thus not public.
        return [list\
            [list -index {}]\
            [list -deletecommand {} {}]\
            [list -draggable 0 0]\
            [list -label {}]\
            [list -raised 0 0]\
        ]
    }

    proc set-deletecommand {this value} {}

    proc set-draggable {this value} {}

    proc set-index {this value} {
        variable index

        if {$composite::($this,complete)} {
            error {option -index cannot be set dynamically}
        }
        if {[string length $value] == 0} {                                                         ;# find a free index for the page
            foreach {page value} [array get index] {set taken($value) {}}
            set value 0; while {[info exists taken($value)]} {incr value}
        }                                                                                      ;# else forced value from a save file
        set index($this) $value
        set ($this,left) [expr {$value * $global::pagesWidth}]                                        ;# left coordinate of the page
    }

    proc set-label {this value} {
        $($this,book) itemconfigure $this -text $value
    }

    # private procedure, used internally to store state so that it can eventually be used when creating pages from a save file
    proc set-raised {this value} {}

    proc raised {this} {
        if {![info exists global::scroll]} return
        set x [lindex [$global::canvas xview] 0]          ;# for some reason, the horizontal view is changed by the operations below
        $global::canvas configure\
            -scrollregion [list $($this,left) 0 [expr {$($this,left) + $global::canvasWidth}] $global::canvasHeight]
        pack $widget::($global::scroll,path) -in $($this,page) -fill both -expand 1          ;# display scrolled canvas in this page
        $global::canvas xview moveto $x                                                             ;# keep the same horizontal view
    }

    proc editLabel {this} {
        set book $($this,book)
        foreach {x top right bottom} [$($this,canvas) bbox $this:text] {}
        set y [expr {($top + $bottom) / 2}]
        set entry [entry .pageLabel\
            -borderwidth 0 -highlightthickness 0 -width 0 -font [$book cget -font]\
            -validate key -validatecommand "$book itemconfigure $this -text %P; list 1"\
        ]                                                               ;# use text as tab label as it is edited for better feedback
        lifoLabel::push $global::messenger {enter page tab label (Return to valid, Escape to abort)}
        # when Return or Enter is pressed, use new text as label, else if Escape is pressed, abort:
        foreach key {<KP_Enter> <Return>} {
            bind $entry $key "composite::configure $this -label \[%W get\]; destroy %W; lifoLabel::pop $global::messenger"
        }
        bind $entry <Escape>\
            "page::set-label $this [list $composite::($this,-label)]; destroy %W; lifoLabel::pop $global::messenger"
        $entry insert 0 $composite::($this,-label)
        $entry selection range 0 end                                  ;# initially select all characters, which is a common behavior
        place $entry -in $($this,canvas) -anchor w -x $x -y $y                                     ;# place entry over existing text
        focus $entry
        ::update idletasks                                                                     ;# so entry is visible and grab works
        grab $entry
    }

    proc dropRegion {this} {
        foreach {left top right bottom} [$($this,canvas) bbox $this:text] {}    ;# tab text area (from BWidget notebook source code)
        set X [winfo rootx $($this,canvas)]; set Y [winfo rooty $($this,canvas)]
        return [list [incr left $X] [incr top $Y] [incr right $X] [incr bottom $Y]]                               ;# absolute region
    }

    proc supportedTypes {this} {                                                                               ;# same as thresholds
        return [thresholds::supportedTypes 0]
    }

    proc monitorCell {this array row column} {
        variable ${this}monitored

        set cell ${array}($row,$column)
        set ${this}monitored($cell) {}
    }

    proc forgetAllMonitoredCells {this} {
        variable ${this}monitored

        catch {unset ${this}monitored}
        if {[info exists ($this,thresholds)]} {delete $($this,thresholds); unset ($this,thresholds)}
        if {[info exists ($this,sequencer)]} {delete $($this,sequencer); unset ($this,sequencer)}
        if {[info exists ($this,tip)]} {switched::configure $($this,tip) -text {}}
        $($this,book) itemconfigure $this -background {}                                                    ;# resets tab background
    }

    proc update {this array} {}                                          ;# nothing to do, only threshold conditions are waited upon

    proc cells {this} {
        variable ${this}monitored

        return [array names ${this}monitored]
    }

    proc initializationConfiguration {this} {
        variable index

        return [list -index $index($this) -label $composite::($this,-label) -raised $composite::($this,-raised)]
    }

    proc thresholdCondition {this array row column color level summary} {
        variable ${this}monitored

        set cell ${array}($row,$column)
        if {![info exists ${this}monitored($cell)]} return
        if {![info exists ($this,thresholds)]} {                                                  ;# create manager only when needed
            set ($this,thresholds) [new thresholdsManager]
        }
        thresholdsManager::condition $($this,thresholds) $cell $color $level $summary
        foreach {colors summaries} [thresholdsManager::colorsAndTexts $($this,thresholds)] {}
        if {[info exists ($this,sequencer)]} {delete $($this,sequencer); unset ($this,sequencer)}
        if {[llength $colors] == 0} {
            $($this,book) itemconfigure $this -background {}                                       ;# restore default tab background
        } elseif {[llength $colors] == 1} {
            $($this,book) itemconfigure $this -background [lindex $colors 0]                                ;# colors tab background
        } else {                                                                         ;# display the different colors in sequence
            set ($this,sequencer) [new sequencer 1000 %c $colors "$($this,book) itemconfigure $this -background %c"]
            sequencer::start $($this,sequencer)
        }
        if {[llength $summaries] == 0} {
            if {[info exists ($this,tip)]} {switched::configure $($this,tip) -text {}}
        } else {
            if {![info exists ($this,tip)]} {
                set ($this,tip) [new widgetTip -path $($this,canvas) -itemortag p:$this]
            }
            set text {}
            set number 0
            foreach summary $summaries {
                if {$number < 3} {                                                                ;# display a maximum of 3 messages
                    if {$number > 0} {append text \n}
                    append text $summary
                }
                incr number
            }
            if {$number > 3} {append text \n...}                                  ;# give a visual clue that there are more messages
            switched::configure $($this,tip) -text $text
        }
    }

    proc manageable {this} {return 0}                                                       ;# pages manage their display themselves

    proc monitored {this cell} {
        variable ${this}monitored

        return [info exists ${this}monitored($cell)]
    }

}


class pages {

    proc pages {this} error                                                                                     ;# object less class

    proc manageScrolledCanvas {} {                                                                                     ;# idempotent
        if {[info exists (book)]} {
            grid forget $widget::($global::scroll,path)                                            ;# let pages display canvas areas
            grid $(book) -row 2 -column 0 -sticky nsew
            foreach page [$(book) pages] {    ;# raise the page that is supposed to be raised so that restoring from save file works
                if {[composite::cget $page -raised]} {
                    $(book) raise $page
                    break                                                                        ;# only a single page can be raised
                }
            }
            raise $widget::($global::scroll,path) $(book)                                      ;# so that scrolled canvas is visible
        } else {
            grid $widget::($global::scroll,path) -row 2 -column 0 -sticky nsew                    ;# display directly in main window
        }
    }

    proc deleted {page} {                                                                                   ;# page was just deleted
        if {[llength [$(book) pages]] == 0} {                                                                       ;# no more pages
            ::delete $(drag)
            destroy $(book)                                                                                          ;# destroy book
            unset (book) (canvas) (drag)
            # all objects must be eventually moved to the sole main page, otherwise they could become invisible:
            foreach handle [canvasWindowManager::handles $global::windowManager] {
                foreach {x y} [canvasWindowManager::handles::getGeometry $handle] break
                canvasWindowManager::handles::move $handle [expr {round($x) % $global::pagesWidth}] $y
            }
            set x [lindex [$global::canvas xview] 0]       ;# for some reason, the horizontal view is changed by the operation below
            $global::canvas configure -scrollregion [list 0 0 $global::canvasWidth $global::canvasHeight]
            $global::canvas xview moveto $x                                                         ;# keep the same horizontal view
            manageScrolledCanvas
        } else {
            $(book) raise [$(book) pages 0]                                                ;# always raise the first page by default
        }
    }

    proc closestPageTopLeftCorner {x} {                                                                          ;# from an abscissa
        return [list [expr {round(double($x) / $global::pagesWidth) * $global::pagesWidth}] 0]
    }

    proc dragData {format} {                                     ;# return page to be deleted to the eraser, format is always OBJECT
        set page [$(book) raise]                                                    ;# active page or nothing if no page is selected
        if {[string length $page] == 0} {return {}}
        # deleting the last remaining page is always allowed:
        if {([llength [$(book) pages]] <= 1) || [canvasWindowManager::currentPageEmpty $global::windowManager]} {
            return $page
        } else {    ;# warn the user that there must not be any table or viewer in the page for the page to be allowed to be deleted
            lifoLabel::flash $global::messenger {a page must be empty to be deleted}
            bell
            return {}
        }
    }

    proc validateDrag {x y} {
        foreach page [$(book) pages] {                                                            ;# mouse pointer must lie in a tab
            if {![composite::cget $page -draggable]} continue
            foreach {left top right bottom} [$(canvas) bbox p:$page] {}
            if {($x > $left) && ($x < $right) && ($y > $top) && ($y < $bottom)} {return 1}                               ;# in a tab
        }
        return 0                                                                                               ;# outside a page tab
    }

    proc labelsSide {value} {                                                                               ;# must be top or bottom
        if {![info exists (book)]} return
        $(book) configure -side $value
    }

    proc transferHandles {targetPage} {
        # a viewer was dropped into that page tab, so move the viewer to the upper left corner of that page:
        canvasWindowManager::handles::move $dragSite::data(HANDLES) $page::($targetPage,left) 0
        monitorActiveCells                                                                          ;# refresh pages monitored cells
    }

    proc book {} {                                                                    ;# return book, create it if it does not exist
        if {![info exists (book)]} {
            set (book) [NoteBook .book\
                -background $viewer::(background) -borderwidth 1 -internalborderwidth 0 -font $font::(mediumNormal)\
                -side $global::pagesTabPosition\
            ]
            $(book) bindtabs <Double-ButtonPress-1> page::editLabel
            $(book) bindtabs <ButtonPress-3> page::editLabel
            set (canvas) $(book).c                                                          ;# from the BWidget notebook source code
            set (drag) [::new dragSite -path $(canvas) -validcommand pages::validateDrag]    ;# for pages deletion by drop in eraser
            dragSite::provide $(drag) OBJECTS pages::dragData
            manageScrolledCanvas
        }
        return $(book)
    }

    proc edit {page} {
        $(book) see $page
        ::update idletasks                                           ;# make sure tab label is in place before attempting to edit it
        after idle "page::editLabel $page"
    }

    proc tagOrItemPage {value} {                           ;# return the page containing a canvas item or tag, or nothing if failure
        if {![info exists (book)]} {return {}}
        foreach {left top right bottom} [$global::canvas bbox $value] {}
        foreach page [$(book) pages] {
            set x [expr {$right - $page::($page,left)}]
            if {($x >= 0) && ($x < $global::pagesWidth)} {                   ;# right side must be in page area but not in next page
                return $page
            }
        }
        return {}
    }

    proc cellsWithActiveThreshold {pageCellsName} {                        ;# fill array with page as key and list of cells as value
        upvar 1 $pageCellsName pageCells

        foreach cell [thresholds::activeCells] {
            foreach viewer [viewer::monitoring $cell] {                                                           ;# viewer or table
                set page [canvasWindowManager::viewerPage $::global::windowManager $viewer]
                if {[string length $page] == 0} continue                                       ;# if an unmanaged viewer for example
                set ${page}($cell) {}
            }
            foreach table [dataTable::monitoring $cell] {
                set page [canvasWindowManager::viewerPage $::global::windowManager $table]
                if {[string length $page] == 0} continue
                set ${page}($cell) {}                        ;# suppress duplicates (viewers derived from table or including tables)
            }
        }
        foreach page [info locals] {
            if {![string is integer -strict $page]} continue                                       ;# a page is an object identifier
            set pageCells($page) [array names $page]
        }
    }

    # Since a page is a viewer, it is notified when a cell threshold condition occurs (color, level change, ...).
    # The following procedure is idempotent, and is invoked in the following cases:
    #  - thresholds updated in their dialog box
    #  - module unloaded
    #  - table or viewer moved between pages
    #  - viewer deleted
    #  - first page created
    # When a module is loaded, its tables cells do not have thresholds set yet.
    # When a viewer is created, it contains no cells or cells located in the same page, thus already accounted for.
    # When a page is created, it is empty of any table or viewer (except for the first page).
    proc monitorActiveCells {} {
        if {![info exists (book)]} return
        foreach page [$(book) pages] {
            page::forgetAllMonitoredCells $page                                                      ;# completely refresh all pages
        }
        cellsWithActiveThreshold data
        foreach {page cells} [array get data] {
            viewer::view $page $cells
        }
    }

    ### hack so that if there are several pages, a table from first page does not appear (as a ghost) on a second (raised) page ###
    proc eventuallyRefresh {} {
        if {![info exists (book)] || ([llength [$(book) pages]] < 2)} return             ;# only useful when there are several pages
        set region [$global::canvas cget -scrollregion]
        $global::canvas configure -scrollregion {0 0 0 0}
        update idletasks
        $global::canvas configure -scrollregion $region
    }

}
