# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: store.tcl,v 1.39 2004/02/06 23:19:42 jfontain Exp $

# Remember cells the data of which is to be stored for history purposes, in a database, by the moomps daemon.
# A dialog box is used with a table as a drop site for data cells.


class store {

    variable number
    set column 0
    foreach title {label active current comment} {                           ;# note: current column data is only for dialog box use
        set number($title) $column
        incr column
    }

    proc store {this args} switched {$args} viewer {} {
        variable singleton

        if {[info exists singleton]} {
            error {only 1 store object can exist}
        }
        switched::complete $this
    }

    proc ~store {this} {
        error {not implemented}
    }

    proc options {this} {
        return [list\
            [list -configurations {} {}]\
            [list -comments {} {}]\
        ]
    }

    proc set-configurations {this value} {      ;# list of lists of switch/value pairs from save file (used starting at moodss 17.7)
        set ($this,initializeIndex) 0                                        
    }

    proc set-comments {this value} {                     ;# list of comments, one per cell, from save file (used before moodss 17.7)
        set ($this,initializeIndex) 0
    }

    proc setData {dataName row cell active comment} {
        variable number
        upvar 1 $dataName data

        viewer::parse $cell array cellRow cellColumn type
        set label [viewer::label $array $cellRow $cellColumn 1]
        set data($row,-1) $cell
        set data($row,$number(label)) $label
        set data($row,$number(active)) $active
        set data($row,$number(current)) {}                                                                  ;# updated in dialog box
        set data($row,$number(comment)) $comment
    }

    proc sortedRows {dataName} {
        upvar 1 $dataName data

        set rows {}
        foreach name [array names data *,-1] {                                                    ;# cells are kept in hidden column
            lappend rows [lindex [split $name ,] 0]
        }
        return [lsort -integer $rows]                                                                           ;# in creation order
    }

    proc supportedTypes {this} {
        return $global::dataTypes                                                                                       ;# all types
    }

    # invoked by core during initialization from save file only, dropped cells must be handled below by dialog box:
    proc monitorCell {this array row column} {
        variable data
        variable number

        viewer::registerTrace $this $array
        set index $($this,initializeIndex)
        incr ($this,initializeIndex)
        set length [llength $switched::($this,-configurations)]
        if {$length > 0} {                                                                   ;# new format (starting at moodss 17.7)
            array set option [lindex $switched::($this,-configurations) $index]
            setData data $index ${array}($row,$column) $option(-active) $option(-comment)
            if {$($this,initializeIndex) == $length} {                                           ;# done initializing from save file
                switched::configure $this -configurations {}
            }
        } else {
            setData data $index ${array}($row,$column) 1 [lindex $switched::($this,-comments) $index]
            if {$($this,initializeIndex) == $length} {                                           ;# done initializing from save file
                switched::configure $this -comments {}
            }
        }
        if {$($this,initializeIndex) == $length} {unset ($this,initializeIndex)}
        if {[string first ? $data($index,$number(label))] >= 0} {                                  ;# label cannot be determined yet
            set ($this,relabel,$index) {}
        }
        set ($this,register,$index) {}                                                       ;# register with the database once only
    }

    proc update {this array} {     ;# if array is void, it means to eventually update database static data (invoked from dialog box)
        variable data
        variable number

        set externalUpdate [string length $array]
        foreach {name cell} [array get data *,-1] {                                               ;# cells are kept in hidden column
            if {$externalUpdate && ([string first $array $cell] != 0)} continue          ;# check that cell belongs to updated array
            set row [lindex [split $name ,] 0]
            viewer::parse $cell ignore cellRow cellColumn type
            if {[info exists ($this,relabel,$row)]} {                                      ;# if label is not yet defined, update it
                set label [viewer::label $array $cellRow $cellColumn 1]
                set data($row,$number(label)) $label
                if {[string first ? $label] < 0} {                                                   ;# label now completely defined
                    unset ($this,relabel,$row)
                    set ($this,register,$row) {}                                                        ;# eventually register again
                }
            }
            set database $global::database
            if {$database == 0} continue                                                      ;# no history to be stored in database
            if {!$data($row,$number(active))} continue                         ;# no history to be stored for this cell at this time
            set label $data($row,$number(label))
            set comment $data($row,$number(comment))
if {$global::withGUI} {
            if {[catch {set instance $($this,databaseInstance,$array)}]} {            ;# module instance registration yet to be done
                set instance [database::register $database [modules::instanceData $array]]
                if {[string length $database::($database,error)] > 0} {                 ;# any database error is fatal at this point
                    traceDialog {moodss fatal error: database module instance registration} $database::($database,error) 1
                    _exit 1                                                                                                 ;# abort
                }
                set ($this,databaseInstance,$array) $instance
            }
            if {[info exists ($this,register,$row)]} {       ;# note: do not prefix label with module identifier to avoid redundancy
                database::monitor $database $instance $cellRow $cellColumn [viewer::label $array $cellRow $cellColumn 0] $comment
                unset ($this,register,$row)
            }
            if {$externalUpdate} {                                                ;# only update static data when invoked internally
                set value ?; catch {set value [set $cell]}                                   ;# cell data may not or no longer exist
                database::update $database $instance $cellRow $cellColumn $value
            }
} else {
            if {[catch {set instance $($this,databaseInstance,$array)}]} {            ;# module instance registration yet to be done
                set instance [$database register [modules::instanceData $array]]           ;# use database object in its interpreter
                if {[string length [$database error]] > 0} {
                    exit 1                                                                                            ;# fatal error
                }
                set ($this,databaseInstance,$array) $instance
            }
            if {[info exists ($this,register,$row)]} {       ;# note: do not prefix label with module identifier to avoid redundancy
                $database monitor $instance $cellRow $cellColumn [viewer::label $array $cellRow $cellColumn 0] $comment
                unset ($this,register,$row)
            }
            # note: there are only external (core) updates in daemon mode
            set value ?; catch {set value [set $cell]}                                       ;# cell data may not or no longer exist
            $database update $instance $cellRow $cellColumn $value
}
        }
    }

    proc cells {this} {
        variable data

        set cells {}
        foreach row [sortedRows data] {
            lappend cells $data($row,-1)
        }
        return $cells
    }

    proc initializationConfiguration {this} {        ;# return a list of comments, one for each stored cell, in the cells list order
        variable number
        variable data

        set arguments {}
        foreach row [sortedRows data] {
            lappend arguments [list -active $data($row,$number(active)) -comment $data($row,$number(comment))]
        }
        return [list -configurations $arguments]
    }

    proc manageable {this} {return 0}                                           ;# dialog box is displayed and managed locally below

if {$global::withGUI} {

    proc reload {dataName} {                                  ;# data comes from dialog box table that the user edited and validated
        variable data
        variable singleton
        upvar 1 $dataName new

        reset $singleton
        array set data [array get new]
        foreach row [sortedRows data] {
            viewer::parse $data($row,-1) array dummy dummy dummy
            viewer::registerTrace $singleton $array
            set ($singleton,register,$row) {}           ;# register with the database for new cells or in case comments were changed
            store::update $singleton {}                                                    ;# eventually update static database data
        }
    }

    proc monitored {this cell} {
        variable data

        foreach {name monitored} [array get data *,-1] {
            if {[string equal $monitored $cell]} {
                return 1
            }
        }
        return 0
    }

    proc reset {this} {                                                       ;# return to original state when singleton was created
        variable data

        foreach row [sortedRows data] {
            viewer::parse $data($row,-1) array dummy dummy dummy
            viewer::unregisterTrace $this $array
        }
        catch {unset data}
        switched::configure $this -comments {}
    }

}

}

set ::store::singleton [new store]


if {$global::withGUI} {

class store {

    proc edit {writable destroyCommand} {
        if {[info exists (dialog)]} {                                                                                      ;# exists
            raise $widget::($dialog::($(dialog),dialog),path)                                                     ;# make it visible
        } else {
            append destroyCommand "\nunset store::(dialog)"
            set (dialog) [new dialog . $writable $destroyCommand]
        }
    }

    proc setCellColor {this cell color} {
        variable ${this}data

        if {![info exists (dialog)]} return                                                                    ;# nothing to display
        dialog::setCellColor $(dialog) $cell $color
    }

    class dialog {

        variable help
        set help(label) {data cell identification}
        set help(active) {whether data cell history should be recorded in database}
        set help(current) {current value of data cell}
        set help(comment) {user editable comment}

        proc dialog {this parentPath writable {deleteCommand {}}} viewer {} {
            variable ${this}data

            set dialog [new dialogBox .\
                -buttons hoc -default o -title [mc {moodss: Database History Cells}]\
                -helpcommand {generalHelpWindow #menus.edit.database} -x [winfo pointerx .] -y [winfo pointery .]\
                -grab release -otherbuttons delete -command "set store::dialog::($this,valid) 1" -deletecommand "delete $this"\
            ]
            lappend ($this,tips) [linkedHelpWidgetTip $composite::($dialog,help,path)]
            composite::configure $dialog delete -text Delete -underline 0 -command "store::dialog::delete $this" -state disabled
            set frame [frame $widget::($dialog,path).frame]
            set table [createTable $this $frame]
            set ($this,drop) [new dropSite -path $selectTable::($table,tablePath) -formats DATACELLS\
                -command "store::dialog::dropped $this \$dragSite::data(DATACELLS)"\
            ]
            pack $widget::($table,path) -anchor nw -fill both -expand 1
            wm geometry $widget::($dialog,path) 400x300
            dialogBox::display $dialog $frame
            set ($this,table) $table
            set ($this,dialog) $dialog
            array set ${this}data [array get store::data]                                    ;# copy valid data in case user cancels
            selectTable::rows $table [llength [array names ${this}data *,-1]]
            initialize $this [store::sortedRows ${this}data] $writable
            selectTable::refreshBorders $table                                                ;# needed if there are multi-line rows
            selectTable::adjustTableColumns $table
            colorRows $this
            set ($this,valid) 0                                                            ;# whether the user validated its choices
            set ($this,deleteCommand) $deleteCommand
        }

        proc ~dialog {this} {
            variable ${this}data

            if {$($this,valid)} {                                                                      ;# user validated its choices
                store::reload ${this}data
            }
            eval ::delete $($this,tips) $($this,drop) $($this,table)
            catch {unset ${this}data}
            if {[string length $($this,deleteCommand)] > 0} {
                uplevel #0 $($this,deleteCommand)                                           ;# always invoke command at global level
            }
        }

        proc createTable {this parentPath} {
            variable ${this}data
            variable help

            set table [new selectTable $parentPath\
                -selectcommand "store::dialog::select $this" -variable store::dialog::${this}data -titlerows 1 -roworigin -1\
            ]
            set path $selectTable::($table,tablePath)
            set column 0
            foreach title {label active current comment} {                                                           ;# table titles
                set ${this}data(-1,$column) $title
                incr column
            }
            # set number of columns according to title row data
            composite::configure $table -columns [llength [array names ${this}data -1,*]]
            foreach {cell title} [array get ${this}data -1,*] {
                set label [label $path.$cell -font $font::(mediumBold) -text $title]
                selectTable::windowConfigure $table $cell -window $label -padx 1 -pady 1 -sticky nsew
                lappend ($this,tips) [new widgetTip -path $label -text $help($title)]
            }
            selectTable::tag $table configure disabled -state disabled
            selectTable::tag $table col disabled $store::number(label)
            return $table
        }

        proc dropped {this cells} {                                                           ;# cells is a list of data array cells
            variable ${this}data

            set table $($this,table)
            foreach {name cell} [array get ${this}data *,-1] {                                    ;# cells are kept in hidden column
                set saved($cell) {}
            }
            set rows [store::sortedRows ${this}data]
            set length [llength $rows]
            if {$length == 0} {
                set last -1
            } else {
                set last [lindex $rows end]
            }
            set row $last
            set new {}
            foreach cell $cells {
                if {[info exists saved($cell)]} continue                                                            ;# already saved
                viewer::parse $cell array ignore ignore ignore
                set module [modules::identifier $array]
                if {[string length $module] == 0} {                                ;# ignore cells not attached to a module instance
                    lifoLabel::flash $global::messenger {data does not belong do an original module table}
                    bell
                    continue
                }
                if {[string equal $module trace]} {                                                     ;# ignore trace module cells
                    lifoLabel::flash $global::messenger {cannot monitor cells from trace module}
                    bell
                    continue
                }
                store::setData ${this}data [incr row] $cell 1 {}
                # row height is number of lines
                selectTable::height $table $row [linesCount [set ${this}data($row,$store::number(label))]]
                lappend new $row
                incr length
            }
            if {[llength $new] > 0} {                                                             ;# one or more new rows were added
                selectTable::rows $table $length                                                              ;# including title row
                initialize $this $new
                selectTable::refreshBorders $table
                selectTable::adjustTableColumns $table
                # color rows according to threshold condition (do it last since a tktable bug undoes it when number of rows changes)
                colorRows $this
                update $this {}
            }
        }

        proc select {this row} {
            set topPath $widget::($($this,dialog),path)
            set button $composite::($($this,dialog),delete,path)
            $button configure -state normal
            bind $topPath <Alt-KeyPress-d> "$button configure -relief sunken"      ;# make sure that only this button sees the event
            bind $topPath <Alt-KeyRelease-d> "$button configure -relief raised; $button invoke"
            return 1
        }

        proc delete {this} {
            variable ${this}data

            set table $($this,table)
            set row [selectTable::selected $table]
            if {[string length $row] == 0} return
            set path $selectTable::($table,tablePath)
            foreach index [store::sortedRows ${this}data] {                                                    ;# delete all entries
                destroy $path.$index,active $path.$index,comment
            }
            viewer::parse [set ${this}data($row,-1)] array dummy dummy dummy
            viewer::unregisterTrace $this $array
            array unset ${this}data $row,*                                                                ;# delete related row data
            array set data [array get ${this}data]
            unset ${this}data
            set row 0; set rows {}
            foreach index [store::sortedRows data] {
                set ${this}data($row,-1) $data($index,-1) 
                set column $store::number(label); set ${this}data($row,$column) $data($index,$column)
                set column $store::number(active); set ${this}data($row,$column) $data($index,$column)
                set column $store::number(comment); set ${this}data($row,$column) $data($index,$column)
                lappend rows $row; incr row
            }
            selectTable::rows $table $row
            initialize $this $rows
            selectTable::clear $table
            selectTable::refreshBorders $table                                                ;# needed if there are multi-line rows
            selectTable::adjustTableColumns $table
            colorRows $this                                                  ;# possibly recolor rows since indexes may have changed
            set topPath $widget::($($this,dialog),path)
            bind $topPath <Alt-KeyPress-d> {}; bind $topPath <Alt-KeyRelease-d> {}
            composite::configure $($this,dialog) delete -state disabled
        }

        proc setCellColor {this cell color} {                                          ;# implementation of the base class procedure
            variable ${this}data

            foreach {name value} [array get ${this}data *,-1] {                                   ;# cells are kept in hidden column
                if {[string equal $value $cell]} {                                                                  ;# cell is saved
                    colorRow $this [lindex [split $name ,] 0] $color
                    return
                }
            }
        }

        proc colorRow {this row color} {                                ;# actually only the cell current value column changes color
            # note: no need to handle special corner case since the last column contains data that is not subject to highlighting
            set cell $row,$store::number(current)
            if {[string length $color] == 0} {
                selectTable::tag $($this,table) cell {} $cell                                                    ;# reset cell color
            } else {
                selectTable::tag $($this,table) configure color$color -background $color
                selectTable::tag $($this,table) cell color$color $cell
            }
        }

        proc colorRows {this} {                                                ;# color all rows according to cells threshold colors
            variable ${this}data

            foreach {name cell} [array get ${this}data *,-1] {                                    ;# cells are kept in hidden column
                viewer::parse $cell array row column type
                colorRow $this [lindex [split $name ,] 0] [viewer::cellThresholdColor $array $row $column]
            }
        }

        proc initialize {this rows {writable 1}} {
            variable ${this}data

            set table $($this,table)
            set path $selectTable::($table,tablePath)
            set background [$path cget -background]
            foreach row $rows {
                set cell [set ${this}data($row,-1)]
                viewer::parse $cell array dummy dummy dummy
                viewer::registerTrace $this $array                                     ;# monitor data cell for current value column
                set cell $row,$store::number(active)
                set button [checkbutton $path.$row,active\
                    -activebackground $background -highlightthickness 0 -variable store::dialog::${this}data($cell) -takefocus 0\
                ]
                bind $button <ButtonRelease-1> "selectTable::select $table $row"
                selectTable::windowConfigure $table $cell -window $button -padx 1 -pady 1 -sticky nsew
                set cell $row,$store::number(comment)
                set entry [entry $path.$row,comment\
                    -font $font::(mediumNormal) -textvariable store::dialog::${this}data($cell) -borderwidth 0\
                    -highlightthickness 0\
                ]
                if {!$writable} {
                    $entry configure -state disabled
                }
                bind $entry <FocusIn> "selectTable::select $table $row"
                selectTable::windowConfigure $table $cell -window $entry -padx 1 -pady 1 -sticky nsew
                # row height is number of lines
                selectTable::height $table $row [linesCount [set ${this}data($row,$store::number(label))]]
            }
            update $this {}
        }

        proc update {this array} {                            ;# if array is empty, it is an internal invocation to update all cells
            variable ${this}data

            set externalUpdate [string length $array]
            foreach {name cell} [array get ${this}data *,-1] {                                    ;# cells are kept in hidden column
                if {$externalUpdate && ([string first $array $cell] != 0)} continue      ;# check that cell belongs to updated array
                set row [lindex [split $name ,] 0]
                set value ?
                catch {set value [set $cell]}
                set ${this}data($row,$store::number(current)) $value
            }
        }

        proc saved {this} {return 0}                                                                  ;# no need to save this viewer

        proc manageable {this} {return 0}                                           ;# dialog box is obviously not managed in canvas

        proc reset {this} {                                                ;# invoked by core for example when clearing display, ...
            ::delete $($this,dialog)                                        ;# delete dialog object which in turn delete this object
        }

    }

}

}
