=begin

= Name
mooix.rb - Mooix bindings for Ruby

= Synopsis

  #!/usr/bin/ruby
  require "mooix"
  require "mooix/thing"
  . . .
  Mooix.run do
  . . .
  end

= Description

mooix.rb provides a Ruby interface to <((<Mooix|URL:http://mooix.net>))>.

=end

=begin
= Variables
--- $autoconvert
    Set this to true to convert strings containing only digits and
    decimals to integers. (default)
=end

$autoconvert = true

=begin
= Mooix module

This module holds various assorted variables and methods which, though
essential, did not fit easily within the ((|Thing|)) class.

= Variables
=end

require "mooix/thing"

module Mooix

  class SetfieldFailure <Exception; end

=begin
--- Mooix::OK
    Return code indicating success.
=end

  OK = 0

=begin
--- Mooix::FAIL
    Return code indicating failure. Method handled command, but action
    failed.
=end

  FAIL = 10

=begin
--- Mooix::SKIP
    Return code indicating neither success nor failure, but that this
    method is incapable of handling the task for which it was called,
    and another should be tried instead.
=end

  SKIP = 20

=begin
--- Mooix::EXIT
    Return code indicating that the user should be logged out.
=end

  EXIT = 30

=begin
--- Mooix::SETIT
    Return code indicating that the object referred to by ((|it|)) or
    ((|them|)) must be changed. Return value is the object to which
    ((|it|)) now refers.
=end

  SETIT = 40

=begin
--- Mooix::SETITREF
    Return code indicating that the object referred to by ((|it|)) or
    ((|them|)) must be changed. Return value is the object to which
    ((|it|)) now refers. Unlike ((|Mooix::SETIT|)), this object must
    be usable as a reference.
=end

  SETITREF = 50

=begin
--- Mooix::ROOT
    String indicating the root of the local object hierarchy.
=end

  ROOT = "$(localstatedir)/lib/mooix"

=begin
--- Mooix::ROOT
    Root object of the local hierarchy.
=end

  Root = Thing.new(ROOT)

  def Mooix.find_method_path(name, dir, mixin = nil)
    if (mixin == nil)
      return dir if File.exists?("#{dir}/#{name}")
      return find_method_path(name, "#{dir}/parent") if File.directory?("#{dir}/parent")
      return nil
    else
      if File.directory?(dir+"/"+mixin)
        mdir = find_method_path(name, dir+"/"+mixin)
        return nil if mdir == nil
        return mdir
      else
        return find_method_path(name, dir+"/parent", mixin) if File.directory?(dir+"/parent")
      end
    end
  end
  def Mooix.escape(str)
    str = str.gsub(/\\\\/, "\\")
    str = "\"#{str}\"" if str.include?('\n')
    return str
  end
  def Mooix.unescape(val)
    str2 = val.gsub(/\\\\/) { '\\' }
    str = str2.gsub(/\\n/, "\\n")
    if str[0,1] == '"' and str[str.length-1,1] == '"'
      str[0,1] = "" 
      str[str.length-1,1] = ""
    end
    return str
  end
  def Mooix.parse_mooix(val)
    if val =~ /^mooix\:(.+)/
    newthing = Thing.new($1) 
      newthing = Thinglist.new($1) if newthing.has?("list")
      return newthing
    end
    val = unescape(val)
    return val.to_i if $autoconvert and val =~ /^\d+$/
    return val
  end
  def Mooix.ruby_to_mooix(val)
    str = ""
    val = val.to_a.flatten if val.instance_of?(Hash)
    if val.instance_of?(Array)
      val.each { |elt| str += ruby_to_mooix(elt) }
    else
      str += escape(val.to_s) + "\n"
    end
    return str
  end

=begin
= Functions
=end

=begin
--- Mooix.return(*val)
Returns ((|val|)) to calling method. As the standard ((|return|))
returns from functions, and Mooix methods run within blocks, the stock
function does not work.
    :Parameters:
      : ((|*val|))
        arguments to the overridden method.
    :Returns:
      ((|val|)) to caller.
=end

  def Mooix.return(*val)
    return_with(val, OK)
  end

=begin
--- Mooix.return_with(*val, ret)
Returns ((|val|)) to calling method. As the standard ((|return|))
returns from functions, and Mooix methods run within blocks, the stock
function does not work. Unlike ((|Mooix::return|)), this method also
returns a return code.
    :Parameters:
      : ((|*val|))
        arguments to the overridden method.
      : ((|*ret|))
        status code.
    :Returns:
      ((|val|)) to caller.
=end

  def Mooix.return_with(val, ret)
    if val.instance_of?(Array)
      val.map! { |elt| ruby_to_mooix(elt) }
      $stdout.write(val.join("\n"))
    else
      $stdout.write(ruby_to_mooix(val))
    end
    exit(ret)
  end

=begin
--- Mooix.run{ . . . }
All Mooix Ruby methods must be wrapped in Mooix.run blocks, from which
they return by calling ((|Mooix.return|)).
=end

  def Mooix.run(&block)
    begin
      $debug = ENV["MOOIX_DEBUG"]
      $debug = parse_mooix($debug) if $debug != nil
      $in_run = true
      args = $stdin.readlines.each { |val| val.chomp! }
      here = ENV['THIS']
      $ZERO = if ENV.key?('METHOD')
	ENV['METHOD']
      else
	$0
      end
      $stderr.puts("Setting $ZERO is <#{$ZERO}>") if $debug != nil
      $root = here
      $stderr.puts("Doing initial chdir to #{here}") if $debug != nil
      Dir.chdir(here)
      #this = Thing.new(".")
      this = Thing.new(here, true)
	this.args = args.map { |v| parse_mooix(v) }
      this.instance_eval(&block)
      rescue Errno::EACCES, SetfieldFailure
      # This -5 is probably a bad idea, but it seems to work for now.
      $stderr.puts("Error: #{$!} at #{$!.backtrace[-5]}\nCalltrace:")
      this.calltrace.each{ |x| $stderr.puts("\t#{x}") }
    rescue SystemExit => sx
      raise sx
    rescue Exception => err
      $stderr.puts("Error: #{err.to_s}")
      $stderr.puts(err.backtrace.map { |elt| "\t"+elt+"\n" })
      puts("Method failure.")
      exit(Mooix::FAIL)
    end
    ensure $in_run = false
  end

=begin
--- Mooix.background [do . . . end]
Convenience wrapper around ((|fork|)). If given a block, runs its
contents in the background, else returns the pid of the child process.
    :Returns:
      pid of child process.
=end

  def Mooix.background
    pid = fork
    return pid if pid != nil
    $stdin.close
    $stdout.close
    File.new("/dev/null", "r") #new stdin
    File.new("/dev/null", "w") #new stdout
    if block_given?
      yield
      exit!(0)
    else
      return nil
    end
  end
end

=begin
= Authors

* Abraham Egnor
* Nolan Darilek <((<nolan@thewordnerd.info|URL:mailto:nolan@thewordnerd.info>))>

Copyright (c) 2003, Abraham Egnor and Nolan Darilek.
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.
* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=end
