/*
 * Mooix proxy wrapper library.
 * Public utility functions.
 *
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the modified BSD license given in full in the file
 * COPYRIGHT.
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <libgen.h>

#include "libmooproxy.h"
#include "proxy.h"

/* The device and inode of the object the method is running on. */
dev_t this_dev;
ino_t this_inode;

/*
 * Called after cd's, to see if we're still in the object in THISFD. Updates
 * in_object appropriatly.
 */
void check_in_object (void) {
	struct stat buf;
	stat(".", &buf);
	if (buf.st_ino == this_inode && buf.st_dev == this_dev) {
		in_object = 1;
		mooix_debug(PRIORITY_DEBUG, "still in object..");
	}
	else {
		mooix_debug(PRIORITY_DEBUG, "no longer in object");
		in_object = 0;
	}
}

/* Enable proxying if THISFD is open, or if forced. */
void mooix_proxy_enable (int force) {
	struct stat buf;
	
	if (fstat(THISFD, &buf) != 0 && ! force)
		proxy_enabled = 0;
	else
		proxy_enabled = 1;
	mooix_debug(PRIORITY_DEBUG, "proxy: %i", proxy_enabled, NULL);
	this_dev = buf.st_dev;
	this_inode = buf.st_ino;
}

/* Called at library load time. Requires linking with -nostdlib. */
void _init (void) {
	mooix_debug(PRIORITY_DEBUG, "startup", NULL);
	proxysock = -1;
	
	/* Collect some state info we'll probably need later. */
	mooix_proxy_enable(0);
	in_object = 0;
	umask(mask = umask(0));
	mooix_debug(PRIORITY_DEBUG, "startup complete", NULL);
}

#ifdef DEBUG
/* Output a debug message if MOOIX_PROXY_DEBUG is set to priority or higher. */
/* (Or if ALWAYS_DEBUG is set. */
void mooix_debug (int priority, const char *fmt, ...) {
#ifndef ALWAYS_DEBUG
	char *debuglevel = getenv("MOOIX_PROXY_DEBUG");
	if (priority == PRIORITY_ERROR || 
	    (debuglevel && atoi(debuglevel) >= priority)) {
#endif
		/* It's important that the entire debug message be written
		 * atomically, with one syscall. Some regression tests depend
		 * on it. */
		char *fullfmt = malloc((strlen(fmt) + 29) * sizeof(char));
		va_list ap;
		va_start(ap, fmt);
		sprintf(fullfmt, "mooix[%i]: %s\n", getpid(), fmt);
		vfprintf(stderr, fullfmt, ap);
		free(fullfmt);
#ifndef ALWAYS_DEBUG
	}
#endif
}
#endif /* DEBUG */

/* Checks to see if the passed filename is part of a mooix oject. */
int mooix_obj (const char *filename) {
	char *dir, *moofile=malloc(PATH_MAX + 16);
	struct stat buf;

	strcpy(moofile, filename);
	dir = dirname(moofile);
	if (dir != moofile)
		strcpy(moofile, dir);
	strcat(moofile, "/" MOOFILE);
	if (stat(moofile, &buf) == 0) {
		free(moofile);
		return 1;
	}
	else {
		free(moofile);
		return 0;
	}
}

/*
 * Checks to see if the given filename is part of the mooix object that
 * THISFD is open to. If so, returns the basename of the filename.
 * 
 * The follow_symlinks should be set only if the operation that will be
 * performed on the filename is one that follows symlinks, like chmod. But
 * not if it's one that operates on symlinks, like unlink does.
 * 
 * The returned value is always freshly malloced, and should be freed after
 * use.
 *
 * NULL on failure, of course.
 */
char *is_field (const char *filename, int follow_symlinks) {
	char *file, *dir, *base;
	struct stat buf;
	
	/* If we're in the object, just look for relative filenames as some
	 * easy cases. */
	if (in_object) {
		if (strchr(filename, '/') == NULL &&
		    strcmp(filename, "..") != 0) {
			return strdup(filename);
		}
	
		if (filename[0] == '.' && filename[1] == '/' &&
		    strchr(filename + 2, '/') == NULL &&
		    strcmp(filename + 2, "..") != 0) {
			return strdup(filename + 2);
		}
	}
	
	/* Either we're not in an object, or the path is not simple. So,
	 * get the dirname of it, open that, and compare to the object's
	 * stats. This is sorta expensive, and so clients should be
	 * steered away from passing in filenames like this, and from cding
	 * out of objects.
	 */
	file = strdup(filename);
	dir = dirname(file);
	if (stat(dir, &buf) != 0) {
		free(file);
		return NULL;
	}
	free(file);
	if (buf.st_dev == this_dev && buf.st_ino == this_inode) {
//		fprintf(stderr, "abs: %s\n", filename);
		/* Alright, so it _is_ the same. Get the basename, and
		 * return that.. */
		file = strdup(filename);
		base = basename(file);
		if (strcmp(base, "..") == 0)
			return NULL;
		base = strdup(base); /* because basename sucks eggs */
		free(file);
		return base;
	}
	
	/* Maybe they're referring to the object via its full path, and are
	 * operating on the object directory itself. Check with stat. */
	if (stat(filename, &buf) == 0 && 
	    buf.st_dev == this_dev && buf.st_ino == this_inode) {
		if (follow_symlinks || 
		    (lstat(filename, &buf) == 0 && ! S_ISLNK(buf.st_mode))) {
			return strdup(".");
		}
	}
	
	return NULL;
}
