#!/usr/bin/perl -T
# Log the user who is logged in to the current tty into the moo.
# This needs to be able to chgrp the user's tty, and it needs to run moo
# methods. Unfortunatly, that means it needs to be suid root.
use warnings;
use strict;
use POSIX q{ttyname};
use Mooix::Thing;
use Mooix::Root;
use Mooix::Conf;
use UNIVERSAL q{isa};
# This is optional, but nice.
eval "use Sys::Utmp";
my $have_utmp=$@ ? 0 : 1;

my $devdir="/dev/";

# Use the SAFEPATH, so tainting is happy.
$ENV{PATH}=$Mooix::Conf::field{safepath};
# Make %ENV safer
delete @ENV{qw(IFS CDPATH ENV BASH_ENV)};   

my $realtty = ttyname(0) || die "cannot find your tty: $!";

my $sessionmanager=$Mooix::Root->system->sessionmanager;
my $sessionparent=$Mooix::Root->sessions->tty;

# Get the hostname out of the utmp file.
my $hostname="unknown";
if ($have_utmp) {
	my $utmp = Sys::Utmp->new();
	while (my $utent = $utmp->getutent()) {
		if ($utent->user_process && $devdir.$utent->ut_line eq $realtty) {
			$hostname=$utent->ut_host;
			last;
		}
	}
	$utmp->endutent;
}

my $username=getpwuid($<);

# See if there is a preferred name, and sanitize it.
my $name=shift;
if (defined $name) {
	if ($name=~m/^([a-zA-Z0-9-_ ]+)$/) {
		$name=$1;
	}
	else {
		die "Badly formed name, \"$name\"\n";
	}
}

# Cleanup on HUP.
$SIG{HUP} = \&cleanup;
# Ignore these. ctrl-c doesn't normally kill shells.
# Use a handler that odes nothing so the ignoring will not propigate to
# children.
$SIG{QUIT} = $SIG{INT} = $SIG{TERM} = sub {};
# I'd like to support suspend, but readline has issues suspending properly.
# Instead, send a SIGCONT to everything in the process group. This is
# because runmeth probably got stopped too, and it needs to be restarted.
$SIG{TSTP} = sub { kill(&POSIX::SIGCONT, 0) };
# Deliver sigwinches through to the session. A trifle expensive.
my $session;
sub winch {
	# Ignore other sigwinches until we're done.
	$SIG{WINCH} = 'IGNORE';
	if ($session) {
		my $pid=fork();
		if (! $pid) {
			$session->resize;
			exit;
		}
		waitpid($pid, 0);
	}
	$SIG{WINCH} = \&winch;
}
$SIG{WINCH} = \&winch;

# Create a session. Have to do this first so that its tty field can be
# copied in, a process that only works outside the moo.
$session = $sessionmanager->makesession(
	parent => $sessionparent,
	realtty => $realtty,
	term => $ENV{TERM},
	unixuser => $username,
	auth_username => $username,
);
if (! isa($session, "Mooix::Thing")) {
	die "Unable to set up a session. Is the moo running?\n";
}
$session->untaint;
# The tty is set up by a setuid root helper program.
if (system("mooix-pty-helper", $session->id) != 0) {
	cleanup();
	die "mooix-pty-helper refused to copy your tty";
}

# Ask the session manager to log the user in.
my @sessargs=(session => $session, auth_username => $username,
              auth_hostname => $hostname);
push @sessargs, name => $name if defined $name;
if ($sessionmanager->login(@sessargs)) {
	$session->avatar->untaint->parser_parse(session => $session);
}

# Fini.
cleanup();
my $cleaning=0;
sub cleanup {
	return if $cleaning;
	$cleaning=1;
	if ($session and -d $session->id) {
		# Ignore signals during cleanup. Note use of "IGNORE"; this
		# should make the signal mask propigate to children, like
		# runmeth, and keep them from catching these signals, too.
		$SIG{TSTP} = $SIG{HUP} = 'IGNORE';
		$sessionmanager->logout(session => $session);
	}
}
