/*
	event.c

	Generalised event handling for motion

	Copyright Jeroen Vreeken, 2002
	This software is distributed under the GNU Public License Version 2
	see also the file 'COPYING'.

*/

#include "motion.h"
#include "event.h"
#include "video.h"
#include "picture.h"
#include "ffmpeg.h"

/*
 *	Various functions (most doing the actual action)
 */

static void send_sms(char *sms_nr, struct tm *currenttime)
{
	char message[150];

	if (!fork()) {
		printf("Sending SMS message to %s\n", sms_nr);
		sprintf(message, "Motion detected %04d-%02d-%02d %02d:%02d:%02d\n",
		    currenttime->tm_year+1900, currenttime->tm_mon +1,
		    currenttime->tm_mday, currenttime->tm_hour,
		    currenttime->tm_min, currenttime->tm_sec);
		execlp("sms_client", "sms_client", sms_nr, message, 0);
		perror("Unable to send SMS message");
		exit(0);
	}
}

static void send_mail(char *mail_address, struct tm *currenttime)
{
	char message[150];
	int fdpipe[2];

	if (!fork()) {
		printf("Sending mail to %s\n", mail_address);
		sprintf(message, "This is an automated message generated by motion\n\n"
		    "Motion detected: %04d-%02d-%02d %02d:%02d:%02d\n"
		    "\n",
		    currenttime->tm_year+1900, currenttime->tm_mon+1,
		    currenttime->tm_mday, currenttime->tm_hour,
		    currenttime->tm_min, currenttime->tm_sec);
		pipe(fdpipe);
		close(0);
		dup(fdpipe[0]);
		close(fdpipe[0]);
		write(fdpipe[1], message, strlen(message));
		close(fdpipe[1]);
		execlp("mail", "mail", mail_address, "-s", "MOTION ALERT", 0);
		perror("Unable to send message");
		exit(1);
	}
}

/* Execute 'command' with 'arg' as its argument.
   if !arg command is started with no arguments */
void exec_command (char *command, char *arg)
{
	if (!fork()) {
		vid_close();
		if (!arg) {
			printf("Executing external command '%s'\n", command);
			execlp(command, command, 0);
		} else {
			printf("Executing external command '%s %s'\n", command, arg);
			execlp(command, command, arg, 0);
		}
		perror("Unable to start external command");
		exit(1);
	}
}

#ifdef HAVE_MYSQL
void put_mysql (struct config *conf, MYSQL *db, char *filebase, struct tm *currenttime, int type)
{
	char sqlquery[512];
	sprintf(sqlquery, "insert into security set filename='%s', minute=%d, hour=%d, day=%d, month=%d, year=%d, type=%d",
	    filebase,
	    currenttime->tm_min, currenttime->tm_hour,
	    currenttime->tm_mday, currenttime->tm_mon+1,
	    currenttime->tm_year+1900, type);
	if (mysql_query(db, sqlquery)) {
		printf("Error querying mysql");
		exit(0);
	}
}
#endif

#ifdef HAVE_PGSQL
void put_pgsql (struct config *conf, PGconn *conn, char *filebase, struct tm *currenttime, int type)
{
	char sqlquery[512];
	PGresult *res;

	//use proper sql, and not some strange mysql version :-P
	sprintf(sqlquery, 
	    "insert into security(filename, minute, hour, day, month, year, type) values('%s', '%d', '%d', '%d', '%d', '%d', '%d')",
	    filebase,
	    currenttime->tm_min, currenttime->tm_hour,
	    currenttime->tm_mday, currenttime->tm_mon+1,
	    currenttime->tm_year+1900, type);
	res = PQexec(conn, sqlquery);
	if (PQresultStatus(res) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "PGSQL query failed: %s\n", sqlquery);
		PQclear(res);
		exit(1);
	}
}
#endif


/* 
 *	Event handlers
 */

static void event_newfile(int type, struct context *cnt, char *filename, void *mime, struct tm *tm)
{
	printf("File of type %s saved to: %s\n", (char *)mime, filename);
}

static void event_motion(int type, struct context *cnt, char *filename, void *mime, struct tm *tm)
{
	if (!cnt->conf.quiet)
		printf("\a");
	printf("Motion detected\n");
}

static void onsave_command(int type, struct context *cnt, char *filename, void *mime, struct tm *tm)
{
	if (!strncmp(mime, "image", 5))
		if (cnt->conf.onsavecommand)
			exec_command(cnt->conf.onsavecommand, filename);
	if (!strncmp(mime, "video", 5))
		if (cnt->conf.onmpegcommand) {
			exec_command(cnt->conf.onmpegcommand, filename);
	}
}


#ifdef HAVE_MYSQL
static void event_mysql_newfile(int type, struct context *cnt, char *filename, void *mime, struct tm *tm)
{
	int sqltype=0;

	if (!strncmp(mime, "image", 5)) {
		sqltype=1;
		if (filename[strlen(filename)-5]=='m')
			sqltype=4;
		if (!strncmp(filename+strlen(filename)-12, "snapshot", 8))
			sqltype=2;
	}

	if (cnt->conf.mysql_db && sqltype)
		put_mysql(&cnt->conf, cnt->database, filename, tm, sqltype);
	
}
#endif

#ifdef HAVE_PGSQL
static void event_pgsql_newfile(int type, struct context *cnt, char *filename, void *mime, struct tm *tm)
{
	int sqltype=0;

	if (!strncmp(mime, "image", 5)) {
		sqltype=1;
		if (filename[strlen(filename)-5]=='m')
			sqltype=4;
		if (!strncmp(filename+strlen(filename)-12, "snapshot", 8))
			sqltype=2;
	}

	if (cnt->conf.pgsql_db)
		put_pgsql(&cnt->conf, cnt->database_pg, filename, cnt->currenttime, 4);
}
#endif

static void event_firstmotion(int type, struct context *cnt, char *dummy, void *dummy2, struct tm *tm)
{
	if (cnt->conf.sms_nr)
		send_sms(cnt->conf.sms_nr, tm);
	if (cnt->conf.mail_address)
		send_mail(cnt->conf.mail_address, tm);
	if (cnt->conf.externcommand)
		exec_command(cnt->conf.externcommand, 0);
}

static void event_stop_webcam(int type, struct context *cnt, char *dmy, void *dmy2, struct tm *tm)
{
	webcam_stop(cnt);
}

static void event_webcam_put(int type, struct context *cnt, char *img, void *devpipe, struct tm *tm)
{
	if (cnt->conf.webcam_port) {
		if (type==EVENT_IMAGE_DETECTED && cnt->conf.webcam_motion)
			webcam_put(cnt, cnt->imgs.new);
		if (type==EVENT_IMAGE && !cnt->conf.webcam_motion)
			webcam_put(cnt, img);
	}
}

static void event_vid_putpipe(int type, struct context *cnt, char *img, void *devpipe, struct tm *tm)
{
	vid_putpipe(*(int *)devpipe, img, cnt->imgs.size);
}

void mkdir_check(const char *pathname, mode_t mode)
{
	if (!mkdir(pathname, mode))
		return;
	else {
		if (errno==EEXIST)
			return;
		else {
			perror("Error while creating directory");
			exit(0);
		}
	}
}

void mkfilebase(struct context *cnt)
{
	struct tm *currenttime=cnt->currenttime;
	char filebase[255];

	sprintf(filebase, "%s/", cnt->conf.filepath);
	sprintf(filebase, "%s%04d/", filebase, currenttime->tm_year+1900);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_mon+1);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_mday);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_hour);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_min);
	mkdir_check(filebase, 0755);
}

void mkfilepath(struct context *cnt, char *c)
{
	struct tm *currenttime=cnt->currenttime;

	sprintf(c, "%04d/%02d/%02d/%02d/%02d/", currenttime->tm_year+1900,
	    currenttime->tm_mon+1, currenttime->tm_mday,
	    currenttime->tm_hour, currenttime->tm_min);
}

char *imageext(struct context *cnt)
{
	if (cnt->conf.ppm)
		return "ppm";
	return "jpg";
}

static void event_image_detect(int type, struct context *cnt, char *dummy, void *dummy2, struct tm *tm)
{
	struct config *conf=&cnt->conf;
	char fullfilename[255];
	char filename[255];
	char fullfilenamem[255];
	char filenamem[255];
	char filepath[255];

	if (conf->motion_img || conf->new_img) {
		if (!conf->oldlayout) {
			mkfilebase (cnt);
			mkfilepath (cnt, filepath);
			sprintf(filename, "%s%02d-%02d", filepath, cnt->currenttime->tm_sec, cnt->shots);
		} else {
			sprintf(filename, "%02d-%04d%02d%02d%02d%02d%02d-%02d",
			    cnt->event_nr,
			    cnt->currenttime->tm_year+1900,
			    cnt->currenttime->tm_mon+1,
			    cnt->currenttime->tm_mday,
			    cnt->currenttime->tm_hour,
			    cnt->currenttime->tm_min,
			    cnt->currenttime->tm_sec,
			    cnt->shots);
		}
		sprintf(filenamem, "%sm", filename);
		sprintf(fullfilename, "%s/%s.%s", cnt->conf.filepath, filename, imageext(cnt));
		sprintf(fullfilenamem, "%s/%s.%s", cnt->conf.filepath, filenamem, imageext(cnt));
	}
	if (conf->motion_img) {
		cnt->filesmc++;
		cnt->filesm=realloc(cnt->filesm, cnt->filesmc*(sizeof(void *)));
		cnt->filesm[cnt->filesmc-1]=malloc(strlen(fullfilenamem)+1);
		strcpy(cnt->filesm[cnt->filesmc-1], fullfilenamem);
		cnt->filesmrate=realloc(cnt->filesmrate, cnt->filesmc*(sizeof(void *)));
		cnt->filesmrate[cnt->filesmc-1]=cnt->lastrate;
		put_picture(cnt, fullfilenamem, cnt->imgs.out);
	
	}
	if (conf->new_img) {
		cnt->filesc++;
		cnt->files=realloc(cnt->files, cnt->filesc*(sizeof(void *)));
		cnt->files[cnt->filesc-1]=malloc(strlen(fullfilename)+1);
		strcpy(cnt->files[cnt->filesc-1], fullfilename);
		cnt->filesrate=realloc(cnt->filesrate, cnt->filesc*(sizeof(void *)));
		cnt->filesrate[cnt->filesc-1]=cnt->lastrate;
		put_picture(cnt, fullfilename, cnt->imgs.new);
	}
}

static void event_image_snapshot(int type, struct context *cnt, char *dummy, void *dummy2, struct tm *tm)
{
	char filepath[255];
	char filename[255];
	char fullfilename[255];
	char linkpath[255];

	sprintf(linkpath, "%s/lastsnap.%s", cnt->conf.filepath, imageext(cnt));
	remove(linkpath);
	if (!cnt->conf.snap_overwrite) {
		if (!cnt->conf.oldlayout) {
			mkfilebase (cnt);
			mkfilepath (cnt, filepath);
			sprintf(filename, "%s%02d-snapshot.%s", filepath, cnt->currenttime->tm_sec, imageext(cnt));
		} else {
			sprintf(filename, "%02d-%04d%02d%02d%02d%02d%02d-snapshot.%s",
			    cnt->event_nr,
			    cnt->currenttime->tm_year+1900,
			    cnt->currenttime->tm_mon+1,
			    cnt->currenttime->tm_mday,
			    cnt->currenttime->tm_hour,
			    cnt->currenttime->tm_min,
			    cnt->currenttime->tm_sec,
			    imageext(cnt));
		}
		sprintf(fullfilename, "%s/%s", cnt->conf.filepath, filename);
		symlink(filename, linkpath);
	} else {
		sprintf(fullfilename, "%s/lastsnap.%s", cnt->conf.filepath, imageext(cnt));
	}
	put_picture(cnt, fullfilename, cnt->imgs.new);
	cnt->snapshot=0;
}

#ifdef HAVE_FFMPEG

void mkmpegfilebase(struct context *cnt)
{
	struct tm *currenttime=cnt->currenttime;
	char filebase[255];

	sprintf(filebase, "%s/", cnt->conf.filepath);
	sprintf(filebase, "%s%04d/", filebase, currenttime->tm_year+1900);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_mon+1);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_mday);
	mkdir_check(filebase, 0755);
}

void mktimelapsfilebase(struct context *cnt)
{
	struct tm *currenttime=cnt->currenttime;
	char filebase[255];

	sprintf(filebase, "%s/", cnt->conf.filepath);
	sprintf(filebase, "%s%04d/", filebase, currenttime->tm_year+1900);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_mon+1);
	mkdir_check(filebase, 0755);
}

void mkmpegfilepath(struct context *cnt, char *c)
{
	struct tm *currenttime=cnt->currenttime;

	sprintf(c, "%s/%04d/%02d/%02d/", cnt->conf.filepath,
	    currenttime->tm_year+1900, currenttime->tm_mon+1, currenttime->tm_mday);
}

void mktimelapsfilepath(struct context *cnt, char *c)
{
	struct tm *currenttime=cnt->currenttime;

	sprintf(c, "%s/%04d/%02d/", cnt->conf.filepath,
	    currenttime->tm_year+1900, currenttime->tm_mon+1);
}

void grey2yuv420p(unsigned char *u, unsigned char *v, int width, int height)
{
	memset(u, 128, width*height/4);
	memset(v, 128, width*height/4);
}

void rgb2yuv420p(unsigned char *src, unsigned char *dst, int width, int height)
{
	unsigned char *y, *u, *v;
	unsigned char *r, *g, *b;
	int size, i=0;

	b=src;
	g=b+1;
	r=g+1;
	y=dst;
	u=y+width*height;
	v=u+(width*height)/4;
	memset(u, 0, width*height/4);
	memset(v, 0, width*height/4);

	size=width*height;
	while (i<size) {
		*y++=(9796**r+19235**g+3736**b)>>15;
		*u+=((-4784**r-9437**g+14221**b)>>17)+32;
		*v+=((20218**r-16941**g-3277**b)>>17)+32;
		r+=3;
		g+=3;
		b+=3;
		*y++=(9796**r+19235**g+3736**b)>>15;
		*u+=((-4784**r-9437**g+14221**b)>>17)+32;
		*v+=((20218**r-16941**g-3277**b)>>17)+32;
		r+=3;
		g+=3;
		b+=3;
		u++;
		v++;
		i+=2;
		if (0==i%width && 0!=i%(width*2)) {
			u-=width/2;
			v-=width/2;
		}
	}
}

static void event_ffmpeg_newfile(int type, struct context *cnt, char *dummy, void *dummy2, struct tm *currenttime)
{
	int width=cnt->imgs.width;
	int height=cnt->imgs.height;
	char newfilename[255];
	char motionfilename[255];
	unsigned char *convbuf, *y, *u, *v;
	int fps;

	if (!cnt->conf.ffmpeg_cap_new && !cnt->conf.ffmpeg_cap_motion)
		return;

	if (!cnt->conf.oldlayout) {
		mkmpegfilebase(cnt);
		mkmpegfilepath(cnt, newfilename);
		sprintf(motionfilename, "%s/%02d%02d%02dm.mpg", newfilename,
		    currenttime->tm_hour, currenttime->tm_min, currenttime->tm_sec);
		sprintf(newfilename, "%s/%02d%02d%02d.mpg", newfilename,
		    currenttime->tm_hour, currenttime->tm_min, currenttime->tm_sec);
	} else {
		sprintf(motionfilename, "%s/%02d-%04d%02d%02d%02d%02d%02dm.mpg",
		    cnt->conf.filepath,
		    cnt->event_nr,
		    cnt->currenttime->tm_year+1900,
		    cnt->currenttime->tm_mon+1,
		    cnt->currenttime->tm_mday,
		    cnt->currenttime->tm_hour,
		    cnt->currenttime->tm_min,
		    cnt->currenttime->tm_sec
		);
		sprintf(newfilename, "%s/%02d-%04d%02d%02d%02d%02d%02d.mpg",
		    cnt->conf.filepath,
		    cnt->event_nr,
		    cnt->currenttime->tm_year+1900,
		    cnt->currenttime->tm_mon+1,
		    cnt->currenttime->tm_mday,
		    cnt->currenttime->tm_hour,
		    cnt->currenttime->tm_min,
		    cnt->currenttime->tm_sec
		);
	}

	if (cnt->conf.ffmpeg_cap_new) {
		if (cnt->imgs.type==VIDEO_PALETTE_GREY) {
			convbuf=malloc((width*height)/2);
			y=cnt->imgs.new;
			u=convbuf;
			v=convbuf+(width*height)/4;
		} else if (cnt->imgs.type==VIDEO_PALETTE_RGB24) {
			convbuf=malloc((width*height*3)/2);
			y=convbuf;
			u=convbuf+width*height;
			v=u+(width*height)/4;
		} else {
			convbuf=NULL;
			y=cnt->imgs.new;
			u=cnt->imgs.new+width*height;
			v=u+(width*height)/4;
		}
		if (cnt->conf.low_cpu)
			fps=cnt->conf.frame_limit;
		else
			fps=cnt->lastrate;
		if (fps>30)
			fps=30;
		if (fps<2)
			fps=2;
		cnt->ffmpeg_new=ffmpeg_open(newfilename, y, u, v, cnt->imgs.width, cnt->imgs.height, fps, cnt->conf.ffmpeg_bps);
		cnt->ffmpeg_new->udata=convbuf;
		event(EVENT_FILECREATE, cnt, newfilename, "video/mpeg", NULL);
	}
	if (cnt->conf.ffmpeg_cap_motion) {
		if (cnt->imgs.type==VIDEO_PALETTE_GREY) {
			convbuf=malloc((width*height)/2);
			y=cnt->imgs.out;
			u=convbuf;
			v=convbuf+(width*height)/4;
			grey2yuv420p(u, v, width, height);
		} else if (cnt->imgs.type==VIDEO_PALETTE_RGB24) {
			convbuf=malloc((width*height*3)/2);
			y=convbuf;
			u=convbuf+width*height;
			v=u+(width*height)/4;
		} else {
			y=cnt->imgs.out;
			u=cnt->imgs.out+width*height;
			v=u+(width*height)/4;
			convbuf=NULL;
		}
		if (cnt->conf.low_cpu)
			fps=cnt->conf.frame_limit;
		else
			fps=cnt->lastrate;
		if (fps>30)
			fps=30;
		if (fps<2)
			fps=2;
		cnt->ffmpeg_motion=ffmpeg_open(motionfilename, y, u, v, cnt->imgs.width, cnt->imgs.height, fps, cnt->conf.ffmpeg_bps);
		cnt->ffmpeg_new->udata=convbuf;
		event(EVENT_FILECREATE, cnt, motionfilename, "video/mpeg", NULL);
	}
}

static void event_ffmpeg_timelaps(int type, struct context *cnt, char *dummy, void *dummy2, struct tm *currenttime)
{
	int width=cnt->imgs.width;
	int height=cnt->imgs.height;
	char timelapsfilename[255];
	unsigned char *convbuf, *y, *u, *v;


	if (!cnt->ffmpeg_timelaps) {
		if (!cnt->conf.oldlayout) {
			mktimelapsfilebase(cnt);
			mktimelapsfilepath(cnt, timelapsfilename);
			sprintf(timelapsfilename, "%s/%02d-timelaps.mpg", timelapsfilename,
			    currenttime->tm_mday);
		} else {
			sprintf(timelapsfilename, "%s/%02d-%04d%02d%02d-timelaps.mpg",
			    cnt->conf.filepath,
			    cnt->event_nr,
			    cnt->currenttime->tm_year+1900,
			    cnt->currenttime->tm_mon+1,
			    cnt->currenttime->tm_mday
			);
		}
		if (cnt->imgs.type==VIDEO_PALETTE_GREY) {
			convbuf=malloc((width*height)/2);
			y=cnt->imgs.new;
			u=convbuf;
			v=convbuf+(width*height)/4;
		} else if (cnt->imgs.type==VIDEO_PALETTE_RGB24) {
			convbuf=malloc((width*height*3)/2);
			y=convbuf;
			u=convbuf+width*height;
			v=u+(width*height)/4;
		} else {
			convbuf=NULL;
			y=cnt->imgs.new;
			u=cnt->imgs.new+width*height;
			v=u+(width*height)/4;
		}
		cnt->ffmpeg_timelaps=ffmpeg_open(timelapsfilename, y, u, v, cnt->imgs.width, cnt->imgs.height, 24, cnt->conf.ffmpeg_bps);
		cnt->ffmpeg_timelaps->udata=convbuf;
		event(EVENT_FILECREATE, cnt, timelapsfilename, "video/mpeg", NULL);
	}
	if (cnt->imgs.type==VIDEO_PALETTE_RGB24) {
		rgb2yuv420p(cnt->imgs.new, cnt->ffmpeg_timelaps->udata, cnt->imgs.width, cnt->imgs.height);
	}
	ffmpeg_put_image(cnt->ffmpeg_timelaps);
}

static void event_ffmpeg_put(int type, struct context *cnt, char *dummy, void *dummy2, struct tm *tm)
{
	if (cnt->ffmpeg_new) {
		if (cnt->imgs.type==VIDEO_PALETTE_RGB24)
			rgb2yuv420p(cnt->imgs.new, cnt->ffmpeg_new->udata, cnt->imgs.width, cnt->imgs.height);
		ffmpeg_put_image(cnt->ffmpeg_new);
	}
	if (cnt->ffmpeg_motion) {
		if (cnt->imgs.type==VIDEO_PALETTE_RGB24)
			rgb2yuv420p(cnt->imgs.out, cnt->ffmpeg_motion->udata, cnt->imgs.width, cnt->imgs.height);
		ffmpeg_put_image(cnt->ffmpeg_motion);
	}
}

static void event_ffmpeg_closefile(int type, struct context *cnt, char *dummy, void *dummy2, struct tm *tm)
{
	if (cnt->ffmpeg_new) {
		if (cnt->ffmpeg_new->udata)
			free(cnt->ffmpeg_new->udata);
		ffmpeg_close(cnt->ffmpeg_new);
		cnt->ffmpeg_new=NULL;
	}
	if (cnt->ffmpeg_motion) {
		if (cnt->ffmpeg_new->udata)
			free(cnt->ffmpeg_motion->udata);
		ffmpeg_close(cnt->ffmpeg_motion);
		cnt->ffmpeg_motion=NULL;
	}
}

static void event_ffmpeg_timelapsend(int type, struct context *cnt, char *dummy, void *dummy2, struct tm *tm)
{
	if (cnt->ffmpeg_timelaps) {
		if (cnt->ffmpeg_timelaps->udata)
			free(cnt->ffmpeg_timelaps->udata);
		ffmpeg_close(cnt->ffmpeg_timelaps);
		cnt->ffmpeg_timelaps=NULL;
	}
}

#endif /* HAVE_FFMPEG */

/*  
 *	Starting point for all events
 */

struct event_handlers {
	int type;
	event_handler handler;
};

struct event_handlers event_handlers[] = {
	{
	EVENT_FILECREATE,
	event_newfile
	},
	{
	EVENT_FILECREATE,
	onsave_command
	},
#ifdef HAVE_MYSQL
	{
	EVENT_FILECREATE,
	event_mysql_newfile
	},
#endif
#ifdef HAVE_PGSQL
	{
	EVENT_FILECREATE,
	event_pgsql_newfile
	},
#endif
	{
	EVENT_MOTION,
	event_motion
	},
	{
	EVENT_FIRSTMOTION,
	event_firstmotion
	},
	{
	EVENT_IMAGE_DETECTED,
	event_image_detect
	},
	{
	EVENT_IMAGE_SNAPSHOT,
	event_image_snapshot
	},
	{
	EVENT_IMAGE | EVENT_IMAGEM,
	event_vid_putpipe
	},
	{
	EVENT_IMAGE | EVENT_IMAGE_DETECTED,
	event_webcam_put
	},
#ifdef HAVE_FFMPEG
	{
	EVENT_FIRSTMOTION,
	event_ffmpeg_newfile
	},
	{
	EVENT_IMAGE_DETECTED,
	event_ffmpeg_put
	},
	{
	EVENT_ENDMOTION,
	event_ffmpeg_closefile
	},
	{
	EVENT_TIMELAPS,
	event_ffmpeg_timelaps
	},
	{
	EVENT_TIMELAPSEND,
	event_ffmpeg_timelapsend
	},
#endif
	{
	EVENT_STOP,
	event_stop_webcam
	},
	{0, NULL}
};

void event (int type, struct context *cnt, char *filename, void *eventdata, struct tm *tm)
{
	int i=-1;

	while (event_handlers[++i].handler) {
		if (type & event_handlers[i].type)
			event_handlers[i].handler(type, cnt, filename, eventdata, tm);
	}
}
