/**********************************************************************
 *
 * ffmpeg.c
 *
 * This software is distributed under the GNU Public License version 2
 * See also the file 'COPYING'.
 *
 * Most of this was taken directly from output_example.c from ffmpeg.
 *
 **********************************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#ifdef HAVE_FFMPEG

#include "motion.h"
#include "ffmpeg.h"

/* ffmpeg init initialize the codec. If the codec is mpeg1 the old libavcodec
 * method is used. Otherwise we use the newer libavframe method.
 * We initialize for both methods
 */
void ffmpeg_init()
{
	avcodec_init();
	/*register_avcodec(&mpeg1video_encoder);*/
	av_register_all();
	return;
}


/* ffmpeg_open opens a new mpeg file and sets up the codecs used by ffmpeg
 * ffmpeg_open calls either an mpeg1 version based on libavcodec
 * or an mpeg4 version based on libavframe
 */
struct ffmpeg *ffmpeg_open(char *ffmpeg_video_codec, char *filename, unsigned char *y, unsigned char *u, unsigned char *v, int width, int height, int rate, int bps, int vbr)
{
	if (strcmp(ffmpeg_video_codec, "mpeg1") == 0) {
		return ffmpeg_open_mpeg1(ffmpeg_video_codec, filename, y, u, v, width, height, rate, bps, vbr);
	} else {
		return ffmpeg_open_mpeg4(ffmpeg_video_codec, filename, y, u, v, width, height, rate, bps, vbr);
	}
}

/* mpeg4 version of ffmpeg_open */
struct ffmpeg *ffmpeg_open_mpeg4(char *ffmpeg_video_codec, char *filename, unsigned char *y, unsigned char *u, unsigned char *v, int width, int height, int rate, int bps, int vbr)
{
	AVCodecContext *c;
	AVCodec *codec;
	struct ffmpeg *ffmpeg;

	/* Allocate space for our ffmpeg structure.
	 * This structure contains all the codec and image information
	 * we need to generate movies.
	 * FIXME when motion exits we should close the movie to ensure that
	 * ffmpeg is freed  */
	if ((ffmpeg = malloc(sizeof(struct ffmpeg))) == NULL) return NULL;
	memset(ffmpeg, 0, sizeof(*ffmpeg));

	ffmpeg->y = y;
	ffmpeg->u = u;
	ffmpeg->v = v;
	ffmpeg->width = width;
	ffmpeg->height = height;
	ffmpeg->vbr = vbr;
	strcpy(ffmpeg->codec, ffmpeg_video_codec);

	/* allocation the output media context */
	ffmpeg->oc = av_mallocz(sizeof(AVFormatContext));
	if (!ffmpeg->oc) {
		syslog(LOG_ERR, "Memory error while allocating output media context: %m");
		ffmpeg_cleanups(ffmpeg);
		return (NULL);
	}

	/* Setup output format
	 * Here, we use guess format to automatically setup the codec information.
	 * If we are using msmpeg4, manually set that codec here.
	 * We also, dynamically add the file extension to the movie here.  This was
	 * done to support both mpeg1 and mpeg4 codecs since they have different extensions.
	 */
	
	/* mpeg1 is not working with the libavframe method - but the mpeg1new value is
	 * temporarily here for testing. It will be removed in the released version.
	 */
	if (strcmp(ffmpeg_video_codec, "mpeg1new") == 0) {
		ffmpeg->oc->oformat = guess_format("mpeg", NULL, NULL);
		if (!ffmpeg->oc->oformat) {
			syslog(LOG_ERR, "Could not guess format for mpeg1 mpeg");
			ffmpeg_cleanups(ffmpeg);
			return (NULL);
		}
		sprintf(filename, "%s.mpg", filename);
	} else if (strcmp(ffmpeg_video_codec, "mpeg4") == 0) {
		ffmpeg->oc->oformat = guess_format("avi", NULL, NULL);
		if (!ffmpeg->oc->oformat) {
			syslog(LOG_ERR, "Could not guess format for mpeg4 avi");
			ffmpeg_cleanups(ffmpeg);
			return (NULL);
		}
		sprintf(filename, "%s.avi", filename);
	} else if (strcmp(ffmpeg_video_codec, "msmpeg4") == 0) {
		ffmpeg->oc->oformat = guess_format("avi", NULL, NULL);
		if (!ffmpeg->oc->oformat) {
			syslog(LOG_ERR, "Could not guess format for msmpeg4 avi");
			ffmpeg_cleanups(ffmpeg);
			return (NULL);
		}
		ffmpeg->oc->oformat->video_codec = CODEC_ID_MSMPEG4V2;
		sprintf(filename, "%s.avi", filename);
	} else {
		syslog(LOG_ERR, "ffmpeg_video_codec option value %s is not supported", ffmpeg_video_codec);
		ffmpeg_cleanups(ffmpeg);
		return (NULL);
	}
	snprintf(ffmpeg->oc->filename, sizeof(ffmpeg->oc->filename), "%s", filename);

	/* Create a new video stream and initialize the codecs */
	ffmpeg->video_st = NULL;
	if (ffmpeg->oc->oformat->video_codec != CODEC_ID_NONE) {
		ffmpeg->video_st = av_new_stream(ffmpeg->oc, 0);
		if (!ffmpeg->video_st) {
			syslog(LOG_ERR, "av_new_stream - could not alloc stream: %m");
			ffmpeg_cleanups(ffmpeg);
			return (NULL);
		}
	}
	c = &ffmpeg->video_st->codec;
	c->codec_id = ffmpeg->oc->oformat->video_codec;
	c->codec_type = CODEC_TYPE_VIDEO;

	// FIXME: It doesn't like our framerate apparently
	//c->strict_std_compliance = -1;

	/* Set default parameters */
	c->bit_rate = bps;
	c->width = width;
	c->height = height;
	c->frame_rate = rate;
	c->frame_rate_base = 1;
	c->gop_size = 12; /* emit one intra frame every twelve frames at most */
	if (vbr) {
		c->flags |= CODEC_FLAG_QSCALE;
	}
	if (c->codec_id == CODEC_ID_MPEG2VIDEO) {
		/* just for testing, we also add B frames */
		c->max_b_frames = 2;
	}
	if (c->codec_id == CODEC_ID_MPEG1VIDEO) {
		/* needed to avoid using macroblocks in which some coeffs overflow
			 this doesnt happen with normal video, it just happens here as the
			 motion of the chroma plane doesnt match the luma plane */
		c->mb_decision=2;
	}
	// some formats want stream headers to be seperate
	if(!strcmp(ffmpeg->oc->oformat->name, "mp4") || !strcmp(ffmpeg->oc->oformat->name, "mov") ||
	           !strcmp(ffmpeg->oc->oformat->name, "3gp"))
		c->flags |= CODEC_FLAG_GLOBAL_HEADER;

	/* set the output parameters (must be done even if no parameters). */
	if (av_set_parameters(ffmpeg->oc, NULL) < 0) {
		syslog(LOG_ERR, "ffmpeg av_set_parameters error: Invalid output format parameters");
		ffmpeg_cleanups(ffmpeg);
		return (NULL);
	}

	/* Dump the format settings.  This shows how the various streams relate to each other */
	//dump_format(ffmpeg->oc, 0, filename, 1);

	/* Now that all the parameters are set, we can open the video
		codec and allocate the necessary encode buffers */
	codec = avcodec_find_encoder((&ffmpeg->video_st->codec)->codec_id);
	if (!codec) {
		syslog(LOG_ERR, "Codec not found: %m");
		ffmpeg_cleanups(ffmpeg);
		return (NULL);
	}

	/* open the codec */
	if (avcodec_open(&ffmpeg->video_st->codec, codec) < 0) {
		syslog(LOG_ERR, "avcodec_open - could not open codec: %m");
		ffmpeg_cleanups(ffmpeg);
		return (NULL);
	}

	ffmpeg->video_outbuf = NULL;
	if (!(ffmpeg->oc->oformat->flags & AVFMT_RAWPICTURE)) {
		/* allocate output buffer */
		/* XXX: API change will be done */
		ffmpeg->video_outbuf_size = 200000;
		ffmpeg->video_outbuf = malloc(ffmpeg->video_outbuf_size);
	}

	/* allocate the encoded raw picture */
	ffmpeg->picture = avcodec_alloc_frame();
	if (!ffmpeg->picture) {
		syslog(LOG_ERR, "avcodec_alloc_frame - could not alloc frame: %m");
		ffmpeg_cleanups(ffmpeg);
		return (NULL);
	}

	if (ffmpeg->vbr)
		ffmpeg->picture->quality = ffmpeg->vbr;

	ffmpeg->picture->data[0] = y;
	ffmpeg->picture->data[1] = u;
	ffmpeg->picture->data[2] = v;
	ffmpeg->picture->linesize[0] = ffmpeg->width;
	ffmpeg->picture->linesize[1] = ffmpeg->width / 2;
	ffmpeg->picture->linesize[2] = ffmpeg->width / 2;

	/* open the output file, if needed */
	if (!(ffmpeg->oc->oformat->flags & AVFMT_NOFILE)) {
		if (url_fopen(&ffmpeg->oc->pb, filename, URL_WRONLY) < 0) {
			/* path did not exist? */
			if (errno == ENOENT) {

				/* create path for file... */
				if (create_path(filename) == -1) {
					ffmpeg_cleanups(ffmpeg);
					return (NULL);
				}

				/* and retry opening the file */
				if (url_fopen(&ffmpeg->oc->pb, filename, URL_WRONLY) < 0) {
					syslog(LOG_ERR, "url_fopen - error opening file %s: %s", filename, strerror(errno));
					ffmpeg_cleanups(ffmpeg);
					return (NULL);
				}
				/* Permission denied */
			} else if (errno ==  EACCES) {
				syslog(LOG_ERR, "url_fopen - error opening file %s: %s ... check access rights to target directory", filename, strerror(errno));
				/* create path for file... */
				ffmpeg_cleanups(ffmpeg);
				return (NULL);
			} else {
				syslog(LOG_ERR, "Error opening file %s: %s", filename, strerror(errno));
				ffmpeg_cleanups(ffmpeg);
				return (NULL);
			}
		}
	}

	/* write the stream header, if any */
	av_write_header(ffmpeg->oc);

	return ffmpeg;
}

/* mpeg1 version of ffmpeg_open */
struct ffmpeg *ffmpeg_open_mpeg1(char *ffmpeg_video_codec, char *filename, unsigned char *y, unsigned char *u, unsigned char *v, int width, int height, int rate, int bps, int vbr)
{
	AVCodec *codec;
	AVCodecContext *c;
	AVFrame *picture;

	struct ffmpeg *ffmpeg;

	if ((ffmpeg = mymalloc(sizeof(struct ffmpeg))) == NULL) return NULL;
	memset(ffmpeg, 0, sizeof(*ffmpeg));

	ffmpeg->vbr = vbr;
	/* find the mpeg1 video encoder */
	codec = avcodec_find_encoder(CODEC_ID_MPEG1VIDEO);
	if (!codec) {
		ffmpeg_cleanups(ffmpeg);
		return NULL;
	}

	c = avcodec_alloc_context();
	ffmpeg->c=c;
	picture = avcodec_alloc_frame();
	ffmpeg->picture=picture;
	strcpy(ffmpeg->codec, ffmpeg_video_codec);

	/* put sample parameters */
	c->bit_rate = bps; /* bit per second */

	/* resolution must be a multiple of two */
	c->width  = width;  
	c->height = height;

	/* frames per second */
	c->frame_rate = rate;  /* default rate is only 5 frames per second */
	c->frame_rate_base = 1;
	c->gop_size = 10; /* emit one intra frame every ten frames */
	if (vbr) {
		c->flags |= CODEC_FLAG_QSCALE;
	}
//	c->strict_std_compliance = -1; /* allow mpegs with low framerates */
//	c->qmin = 3;
//	c->max_qdiff = 3;
//	c->qblur = 0.5;
//	c->qcompress = 0.5;
//	c->get_psnr = 0;
//	c->bit_rate_tolerance = bps;
//	c->me_method = 0;
///	c->flags = 0;
	if (ffmpeg->vbr)
		ffmpeg->picture->quality = ffmpeg->vbr;


	/* open it */
	if (avcodec_open(c, codec) < 0) {
		ffmpeg_cleanups(ffmpeg);
		return NULL;
	}

	sprintf(filename, "%s.mpg", filename);
	if (!(ffmpeg->f = myfopen(filename, "ab"))) {
		if (errno ==  EACCES) {
			syslog(LOG_ERR, "Error opening file %s: %s - Check access rights to target directory", filename, strerror(errno));
		} else {
			syslog(LOG_ERR, "Error opening file %s: %s", filename, strerror(errno));
		}
		ffmpeg_cleanups(ffmpeg);
		return NULL;
	}
	
	/* alloc image and output buffer */
	ffmpeg->video_outbuf_size = width * height * 10; // large outbuf_buf
	if ((ffmpeg->video_outbuf = (uint8_t*) mymalloc(ffmpeg->video_outbuf_size)) == NULL){
		ffmpeg_cleanups(ffmpeg);
		return NULL;
	}

	ffmpeg->picture->data[0] = y;
	ffmpeg->picture->data[1] = u;
	ffmpeg->picture->data[2] = v;
	ffmpeg->picture->linesize[0] = c->width;
	ffmpeg->picture->linesize[1] = c->width / 2;
	ffmpeg->picture->linesize[2] = c->width / 2;

	return ffmpeg;
}


/*
  Clean up ffmpeg struct is something was wrong
*/
void ffmpeg_cleanups(struct ffmpeg *ffmpeg)
{
	if (strcmp(ffmpeg->codec, "mpeg1") == 0) {
		if (ffmpeg->f) fclose(ffmpeg->f);
		if (ffmpeg->video_outbuf) free(ffmpeg->video_outbuf);
		if (ffmpeg->c) avcodec_close(ffmpeg->c);
		free(ffmpeg->c);
		if (ffmpeg->picture) free(ffmpeg->picture);
		free(ffmpeg);
	} else {
		int i;

		/* close each codec */
		if (ffmpeg->video_st) {
			avcodec_close(&ffmpeg->video_st->codec);
			av_free(ffmpeg->picture);
			ffmpeg->picture = NULL;
			av_free(ffmpeg->video_outbuf);
			ffmpeg->video_outbuf = NULL;
		}

		/* free the streams */
		for (i = 0; i < ffmpeg->oc->nb_streams; i++) {
			av_freep(&ffmpeg->oc->streams[i]);
		}
/*
		if (!(ffmpeg->oc->oformat->flags & AVFMT_NOFILE)) {
			// close the output file 
			if (ffmpeg->oc->pb) url_fclose(&ffmpeg->oc->pb);
		}
*/
		/* free the stream */
		av_free(ffmpeg->oc);
		ffmpeg->oc = NULL;
		
		free(ffmpeg);
		ffmpeg = NULL;
	}
}

/* ffmpeg_close closes the mpeg file and frees the memory that has been used by ffmpeg
 * ffmpeg_close calls either an mpeg1 version based on libavcodec
 * or an mpeg4 version based on libavframe
 */
void ffmpeg_close(struct ffmpeg *ffmpeg)
{
	if (strcmp(ffmpeg->codec, "mpeg1") == 0) {
		ffmpeg_close_mpeg1(ffmpeg);
	} else {
		ffmpeg_close_mpeg4(ffmpeg);
	}
}

/* mpeg4 version of ffmpeg_close */
void ffmpeg_close_mpeg4(struct ffmpeg *ffmpeg)
{
	int i;

	/* close each codec */
	if (ffmpeg->video_st) {
		avcodec_close(&ffmpeg->video_st->codec);
		av_free(ffmpeg->picture);
		ffmpeg->picture = NULL;
		av_free(ffmpeg->video_outbuf);
		ffmpeg->video_outbuf = NULL;
	}

	/* write the trailer, if any */
	av_write_trailer(ffmpeg->oc);

	/* free the streams */
	for (i = 0; i < ffmpeg->oc->nb_streams; i++) {
		av_freep(&ffmpeg->oc->streams[i]);
	}

	if (!(ffmpeg->oc->oformat->flags & AVFMT_NOFILE)) {
		/* close the output file */
		url_fclose(&ffmpeg->oc->pb);
	}


	/* free the stream */
	av_free(ffmpeg->oc);
	ffmpeg->oc = NULL;

	free(ffmpeg);
	ffmpeg = NULL;
}

/* mpeg1 version of ffmpeg_close */
void ffmpeg_close_mpeg1(struct ffmpeg *ffmpeg)
{
	/* add sequence end code to have a real mpeg file */
	ffmpeg->video_outbuf[0] = 0x00;
	ffmpeg->video_outbuf[1] = 0x00;
	ffmpeg->video_outbuf[2] = 0x01;
	ffmpeg->video_outbuf[3] = 0xb7;
	fwrite(ffmpeg->video_outbuf, 1, 4, ffmpeg->f);
	fclose(ffmpeg->f);
	free(ffmpeg->video_outbuf);
	avcodec_close(ffmpeg->c);
	free(ffmpeg->c);
	free(ffmpeg->picture);
	free(ffmpeg);
}


/* ffmpeg_put_image adds a frame to the mpeg file pointed
 * to by ffmpeg->f.
 * It uses the image that ffmpeg->picture points to which
 * makes this function less useful when the current image is
 * in a ringbuffer. The function ffmpeg_put_other_image allows
 * you to specify a specific image location by YUV.
 * ffmpeg_put_image calls either an mpeg1 version based on
 * libavcodec or an mpeg4 version based on libavframe
 */
void ffmpeg_put_image(struct ffmpeg *ffmpeg) 
{
	if (strcmp(ffmpeg->codec, "mpeg1") == 0) {
		ffmpeg_put_image_mpeg1(ffmpeg);
	} else {
		ffmpeg_put_image_mpeg4(ffmpeg);
	}
	return;
}

/* mpeg4 version of ffmpeg_put_image */
void ffmpeg_put_image_mpeg4(struct ffmpeg *ffmpeg) 
{
	int out_size, ret;
	AVFrame *picture_ptr;

	picture_ptr = ffmpeg->picture;

	if (ffmpeg->oc->oformat->flags & AVFMT_RAWPICTURE) {
		/* raw video case. The API will change slightly in the near future for that */
		ret = av_write_frame(ffmpeg->oc, ffmpeg->video_st->index,
				(uint8_t *)picture_ptr, sizeof(AVPicture));
	} else {
		/* encode the image */
		out_size = avcodec_encode_video(&ffmpeg->video_st->codec, ffmpeg->video_outbuf, ffmpeg->video_outbuf_size, picture_ptr);
		/* if zero size, it means the image was buffered */
		if (out_size != 0) {
			/* write the compressed frame in the media file */
			/* XXX: in case of B frames, the pts is not yet valid */
			ret = av_write_frame(ffmpeg->oc, ffmpeg->video_st->index, ffmpeg->video_outbuf, out_size);
		} else {
			ret = 0;
		}
	}
	if (ret != 0) {
		syslog(LOG_ERR, "Error while writing video frame: %m");
		return;
	}
}

/* mpeg1 version of ffmpeg_put_image */
void ffmpeg_put_image_mpeg1(struct ffmpeg *ffmpeg) 
{
	int out_size;

	out_size = avcodec_encode_video(
	    ffmpeg->c, 
	    ffmpeg->video_outbuf, 
	    ffmpeg->video_outbuf_size, 
	    ffmpeg->picture);

	fwrite(ffmpeg->video_outbuf, 1, out_size, ffmpeg->f);
}


/* ffmpeg_put_other_image adds a frame to the mpeg file pointed
 * to by ffmpeg->f.
 * It uses the image that given by pointers to the Y, U and V area
 * of an YUV encoded image.
 * This function is used when the current image is in a ringbuffer.
 * The function ffmpeg_put_image uses an image location pointed to by
 * the picture member of an ffmpeg structure (ffmpeg->picture).
 * ffmpeg_put_other_image calls either an mpeg1 version based on
 * libavcodec or an mpeg4 version based on libavframe
 */
void ffmpeg_put_other_image(struct ffmpeg *ffmpeg, unsigned char *y, unsigned char *u, unsigned char *v)
{
	if (strcmp(ffmpeg->codec, "mpeg1") == 0) {
		ffmpeg_put_other_image_mpeg1(ffmpeg, y, u, v);
	} else {
		ffmpeg_put_other_image_mpeg4(ffmpeg, y, u, v);
	}	
}

/* mpeg4 version of ffmpeg_put_other_image */
void ffmpeg_put_other_image_mpeg4(struct ffmpeg *ffmpeg, unsigned char *y, unsigned char *u, unsigned char *v)
{
	int out_size, ret;
	AVFrame *picture, *picture_ptr;

	/* allocate the encoded raw picture */
	picture = avcodec_alloc_frame();
	if (!picture) {
		syslog(LOG_ERR, "Could not alloc frame: %m");
		return;
	}

	if (ffmpeg->vbr)
		picture->quality = ffmpeg->vbr;

	picture->data[0] = y;
	picture->data[1] = u;
	picture->data[2] = v;
	picture->linesize[0] = ffmpeg->width;
	picture->linesize[1] = ffmpeg->width / 2;
	picture->linesize[2] = ffmpeg->width / 2;

	picture_ptr = picture;

	if (ffmpeg->oc->oformat->flags & AVFMT_RAWPICTURE) {
		/* raw video case. The API will change slightly in the near future for that */
		ret = av_write_frame(ffmpeg->oc, ffmpeg->video_st->index,
				(uint8_t *)picture_ptr, sizeof(AVPicture));
	} else {
		/* encode the image */
		out_size = avcodec_encode_video(&ffmpeg->video_st->codec, ffmpeg->video_outbuf, ffmpeg->video_outbuf_size, picture_ptr);
		/* if zero size, it means the image was buffered */
		if (out_size != 0) {
			/* write the compressed frame in the media file */
			/* XXX: in case of B frames, the pts is not yet valid */
			ret = av_write_frame(ffmpeg->oc, ffmpeg->video_st->index, ffmpeg->video_outbuf, out_size);
		} else {
			ret = 0;
		}
	}
	if (ret != 0) {
		syslog(LOG_ERR, "Error while writing video frame: %m");
		return;
	}

	av_free(picture);
}

/* mpeg1 version of ffmpeg_put_other_image */
void ffmpeg_put_other_image_mpeg1(struct ffmpeg *ffmpeg, unsigned char *y, unsigned char *u, unsigned char *v)
{
	int out_size;
	AVFrame *picture = avcodec_alloc_frame();

	if (ffmpeg->vbr)
		picture->quality = ffmpeg->vbr;
	
	picture -> data[0] = y;
	picture -> data[1] = u;
	picture -> data[2] = v;
	picture -> linesize[0] = ffmpeg -> c -> width;
	picture -> linesize[1] = ffmpeg -> c -> width / 2;
	picture -> linesize[2] = ffmpeg -> c -> width / 2;

	out_size = avcodec_encode_video(
	    ffmpeg->c, 
	    ffmpeg->video_outbuf, 
	    ffmpeg->video_outbuf_size, 
	    picture);

	fwrite(ffmpeg->video_outbuf, 1, out_size, ffmpeg->f);

	free(picture);
}

#endif /* HAVE_FFMPEG */
