/*	motion.c
 *
 *	Detect changes in a video stream.
 *	Copyright 2000 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"

#ifdef __freebsd__
#include "video_freebsd.h"
#else
#include "video.h"
#endif /* __freebsd__ */

#include "conf.h"
#include "alg.h"
#include "track.h"
#include "event.h"
#include "picture.h"
#include "ffmpeg.h"

struct context **cnt=NULL;

int threads_running=0;
int restart=0;

typedef struct
{
	pthread_mutex_t lock;
	char **list;
	int size, maxsize;
	char finish;
} storethread;

static void context_init (struct context *cnt)
{
	cnt->threadnr=0;

	cnt->snapshot=0;
	cnt->makemovie=0;
	cnt->finish=0;

	cnt->noise=255;

	cnt->shots=0;
	cnt->event_nr=0;
	cnt->prev_event=0;
	cnt->currenttime=NULL;
	cnt->lasttime=0;
	cnt->eventtime=0;
	cnt->lastshot=0;
	cnt->lastrate=25;
	cnt->filesc=0;
	cnt->filesmc=0;
	cnt->files=NULL;
	cnt->filesm=NULL;
	cnt->filesrate=NULL;
	cnt->filesmrate=NULL;

	memcpy(&cnt->track, &track_template, sizeof(struct trackoptions));
	cnt->moved=0;

	cnt->switched=0;
	cnt->pause=0;
	cnt->stream_count=0;

	cnt->precap_nr=0;
	cnt->precap_cur=0;

	cnt->pipe=-1;
	cnt->mpipe=-1;

#if defined(HAVE_MYSQL) || defined(HAVE_PGSQL)
	cnt->sql_mask=0;
#endif
	
#ifdef HAVE_MYSQL
	cnt->database=NULL;
#endif /* HAVE_MYSQL */

#ifdef HAVE_PGSQL
	cnt->database_pg=NULL;
#endif /* HAVE_PGSQL */

#ifdef HAVE_FFMPEG
	cnt->ffmpeg_new=NULL;
	cnt->ffmpeg_motion=NULL;
	cnt->ffmpeg_timelapse=NULL;
	cnt->newfilename[0] = '\0';
	cnt->motionfilename[0] = '\0';
	cnt->timelapsefilename[0] = '\0';
#endif /* HAVE_FFMPEG */

	cnt->predict_sets=NULL;
	cnt->predict.length=0;
}


static char *get_pgm (FILE *picture, int width, int height)
{
	int x=0 ,y=0, maxval;
	char line[256];
	char *image;

	line[255]=0;
	fgets(line, 255, picture);
	if (strncmp(line, "P5", 2)) {
		syslog(LOG_ERR, "This is not a ppm file, starts with: '%s'", line);
		return NULL;
	}
	/* skip comment */
	line[0]='#';
	while (line[0]=='#') fgets(line, 255, picture);
	/* check size */
	sscanf(line, "%d %d", &x, &y);
	if (x!=width || y!=height) {
		syslog(LOG_ERR, "Wrong image size: %dx%d should be %dx%d", x, y, width, height);
		return NULL;
	}
	/* Maximum value */
	line[0]='#';
	while (line[0]=='#') fgets(line, 255, picture);
	sscanf(line, "%d", &maxval);
	/* read data */
	image=mymalloc(width*height);
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			fread(&image[y*width+x], 1, 1, picture);
			image[y*width+x]=(int)image[y*width+x]*255/maxval;
		}
	}
	return image;
}

static void cleanup (struct config *conf, char *filename)
{
	char tmp[PATH_MAX];

	remove(filename);
	if (!conf->oldlayout) {
		strcpy(tmp, filename);
		tmp[strlen(conf->filepath)+1+16]=0;
		rmdir(tmp);
		tmp[strlen(conf->filepath)+1+13]=0;
		rmdir(tmp);
		tmp[strlen(conf->filepath)+1+10]=0;
		rmdir(tmp);
		tmp[strlen(conf->filepath)+1+7]=0;
		rmdir(tmp);
		tmp[strlen(conf->filepath)+1+4]=0;
		rmdir(tmp);
	}
}

static void create_mpeg (struct context *cnt, int event_nr, int filesc, char **files, int *rate)
{
	int i, j;
	char params[255];
	char eventname[255];
	char mpegfilename[255];
	FILE *paramfile=NULL;
	char *eventtime;
	struct config *conf=&cnt->conf;

	if (!filesc) return;
	if (!fork()) {
#ifndef WITHOUT_V4L
		vid_close();
#endif	
		if (conf->oldlayout) {
			eventtime=files[0]+strlen(cnt->conf.filepath)+1;
			while (*eventtime++!='-');
			strncpy(eventname, eventtime, 14);
			if (files[0][strlen(files[0])-5]=='m') {
				eventname[14]='m';
				eventname[15]=0;
			} else eventname[14]=0;
		} else {
			strncpy(eventname, files[0]+strlen(cnt->conf.filepath)+1, 11);/* daydir */
			strncpy(eventname+11, files[0]+11+strlen(cnt->conf.filepath)+1, 2);/* hour */
			strncpy(eventname+13, files[0]+14+strlen(cnt->conf.filepath)+1, 2);/* minute */
			strncpy(eventname+15, files[0]+17+strlen(cnt->conf.filepath)+1, 2);/* second */
			if (files[0][strlen(files[0])-5]=='m') {
				eventname[17]='m';
				eventname[18]=0;
			} else eventname[17]=0;
		}
		sprintf(params, "%s/%s.params", cnt->conf.filepath, eventname);
		sprintf(mpegfilename, "%s/%s.mpg", cnt->conf.filepath, eventname);
		paramfile=myfopen(params, "w");
		if(paramfile==NULL) {
			syslog(LOG_ERR, "couldn't open %s\n", params);
			exit(0);
		}
		fprintf(paramfile,
			"PATTERN IBBPBBPBBPBBPBBP\n"
			"OUTPUT %s\n"
			"BASE_FILE_FORMAT ", mpegfilename);
		if (!strncmp("ppm", files[0]+strlen(files[0])-3, 3))
			fprintf(paramfile, "PPM\n");
		else 
			fprintf(paramfile, "JPEG\n");
		fprintf(paramfile,
			"INPUT_CONVERT *\n"
			"GOP_SIZE 30\n"
			"SLICES_PER_FRAME 1\n"
			"INPUT_DIR /\n"
			"INPUT\n");
		for (i=0; i<filesc; i++) {
			if (!conf->adjustrate || rate[i]>25) {
				fprintf(paramfile, "%s\n", files[i]);
			} else {
				fprintf(stderr, "%d %d\n", rate[i], 25/(rate[i] ? rate[i]: 1));
				for (j=0; j<25/(rate[i] ? rate[i]: 2); j++) {
					fprintf(paramfile, "%s\n", files[i]);
					fflush(paramfile);
				}
			}
		}
		fprintf(paramfile,
			"END_INPUT\n"
			"PIXEL HALF\n"
			"RANGE 10\n"
			"PSEARCH_ALG LOGARITHMIC\n"
			"BSEARCH_ALG CROSS2\n"
			"IQSCALE 4\n"
			"PQSCALE 5\n"
			"BQSCALE 13\n"
			"FORCE_ENCODE_LAST_FRAME\n"
			"REFERENCE_FRAME ORIGINAL\n"/*DECODED*/
			"FRAME_RATE 25\n");
		fclose(paramfile);

		if (!fork()) {
			nice(5);
			execlp(cnt->conf.mpeg_encode_bin, "mpeg_encode", params, 0);
			syslog(LOG_ERR, "Failed to start mpeg_encode: %m");
			exit(1);
		}
		if (wait(0)==-1) {
			syslog(LOG_ERR, "mpeg creation unsuccesfull: %m");
			exit(1);
		}
		event(EVENT_FILECREATE, cnt, mpegfilename, (void *)FTYPE_MPEG, NULL);
		for (i=0; i<filesc; i++) {
			if (conf->jpg_cleanup) 
				cleanup(conf, files[i]);
			free(files[i]);
		}
		if (files)
			free(files);
		if (rate)
			free(rate);
		cleanup(conf, params);
		exit(0);
	}
	return;
}


/*	Our SIGNAL-Handler
 *	We need this to handle alarms and external signals
 */
static void sig_handler (int signo)
{
	int i;

	switch(signo) {
		case SIGHUP: {
			syslog(LOG_DEBUG, "Received SIGHUP, preparing restart\n");
			restart=1;
			if (cnt) {
				i=-1;
				while (cnt[++i]) {
					cnt[i]->makemovie=1;
					cnt[i]->finish=1;
					if(cnt[i]->pipe != -1)
						close(cnt[i]->pipe);
					if(cnt[i]->mpipe != -1)
						close(cnt[i]->mpipe);
				}
			}
			signal(SIGHUP, sig_handler);
			return;
		}
		case SIGALRM: {
			/* Somebody (maybe we ourself) wants us to make a snapshot
			 * This feature triggers snapshots on ALL threads that have
			 * snapshot_interval different from 0.
			 */
			if (cnt) {
				i=-1;
				while (cnt[++i]) {
					if (cnt[i]->conf.snapshot_interval) {
						cnt[i]->snapshot=1;
					}
				}
			}
			/* Set everything for the next snapshot */
			signal(SIGALRM, sig_handler);
			return;
		}
		case SIGUSR1: {
			/* Ouch! We have been hit from the outside! Someone wants us to
			   make a movie! */
			if (cnt) {
				i=-1;
				while (cnt[++i])
					cnt[i]->makemovie=1;
			}
			signal(SIGUSR1, sig_handler);
			return;
		}
		case SIGINT:
		case SIGQUIT:
		case SIGTERM: {

			/* Somebody wants us to quit! We should better finish the actual
			    movie and end up! */
			signal(SIGINT, sig_handler);
			signal(SIGQUIT, sig_handler);
			signal(SIGTERM, sig_handler);
			if (cnt) {
				i=-1;
				while (cnt[++i]) {
					cnt[i]->makemovie=1;
					cnt[i]->finish=1;
					if(cnt[i]->pipe != -1)
						close(cnt[i]->pipe);
					if(cnt[i]->mpipe != -1)
						close(cnt[i]->mpipe);
				}
			}
			return;
		}
		default:
			syslog(LOG_ERR, "unknown signal: %d\n", signo);
			if (signo==11)
				exit(0);
	}

	return;
}


static int motion_detected (struct context *cnt, int diffs, int dev, int devpipe, int devmpipe, char *newimg)
{
	struct config *conf=&cnt->conf;
	struct images *imgs=&cnt->imgs;
	struct coord location;

	if (diffs) {
		if (conf->locate || cnt->track.type || conf->switchfilter)
			location=alg_locate_center(imgs, imgs->width, imgs->height);
		if (conf->locate)
			alg_locate(&location, imgs, imgs->width, imgs->height, newimg);
		if (conf->switchfilter) {
			diffs=alg_switchfilter(cnt, diffs, &location, newimg);
			if (diffs <= cnt->threshold )
				return 0;
		}
	}
	alg_predict(&location, cnt, diffs, cnt->event_nr!=cnt->prev_event);
	event(EVENT_MOTION, cnt, NULL, NULL, cnt->currenttime);
	cnt->lasttime=cnt->currenttimep;
	/* Take action if necessary */
	if (cnt->event_nr!=cnt->prev_event) {
		int i, tmpshots;
		struct tm tmptime;

		event(EVENT_FIRSTMOTION, cnt, newimg, NULL, cnt->currenttime);
		cnt->prev_event=cnt->event_nr;
		cnt->eventtime=cnt->currenttimep;

		//DEBUG CODE syslog(LOG_DEBUG, "Motion detected (number of changed pixels: %d)", diffs);
		
		
		/* pre_capture frames are written as jpegs and to the ffmpeg film
		 * We store the current cnt->shots temporarily until we are done with
		 * the pre_capture stuff
		 */
		 
		tmpshots = cnt->shots;
		
		for (i=cnt->precap_cur; i < cnt->precap_nr; i++) {
			memcpy(&tmptime, localtime((cnt->imgs.timestamp + i)), sizeof(struct tm));
			cnt->shots = *(cnt->imgs.shotstamp + i);
			event(EVENT_IMAGE_DETECTED, cnt,
			    cnt->imgs.new + (cnt->imgs.size * i), NULL, &tmptime);
		}
		
		if (cnt->precap_cur) {
			memcpy(&tmptime, localtime((cnt->imgs.timestamp+cnt->precap_nr)), sizeof(struct tm));
			cnt->shots = *(cnt->imgs.shotstamp + cnt->precap_nr);
			event(EVENT_IMAGE_DETECTED, cnt,
			    cnt->imgs.new + (cnt->imgs.size * cnt->precap_nr),
			    NULL, &tmptime);
		}

		for (i=0; i < cnt->precap_cur-1; i++) {
			memcpy(&tmptime, localtime((cnt->imgs.timestamp+i)), sizeof(struct tm));
			cnt->shots = *(cnt->imgs.shotstamp + i);
			event(EVENT_IMAGE_DETECTED, cnt,
			    cnt->imgs.new + (cnt->imgs.size * i),
			    NULL, &tmptime);
		}
		
		cnt->shots = tmpshots;
	}

	if (cnt->shots < conf->frame_limit && cnt->currenttimep-cnt->lastshot >= conf->mingap )
	{
		cnt->lastshot=cnt->currenttimep;

		/* Output the latest picture 'image_new' or image_out for motion picture.
		 */
		event(EVENT_IMAGE_DETECTED, cnt, newimg, NULL, cnt->currenttime);
	}

	if (cnt->track.type != 0 && diffs != 0)
	{
		cnt->moved=track_move(cnt, dev, devpipe, devmpipe, &location, imgs, 0);
	}

	return diffs;
}


static void *motion_loop(void *arg)
{
	struct context *cnt=arg;
	int i, j, diffs, detecting_motion=0;
	struct tm curtime;
	time_t lastframe=0;
	FILE *picture;
	int postcap=0;
	int frame_buffer_size;

	int passflag=0;
	long int *rolling_average_data;
	long int rolling_average_limit, required_frame_time, frame_delay;
	int rolling_frame=0;
	struct timeval tv1, tv2;
	unsigned long int rolling_average, elapsedtime, splittime, timenow=0, timebefore=0;

	cnt->currenttime=&curtime;

	syslog(LOG_DEBUG, "Thread PID: %d\n", getpid());

	if (!cnt->conf.filepath)
		cnt->conf.filepath=".";

	/* Work out expected frame rate based on config setting */
	if (cnt->conf.frame_limit)
		required_frame_time=1000000L/cnt->conf.frame_limit;
	else
		required_frame_time=0;
	frame_delay=required_frame_time;
	
	/* Reserve enough space for a 10 second timing history buffer */
	rolling_average_limit=10*cnt->conf.frame_limit;
	rolling_average_data=mymalloc(sizeof(long int)*rolling_average_limit);
	if (!rolling_average_data)
	{
		syslog(LOG_ERR, "Unable to allocate memory [rolling_average_data]");
		exit(1);
	}
	
	/* Preset history buffer with expected frame rate */
	for (j=0; j< rolling_average_limit; j++)
		rolling_average_data[j]=required_frame_time;

	/* set the device settings */
	cnt->video_dev=vid_start (cnt, cnt->conf.width, cnt->conf.height, cnt->conf.input, cnt->conf.norm, cnt->conf.frequency);
	if (cnt->video_dev==-1) {
		syslog(LOG_ERR, "Capture error %s",strerror(errno));
		exit(1);
	}
	cnt->imgs.new=mymalloc(cnt->imgs.size);
	cnt->imgs.ref=mymalloc(cnt->imgs.size);
	cnt->imgs.out=mymalloc(cnt->imgs.size);
	cnt->imgs.labels=mymalloc(cnt->imgs.motionsize*sizeof(cnt->imgs.labels));
	cnt->imgs.labelsize=mymalloc((cnt->imgs.motionsize/2+1)*sizeof(cnt->imgs.labelsize));
	cnt->imgs.timestamp=mymalloc(sizeof(time_t));
	cnt->imgs.shotstamp=mymalloc(sizeof(int));

	/* Allow videodevice to settle in */
	sleep(1);

	/* Capture first image, or we will get an alarm on start */
	if (!(vid_next(cnt, cnt->video_dev, cnt->imgs.new, cnt->conf.width, cnt->conf.height))) {
		syslog(LOG_ERR, "Capture error %s",strerror(errno));	
		exit(1);
	}

	/* create a reference frame */
	memcpy(cnt->imgs.ref, cnt->imgs.new, cnt->imgs.size);

#ifndef WITHOUT_V4L
	/* open video loopback devices if enabled */
	if (cnt->conf.vidpipe) {
		printf("Opening video loopback device for normal pictures\n");
		cnt->pipe=vid_startpipe(cnt->conf.vidpipe, cnt->conf.width, cnt->conf.height, cnt->imgs.type);
		if (cnt->pipe < 0) {
			syslog(LOG_ERR, "Failed to open video loopback\n");
			exit(1);
		}
	}
	if (cnt->conf.motionvidpipe) {
		printf("Opening video loopback device for motion pictures\n");
		cnt->mpipe=vid_startpipe(cnt->conf.motionvidpipe, cnt->conf.width, cnt->conf.height, cnt->imgs.type);
		if (cnt->mpipe < 0) {
			syslog(LOG_ERR, "Failed to open video loopback\n");
			exit(1);
		}
	}
#endif /*WITHOUT_V4L*/

#ifdef HAVE_MYSQL
	if(cnt->conf.mysql_db) {
		cnt->database=(MYSQL *) mymalloc(sizeof(MYSQL));
		mysql_init(cnt->database);
		if (!mysql_real_connect(cnt->database, cnt->conf.mysql_host, cnt->conf.mysql_user,
		    cnt->conf.mysql_password, cnt->conf.mysql_db, 0, NULL, 0)) {
			syslog(LOG_ERR, "Cannot connect to database %s on host %s with user %s", cnt->conf.mysql_db, cnt -> conf.mysql_host, cnt -> conf.mysql_user);
			exit(1);
		}
	}
#endif /* HAVE_MYSQL */
	
#ifdef HAVE_PGSQL
	if (cnt->conf.pgsql_db) {
		char connstring[255];
	
		/* create the connection string. 
		   Quote the values so we can have null values (blank)*/
		snprintf(connstring, 255,
		    "dbname='%s' host='%s' user='%s' password='%s' port='%d'",
		    cnt->conf.pgsql_db, /* dbname */
		    (cnt->conf.pgsql_host ? cnt->conf.pgsql_host : ""), /* host (may be blank) */
		    (cnt->conf.pgsql_user ? cnt->conf.pgsql_user : ""), /* user (may be blank) */
		    (cnt->conf.pgsql_password ? cnt->conf.pgsql_password : ""), /* password (may be blank) */
		    cnt->conf.pgsql_port
		);

		cnt->database_pg = PQconnectdb(connstring);
		if (PQstatus(cnt->database_pg) == CONNECTION_BAD)
		{
			syslog(LOG_ERR, "Connection to database '%s' failed: %s", cnt->conf.pgsql_db, PQerrorMessage(cnt->database_pg));
			exit(1);
		}
	}
#endif /* HAVE_PGSQL */

	/* Load the mask file if any */
	if (cnt->conf.mask_file) {
		if ((picture=fopen(cnt->conf.mask_file, "r"))) {
			cnt->imgs.mask=get_pgm(picture, cnt->conf.width, cnt->conf.height);
			fclose(picture);
		} else {
			syslog(LOG_ERR, "Error opening mask file %s: %m", cnt->conf.mask_file);
		}
		if (!cnt->imgs.mask) {
			syslog(LOG_ERR, "failed to read mask image");
			exit(1);
		}
	} else
		cnt->imgs.mask=NULL;


	/* Set noise level if no noice_tune */
	if (!cnt->conf.noise_tune)
		cnt->noise=cnt->conf.noise;

	/* Set threshold value */
	cnt->threshold=cnt->conf.max_changes;

	/* Set signal handlers */
	signal(SIGALRM, sig_handler);
	signal(SIGUSR1, sig_handler);
	signal(SIGHUP, sig_handler);
	signal(SIGTERM, sig_handler);
	signal(SIGPIPE, SIG_IGN);

	/* Initialize webcam server is webcam port is specified to not 0 */
	if (cnt->conf.webcam_port)
		if ( webcam_init(cnt) == -1 ) {	
			syslog(LOG_ERR, "Problem enabling stream server: %s",strerror(errno));
			printf("Problem enabling stream server: %s\n",strerror(errno));
			cnt->finish=1; cnt->makemovie=0;
                        //exit(0);
		}

	/* Prevent first few frames from triggering motion... */
	cnt->moved=8;
	

	/* MAIN MOTION LOOP BEGINS HERE */
	/* Should go on forever... unless you bought vaporware :) */

	while (!cnt->finish || cnt->makemovie) {
		unsigned char *newimg=NULL;

		/* since we don't have sanity checks done when options are set,
		 * this sanity check must go in the main loop :(, before pre_captures
		 * are attempted. */
		if (cnt->conf.minimum_motion_frames < 1)
			cnt->conf.minimum_motion_frames = 1;
		if (cnt->conf.pre_capture < 0)
			cnt->conf.pre_capture = 0;

		/* Check if our buffer is still the right size */
		frame_buffer_size = cnt->conf.pre_capture + cnt->conf.minimum_motion_frames - 1;
		if (cnt->precap_nr != frame_buffer_size) {
			/* Only decrease if at last position in new buffer */
			if (frame_buffer_size > cnt->precap_nr || frame_buffer_size == cnt->precap_cur) {
				char *tmp;
				time_t *tmp2;
				int *tmp3;
				int smallest;
				smallest = (cnt->precap_nr < frame_buffer_size) ? cnt->precap_nr : frame_buffer_size;
				tmp=mymalloc(cnt->imgs.size*(1+frame_buffer_size));
				tmp2=mymalloc(sizeof(time_t)*(1+frame_buffer_size));
				tmp3=mymalloc(sizeof(int)*(1+frame_buffer_size));
				memcpy(tmp, cnt->imgs.new, cnt->imgs.size*(1+smallest));
				memcpy(tmp2, cnt->imgs.timestamp, sizeof(time_t)*(1+smallest));
				memcpy(tmp3, cnt->imgs.timestamp, sizeof(int)*(1+smallest));
				free(cnt->imgs.new);
				free(cnt->imgs.timestamp);
				free(cnt->imgs.shotstamp);
				cnt->imgs.new=tmp;
				cnt->imgs.timestamp=tmp2;
				cnt->imgs.shotstamp=tmp3;
				cnt->precap_nr=frame_buffer_size;
			}
		}

		/* Get time for current frame */
		cnt->currenttimep=time(NULL);
		
		/* localtime returns static data and is not threadsafe
		 * so we copy the time to a safe place
		 */
		memcpy(cnt->currenttime, localtime(&cnt->currenttimep), sizeof(struct tm));
		
		/* If we have started on a new second we reset the shots variable
		 * lastrate is updated to be the number of the last frame. last rate
		 * is used as the ffmpeg framerate when motion is detected.
		 */
		if (lastframe!=cnt->currenttimep) {
			cnt->lastrate=cnt->shots;
			cnt->shots=-1;
			lastframe=cnt->currenttimep;
		}

		/* Increase the shots variable for each frame captured within this second */
		cnt->shots++;

		/* Store time with pre_captured image*/
		*(cnt->imgs.timestamp+cnt->precap_cur)=cnt->currenttimep;

		/* Store shot number with pre_captured image*/
		*(cnt->imgs.shotstamp+cnt->precap_cur)=cnt->shots;

		/* newimg points to the current image and precap_cur incremented pointing to the
		 * position in the buffer for the NEXT image frame, not the current!!!
		 */
		newimg=cnt->imgs.new+(cnt->imgs.size*(cnt->precap_cur++));
		
		/* If we are at the end of the ring buffer go to the start */
		if (cnt->precap_cur > cnt->precap_nr)
			cnt->precap_cur=0;

		/* Fetch next frame from camera */
		if (!vid_next(cnt, cnt->video_dev, newimg, cnt->conf.width, cnt->conf.height))
			break;
		
		/* Get current time and preserver last time for frame interval calc. */
		timebefore=timenow;
		gettimeofday(&tv1, NULL);
		timenow=tv1.tv_usec+1000000L*tv1.tv_sec;

#if defined(HAVE_MYSQL) || defined(HAVE_PGSQL)

		/* Set the sql mask file according to the SQL config options
		 * We update it for every frame in case the config was updated
		 * via remote control.
		 */
		cnt->sql_mask = cnt->conf.sql_log_image * (FTYPE_IMAGE + FTYPE_IMAGE_MOTION) +
		                cnt->conf.sql_log_snapshot * FTYPE_IMAGE_SNAPSHOT +
		                cnt->conf.sql_log_mpeg * (FTYPE_MPEG + FTYPE_MPEG_MOTION) +
		                cnt->conf.sql_log_timelapse * FTYPE_MPEG_TIMELAPSE +
		                cnt->conf.sql_log_prediction * FTYPE_PREDICT;
		
#endif /* defined(HAVE_MYSQL) || defined(HAVE_PGSQL) */

		/* The actual motion detection takes place in the following
		 * diffs is the number of pixels detected as changed
		 * Make a differences picture in image_out
		 */
		if (cnt->threshold && !cnt->pause) {
			/* if we've already detected motion and we want to see if there's
			 * still motion, don't bother trying the fast one first. IF there's
			 * motion, the alg_diff_hybrid will trigger alg_diff_standard
			 * anyway
			 */
			if (detecting_motion)
				diffs=alg_diff_standard(cnt, newimg);
			else
				diffs=alg_diff(cnt, newimg);
 
 			/* Despeckle feature is run now */
			if (cnt->conf.despeckle && diffs > 0) {
				int olddiffs = diffs;
				diffs=alg_despeckle(cnt);
				if (cnt->conf.always_diff)
					printf("Despeckle %d (%s) from %d to %d\n",
						cnt->threadnr, cnt->conf.despeckle,
						olddiffs, diffs);
			}
		} else
			diffs=0;
#ifndef WITHOUT_V4L			
		/* Auto-brightness feature */
		if (cnt->conf.autobright)
			vid_autobright(&cnt->conf, cnt->video_dev, newimg, cnt->conf.width, cnt->conf.height);
#endif
		if (cnt->moved) {
			cnt->moved--;
			diffs=0;
		}
		
		/* Lightswitch feature - has lightintensity changed?
		 * This can happen due to change of light conditions or due to a sudden change of the camera
		 * sensitivity. If alg_lightswitch returns 
		 */
		if (cnt->conf.lightswitch) {
			if (alg_lightswitch(cnt, diffs)) {
				printf("Lightswitch detected\n");
				if (cnt->moved<5)
					cnt->moved = 5;
				diffs = 0;
			}
		}
		
		/* Is the mpeg movie to long? Then make movies
		 * First test for max mpegtime
		 */
		if (cnt->conf.maxmpegtime && cnt->event_nr==cnt->prev_event)
			if (cnt->currenttimep-cnt->eventtime>= cnt->conf.maxmpegtime)
				cnt->makemovie=1;

		/* Now test for quiet longer than 'gap' OR make movie as decided in
		 * previous statement.
		 */
		if ((cnt->currenttimep-cnt->lasttime >= cnt->conf.gap) || cnt->makemovie) {
			if (cnt->event_nr == cnt->prev_event || cnt->makemovie) {
				event(EVENT_ENDMOTION, cnt, NULL, localtime(&cnt->eventtime), cnt->currenttime);
				
				/* if tracking is enabled we track a little */
				if (cnt->track.type) {
					printf("%d %d %d\n", cnt->prev_event, cnt->event_nr, cnt->makemovie);
					cnt->moved=track_center(cnt, cnt->video_dev, 0, 0, 0);
				}
				
				cnt->makemovie=0;
				cnt->event_nr++;
				
				/* make mpeg using old Berkeley mpeg_encode */
				if (cnt->conf.mpeg) {
					if (cnt->filesc)
						create_mpeg(cnt, cnt->prev_event,
							    cnt->filesc,
							    cnt->files, 
							    cnt->filesrate
							    );
					if (cnt->filesmc)
						create_mpeg(cnt, cnt->prev_event,
							    cnt->filesmc, 
							    cnt->filesm, 
							    cnt->filesmrate
							    );
				}
				for (i=0; i<cnt->filesc; i++) free(cnt->files[i]);
				if (cnt->files)
					free(cnt->files);
				for (i=0; i<cnt->filesmc; i++) free(cnt->filesm[i]);
				if (cnt->filesm)
					free(cnt->filesm);
				if (cnt->filesrate)
					free(cnt->filesrate);
				if (cnt->filesmrate)
					free(cnt->filesmrate);
				cnt->filesc=0;
				cnt->files=NULL;
				cnt->filesmc=0;
				cnt->filesm=NULL;
				cnt->filesrate=NULL;
				cnt->filesmrate=NULL;
			}
		}

		/* Old image slowly decays, this will make it even harder on
		 * a slow moving object to stay undetected
		 */
		for (i=cnt->imgs.size-1; i>=0; i--) {
			cnt->imgs.ref[i]=(cnt->imgs.ref[i]+newimg[i])/2;
		}

		/* Add changed pixels in upper right corner of the pictures */
		if (cnt->conf.text_changes){
			char tmp[10];
			sprintf(tmp, "%d", diffs);
			draw_text(newimg, cnt->conf.width-10, 10, cnt->conf.height, cnt->conf.width, tmp);
		}

		/* Add text in lower left corner of the pictures */
		if (cnt->conf.text_left) {
			char tmp[PATH_MAX];
			mystrftime(tmp, sizeof(tmp), cnt->conf.text_left, cnt->currenttime, cnt->event_nr, cnt->shots);
			draw_text(newimg, 10, cnt->conf.height-10, cnt->conf.height, cnt->conf.width, tmp);
		}

		/* Add text in lower right corner of the pictures */
		if (cnt->conf.text_right) {
			char tmp[PATH_MAX];
			mystrftime(tmp, sizeof(tmp), cnt->conf.text_right, cnt->currenttime, cnt->event_nr, cnt->shots);
			draw_text(newimg, cnt->conf.width-10, cnt->conf.height-10, cnt->conf.height, cnt->conf.width, tmp);
		}

		/* Is output_all enabled?
		 * If so, take appropriate action by calling motion_detected() */
		if (cnt->conf.output_all) {
			detecting_motion=1;
			motion_detected(cnt, 0, cnt->video_dev, cnt->pipe, cnt->mpipe, newimg);
		} else if (diffs > cnt->threshold) {
			/* Did we detect motion (like the cat just walked in :) )?
			 * If so, ensure the motion is sustained if minimum_motion_frames
			 * is set, and take action by calling motion_detected().
			 * pre_capture is handled by motion_detected(), and we handle
			 * post_capture here. */
			if (!detecting_motion)
				detecting_motion=1;
			
			detecting_motion++;
			
			if (detecting_motion > cnt->conf.minimum_motion_frames) {
				if (motion_detected(cnt, diffs, cnt->video_dev, cnt->pipe, cnt->mpipe, newimg))
					postcap=cnt->conf.post_capture;
			}
		} else if (postcap) {
			motion_detected(cnt, 0, cnt->video_dev, cnt->pipe, cnt->mpipe, newimg);
			postcap--;
		} else {
			detecting_motion=0;
		}

		/* if noise tuning was selected, do it now. but only when
		 * no frames have been recorded
		 */
		if (cnt->conf.noise_tune && cnt->shots==0)
		{
			if (!detecting_motion)
			{
				alg_noise_tune(cnt, newimg);
			}
			/* if motion was detected, reset the noise-level */
			else
			{
				cnt->noise = (cnt->noise + cnt->conf.noise) / 2;
			}
		}

		/* if we are not noise tuning lets make sure that remote controlled
		 * changes of noise_level are used.
		 */
		if (!cnt->conf.noise_tune)
			cnt->noise=cnt->conf.noise;

		/* threshold tuning if enabled 
		 * if we are not threshold tuning lets make sure that remote controlled
		 * changes of threshold are used.
		 */
		if (cnt->conf.threshold_tune)
			alg_threshold_tune(cnt, diffs, detecting_motion);
		else
			cnt->threshold=cnt->conf.max_changes;

		/* If enabled output number of motion detected pixels to console */
		if (cnt->conf.always_diff)
			printf("changes: %d\n", diffs);

		/* Prevent the motion created by moving camera or sudden light intensity
		 * being detected by creating a fresh reference frame
		 */
		if ((cnt->track.type || cnt->conf.lightswitch) && cnt->moved)
			memcpy(cnt->imgs.ref, newimg, cnt->imgs.size);

		/* Did we get triggered to make a snapshot xmlrpc? Then snap
		 * If snapshot_interval is not zero and
		 * time since epoch MOD snapshot_interval = 0  then snap
		 * Note: Negative value means SIGALRM snaps is enabled
		 * XMLRPC snaps are always enabled.
		 */
		if ( (cnt->conf.snapshot_interval > 0 && cnt->shots==0 &&
		      mktime(cnt->currenttime)%cnt->conf.snapshot_interval==0
		     ) || cnt->snapshot) {
			event(EVENT_IMAGE_SNAPSHOT, cnt, newimg, NULL, cnt->currenttime);
			cnt->snapshot=0;
		}
		
#ifdef HAVE_FFMPEG
		/* Timelapse feature */ 
		if (cnt->conf.timelapse) {			

			/* If ffmpeg timelapse is enabled and time since epoch MOD ffmpeg_timelaps = 0
			 * add a timelapse frame to the timelapse mpeg.
			 */
			if (cnt->shots == 0 && mktime(cnt->currenttime)%cnt->conf.timelapse==0)
				event(EVENT_TIMELAPSE, cnt, newimg, NULL, cnt->currenttime);
	
			/* Check to see if we should start a new timelapse file. We start one when 
			   we are on the first shot, and and the seconds are zero. We must use the seconds
			   to prevent the timelapse file from getting reset multiple times during the minute.
			*/ 
			if (cnt->currenttime->tm_min == 0 && cnt->currenttime->tm_sec == 0 && cnt->shots == 0) {
				
				if (strcasecmp(cnt->conf.timelapse_mode,"manual") == 0)
				;/* No action */ 

				/* If we are daily, raise timelapseend event at midnight */ 
				else if (strcasecmp(cnt->conf.timelapse_mode,"daily") == 0) {
					if (cnt->currenttime->tm_hour == 0)
						event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}

				/* handle the hourly case */ 	
				else if (strcasecmp(cnt->conf.timelapse_mode,"hourly") == 0) {
					event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}

				/* If we are weekly-sunday, raise timelapseend event at midnight on sunday */ 
				else if (strcasecmp(cnt->conf.timelapse_mode,"weekly-sunday") == 0) {
					if (cnt->currenttime->tm_hour == 0 && cnt->currenttime->tm_wday == 0)
						event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}
				
				/* If we are weekly-monday, raise timelapseend event at midnight on monday */ 
				else if (strcasecmp(cnt->conf.timelapse_mode,"weekly-monday") == 0) {
					if (cnt->currenttime->tm_hour == 0 && cnt->currenttime->tm_wday == 1)
						event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}
				
				/* If we are weekly-monday, raise timelapseend event at midnight on monday */
				else if (strcasecmp(cnt->conf.timelapse_mode,"monthly") == 0) {
					if (cnt->currenttime->tm_mday == 0 && cnt->currenttime->tm_hour == 0)
						event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
				}
				
				/* If invalid we report in syslog once and continue in manual mode */
				else {
					syslog(LOG_ERR, "invalid timelapse_mode argument '%s'",cnt->conf.timelapse_mode);
					syslog(LOG_ERR, "defaulting to manual timelapse mode");
					conf_cmdparse(&cnt, "ffmpeg_timelapse_mode","manual");
				}
			}
		}
		/* if timelapse mpeg is in progress but conf.timelapse is zero then close timelapse file
		 * This is an important feature that allows manual roll-over of timelapse file using xmlrpc
		 * via a cron job
		 */
		else if (cnt->ffmpeg_timelapse)
			event(EVENT_TIMELAPSEEND, cnt, NULL, NULL, cnt->currenttime);
#endif /* HAVE_FFMPEG */

		/* feed last image and motion image to video device pipes */
		event(EVENT_IMAGE, cnt, newimg, &cnt->pipe, cnt->currenttime);
		event(EVENT_IMAGEM, cnt, cnt->imgs.out, &cnt->mpipe, cnt->currenttime);

		/* Get latest time to calculate time taken to process video data */
		gettimeofday(&tv2, NULL);
		splittime=tv2.tv_usec+1000000L*tv2.tv_sec;
		elapsedtime=splittime-timenow;
		/* Update history buffer but ignore first pass as timebefore
		   variable will be inaccurate
		 */
		if (passflag)
			rolling_average_data[rolling_frame]=timenow-timebefore;
		rolling_frame++;
		passflag=1;
		if (rolling_frame>=rolling_average_limit)
			rolling_frame=0;
		rolling_average=0L;
		/* Calculate 10 second average and use deviation in delay calculation */
		for (j=0; j<rolling_average_limit; j++)
			rolling_average+=rolling_average_data[j];
		rolling_average/=rolling_average_limit;
		frame_delay=required_frame_time-elapsedtime-(rolling_average-required_frame_time);

		if (frame_delay>0) {
			/* Apply delay to meet frame time */
			if (frame_delay>required_frame_time)
				frame_delay=required_frame_time;
			/* Commented out - only for special debugging
			if (!cnt->conf.quiet)
				printf("Split Time is %7lu. Applying %6lu delay. Avg Frame Time %7lu Frame Error %ld\n",
				    elapsedtime, frame_delay, rolling_average, rolling_average-required_frame_time);
			*/
			usleep(frame_delay);
		}

		/* This will limit the framerate to 1 frame while not detecting
		   motion. Using a different motion flag to allow for multiple frames per second
		 */

		if (cnt->conf.low_cpu && !detecting_motion) {
			/* Recalculate remaining time to delay for a total of 1/low_cpu seconds */
			if (required_frame_time+elapsedtime<(1000000/cnt->conf.low_cpu)) {
				frame_delay=(1000000/cnt->conf.low_cpu)-required_frame_time-elapsedtime;
				usleep(frame_delay);
			}
			/* Correct frame times to ensure required_frame_time is maintained */
			gettimeofday(&tv1, NULL);
			timenow=(tv1.tv_usec+(1000000L/cnt->conf.low_cpu)*tv1.tv_sec)-required_frame_time;
		}
	}
	
	printf("Thread exiting\n");
	if (!cnt->finish)
		syslog(LOG_ERR, "Somebody stole the video device, lets hope we got his picture");
	event(EVENT_STOP, cnt, NULL, NULL, NULL);
	threads_running--;
	return NULL;
}

int main (int argc, char **argv)
{
	int i, j;
	int webcam_port;
	pthread_attr_t thread_attr;
	pthread_t thread_id;

	/* cnt is an array of pointers to the context structures for each thread.
	 * First we reserve room for a pointer to thread 0's context structure
	 * and a NULL pointer which indicates that end of the array of pointers to
	 * thread context structures.
	 */
	cnt=mymalloc(sizeof(struct context *)*2);
	
	/* Now we reserve room for thread 0's context structure and let cnt[0] point to it */
	cnt[0]=mymalloc(sizeof(struct context));
	
	/* Populate context structure with start/default values */
	context_init(cnt[0]);
	
	/* cnt[1] pointing to zero indicates no more thread context structures - they get added later */
	cnt[1]=NULL;

	/* Enable automatic zombie reaping */
	signal(SIGCHLD, SIG_IGN);

	/* Command line arguments are being pointed to from cnt[0] and we call conf_load which loads
	 * the config options from motion.conf, thread config files and the command line.
	 */
	cnt[0]->conf.argv=argv;
	cnt[0]->conf.argc=argc;
	cnt=conf_load(cnt);

#ifdef HAVE_FFMPEG
	ffmpeg_init();
#endif /* HAVE_FFMPEG */

	if (cnt[0]->daemon) {
		if (fork()) {
			printf("motion: going to daemon mode\n");
			exit(0);
		}
		signal(SIGQUIT, sig_handler);

#ifdef __freebsd__
		setpgrp(0,getpid());		
#else
		setpgrp();
#endif /* __freebsd__ */

		if ((i=open("/dev/tty", O_RDWR)) >= 0) {
			ioctl(i, TIOCNOTTY, NULL);
			close(i);
		}
		setsid();
		i = open("/dev/null", O_RDONLY);
		if(i != -1) {
			dup2(i, STDIN_FILENO);
			close(i);
		}
		i = open("/dev/null", O_WRONLY);
		if(i != -1) {
			dup2(i, STDOUT_FILENO);
			dup2(i, STDERR_FILENO);
			close(i);
		}
		signal(SIGTTOU, SIG_IGN);
		signal(SIGTTIN, SIG_IGN);
		signal(SIGTSTP, SIG_IGN);

		syslog(LOG_INFO, "motion running as daemon process");
		if (cnt[0] -> conf.low_cpu)
			syslog(LOG_INFO, "capturing %d frames/s when idle", cnt[0] -> conf.low_cpu);
	}

	pthread_attr_init(&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
#ifndef WITHOUT_V4L
	vid_init();
#endif
	do {
		if (restart) {
			i=-1;
			while (cnt[++i])
				free(cnt[i]);
			free(cnt);
			cnt=mymalloc(sizeof(struct context *)*2);
			cnt[0]=mymalloc(sizeof(struct context));
			context_init(cnt[0]);
			cnt[1]=NULL;

			signal(SIGCHLD, SIG_IGN);

			cnt[0]->conf.argv=argv;
			cnt[0]->conf.argc=argc;
			cnt=conf_load(cnt);
			restart=0;
		}

		
		/* Check the webcam port number for conflicts. Two threads cannot use the same port number
		 * for the webcam. If a duplicate port is found the webcam feature gets disabled (port =0)
		 * for this thread and a warning is written to console and syslog.
		 */
		i = 0;
		while (cnt[++i]) {
			webcam_port = cnt[i]->conf.webcam_port;
			printf("Webcam port for thread %d is %d\n", i, webcam_port);
			j = i;
			while (cnt[++j]) {
				if (cnt[j]->conf.webcam_port == webcam_port && webcam_port != 0) {
					cnt[j]->conf.webcam_port = 0;
					printf("Webcam port number %d for thread %d conflicts with thread %d\n",
					        webcam_port, j, i);
					printf("Webcam feature for thread %d is disabled.\n", j);
					syslog(LOG_ERR, "Webcam port number %d for thread %d conflicts with thread %d\n",
					       webcam_port, j, i);
					syslog(LOG_ERR, "Webcam feature for thread %d is disabled.\n", j);	
				}
			}
		}

		i=-1;
		while (cnt[++i]) {
			/* first cnt is global if 'thread' option is used */
			if (!i && cnt[i+1])
				continue;
			printf("Thread%d device: %s input: %d\n",
			    cnt[i]->threadnr,
			    cnt[i]->conf.netcam_url?cnt[i]->conf.netcam_url:cnt[i]->conf.video_device,
			    cnt[i]->conf.netcam_url?-1:cnt[i]->conf.input
			);
			threads_running++;
			pthread_create(&thread_id, &thread_attr, &motion_loop, cnt[i]);
		}
		
#ifdef HAVE_XMLRPC
		if (cnt[0]->conf.control_port)
			pthread_create(&thread_id, &thread_attr, &motion_control, cnt);
#endif /* HAVE_XMLRPC */

		syslog(LOG_DEBUG, "waiting for threads to finish, pid: %d\n", getpid());
		while(threads_running > 0) {
			sleep(1);
		}
		syslog(LOG_DEBUG, "threads finished\n");
		if (restart)
			sleep(2);

	} while (restart);

	syslog(LOG_INFO, "motion terminating");


	/* Cleanup some memory before we exit */

	i=-1;
	while (cnt[++i]){
		if (cnt[i]->imgs.out)
			free(cnt[i]->imgs.out);
		if (cnt[i]->imgs.ref)
			free(cnt[i]->imgs.ref);
		if (cnt[i]->imgs.new)
			free(cnt[i]->imgs.new);
		if (cnt[i]->imgs.shotstamp)
			free(cnt[i]->imgs.shotstamp); 
		if (cnt[i])
			free(cnt[i]);
	}
	free(cnt);
#ifndef WITHOUT_V4L
	/* close devices ? */
	vid_close();
	/* That should clean viddev structs */
	vid_cleanup();
#endif /* WITHOUT_V4L */
	return 0;
}

/* allocate some memory and check if that succeeded or not. if it failed
 * do some errorlogging and bail out
 */
void * mymalloc(size_t nbytes)
{
	void *dummy = malloc(nbytes);
	if (!dummy) {
		syslog(LOG_EMERG, "Could not allocate %d bytes of memory!", nbytes);
		exit(1);
	}

	return dummy;
}

/* reallocate some memory and check if that succeeded or not. if it failed
 * do some errorlogging and bail out
 */
void * myrealloc(void *ptr, size_t size, char *desc)
{
	void *dummy = NULL;

	if (size == 0)
	{
		free(ptr);
		syslog(LOG_WARNING, "Warning! Function %s tries to resize memoryblock at %p to %d bytes!", desc, ptr, size);
	}
	else
	{
		dummy = realloc(ptr, size);
		if (!dummy) {
			syslog(LOG_EMERG, "Could not resize memory-block at offset %p to %d bytes (function %s)!", ptr, size, desc);
			exit(1);
		}
	}

	return dummy;
}

/* this function creates a whole path:
 * this/is/an/example/
 * /this/is/an/example/
 * Warning: a path *must* end with a slash!
 * etc.
 */
int create_path(char *path)
{
	char *start;
	mode_t mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;

	if (path[0] == '/')
		start = strchr(path + 1, '/');
	else
		start = strchr(path, '/');

	while(start)
	{
		char *buffer = strdup(path);
		buffer[start-path] = 0x00;

		if (mkdir(buffer, mode) == -1 && errno != EEXIST)
		{
			syslog(LOG_ERR, "Problem creating directory %s", buffer);
			free(buffer);
			return -1;
		}

		free(buffer);

		start = strchr(start + 1, '/');
	}

	return 0;
}

/* this function opens a file, if that failed because of an ENOENT error (which is: path
 * does not exist), the path is created and then things are tried again.
 * this is faster then trying to create that path over and over again. if someone
 * removes the path after it was created, myfopen will recreate the path automatically
 */
FILE * myfopen(char *path, char *mode)
{
	/* first, just try to open the file */
	FILE *dummy = fopen(path, mode);

	/* could not open file... */
	if (!dummy)
	{
		/* path did not exist? */
		if (errno == ENOENT)
		{
			//DEBUG CODE  syslog(LOG_DEBUG, "Could not open file %s directly; path did not exist. Creating path & retrying.", path);

			/* create path for file... */
			if (create_path(path) == -1)
			{
				return NULL;
			}

			/* and retry opening the file */
			dummy = fopen(path, mode);
			if (dummy)
				return dummy;
		}

		/* two possibilities
		 * 1: there was an other error while trying to open the file for the first time
		 * 2: could still not open the file after the path was created
		 */
		syslog(LOG_ERR, "Error opening file %s with mode %s: %s", path, mode, strerror(errno));

		return NULL;
	}

	return dummy;
}

/* 
   strftime(3) but with %v for eVent and %q for shots
*/
size_t mystrftime(char *s, size_t max, char *userformat,
                  const struct tm *tm, int event_nr, int shots)
{
	char *pos, *posend, format[PATH_MAX]="";
	
	/* First replace all %v by event number min 2 digits/leading zero*/
	pos = userformat;
	
	while ((posend = strstr(pos, "%v")))
	{
		strncat(format, pos, posend - pos);
		sprintf(format, "%s%02d", format, event_nr);
		pos = posend + 2;
	}
	
	strcat(format, pos);
	
	pos = format;
	while ((pos = strstr(format, "%q"))) {
		pos[0] = '0' + shots / 10;
		pos[1] = '0' + shots % 10;
	}
	return strftime(s, max, format, tm);
}
