/*	video.c
 *
 *	Video stream functions for motion.
 *	Copyright 2000 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */

/* Common stuff: */
#include "motion.h"
#include "video.h"
#include "conf.h"

#ifndef WITHOUT_V4L
#include "pwc-ioctl.h" 

/* for the v4l stuff: */
#include "sys/mman.h"
#include "sys/ioctl.h"
#endif /* WITHOUT_V4L */

/* for rotation */
#include "rotate.h"

#ifndef WITHOUT_V4L
void yuv422to420p(unsigned char *map, unsigned char *cap_map, int width, int height)
{
	unsigned char *src, *dest, *src2, *dest2;
	int i, j;

	/* Create the Y plane */
	src=cap_map;
	dest=map;
	for (i=width*height; i; i--) {
		*dest++=*src;
		src+=2;
	}
	/* Create U and V planes */
	src=cap_map+1;
	src2=cap_map+width*2+1;
	dest=map+width*height;
	dest2=dest+(width*height)/4;
	for (i=height/2; i; i--) {
		for (j=width/2; j; j--) {
			*dest=((int)*src+(int)*src2)/2;
			src+=2;
			src2+=2;
			dest++;
			*dest2=((int)*src+(int)*src2)/2;
			src+=2;
			src2+=2;
			dest2++;
		}
		src+=width*2;
		src2+=width*2;
	}
}

void rgb24toyuv420p(unsigned char *map, unsigned char *cap_map, int width, int height)
{
	unsigned char *y, *u, *v;
	unsigned char *r, *g, *b;
	int i, loop;

	b=cap_map;
	g=b+1;
	r=g+1;
	y=map;
	u=y+width*height;
	v=u+(width*height)/4;
	memset(u, 0, width*height/4);
	memset(v, 0, width*height/4);

	for(loop=0; loop<height; loop++) {
		for(i=0; i<width; i+=2) {
			*y++=(9796**r+19235**g+3736**b)>>15;
			*u+=((-4784**r-9437**g+14221**b)>>17)+32;
			*v+=((20218**r-16941**g-3277**b)>>17)+32;
			r+=3;
			g+=3;
			b+=3;
			*y++=(9796**r+19235**g+3736**b)>>15;
			*u+=((-4784**r-9437**g+14221**b)>>17)+32;
			*v+=((20218**r-16941**g-3277**b)>>17)+32;
			r+=3;
			g+=3;
			b+=3;
			u++;
			v++;
		}

		if ((loop & 1) == 1)
		{
			u-=width/2;
			v-=width/2;
		}
	}
}

/*******************************************************************************************
	Video4linux capture routines
*/


static char *v4l_start (struct video_dev *viddev, int width, int height, int input, int norm, unsigned long freq)
{
	int dev=viddev->fd;
	struct video_capability vid_caps;
	struct video_channel vid_chnl;
	struct video_tuner vid_tuner;
	struct video_window vid_win;
	struct video_mbuf vid_buf;
	struct video_mmap vid_mmap;
	char *map;

	if (ioctl (dev, VIDIOCGCAP, &vid_caps) == -1) {
		syslog(LOG_ERR, "ioctl (VIDIOCGCAP): %m");
		return (NULL);
	}
	if (vid_caps.type & VID_TYPE_MONOCHROME) viddev->v4l_fmt=VIDEO_PALETTE_GREY;
	if (input != IN_DEFAULT) {
		vid_chnl.channel = input;
		if (ioctl (dev, VIDIOCGCHAN, &vid_chnl) == -1) {
			syslog(LOG_ERR, "ioctl (VIDIOCGCHAN): %m");
		} else {
			vid_chnl.channel = input;
			vid_chnl.norm    = norm;
			if (ioctl (dev, VIDIOCSCHAN, &vid_chnl) == -1) {
				syslog(LOG_ERR, "ioctl (VIDIOCSCHAN): %m");
				return (NULL);
			}
		}
	}
	if (freq) {
		if (ioctl (dev, VIDIOCGTUNER, &vid_tuner)==-1) {
			syslog(LOG_ERR, "ioctl (VIDIOCGTUNER): %m");
		} else {
			if (vid_tuner.flags & VIDEO_TUNER_LOW) {
				freq=freq*16; /* steps of 1/16 KHz */
			} else {
				freq=(freq*10)/625;
			}
			if (ioctl(dev, VIDIOCSFREQ, &freq)==-1) {
				syslog(LOG_ERR, "ioctl (VIDIOCSFREQ): %m");
				return (NULL);
			}
			printf("Frequency set\n");
		}
	}
	if (ioctl (dev, VIDIOCGMBUF, &vid_buf) == -1) {
		syslog(LOG_WARNING, "no mmap falling back on read");
		if (ioctl (dev, VIDIOCGWIN, &vid_win)== -1) {
			syslog(LOG_ERR, "ioctl VIDIOCGWIN: %m");
			return (NULL);
		}
		vid_win.width=width;
		vid_win.height=height;
		vid_win.clipcount=0;
		if (ioctl (dev, VIDIOCSWIN, &vid_win)== -1) {
			syslog(LOG_ERR, "ioctl VIDIOCSWIN: %m");
			return (NULL);
		}
		syslog(LOG_WARNING, "V4L capturing using read is deprecated!!!!");
		syslog(LOG_WARNING, "Send an email to pe1rxq@amsat.org if your device does not support mmap");
		syslog(LOG_WARNING, "Please also give the kernel and driver versions");
		exit(1);
		viddev->v4l_read_img=1;
		map=mymalloc(width*height*3);
		viddev->v4l_maxbuffer=1;
		viddev->v4l_curbuffer=0;
		viddev->v4l_buffers[0]=map;
		//goto done;
	}else{
		map=mmap(0, vid_buf.size, PROT_READ|PROT_WRITE, MAP_SHARED, dev, 0);
		if (vid_buf.frames>1) {
			viddev->v4l_maxbuffer=2;
			viddev->v4l_buffers[0]=map;
			viddev->v4l_buffers[1]=map+vid_buf.offsets[1];
		} else {
			viddev->v4l_buffers[0]=map;
			viddev->v4l_maxbuffer=1;
		}

		if ((unsigned char *)-1 == (unsigned char *)map) {
			return (NULL);
		}
		viddev->v4l_curbuffer=0;
		vid_mmap.format=viddev->v4l_fmt;
		vid_mmap.frame=viddev->v4l_curbuffer;
		vid_mmap.width=width;
		vid_mmap.height=height;
		if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
			syslog(LOG_DEBUG, "Failed with YUV420P, trying YUV422 palette\n");
			viddev->v4l_fmt=VIDEO_PALETTE_YUV422;
			vid_mmap.format=viddev->v4l_fmt;
			/* Try again... */
			if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
				syslog(LOG_DEBUG, "Failed with YUV422, trying RGB24 palette\n");
				viddev->v4l_fmt=VIDEO_PALETTE_RGB24;
				vid_mmap.format=viddev->v4l_fmt;
				/* Try again... */
				if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
					syslog(LOG_ERR, "ioctl VIDIOCMCAPTURE: %m");
					return (NULL);
				}
			}
		}
	}

	switch (viddev->v4l_fmt) {
		case VIDEO_PALETTE_YUV420P:
			viddev->v4l_bufsize=(width*height*3)/2;
			break;
		case VIDEO_PALETTE_YUV422:
			viddev->v4l_bufsize=(width*height*2);
			break;
		case VIDEO_PALETTE_RGB24:
			viddev->v4l_bufsize=(width*height*3);
			break;
		case VIDEO_PALETTE_GREY:
			viddev->v4l_bufsize=width*height;
			break;
	}
	return map;
}

static char *v4l_next (struct video_dev *viddev, char *map, int width, int height)
{
	int dev=viddev->fd;
	int frame=viddev->v4l_curbuffer;
	struct video_mmap vid_mmap;
	char *cap_map=NULL;

	sigset_t  set, old;

	if (viddev->v4l_read_img) {
		if (read(dev, map, viddev->v4l_bufsize) != viddev->v4l_bufsize)
			return NULL;
	} else {
		vid_mmap.format=viddev->v4l_fmt;
		vid_mmap.width=width;
		vid_mmap.height=height;
		/* Block signals during IOCTL */
		sigemptyset (&set);
		sigaddset (&set, SIGCHLD);
		sigaddset (&set, SIGALRM);
		sigaddset (&set, SIGUSR1);
		sigaddset (&set, SIGTERM);
		sigaddset (&set, SIGHUP);
		pthread_sigmask (SIG_BLOCK, &set, &old);

		cap_map=viddev->v4l_buffers[viddev->v4l_curbuffer];
		viddev->v4l_curbuffer++;
		if (viddev->v4l_curbuffer >= viddev->v4l_maxbuffer)
			viddev->v4l_curbuffer=0;
		vid_mmap.frame=viddev->v4l_curbuffer;

		if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
			syslog(LOG_ERR, "mcapture error in proc %d: %m", getpid());
			sigprocmask (SIG_UNBLOCK, &old, NULL);
			return (NULL);
		}

		vid_mmap.frame=frame;
		if (ioctl(dev, VIDIOCSYNC, &vid_mmap.frame) == -1) {
			syslog(LOG_ERR, "sync error in proc %d: %m", getpid());
			sigprocmask (SIG_UNBLOCK, &old, NULL);
		}
		
		pthread_sigmask (SIG_UNBLOCK, &old, NULL);	/*undo the signal blocking*/
	}
	if (!viddev->v4l_read_img) {
		switch (viddev->v4l_fmt) {
			case VIDEO_PALETTE_RGB24:
				rgb24toyuv420p(map, cap_map, width, height);
				break;
			case VIDEO_PALETTE_YUV422:
				yuv422to420p(map, cap_map, width, height);
				break;
			default:
				memcpy(map, cap_map, viddev->v4l_bufsize);
		}
	}

	return map;
}

void v4l_set_input (struct video_dev *viddev, char *map, int width, int height, int input, int norm, int skip, unsigned long freq)
{
	int dev=viddev->fd;
	int i;
	struct video_channel vid_chnl;
	struct video_tuner vid_tuner;
	unsigned long frequnits = freq;
	
	if (input != viddev->input || width != viddev->width || height!=viddev->height || freq!=viddev->freq) {
		if (freq) {
			if (ioctl (dev, VIDIOCGTUNER, &vid_tuner)==-1) {
				syslog(LOG_ERR, "ioctl (VIDIOCGTUNER): %m");
			} else {
				if (vid_tuner.flags & VIDEO_TUNER_LOW) {
					frequnits=freq*16; /* steps of 1/16 KHz */
				} else {
					frequnits=(freq*10)/625;
				}
				if (ioctl(dev, VIDIOCSFREQ, &frequnits)==-1) {
					syslog(LOG_ERR, "ioctl (VIDIOCSFREQ): %m");
					return;
				}
			}
		}

		vid_chnl.channel = input;
		if (ioctl (dev, VIDIOCGCHAN, &vid_chnl) == -1) {
			syslog(LOG_ERR, "ioctl (VIDIOCGCHAN): %m");
		} else {
			//DEBUG CODE syslog(LOG_DEBUG, "input: %d", input);
			vid_chnl.channel = input;
			vid_chnl.norm = norm;
			if (ioctl (dev, VIDIOCSCHAN, &vid_chnl) == -1) {
				syslog(LOG_ERR, "ioctl (VIDIOCSCHAN): %m");
				return;
			}
		}
		viddev->input=input;
		viddev->width=width;
		viddev->height=height;
		viddev->freq=freq;
		/* skip a few frames if needed */
		for (i=0; i<skip; i++)
			v4l_next (viddev, map, width, height);
	}
}

static int v4l_open_vidpipe(void)
{
	int pipe;
	FILE *vloopbacks;
	char pipepath[255];
	char buffer[255];
	char *loop;
	char *input;
	char *istatus;
	char *output;
	char *ostatus;
	
	vloopbacks=fopen("/proc/video/vloopback/vloopbacks", "r");
	if (!vloopbacks) {
		syslog(LOG_ERR, "Failed to open '/proc/video/vloopback/vloopbacks': %m");
		return -1;
	}
	/* Read vloopback version*/
	fgets(buffer, 255, vloopbacks);
	printf("\t%s", buffer);
	/* Read explaination line */
	fgets(buffer, 255, vloopbacks);
	while (fgets(buffer, 255, vloopbacks)) {
		if (strlen(buffer)>1) {
			buffer[strlen(buffer)-1]=0;
			loop=strtok(buffer, "\t");
			input=strtok(NULL, "\t");
			istatus=strtok(NULL, "\t");
			output=strtok(NULL, "\t");
			ostatus=strtok(NULL, "\t");
			if (istatus[0]=='-') {
				sprintf(pipepath, "/dev/%s", input);
				pipe=open(pipepath, O_RDWR);
				if (pipe>=0) {
					printf("\tInput: /dev/%s\n", input);
					printf("\tOutput: /dev/%s\n", output);
					return pipe;
				}
			}
		}
	}
	return -1;
}

static int v4l_startpipe (char *devname, int width, int height, int type)
{
	int dev;
	struct video_picture vid_pic;
	struct video_window vid_win;

	if (!strcmp(devname, "-")) {
		dev=v4l_open_vidpipe();
	} else {
		dev=open(devname, O_RDWR);
	}
	if (dev < 0)
		return(-1);

	if (ioctl(dev, VIDIOCGPICT, &vid_pic)== -1) {
		syslog(LOG_ERR, "ioctl VIDIOCGPICT: %m");
		return(-1);
	}
	vid_pic.palette=type;
	if (ioctl(dev, VIDIOCSPICT, &vid_pic)== -1) {
		syslog(LOG_ERR, "ioctl VIDIOCSPICT: %m");
		return(-1);
	}
	if (ioctl(dev, VIDIOCGWIN, &vid_win)== -1) {
		syslog(LOG_ERR, "ioctl VIDIOCGWIN: %m");
		return(-1);
	}
	vid_win.height=height;
	vid_win.width=width;
	if (ioctl(dev, VIDIOCSWIN, &vid_win)== -1) {
		syslog(LOG_ERR, "ioctl VIDIOCSWIN: %m");
		return(-1);
	}
	return dev;
}

static int v4l_putpipe (int dev, char *image, int size)
{
	return write(dev, image, size);
}

static void v4l_autobright (int dev, unsigned char *image, int width, int height)
{
	struct video_picture vid_pic;
	int i, j=0, avg=0, offset=0;
	
	for (i=0; i<width*height; i+=100) {
		avg+=*image++;
		j++;
	}
	avg=avg/j;

	if (avg > 140 || avg < 64) {
		if (ioctl(dev, VIDIOCGPICT, &vid_pic)==-1) {
			syslog(LOG_ERR, "ioctl VIDIOCGPICT: %m");
		}
		if (avg > 140) {
			offset=avg-140;
			if (vid_pic.brightness > 100*offset)
				vid_pic.brightness-=100*offset;
		}
		if (avg < 64) {
			offset=64-avg;
			if (vid_pic.brightness < 65535-100*offset)
				vid_pic.brightness+=100*offset;
		}
		printf("auto_brightness: %d\n", vid_pic.brightness);
		if (ioctl(dev, VIDIOCSPICT, &vid_pic)==-1) {
			syslog(LOG_ERR, "ioctl VIDIOCSPICT: %m");
		}
	}
}

/*****************************************************************************
	Wrappers calling the actual capture routines
 *****************************************************************************/

/* big lock for vid_start */
pthread_mutex_t vid_mutex;
/* structure used for per device locking */
struct video_dev **viddevs=NULL;

void vid_init(void)
{
	if (!viddevs) {
		viddevs=mymalloc(sizeof(struct video_dev *));
		viddevs[0]=NULL;
	}

	pthread_mutex_init(&vid_mutex, NULL);
}

/* Called by childs to get rid of open video devices */
void vid_close(void)
{
	int i=-1;

	if (viddevs) {
		while(viddevs[++i]) {
			close(viddevs[i]->fd);
		}
	}
}

void vid_cleanup(void)
{
	int i=-1;
	if (viddevs) {
		while(viddevs[++i])
			free(viddevs[i]);
		free(viddevs);
		viddevs=NULL;
	}
}
#endif /*WITHOUT_V4L*/

int vid_start (struct context *cnt, int width, int height, int input, int norm, unsigned long frequency)
{
	struct config *conf=&cnt->conf;
	int dev=-1;

#ifdef HAVE_CURL
	if (conf->netcam_url)
		return netcam_start(cnt);
#endif

#ifndef WITHOUT_V4L
	{
		int i=-1;
		pthread_mutex_lock(&vid_mutex);

		/* If rotation is 90 or 270 degrees then swap width and height. This
	 	* means that the correct video dimensions will be used, because the
	 	* width and height were swapped when the configuration was loaded.
	 	*/
		if((cnt->conf.rotate_deg == 90) || (cnt->conf.rotate_deg == 270))
		myswap(width, height);

		cnt->imgs.width=cnt->conf.width;
		cnt->imgs.height=cnt->conf.height;

		while (viddevs[++i]) {
			if (!strcmp(conf->video_device, viddevs[i]->video_device)) {
				pthread_mutex_unlock(&vid_mutex);
				cnt->imgs.type=viddevs[i]->v4l_fmt;
				switch (cnt->imgs.type) {
					case VIDEO_PALETTE_GREY:
						cnt->imgs.motionsize=width*height;
						cnt->imgs.size=width*height;
						break;
					case VIDEO_PALETTE_RGB24:
					case VIDEO_PALETTE_YUV422:
						cnt->imgs.type=VIDEO_PALETTE_YUV420P;
					case VIDEO_PALETTE_YUV420P:
						cnt->imgs.motionsize=width*height;
						cnt->imgs.size=(width*height*3)/2;
						break;
				}
				return viddevs[i]->fd;
			}
		}

		viddevs=myrealloc(viddevs, sizeof(struct video_dev *)*(i+2), "vid_start");
		viddevs[i]=mymalloc(sizeof(struct video_dev));
		viddevs[i+1]=NULL;

		pthread_mutexattr_init(&viddevs[i]->attr);
		pthread_mutex_init(&viddevs[i]->mutex, NULL);

		dev=open(conf->video_device, O_RDWR);
		if (dev <0) {
			printf("Failed to open video device %s\n", conf->video_device);
			syslog(LOG_ERR, "failed to open video device %s: %m", conf->video_device);
			exit(1);
		}

		viddevs[i]->video_device=conf->video_device;
		viddevs[i]->fd=dev;
		viddevs[i]->input=input;
		viddevs[i]->height=height;
		viddevs[i]->width=width;
		viddevs[i]->freq=frequency;
		viddevs[i]->owner=-1;

		viddevs[i]->v4l_fmt=VIDEO_PALETTE_YUV420P;
		viddevs[i]->v4l_read_img=0;
		viddevs[i]->v4l_curbuffer=0;
		viddevs[i]->v4l_maxbuffer=1;

		if (!v4l_start (viddevs[i], width, height, input, norm, frequency)) {
			pthread_mutex_unlock(&vid_mutex);
			return -1;
		}
		cnt->imgs.type=viddevs[i]->v4l_fmt;
		switch (cnt->imgs.type) {
			case VIDEO_PALETTE_GREY:
				cnt->imgs.size=width*height;
				cnt->imgs.motionsize=width*height;
			break;
			case VIDEO_PALETTE_RGB24:
			case VIDEO_PALETTE_YUV422:
				cnt->imgs.type=VIDEO_PALETTE_YUV420P;
			case VIDEO_PALETTE_YUV420P:
				cnt->imgs.size=(width*height*3)/2;
				cnt->imgs.motionsize=width*height;
			break;
		}

		pthread_mutex_unlock(&vid_mutex);
	}
#endif /*WITHOUT_V4L*/	
	return dev;
}

char *vid_next (struct context *cnt, int dev, char *map, int width, int height)
{
	struct config *conf=&cnt->conf;
	char *ret=NULL;

#ifdef HAVE_CURL
	if (conf->netcam_url) {
		int retries=0;

		ret=NULL;
		while (!ret && retries++<100)
			ret=netcam_next(cnt, map);
		return ret;
	}
#endif
#ifndef WITHOUT_V4L
	{
		int i=-1;
		/* If rotation is 90 or 270 degrees, then swap width and height. */
		if((conf->rotate_deg == 90) || (conf->rotate_deg == 270))
			myswap(width, height);

		while (viddevs[++i])
			if (viddevs[i]->fd==dev)
				break;

		if (!viddevs[i])
			return NULL;

		if (viddevs[i]->owner!=cnt->threadnr) {
			pthread_mutex_lock(&viddevs[i]->mutex);
			viddevs[i]->owner=cnt->threadnr;
			viddevs[i]->frames=conf->roundrobin_frames;
			cnt->switched=1;
		}

		v4l_set_input (viddevs[i], map, width, height, conf->input, conf->norm, conf->roundrobin_skip, conf->frequency);
		ret=v4l_next (viddevs[i], map, width, height);

		if (--viddevs[i]->frames <= 0) {
			viddevs[i]->owner=-1;
			pthread_mutex_unlock(&viddevs[i]->mutex);
		}
	
		/* Rotate the image if specified. */
		if(conf->rotate_deg > 0) {
			rotate_map(map, viddevs[i]->v4l_bufsize, width, height, conf->rotate_deg);
		}
	}
#endif /*WITHOUT_V4L*/
	return ret;
}

#ifndef WITHOUT_V4L
int vid_startpipe (char *devname, int width, int height, int type)
{
	return v4l_startpipe (devname, width, height, type);
}

int vid_putpipe (int dev, char *image, int size)
{
	return v4l_putpipe (dev, image, size);
}

void vid_autobright (struct config *conf, int dev, char *image, int width, int height)
{
	if (!conf->netcam_url) 
		v4l_autobright(dev, image, width, height);
}
#endif /*WITHOUT_V4L*/
