#ifdef HAVE_XMLRPC
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include "pthread.h"

#include "motion.h"
#include "xmlrpc-httpd.h"

pthread_mutex_t xmlrpc_mutex;
static struct xmlrpc_method_handler **method_handlers=NULL;

void http_returnerror(int errn, int sc);


/***********************************************/

void xmlrpc_httpd_method_add(char *method_name, xmlrpc_async_handler handler, void *userdata)
{
	int i=0;

	pthread_mutex_lock(&xmlrpc_mutex);

	if (!method_handlers) {
		method_handlers=mymalloc(sizeof(struct xmlrpc_method_handler *));
		method_handlers[0]=NULL;
	}
	while (method_handlers[i]) i++;
	/* FIXME the memory reserved below is not freed when motion exits so we
	 * expect the OS to to it for us. This should eventually be fixed.
	 */
	method_handlers=myrealloc(method_handlers, sizeof(struct xmlrpc_method_handler *)*(i+2), "xmlrpc_httpd_method_add");
	method_handlers[i+1]=NULL;
	method_handlers[i]=mymalloc(sizeof(struct xmlrpc_method_handler));
	method_handlers[i]->method_name=method_name;
	method_handlers[i]->handler=handler;
	method_handlers[i]->userdata=userdata;

	pthread_mutex_unlock(&xmlrpc_mutex);
}

void xmlrpc_async_call(char *xml_data, size_t xml_len, int socket)
{
	xmlrpc_env fault;
	char *method_name;
	xmlrpc_mem_block *output;
	xmlrpc_value *param_array;
	struct xmlrpc_httpd_call *call;
	int i;


	xmlrpc_env_init(&fault);
	method_name=NULL;
	param_array=NULL;

	output=xmlrpc_mem_block_new(&fault, 0);
	if (fault.fault_occurred) {
		syslog(LOG_ERR, "xmlrpc_mem_block_new()");
		return;
	}
	
	xmlrpc_parse_call(&fault, xml_data, xml_len, &method_name, &param_array);

	call=mymalloc(sizeof(struct xmlrpc_httpd_call));
	call->method_name=method_name;
	call->output=output;
	call->param_array=param_array;
	call->socket=socket;

	/* call callback */
	if (method_name) {
		pthread_mutex_lock(&xmlrpc_mutex);
		i=-1;
		while (method_handlers[++i])
			if (!strcmp(method_handlers[i]->method_name, method_name)) {
				method_handlers[i]->handler(call, param_array, method_handlers[i]->userdata);
				pthread_mutex_unlock(&xmlrpc_mutex);
				return;
			}
		pthread_mutex_unlock(&xmlrpc_mutex);
	}
		
	http_returnerror(510, call->socket);
	if (call->method_name)
		free(call->method_name);
	if (call->param_array)
		xmlrpc_DECREF(call->param_array);
	free(call);
}

void xmlrpc_httpd_result(xmlrpc_env *env, struct xmlrpc_httpd_call *call, xmlrpc_value *result)
{
	char header[255];

	if (!result) {
		http_returnerror(510, call->socket);
		if (call->method_name)
			free(call->method_name);
		if (call->param_array)
			xmlrpc_DECREF(call->param_array);
		free(call);
		return;
	}
	xmlrpc_serialize_response(env, call->output, result);
	if (env->fault_occurred) {
		syslog(LOG_ERR, "xmlrpc_serialize_response()");
		return;
	}
	if (call->method_name)
		free(call->method_name);
	if (call->param_array)
		xmlrpc_DECREF(call->param_array);
	if (result)
		xmlrpc_DECREF(result);

	/* write result to socket and close it. */
	sprintf(header, "HTTP/1.1 200 OK\n"
			"Server: Motion-xmlrpc-httpd/"VERSION"\n"
			"Connection: close\n"
			"Content-type: text/xml\n"
			"Content-length: %d\n\n",
			xmlrpc_mem_block_size(call->output));
	send(call->socket, header, strlen(header), 0);
	send(call->socket, xmlrpc_mem_block_contents(call->output), xmlrpc_mem_block_size(call->output), 0);
	close(call->socket);

	xmlrpc_mem_block_free(call->output);
	free(call);
}

/************************************************/

void http_returnerror(int errn, int sc)
{
	char error[255];
	
	sprintf(error, "HTTP/1.1 %d\n\n", errn);
	send (sc, error, strlen(error), 0);
	close(sc);
}


struct http_sc {
	int state;
	int sc;
	int headerlen;
	int headerend;
	int contentlen;
	int bodylen;
	char *header;
	char *body;
};

struct http_sc *http_connecthandler(struct http_sc *sock)
{
	int i=0, j, contentok=0, http_err;

	if (sock->state==0) {
		while (!sock->headerend && i>=0) {
			sock->header=myrealloc(sock->header, sock->headerlen+255, "http_connecthandler");
			i=recv(sock->sc, sock->header+sock->headerlen, 255, 0);
			if (i>=0) {
				if (i>4 || sock->headerlen > 4)
					for (j=sock->headerlen>3?sock->headerlen-4:sock->headerlen; j<sock->headerlen+(i-3); j++) {
						if (!memcmp(sock->header+j+2, "\n\n", 2) ||
							!memcmp(sock->header+j, "\r\n\r\n", 4) ) {
								sock->headerend=j+4;
						}
					}
				sock->headerlen+=i;
			} else {
				if (errno==EAGAIN)
					return sock;
				if (sock->header)
					free(sock->header);
				close(sock->sc);
				free(sock);
				return NULL;
			}
		}
		for (i=0; i<sock->headerend; i++) sock->header[i]=toupper(sock->header[i]);
		http_err=0;
		if (sock->headerend < 4) {
			syslog(LOG_ERR, "Invalid request");
			http_err=400;
		}
		if (!http_err && memcmp(sock->header, "POST", 4)) {
			syslog(LOG_ERR, "Invalid method");
			http_err=501;
		}
		if (!http_err) {
			sock->header[sock->headerend-1]=0;
			for (i=0; i<sock->headerend; i++) {
				if (!strncmp("CONTENT-LENGTH", sock->header+i, 14)) {
					j=i;
					while (j<sock->headerend && sock->header[j++]!=':');
					sscanf(sock->header+j, "%d", &sock->contentlen);
				}
				if (!strncmp("CONTENT-TYPE", sock->header+i, 12)) {
					j=i;
					while (j<sock->headerend && sock->header[j]!='\n' && sock->header[j]!='\r') j++;
					if (!strncmp(sock->header+j-8, "TEXT/XML", 8)) {
						contentok=1;
					}
				}
			}

			if (!sock->contentlen || !contentok) {
				syslog(LOG_ERR, "Invalid header");
				http_err=400;
			}
		}
		if (http_err) {
			http_returnerror(http_err, sock->sc);
			close(sock->sc);
			free(sock);
			return NULL;
		}
		sock->body=mymalloc(sock->contentlen);
		if (sock->headerlen-sock->headerend)
			memcpy(sock->body, sock->header+sock->headerend, sock->headerlen-sock->headerend);
		sock->bodylen=sock->headerlen-sock->headerend;
		sock->state++;
	}
	if (sock->state==1) {
		while (sock->bodylen<sock->contentlen) {
			if ((i=recv(sock->sc, sock->body+sock->bodylen, sock->contentlen-sock->bodylen, 0))<0) {
				if (errno==EAGAIN)
					return sock;
				free(sock->header);
				free(sock->body);
				close(sock->sc);
				free(sock);
				return NULL;
			}
			sock->bodylen+=i;
		}
		free(sock->header);
		xmlrpc_async_call(sock->body, sock->bodylen, sock->sc);
		free(sock->body);
		free(sock);
		return NULL;
	}
	return sock;
}

struct http_sc *http_newconnect(int sc)
{
	struct http_sc *new_sc=mymalloc(sizeof(struct http_sc));

	new_sc->state=0;
	new_sc->sc=sc;
	new_sc->headerlen=0;
	new_sc->headerend=0;
	new_sc->contentlen=0;
	new_sc->bodylen=0;
	new_sc->header=NULL;
	new_sc->body=NULL;

	return new_sc;
}

void xmlrpc_httpd_run(struct context *cnt)
{
	int sl;
	int sc;
	int maxfd;
	struct sockaddr_in sin;
	struct http_sc **opensockets=NULL;
	struct http_sc *newsock=NULL;
	int socketscount=0;
	int i;
	socklen_t addrlen;
	fd_set fdread;
	
	pthread_mutex_init(&xmlrpc_mutex, NULL);

	if ((sl=http_bindsock(cnt->conf.control_port, cnt->conf.control_localhost))<0)
		return;
	i=1;
	ioctl(sl, FIONBIO, &i);

	syslog(LOG_DEBUG, "xmlrpc-httpd running, accepting connections");
	while (1) {
		FD_ZERO(&fdread);
		maxfd=sl;
		FD_SET(sl, &fdread);
		for (i=0; i<socketscount; i++) {
			FD_SET(opensockets[i]->sc, &fdread);
			if (opensockets[i]->sc>maxfd)
				maxfd=opensockets[i]->sc;
		}
		select(maxfd+1, &fdread, NULL, NULL, NULL);
		if (FD_ISSET(sl, &fdread)) {
			addrlen=sizeof(struct sockaddr_in);
			sc=accept(sl, (struct sockaddr *)&sin, &addrlen);
			if (sc<0) {
				syslog(LOG_ERR, "Accept(): %m");
				continue;
			}
			i=1;
			ioctl(sc, FIONBIO, &i);
			newsock=http_newconnect(sc);
			if (newsock) {
				socketscount++;
				opensockets=myrealloc(opensockets, sizeof(struct http_sc *)*socketscount, "xmlrpc_httpd_run(1)");
				opensockets[socketscount-1]=newsock;
			}
		}
		for (i=0; i<socketscount; i++) {
			if (FD_ISSET(opensockets[i]->sc, &fdread)) {
				opensockets[i]=http_connecthandler(opensockets[i]);
				if (!opensockets[i]) {
					memmove(opensockets+i, opensockets+i+1, (socketscount-i-1)*sizeof(struct http_sc *));
					socketscount--;
					if (socketscount == 0) {
						free(opensockets);
						opensockets = NULL;
					} else {
						opensockets=myrealloc(opensockets, sizeof(struct http_sc *)*socketscount, "xmlrpc_httpd_run(2)");
					}
				}
			}
		}
	}
}
#endif /* HAVE_XMLRPC */
