//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Plugin "Shares mounting"
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QtGui/QDialog>
#include <QtGui/QAction>
#include <QtGui/QPushButton>
#include <QtGui/QGridLayout>
#include <QtGui/QTreeWidget>
#include <QtGui/QLineEdit>
#include <QtGui/QGroupBox>
#include <QtGui/QCheckBox>
#include <QtGui/QLabel>
#include <QtGui/QRadioButton>
#include <QtCore/QSettings>
#include <QtCore/QFile>
#include <QtCore/QTextStream>
#include <QDebug>
#include <QtPlugin>
#include "sharesmount.h"
#include "../../sources/gui/const.h"
#include "../../sources/gui/choosemountpointwidget.h"

SharesMount::SharesMount(QWidget *parent) {
	currentEditedItem = 0;
	mainWidget = new QDialog(parent);
	mainWidget->setWindowTitle(tr("Shares mounting"));
	mainWidget->setWindowIcon(QIcon(":/icons/share_mount.png"));

	treeWidget = new QTreeWidget;
	treeWidget->setHeaderLabels(QStringList() << tr("Type") << tr("Server") << tr("Remote directory") << tr("Mount point") << tr("Options"));
	connect(treeWidget,SIGNAL(currentItemChanged(QTreeWidgetItem*,QTreeWidgetItem*)),this,SLOT(itemChanged(QTreeWidgetItem *,QTreeWidgetItem *)));

	addButton = new QPushButton(tr("Add"));
	connect(addButton,SIGNAL(clicked()),this,SLOT(add()));
	
	editButton = new QPushButton(tr("Edit"));
	editButton->setEnabled(false);
	connect(editButton,SIGNAL(clicked()),this,SLOT(edit()));
	
	removeButton = new QPushButton(tr("Remove"));
	removeButton->setEnabled(false);
	connect(removeButton,SIGNAL(clicked()),this,SLOT(remove()));
	
	closeButton = new QPushButton(tr("Close"));
	connect(closeButton,SIGNAL(clicked()),mainWidget,SLOT(hide()));

	QAction *mainAction = new QAction(mainWidget);
	mainAction->setText(tr("Shares mounting"));
	mainAction->setIcon(QIcon(":/icons/share_mount.png"));
	actions.append(mainAction);
	connect(mainAction,SIGNAL(triggered()),mainWidget,SLOT(show()));

	nfsType = new QRadioButton(tr("NFS share"));
	nfsType->setChecked(true);
	smbType = new QRadioButton(tr("Samba share"));
	connect(nfsType,SIGNAL(clicked()),this,SLOT(typeChanged()));
	connect(smbType,SIGNAL(clicked()),this,SLOT(typeChanged()));
	
	typeGroupBox = new QGroupBox(tr("Share type"));
	QHBoxLayout *typeGroupBoxLayout = new QHBoxLayout;
	typeGroupBoxLayout->addWidget(nfsType);
	typeGroupBoxLayout->addWidget(smbType);
	typeGroupBox->setLayout(typeGroupBoxLayout);
	typeGroupBox->hide();
	
	smbNameLine = new QLineEdit;
	smbPassLine = new QLineEdit;

	QHBoxLayout *smbOptionsLayout = new QHBoxLayout;
	smbOptionsLayout->addWidget(new QLabel(tr("Name")));
	smbOptionsLayout->addWidget(smbNameLine);
	smbOptionsLayout->addStretch();
	smbOptionsLayout->addWidget(new QLabel(tr("Password")));
	smbOptionsLayout->addWidget(smbPassLine);
	sbmOptionsWidget = new QWidget;
	sbmOptionsWidget->setLayout(smbOptionsLayout);
	
	groupBox = new QGroupBox;
	
	serverLine = new QLineEdit;
	remoteDirLine = new QLineEdit;
	mountPointLine = new ChooseMountPointWidget;
	optionsLine = new QLineEdit;

	optionsLabel = new QLabel(tr("Options"));
	
	addShareButton = new QPushButton(tr("Add share"));
	addShareButton->hide();
	connect(addShareButton,SIGNAL(clicked()),this,SLOT(addShare()));

	editShareButton = new QPushButton(tr("Edit share"));
	editShareButton->hide();
	connect(editShareButton,SIGNAL(clicked()),this,SLOT(editShare()));

	cancelButton = new QPushButton(tr("Cancel"));
	connect(cancelButton,SIGNAL(clicked()),this,SLOT(cancel()));
	
	QHBoxLayout *groupBoxButtonsLayout = new QHBoxLayout;
	groupBoxButtonsLayout->addStretch();
	groupBoxButtonsLayout->addWidget(addShareButton);
	groupBoxButtonsLayout->addWidget(editShareButton);
	groupBoxButtonsLayout->addWidget(cancelButton);

	QGridLayout *groupBoxLayout = new QGridLayout;
	groupBoxLayout->addWidget(new QLabel(tr("Server")),0,0);
	groupBoxLayout->addWidget(serverLine,0,1);
	groupBoxLayout->addWidget(new QLabel(tr("Remote directory")),1,0);
	groupBoxLayout->addWidget(remoteDirLine,1,1);
	groupBoxLayout->addWidget(new QLabel(tr("Mount point")),2,0);
	groupBoxLayout->addWidget(mountPointLine,2,1);
	groupBoxLayout->addWidget(optionsLabel,3,0);
	groupBoxLayout->addWidget(optionsLine,3,1);
	groupBoxLayout->addWidget(sbmOptionsWidget,4,0,1,2);
	groupBoxLayout->addLayout(groupBoxButtonsLayout,5,0,1,2);
	groupBox->setLayout(groupBoxLayout);

	groupBox->hide();

	QHBoxLayout *buttonsLayout = new QHBoxLayout;
	buttonsLayout->addWidget(addButton);
	buttonsLayout->addWidget(editButton);
	buttonsLayout->addWidget(removeButton);
	buttonsLayout->addStretch();
	
	QHBoxLayout *bottomLayout = new QHBoxLayout;
	bottomLayout->addStretch();
	bottomLayout->addWidget(closeButton);
	
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(treeWidget);
	mainLayout->addLayout(buttonsLayout);
	mainLayout->addWidget(typeGroupBox);
	mainLayout->addWidget(groupBox);
	mainLayout->addLayout(bottomLayout);
	mainWidget->setLayout(mainLayout);

	loadSettings();
	searchSharesLinesInFstab();
}

SharesMount::~SharesMount() {
	foreach (QAction *action,actions)
		delete action;
	delete cancelButton;
	delete editShareButton;
	delete addShareButton;
	
	delete nfsType;
	delete smbType;
	delete typeGroupBox;
	
	delete smbNameLine;
	delete smbPassLine;
	delete sbmOptionsWidget;
	
	delete serverLine;
	delete remoteDirLine;
	delete mountPointLine;
	delete optionsLabel;
	delete optionsLine;
	delete groupBox;
	
	delete treeWidget;
	delete addButton;
	delete editButton;
	delete removeButton;
	delete closeButton;
	delete mainWidget;
}

void SharesMount::setParent(QObject *parent) {
	connect(parent,SIGNAL(finished()),this,SLOT(saveSettings()));
}

const QString SharesMount::pluginName() const {
	return tr("Shares mounting");
}

const QString SharesMount::pluginDescription() const {
	return tr("With help of this plugin you can easy mount NFS and Samba shares or set them on auto-mount.");
}
	
QDialog* SharesMount::dialog() const {
	return mainWidget;
}

QDockWidget* SharesMount::dockWidget() const {
	return 0;
}

QMenu* SharesMount::menu() const {
	return 0;
}

QToolBar* SharesMount::toolBar() const {
	return 0;
}

Actions SharesMount::menuActions() const {
	return actions;
}

void SharesMount::saveSettings() {
	QFile fstabFile("/etc/fstab");
	if (fstabFile.open(QIODevice::WriteOnly)) {
		QTextStream fstabStream(&fstabFile);
		fstabStream << fstabTextExceptShares;
		for (int i = 0; i < treeWidget->topLevelItemCount(); i++)
			if (treeWidget->topLevelItem(i) != 0) {
				QTreeWidgetItem *item = treeWidget->topLevelItem(i);
				if (item->text(0) == "Nfs")
					fstabStream << item->text(1) << ':' << item->text(2) << '\t' << item->text(3) << "\tnfs\t" << item->text(4) << '\n';
				else {
					fstabStream << "//" << item->text(1);
					if (!item->text(2).startsWith('/'))
						fstabStream << '/';
					fstabStream << item->text(2) << '\t' << item->text(3) << "\tsmbfs\t" << item->text(4) << "\t0\t0\n";
				}
			}
	} else
		qDebug() << tr("Cannot add Share records to fstab file");
	
	QSettings settings(ORGANIZATION,PROGRAM_NAME);
	settings.setValue("SharesMount/Size",mainWidget->size());
}

void SharesMount::loadSettings() {
	QSettings settings(ORGANIZATION,PROGRAM_NAME);
	mainWidget->resize(settings.value("SharesMount/Size",QSize(600,400)).toSize());
}

void SharesMount::add() {
	groupBox->setTitle(tr("Share addition"));
	addShareButton->show();
	editShareButton->hide();
	typeGroupBox->show();
	groupBox->show();
	typeChanged();

	clearLines();
	
	serverLine->setFocus();
}

void SharesMount::addShare() {
	if (checkLinesContent()) {
		groupBox->hide();
		typeGroupBox->hide();
		QTreeWidgetItem *item = new QTreeWidgetItem(treeWidget);
		setContentToItem(item);
	}
}

void SharesMount::clearLines() {
	serverLine->clear();
	remoteDirLine->clear();
	smbNameLine->clear();
	smbPassLine->clear();
	nfsType->setChecked(true);
	mountPointLine->setMountPoint("");
	optionsLine->setText("defaults");
}

void SharesMount::searchSharesLinesInFstab() {
	QFile fstabFile("/etc/fstab");
	fstabFile.open(QIODevice::ReadOnly);
	QTextStream fstabStream(&fstabFile);
	while (!fstabStream.atEnd()) {
		QString line = fstabStream.readLine();
		if (line.contains("#"))
			continue;
		line.replace("\t"," ");
		QStringList list = line.split(" ");
		list.removeAll("");
		list.removeAll(" ");
		if (list.count() == 4 && list[2] == "nfs") {
			QStringList serverWithRemoteDir = list[0].split(":");
			if (serverWithRemoteDir.count() != 2)
				continue;
			QTreeWidgetItem *item = new QTreeWidgetItem(treeWidget);
			item->setText(0,"Nfs");
			item->setText(1,serverWithRemoteDir[0]);
			item->setText(2,serverWithRemoteDir[1]);
			item->setText(3,list[1]);
			item->setText(4,list[3]);
		} else if (list.count() == 6 && list[2] == "smbfs") {
			if (list[0].startsWith("//")) {
				int endOfServer = list[0].indexOf('/',2);
				QString server = list[0].left(endOfServer).remove(0,2);
				QString remoteDir = list[0].remove("//" + server);
				QTreeWidgetItem *item = new QTreeWidgetItem(treeWidget);
				item->setText(0,"Samba");
				item->setText(1,server);
				item->setText(2,remoteDir);
				item->setText(3,list[1]);
				item->setText(4,list[3]);
			}
		} else
			fstabTextExceptShares += line + '\n';
	}
}

void SharesMount::edit() {
	QTreeWidgetItem *item = treeWidget->currentItem();
	if (item == 0)
		return;
	groupBox->setTitle(tr("Share edition"));
	editShareButton->show();
	removeButton->setEnabled(false);
	addShareButton->hide();
	groupBox->show();
	typeGroupBox->hide();
	clearLines();
	nfsType->setChecked(item->text(0) == "Nfs");
	smbType->setChecked(item->text(0) != "Nfs");
	typeChanged();
	serverLine->setText(item->text(1));
	remoteDirLine->setText(item->text(2));
	mountPointLine->setMountPoint(item->text(3));
	if (nfsType->isChecked())
		optionsLine->setText(item->text(4));
	else {
		QStringList nameWithPass = item->text(4).split(",");
		for (int i = 0; i < 2; i++) {
			if (nameWithPass[i].contains("username="))
				smbNameLine->setText(nameWithPass[i].remove("username="));
			else if (nameWithPass[i].contains("password="))
				smbPassLine->setText(nameWithPass[i].remove("password="));
		}
	}
	currentEditedItem = item;
}

void SharesMount::editShare() {
	if (currentEditedItem == 0)
		return;
	removeButton->setEnabled(true);
	if (checkLinesContent()) {
		setContentToItem(currentEditedItem);
		currentEditedItem = 0;
		groupBox->hide();
	}
}

bool SharesMount::checkLinesContent() {
	if (serverLine->text().isEmpty())
		serverLine->setFocus();
	else if (remoteDirLine->text().isEmpty())
		remoteDirLine->setFocus();
	else if (mountPointLine->mountPoint().isEmpty())
		mountPointLine->setFocus();
	else if (nfsType->isChecked() && optionsLine->text().isEmpty())
		optionsLine->setFocus();
	else if (smbType->isChecked() && smbNameLine->text().isEmpty())
		smbNameLine->setFocus();
	else if (smbType->isChecked() && smbPassLine->text().isEmpty())
		smbPassLine->setFocus();
	else
		return true;
	return false;
}

void SharesMount::setContentToItem(QTreeWidgetItem *item) {
	item->setText(0,nfsType->isChecked() ? "Nfs" : "Samba");
	item->setText(1,serverLine->text());
	item->setText(2,remoteDirLine->text());
	item->setText(3,mountPointLine->mountPoint());
	if (nfsType->isChecked())
		item->setText(4,optionsLine->text());
	else
		item->setText(4,"username=" + smbNameLine->text() + ",password=" + smbPassLine->text());
}

void SharesMount::itemChanged(QTreeWidgetItem *currentItem,QTreeWidgetItem *) {
	if (currentItem == 0)
		currentItem = treeWidget->currentItem();
	editButton->setEnabled(currentItem != 0);
	removeButton->setEnabled(currentItem != 0);
}

void SharesMount::remove() {
	QTreeWidgetItem *item = treeWidget->currentItem();
	if (item == 0)
		return;
	delete item;
	itemChanged(0,0);
}

void SharesMount::typeChanged() {
	optionsLine->setVisible(nfsType->isChecked());
	optionsLabel->setVisible(nfsType->isChecked());
	sbmOptionsWidget->setVisible(!nfsType->isChecked());
}

void SharesMount::cancel() {
	groupBox->hide();
	typeGroupBox->hide();
	itemChanged(0,0);
}

Q_EXPORT_PLUGIN2(sharesmountplugin,SharesMount)
