//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Gui
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QtGui/QLabel>
#include <QtGui/QHeaderView>
#include <QtGui/QHBoxLayout>
#include <QtGui/QPushButton>
#include <QtGui/QCheckBox>
#include <QtGui/QSpinBox>
#include "../core/diskoptionsparser.h"
#include "../core/diskdevice.h"
#include "optionstable.h"
#include "infobutton.h"
#include "specialcomboboxwidgets.h"
#include "permissionswidget.h"

OptionsTable::OptionsTable(DiskDevice *dev,PopupWindow *popupWindow) {
	popup = popupWindow;
	device = dev;
	
	setColumnCount(1);
	horizontalHeader()->setStretchLastSection(true);
	horizontalHeader()->hide();
	verticalHeader()->hide();
	setSelectionBehavior(QAbstractItemView::SelectRows);
	setSelectionMode(QAbstractItemView::SingleSelection);

	setAlternatingRowColors(true);
}

OptionsTable::~OptionsTable() {
	foreach (WidgetRow *widgetRow,widgetRows)
		delete widgetRow;
}

void OptionsTable::addOptions(const QList<Option *>& options,const QString& source) {
	foreach (Option *option,options) {
		if (option->optionType == "CheckBox")
			addCheckBoxOption(option,source);
		else if (option->optionType == "ComboBox" || option->optionType == "Encoding"
				|| option->optionType == "Codepage" || option->optionType == "GID"
				|| option->optionType == "UID" || option->optionType == "Locale")
			addComboBoxOption(option,source);
		else if (option->optionType == "Permissions")
			addPermOption(option,source);
		else if (option->optionType == "SpinBox")
			addSpinBoxOption(option,source);
	}
	resizeRowsToContents();
	foreach (WidgetRow *widgetRow,widgetRows)
		if (widgetRow->type == Perm)
			setRowHeight(widgetRow->row,widgetRow->permWidget->sizeHint().height()+10);
}

void OptionsTable::addCheckBoxOption(Option *option,const QString& source) {
	QCheckBox *checkBox = new QCheckBox(option->optionTitle);
	connect(checkBox,SIGNAL(clicked()),this,SIGNAL(optionChanged()));

	if (option->optionValues.count() == 2)
		foreach (Value *value,option->optionValues)
			if (value->isDefault && value->isChecked) {
				checkBox->setChecked(true);
				break;
			}
	foreach (QString fstabOption, device->fstabOptions().split(","))
		foreach (Value *value, option->optionValues)
			if (value->text == fstabOption) {
				if (option->optionValues.count() == 2) {
					if (!value->isDefault)
						checkBox->setChecked(!checkBox->isChecked());
				} else
					checkBox->setChecked(true);
			}
	
	QWidget *cellWidget = new QWidget;
	QHBoxLayout *cellWidgetLayout = new QHBoxLayout;
	cellWidgetLayout->setMargin(0);
	cellWidget->setLayout(cellWidgetLayout);

	InfoButton *infoButton = 0;
	if (!option->optionDocMark.isEmpty())
		infoButton = new InfoButton(popup);

	if (infoButton) cellWidgetLayout->addWidget(infoButton);
	cellWidgetLayout->addWidget(checkBox);
	cellWidgetLayout->addStretch();
	cellWidgetLayout->addWidget(new QLabel(source));
	int row = rowCount();
	setRowCount(row+1);
	setCellWidget(row,0,cellWidget);

	WidgetRow *widgetRow = new WidgetRow;
	widgetRow->cellWidget = cellWidget;
	widgetRow->type = CheckBox;
	widgetRow->checkBox = checkBox;
	widgetRow->source = source;
	widgetRow->row = row;
	widgetRow->option = option;
	
	widgetRows.append(widgetRow);
}

void OptionsTable::addComboBoxOption(Option *option,const QString& source) {
	QComboBox *comboBox = 0;
	if (option->optionType == "Encoding")
		comboBox = new EncodingWidget(option->optionValues[0]->text);
	else if (option->optionType == "Codepage")
		comboBox = new CodepageWidget;
	else if (option->optionType == "GID")
		comboBox = new GIDWidget;
	else if (option->optionType == "UID")
		comboBox = new UIDWidget;
	else if (option->optionType == "Locale")
		comboBox = new LocaleWidget;
	else {
		comboBox = new QComboBox;
		int currentIndex = 0;
		foreach (Value *value, option->optionValues) {
			comboBox->addItem(option->optionComboBoxItems[currentIndex],value->text);
			currentIndex++;
		}
	}
	connect(comboBox,SIGNAL(currentIndexChanged(int)),this,SIGNAL(optionChanged()));

	if (!device->fstabOptions().isEmpty())
		for (int i = 0; i < comboBox->count(); i++) {
			foreach (QString fstabOption,device->fstabOptions().split(","))
				if (fstabOption == comboBox->itemData(i).toString()) {
					comboBox->setCurrentIndex(i);
					goto exitFromCicle;
				}
			exitFromCicle:
				break;
		}
				
	InfoButton *infoButton = 0;
	if (!option->optionDocMark.isEmpty())
		infoButton = new InfoButton(popup);
	
	QWidget *cellWidget = new QWidget;
	QHBoxLayout *cellWidgetLayout = new QHBoxLayout;
	cellWidgetLayout->setMargin(0);
	cellWidget->setLayout(cellWidgetLayout);

	QLabel *label = new QLabel(option->optionTitle);
	
	if (infoButton)
		cellWidgetLayout->addWidget(infoButton);
	cellWidgetLayout->addWidget(label);
	cellWidgetLayout->addWidget(comboBox);
	cellWidgetLayout->addStretch();
	cellWidgetLayout->addWidget(new QLabel(source));
	int row = rowCount();
	setRowCount(row+1);
	setCellWidget(row,0,cellWidget);

	WidgetRow *widgetRow = new WidgetRow;
	widgetRow->cellWidget = cellWidget;
	widgetRow->type = ComboBox;
	widgetRow->comboBox = comboBox;
	widgetRow->source = source;
	widgetRow->row = row;
	widgetRow->label = label;
	widgetRow->option = option;
	
	widgetRows.append(widgetRow);

}

void OptionsTable::addSpinBoxOption(Option *option,const QString &source) {
	QStringList list = option->optionTitle.split("-=Type=-");
	
	QLabel *leftLabel = new QLabel(list[0]);
	QLabel *rightLabel = new QLabel(list[1]);
	QSpinBox *spinBox = new QSpinBox;
	spinBox->setMaximum(10000);
	connect(spinBox,SIGNAL(valueChanged(int)),this,SIGNAL(optionChanged()));

	if (!option->optionDefaultValue.isEmpty())
		spinBox->setValue(QVariant(option->optionDefaultValue).toInt());
	foreach (QString fstabOption, device->fstabOptions().split(",")) {
		QStringList optionList = fstabOption.split("=");
		if (optionList.count() != 2)
			continue;
		if (option->optionValues[0]->text == optionList[0] + "=")
			spinBox->setValue(QVariant(optionList[1]).toInt());
	}
	
	InfoButton *infoButton = 0;
	if (!option->optionDocMark.isEmpty())
		infoButton = new InfoButton(popup);

	QWidget *cellWidget = new QWidget;
	QHBoxLayout *cellWidgetLayout = new QHBoxLayout;
	cellWidgetLayout->setMargin(0);
	cellWidget->setLayout(cellWidgetLayout);
	
	if (infoButton)
		cellWidgetLayout->addWidget(infoButton);
	cellWidgetLayout->addWidget(leftLabel);
	cellWidgetLayout->addWidget(spinBox);
	cellWidgetLayout->addWidget(rightLabel);
	cellWidgetLayout->addStretch();
	cellWidgetLayout->addWidget(new QLabel(source));
	int row = rowCount();
	setRowCount(row+1);
	setCellWidget(row,0,cellWidget);

	WidgetRow *widgetRow = new WidgetRow;
	widgetRow->cellWidget = cellWidget;
	widgetRow->type = SpinBox;
	widgetRow->spinBox = spinBox;
	widgetRow->source = source;
	widgetRow->row = row;
	widgetRow->label = leftLabel;
	widgetRow->extraLabel = rightLabel;
	widgetRow->option = option;
	
	widgetRows.append(widgetRow);
}

void OptionsTable::addPermOption(Option *option,const QString &source) {
	PermissionsWidget *permWidget = new PermissionsWidget(option);
	connect(permWidget,SIGNAL(permissionsChanged()),this,SIGNAL(optionChanged()));

	foreach (QString fstabOption,device->fstabOptions().split(","))
		if (fstabOption.contains(option->optionValues[0]->text))
			permWidget->setCode(fstabOption.replace(option->optionValues[0]->text,""));
	
	QWidget *cellWidget = new QWidget;
	QHBoxLayout *cellWidgetLayout = new QHBoxLayout;
	cellWidgetLayout->setMargin(0);
	cellWidget->setLayout(cellWidgetLayout);
	
	cellWidgetLayout->addWidget(permWidget);
	cellWidgetLayout->addStretch();
	cellWidgetLayout->addWidget(new QLabel(source));
	int row = rowCount();
	setRowCount(row+1);
	setCellWidget(row,0,cellWidget);

	WidgetRow *widgetRow = new WidgetRow;
	widgetRow->cellWidget = cellWidget;
	widgetRow->type = Perm;
	widgetRow->permWidget = permWidget;
	widgetRow->source = source;
	widgetRow->row = row;
	widgetRow->option = option;
	
	widgetRows.append(widgetRow);
}

void OptionsTable::setFilter(const QString &filterText) {
	foreach (WidgetRow *widgetRow, widgetRows) {
		if (!fixedFilter.contains(widgetRow->source) && !fixedFilter.isEmpty())
			continue;
		if (currentTypeFilter != widgetRow->source && !currentTypeFilter.isEmpty())
			continue;
		switch (widgetRow->type) {
			case CheckBox:
				widgetRow->checkBox->text().contains(filterText,Qt::CaseInsensitive) ? showRow(widgetRow->row) : hideRow(widgetRow->row);
				break;
			case ComboBox:
				widgetRow->label->text().contains(filterText,Qt::CaseInsensitive) ? showRow(widgetRow->row) : hideRow(widgetRow->row);
				break;
			case SpinBox:
				widgetRow->label->text().contains(filterText,Qt::CaseInsensitive)
					|| widgetRow->extraLabel->text().contains(filterText,Qt::CaseInsensitive) ?
					showRow(widgetRow->row) : hideRow(widgetRow->row);
				break;
			case Perm:
				filterText.isEmpty() ? showRow(widgetRow->row) : hideRow(widgetRow->row);
				break;
		}
	}
	currentFilter = filterText;
}

void OptionsTable::setFixedFilter(const QString& filter) {
	setFixedFilter(QStringList() << filter);
}

void OptionsTable::setFixedFilter(QStringList filters) {
	fixedFilter.clear();
	fixedFilter = filters;
	foreach (WidgetRow *widgetRow, widgetRows) {
		if (!fixedFilter.contains(widgetRow->source) && !fixedFilter.isEmpty())
			hideRow(widgetRow->row);
		else
			showRow(widgetRow->row);
	}
}

void OptionsTable::showOptionsFromSource(QString source) {
	foreach (WidgetRow *widgetRow,widgetRows) {
		if (!fixedFilter.contains(widgetRow->source) && !fixedFilter.isEmpty())
			continue;
		if (source == "all") {
			showRow(widgetRow->row);
			continue;
		}
		source == widgetRow->source ? showRow(widgetRow->row) : hideRow(widgetRow->row);
	}
	if (source == "all")
		currentTypeFilter.clear();
	else
		currentTypeFilter = source;
	if (!currentFilter.isEmpty())
		setFilter(currentFilter);
}

QString OptionsTable::options(bool includeDefaultOptions) {
	QString options;

	foreach (WidgetRow *widgetRow,widgetRows) {
		if (!fixedFilter.contains(widgetRow->source) && !fixedFilter.isEmpty())
			continue;
		switch (widgetRow->type) {
			case CheckBox:
				if (widgetRow->option->optionValues.count() == 1) {
					if (widgetRow->checkBox->isChecked())
						options += "," + widgetRow->option->optionValues[0]->text;
				} else {
					if (widgetRow->checkBox->isChecked()) {
						QString defaultValue;
						QString checkedValue;
						foreach (Value *value,widgetRow->option->optionValues) {
							if (value->isDefault)
								defaultValue = value->text;
							if (value->isChecked)
								checkedValue = value->text;
						}
						if ((defaultValue == checkedValue && includeDefaultOptions) || defaultValue != checkedValue)
							options += "," + checkedValue;
					}
					else {
						foreach (Value *value,widgetRow->option->optionValues)
							if (!value->isChecked) {
								if ((value->isDefault && includeDefaultOptions) || !value->isDefault)
									options += "," + value->text;
							}
					}
				}
				break;
			case ComboBox:
				if (widgetRow->comboBox->currentIndex() != 0 || includeDefaultOptions)
					options += ',' + widgetRow->comboBox->itemData(widgetRow->comboBox->currentIndex()).toString();
				break;
			case SpinBox:
				if (includeDefaultOptions || widgetRow->option->optionDefaultValue != QVariant(widgetRow->spinBox->value()).toString())
					options += "," + widgetRow->option->optionValues[0]->text + QVariant(widgetRow->spinBox->value()).toString();
				break;
			case Perm:
				if (includeDefaultOptions || !widgetRow->permWidget->isDefaultValue())
					options += "," + widgetRow->permWidget->codeForFstab();
				break;
		}
	}
	
	if (!options.isEmpty())
		options.remove(0,1);
	return options;
}
