//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Core
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QDir>
#include <QFile>
#include <QTextStream>
#include <QDate>
#include <QTime>
#include <QVariant>
#include <QObject>
#include "diskbackup.h"

const QString DEFAULT_BACKUP_PATH = QDir::homePath() + "/.backups/";

DiskBackUp::DiskBackUp(const QString& path) {
	backupDirPath = path.isEmpty() ? DEFAULT_BACKUP_PATH : path;
	validBackUpPath = true;
	backupDir = new QDir(backupDirPath);
	if (!backupDir->exists())
		if (!backupDir->mkpath(backupDirPath))
			validBackUpPath = false;
	if (validBackUpPath)
		parseBackUpDirectory();
}

DiskBackUp::~DiskBackUp() {
	delete backupDir;
	foreach (BackUp *backup,backupsList)
		delete backup;
}
		
bool DiskBackUp::isValidBackUpPath() {
	return validBackUpPath;
}

BackUp *DiskBackUp::createBackUp(const QString& backupText) {
	if (!validBackUpPath)
		return 0;
	QString fileName = backupFileName();
	QFile newFile(backupDir->absoluteFilePath(fileName));
	if (!newFile.open(QIODevice::WriteOnly))
		return 0;
	QTextStream newStream(&newFile);
	newStream << backupText;
	BackUp *newBackUp = new BackUp;
	newBackUp->content = backupText;
	newBackUp->filePath = newFile.fileName();
	newBackUp->date = fileName.section(" ",1,3);
	newBackUp->time = fileName.section(" ",0,0);
	backupsList.append(newBackUp);
	return newBackUp;
}

bool DiskBackUp::removeBackUp(BackUp *backup) {
	if (!QFile::remove(backup->filePath))
		return false;
	return true;
}

QString DiskBackUp::backupFileName() {
	QString backupName;
	backupName += QTime::currentTime().toString("hh:mm") + " ";
	backupName += QVariant(QDate::currentDate().day()).toString() + " ";
	switch (QDate::currentDate().month()) {
		case 1:
			backupName += QObject::tr("January"); break;
		case 2:
			backupName += QObject::tr("February"); break;
		case 3:
			backupName += QObject::tr("March"); break;
		case 4:
			backupName += QObject::tr("April"); break;
		case 5:
			backupName += QObject::tr("May"); break;
		case 6:
			backupName += QObject::tr("June"); break;
		case 7:
			backupName += QObject::tr("July"); break;
		case 8:
			backupName += QObject::tr("August"); break;
		case 9:
			backupName += QObject::tr("September"); break;
		case 10:
			backupName += QObject::tr("October"); break;
		case 11:
			backupName += QObject::tr("November"); break;
		case 12:
			backupName += QObject::tr("December"); break;
	}
	backupName += " " + QVariant(QDate::currentDate().year()).toString();
	return backupName;
}

void DiskBackUp::setBackUpPath(const QString &backupPath) {
	backupDirPath = backupPath;
	parseBackUpDirectory();
}

QString& DiskBackUp::backupPath() {
	return backupDirPath;
}

const QString& DiskBackUp::backupPath() const {
	return backupDirPath;
}

QList<BackUp *>& DiskBackUp::backups() {
	return backupsList;
}

const QList<BackUp *>& DiskBackUp::backups() const {
	return backupsList;
}

BackUp *DiskBackUp::backUp(int index) const {
	if (backupsList.count() <= index || index < 0)
		return 0;
	return backupsList[index];
}

void DiskBackUp::parseBackUpDirectory() {
	foreach (QString backupFileName,backupDir->entryList(QDir::Files))
		if (backupFileName.contains(QRegExp("[0-9][0-9]:[0-9][0-9]\\s[0-9]\\s*\\s*"))) {
			QFile file(backupDir->absoluteFilePath(backupFileName));
			if (!file.open(QIODevice::ReadOnly)) continue;
			QTextStream stream(&file);

			BackUp *backup = new BackUp;
			backup->content = stream.readAll();
			backup->filePath = file.fileName();
			backup->date = backupFileName.section(" ",1,3);
			backup->time = backupFileName.section(" ",0,0);
			backupsList.append(backup);
		}
}
