/* main routines

   Written by Matthias Hensler
   Copyright WSPse 1999-2001
   eMail: wsp@gmx.de

Created: 1999/06/06
Updated: 2001/06/27
*/

/* Copying:
   This program is free software; you can redistribute it and/or modify it under
   the terms of the GNU Gerneral Public License as published by the Free Soft-
   ware Foundation; either version 2 of License, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILTY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 675 Mass
   Ave, Cambridge, MA 02139, USA.
   */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <ncurses.h>
#include "mp3creat.h"

extern WINDOW *c_newwin(int h, int l, int y, int x, void *proc, int arg1, int arg2);
extern void c_delwin(WINDOW *win);
extern int open_cdrom();
extern int close_cdrom();
extern int init_cd();
extern int build_data_tree(char *cddb_server, char *local_cddb_db,
			   song_typ **ret_tree, BOOL force_sampler);
extern void main_loop(WINDOW *win);
extern void fill_in_songs(WINDOW *win);
extern void free_song_typ(song_typ **anchor);
extern void act_signal_handler();
extern void pop_up_info();
extern void check_config();
extern void setup_stat_win(int max_length);
extern void print_stat_win(char *text);
extern void destroy_stat_win();
extern void popup_error_win(char *tx);
extern void draw_main_win(WINDOW *win, int dummy1, int dummy2);
extern int get_config(char *file);
extern int put_config(char *file, BOOL ask);
extern int start_non_interact(char *batchfile);
extern void calc_tot_frm();
extern void init_config();
extern void news_show_summery(BOOL windowed);
extern int lock_lock_file(char *lockfile, BOOL extension);
extern int lock_unlock_file(char *lockfile, BOOL extension);
extern char *file_glob_in_name(char *name);
extern char *copy_char_str(char *old);
extern BOOL select_yesno_box(char *tx);
extern char *def_cddb_server;
extern char *def_cddb_bank;
extern char *def_cdrom_dev;
extern int  auto_save;
extern int track_last;
extern int cd_drv;
extern BOOL rip_enc_ordered;
extern char *batch_alt_tmp_file;
extern song_typ *lay_global_anchor[2];
extern song_typ *lay_top_anchor[2];
extern song_typ *lay_curr_anchor[2];
extern int lay_select_line[2];
extern int lay_divider;
extern int lay_act_side;
extern int num_ripped_tracks;
extern int del_tmp_on_exit;
extern BOOL careful_batch;
extern BOOL config_para_mp3c;

/* some globals, would not used extensivly */
char *external_config_file;
char *external_cdrom_dev;
char external_version_str[80];

/* used to leave program clearly. just clear screen, end curses-functions
   and shutdown windows */
void shut_down_prg()
{
  clear();
  refresh();
  endwin();
  close_cdrom();

  printf(_("\n\nMP3c, by WSPse 1999-2001, Matthias Hensler, <matthias@wspse.de>\n"
	 "Thanks for using this program\n\n"));
}

/* last resort if an error occured, which could not handled and force to
   quit program, e.g. out of memory. Try to give specified codes.
   Currently used:
   1 - got NULL-pointer from malloc/realloc function
   2 - User-terminate
   3 - got NULL-pointer from ncurses-lib (calling newwin)
 */
void wuuush(int code)
{
  switch(code) {
    case 1:
      /* well, this might force problems, since error win
	 also needs memory ;-) */
      popup_error_win(_("ups, we ran out of memory :-("));
      break;

    case 2:
      popup_error_win(_("terminated..."));
      if(auto_save)
	put_config(external_config_file, FALSE);     /* save configuration */
      break;

    case 3:
      popup_error_win(_("creating new window caused in NULL-pointer (ncurses-bug?!)"));
      break;

    default:
      popup_error_win(_("sorry, unspecified error occured"));
      break;
  }
  
  shut_down_prg();      /* well, program must be killed, but try to do this */
  if(code == 2) exit(EXIT_SUCCESS);    /* clearly, with reseted terminal-   */
  else          exit(EXIT_FAILURE);    /* functions                         */
}

void usage(char *arg)
{
  printf("\n%s", external_version_str);
  printf(_("\n\nUseful arguments for \"%s\":\n"
	 "-i <configfile>\n"
	 "-b <batchfile>\n"
	 "-t <tmp-dir> (only in context with -b)\n"
	 "-d <cdrom-device>\n"
	 "-v\n"
	 "-n (show news)\n"
	 "\nif you start %s with \"-b\" argument given, a\n"
	 "scriptfile will created non-interactivly and program\n"
	 "exits.\n"
	 "\n"), arg, arg);
  exit(EXIT_SUCCESS);
}

/* start of program */
int main(int argc, char **argv)
{
  WINDOW *win;
  song_typ *anchor;
  char *tx;
  int i;
  BOOL batch_mode;
  char *batch_file = NULL;
  BOOL existing_lockfile = FALSE;

  sprintf(external_version_str, "%s-%s", PACKAGE, PRG_VERSION);
  
  /* startup national language support */
#ifdef HAVE_NLS
  setlocale(LC_MESSAGES, "");
  bindtextdomain( (char *) PACKAGE, LOCALEDIR);
  textdomain( (char *) PACKAGE);
  strcat(external_version_str, " (NLS)");
#endif

  strcat(external_version_str, _(" - ENGLISH ("));
  strcat(external_version_str, __DATE__);
  strcat(external_version_str, ")");
  
  external_cdrom_dev   = NULL;
  external_config_file = NULL;
  batch_mode           = FALSE;
  batch_alt_tmp_file   = NULL;

  if(argc > 1) {
    for(i=1;i<argc;i++) {
      if(strncmp(argv[i], "-batch", strlen(argv[i])) == 0) {
	i++;
	if(i == argc) usage(argv[0]);
	batch_file = argv[i];
	batch_mode = TRUE;
      } else if(strncmp(argv[i], "-device", strlen(argv[i])) == 0) {
	i++;
	if(i == argc) usage(argv[0]);
	external_cdrom_dev = argv[i];
      } else if(strncmp(argv[i], "-tempfile", strlen(argv[i])) == 0) {
	i++;
	if(i == argc) usage(argv[0]);
	batch_alt_tmp_file = argv[i];
      } else if(strncmp(argv[i], "-initfile", strlen(argv[i])) == 0) {
	i++;
	if(i == argc) usage(argv[0]);
	external_config_file = argv[i];
      } else if(strncmp(argv[i], "-version", strlen(argv[i])) == 0) {
	printf(_("\nThis is MP3c, an automated, CDDB-based, MP3-encoder"
	       "\nversion V%s, compiled on %s"
	       "\n\nMP3c is written by Matthias Hensler <matthias@wspse.de>"
	       "\nCopyright WSPse 1999-2001, distributed under GNU General Public License"
	       "\n\nMP3c-Homepage: http://mp3c.wspse.de/"
	       "\nDownload:      ftp://ftp.wspse.de/pub/linux/wspse/"
	       "\n\nI want to thank for any suggestion I received (take a look in README-"
	       "\nfile and ChangeLog for details)"
	       "\n\nIf you have any wish for a feature not currently included in MP3c, or"
	       "\nif you found a bug, or an unexpected error, please feel free to contact"
	       "\nme. I would like to implement new features and fix bugs. It is really"
	       "\nhelpful if you can send me a detailed bugreport, and maybe how to"
	       "\nreproduce the problem. If you get a segmentation fault, *please* send me"
	       "\nthe core-dump AND your binary-file (and maybe your config-file)"
	       "\nMy eMail-adress: Matthias Hensler <matthias@wspse.de>"
	       "\n\nThanks for reading this. Hope you like this program.\n\n")
	       , PRG_VERSION, __DATE__);
	exit(EXIT_SUCCESS);
      } else if(strncmp(argv[i], "-news", strlen(argv[i])) == 0) {
	careful_batch = TRUE;
	init_config();
	get_config(external_config_file);
	news_show_summery(FALSE);
	exit(EXIT_SUCCESS);
      } else {
	usage(argv[0]);
      }
    }
  }

  if(batch_mode) {
    start_non_interact(batch_file);
    exit(EXIT_SUCCESS);
  }
  
  /* Initiating interactive startup...
     ...but first check if there is an open terminal.
   */
  if(! isatty(0)) {
    fprintf(stderr, _("MP3c was started in interactive mode without controlling terminal!\n"
		      "This normally causes high load of CPU-time, so MP3c exits now.\n"
		      "(if you think this is wrong, please contact the author wsp@gmx.de)\n"));
    exit(EXIT_FAILURE);
  }

  /* OK, seems that everything is fine, so do startup */
  initscr();                                /* init of curses functions */
  start_color();
  intrflush(stdscr, FALSE);

  init_pair(1, COLOR_WHITE,  COLOR_BLUE);   /* global color definitions */
  init_pair(2, COLOR_YELLOW, COLOR_BLUE);
  init_pair(3, COLOR_BLACK,  COLOR_CYAN);
  init_pair(4, COLOR_YELLOW, COLOR_CYAN);
  init_pair(5, COLOR_RED,    COLOR_YELLOW);
  init_pair(6, COLOR_RED,    COLOR_BLUE);
  init_pair(7, COLOR_BLACK,  COLOR_WHITE);

  lay_top_anchor[0]    = NULL;    /* anchors should be initialized with          */
  lay_global_anchor[0] = NULL;    /* NULL, to prevent dereference of unspecified */
  lay_top_anchor[1]    = NULL;    /* adresses, like "draw_main_win" would do!    */
  lay_global_anchor[1] = NULL;
  lay_divider          = 50;      /* divide window at 50%    */
  				  /* TODO: implement user-   */
			       	  /* setting for this value  */
  lay_act_side         = 0;       /* start left              */
  num_ripped_tracks    = 0;       /* rippedlist empty        */
  
  init_config();                  /* init default parameters */
  act_signal_handler();           /* install signal handler  */
  pop_up_info();                  /* show info window        */

  get_config(external_config_file);    /* try to load configuration      */
  
  check_config();                      /* check configuration            */

  /*  was parallel running allowed? */
  if(! config_para_mp3c) {
    /* if not we now must check the lockfile */
    batch_file = file_glob_in_name(copy_char_str(MP3C_LOCKFILE));
    i = lock_lock_file(batch_file, FALSE);
    free(batch_file);
    batch_file = NULL;

    if(i == 3) {
      if(select_yesno_box(_("An other MP3c process is still running, continue anyway?"))) {
	existing_lockfile = TRUE;
      } else {
	shut_down_prg();
	return EXIT_SUCCESS;
      }
    }
  }

  news_show_summery(TRUE);             /* show hot news for this version */

  win = c_newwin(0,0,0,0, draw_main_win, 0, 0);  /* build main window */
  draw_main_win(win,0,0);
  wrefresh(win);

  /* program is running, now try first initialisation of cdrom, try to get
     TOC data and cddb entry. if this fails just skip it and jump into the
     main routine, its then users task to insert cd, change options (for
     correct device, cddb-server, etc) and reinitialisize. */

  anchor = NULL;
  setup_stat_win(75);

  print_stat_win(_("cdrom initialisation in progress....."));
  if(init_cd() != 0) {
    if(cd_drv == -2) {
      popup_error_win(_("device is ok, but no cd inserted, eh?"));
    } else {
      tx = malloc(sizeof(char) * (strlen(def_cdrom_dev) + 70));
      if(! tx) {
	perror("malloc");
	wuuush(1);
      }
      sprintf(tx, _("uh, device \"%s\" failure (please get permissions and/or devicename)"), def_cdrom_dev);
      popup_error_win(tx);
      free(tx);
    }
  } else {
    print_stat_win(_("cool, you configured it right, ya"));
    build_data_tree(def_cddb_server, def_cddb_bank, &anchor, FALSE);
    print_stat_win(_("well done, initialisation finished successful"));
  }
  
  destroy_stat_win();

  /* build up windows and windowcontent */
  lay_top_anchor[0]    = anchor;
  lay_global_anchor[0] = anchor;
  lay_select_line[0]   = 1;
  lay_select_line[1]   = 1;
  calc_tot_frm();
  fill_in_songs(win);
  
  /* now jump to mainloop and let user do whatever he wants */
  main_loop(win);

  /* mainloop was left, its now our task to exit program clearly */
  c_delwin(win);                      /* first delete mainwindow */

  /* check for remaining tmpfiles (and how to handle these) */
  if(del_tmp_on_exit > 0 && lay_global_anchor[1]) {
    setup_stat_win(60);
    print_stat_win(_("deleting tmpfiles"));
    anchor = lay_global_anchor[1];
    while(anchor) {
      if(anchor->tmp_wav_file && (del_tmp_on_exit == 2 || anchor->on_fly)) {
	print_stat_win(anchor->tmp_wav_file);
	unlink(anchor->tmp_wav_file);
      }
      anchor = anchor->next;
    }
    destroy_stat_win();
  }
  
  free_song_typ(&(lay_global_anchor[0])); /* free memory if      */
  free_song_typ(&(lay_global_anchor[1])); /* there is any in use */

  /* should we save config file? */
  if(auto_save) {
    put_config(external_config_file, FALSE);     /* save configuration */
  }
  
  /*  was parallel running allowed? */
  if((! config_para_mp3c) && (! existing_lockfile)) {
    /* if not we now must clear the lockfile */
    batch_file = file_glob_in_name(copy_char_str(MP3C_LOCKFILE));
    lock_unlock_file(batch_file, FALSE);
    free(batch_file);
    batch_file = NULL;
  }
  
  shut_down_prg();                               /* clear windows      */
  return EXIT_SUCCESS;                           /* shutdown...        */
  /* program ended, thanx for using. Matthias Hensler, WSPse 1999-2001 */
}
