/* mpc_sqr -- Square a complex number.

Copyright (C) 2002, 2005, 2008, 2009 Andreas Enge, Paul Zimmermann, Philippe Th\'eveny

This file is part of the MPC Library.

The MPC Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The MPC Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the MPC Library; see the file COPYING.LIB.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */

#include "mpc-impl.h"

int
mpc_sqr (mpc_ptr rop, mpc_srcptr op, mpc_rnd_t rnd)
{
   int ok;
   mpfr_t u, v;
   mpfr_t x;
      /* rop temporary variable to hold the real part of op,
         needed in the case rop==op */
   mp_prec_t prec;
   int inex_re, inex_im, inexact;

   /* special values: NaN and infinities */
   if (!mpfr_number_p (MPC_RE (op)) || !mpfr_number_p (MPC_IM (op))) {
      if (mpfr_nan_p (MPC_RE (op)) || mpfr_nan_p (MPC_IM (op))) {
         mpfr_set_nan (MPC_RE (rop));
         mpfr_set_nan (MPC_IM (rop));
      }
      else if (mpfr_inf_p (MPC_RE (op))) {
         if (mpfr_inf_p (MPC_IM (op))) {
            mpfr_set_nan (MPC_RE (rop));
            mpfr_set_inf (MPC_IM (rop),
                          MPFR_SIGN (MPC_RE (op)) * MPFR_SIGN (MPC_IM (op)));
         }
         else {
            mpfr_set_inf (MPC_RE (rop), +1);
            if (mpfr_zero_p (MPC_IM (op)))
               mpfr_set_nan (MPC_IM (rop));
            else
               mpfr_set_inf (MPC_IM (rop),
                             MPFR_SIGN (MPC_RE (op)) * MPFR_SIGN (MPC_IM (op)));
         }
      }
      else /* IM(op) is infinity, RE(op) is not */ {
         mpfr_set_inf (MPC_RE (rop), -1);
         if (mpfr_zero_p (MPC_RE (op)))
            mpfr_set_nan (MPC_IM (rop));
         else
            mpfr_set_inf (MPC_IM (rop),
                          MPFR_SIGN (MPC_RE (op)) * MPFR_SIGN (MPC_IM (op)));
      }
      return MPC_INEX (0, 0); /* exact */
   }

   prec = MPC_MAX_PREC(rop);

   /* first check for real resp. purely imaginary number */
   if (mpfr_zero_p (MPC_IM(op)))
   {
      int same_sign = mpfr_signbit (MPC_RE (op)) == mpfr_signbit (MPC_IM (op));
      inex_re = mpfr_sqr (MPC_RE(rop), MPC_RE(op), MPC_RND_RE(rnd));
      inex_im = mpfr_set_ui (MPC_IM(rop), 0ul, GMP_RNDN);
      if (!same_sign)
        mpc_conj (rop, rop, MPC_RNDNN);
      return MPC_INEX(inex_re, inex_im);
   }
   if (mpfr_zero_p (MPC_RE(op)))
   {
      int same_sign = mpfr_signbit (MPC_RE (op)) == mpfr_signbit (MPC_IM (op));
      inex_re = -mpfr_sqr (MPC_RE(rop), MPC_IM(op), INV_RND (MPC_RND_RE(rnd)));
      mpfr_neg (MPC_RE(rop), MPC_RE(rop), GMP_RNDN);
      inex_im = mpfr_set_ui (MPC_IM(rop), 0ul, GMP_RNDN);
      if (!same_sign)
        mpc_conj (rop, rop, MPC_RNDNN);
      return MPC_INEX(inex_re, inex_im);
   }
   /* If the real and imaginary part of the argument have rop very different */
   /* exponent, it is not reasonable to use Karatsuba squaring; compute    */
   /* exactly with the standard formulae instead, even if this means an    */
   /* additional multiplication.                                           */
   if (SAFE_ABS (mp_exp_t, MPFR_EXP (MPC_RE (op)) - MPFR_EXP (MPC_IM (op)))
       > (mp_exp_t) MPC_MAX_PREC (op) / 2)
   {
      mpfr_init2 (u, 2*MPFR_PREC (MPC_RE (op)));
      mpfr_init2 (v, 2*MPFR_PREC (MPC_IM (op)));

      mpfr_sqr (u, MPC_RE (op), GMP_RNDN);
      mpfr_sqr (v, MPC_IM (op), GMP_RNDN); /* both are exact */
      inex_im = mpfr_mul (rop->im, op->re, op->im, MPC_RND_IM (rnd));
      mpfr_mul_2exp (rop->im, rop->im, 1, GMP_RNDN);
      inex_re = mpfr_sub (rop->re, u, v, MPC_RND_RE (rnd));

      mpfr_clear (u);
      mpfr_clear (v);
      return MPC_INEX (inex_re, inex_im);
   }


   mpfr_init (u);
   mpfr_init (v);

   if (rop == op)
   {
      mpfr_init2 (x, MPFR_PREC (op->re));
      mpfr_set (x, op->re, GMP_RNDN);
   }
   else
      x [0] = op->re [0];

   do
   {
      prec += mpc_ceil_log2 (prec) + 5;

      mpfr_set_prec (u, prec);
      mpfr_set_prec (v, prec);

      /* Let op = x + iy. We need u = x+y and v = x-y, rounded away.       */
      /* As this is not implemented in mpfr, we round towards zero and    */
      /* add one ulp if the result is not exact. The error is bounded     */
      /* above by 1 ulp.                                                  */
      /* We first let inexact be 1 if the real part is not computed       */
      /* exactly and determine the sign later.                            */
      inexact = 0;
      if (mpfr_add (u, x, MPC_IM (op), GMP_RNDZ))
         /* we have to use x here instead of MPC_RE (op), as MPC_RE (rop)
            may be modified later in the attempt to assign u to it */
      {
         mpfr_add_one_ulp (u, GMP_RNDN);
         inexact = 1;
      }
      if (mpfr_sub (v, x, MPC_IM (op), GMP_RNDZ))
      {
         mpfr_add_one_ulp (v, GMP_RNDN);
         inexact = 1;
      }

      /* compute the real part as u*v, rounded away                    */
      /* determine also the sign of inex_re                            */
      if (mpfr_sgn (u) == 0 || mpfr_sgn (v) == 0)
        {
          /* as we have rounded away, the result is exact */
          mpfr_set_ui (MPC_RE (rop), 0, GMP_RNDN);
          inex_re = 0;
          ok = 1;
        }
      else if (mpfr_sgn (u) * mpfr_sgn (v) > 0)
        {
          inexact |= mpfr_mul (u, u, v, GMP_RNDU); /* error 5 */
          ok = (!inexact) | mpfr_can_round (u, prec - 3, GMP_RNDU, GMP_RNDZ,
               MPFR_PREC (MPC_RE (rop)) + (MPC_RND_RE (rnd) == GMP_RNDN));
          if (ok)
            {
              inex_re = mpfr_set (MPC_RE (rop), u, MPC_RND_RE (rnd));
              if (inex_re == 0)
                /* remember that u was already rounded */
                inex_re = inexact;
            }
        }
      else
        {
          inexact |= mpfr_mul (u, u, v, GMP_RNDD); /* error 5 */
          ok = (!inexact) | mpfr_can_round (u, prec - 3, GMP_RNDD, GMP_RNDZ,
               MPFR_PREC (MPC_RE (rop)) + (MPC_RND_RE (rnd) == GMP_RNDN));
          if (ok)
            {
              inex_re = mpfr_set (MPC_RE (rop), u, MPC_RND_RE (rnd));
              if (inex_re == 0)
                inex_re = inexact;
            }
        }
   }
   while (!ok);

   /* compute the imaginary part as 2*x*y, which is always possible */
   inex_im = mpfr_mul (MPC_IM (rop), x, MPC_IM (op), MPC_RND_IM (rnd));
   mpfr_mul_2ui (MPC_IM (rop), MPC_IM (rop), 1, GMP_RNDN);

   mpfr_clear (u);
   mpfr_clear (v);

   if (rop == op)
      mpfr_clear (x);

   return MPC_INEX (inex_re, inex_im);
}
