/**
   The command-line version of MRIConvert.
**/

#include <string>
#include <iostream>
#include <vector>

#include <wx/app.h>
#include <wx/cmdline.h>
#include <wx/string.h>
#include <wx/dir.h>
#include <wx/config.h>
#include <wx/stdpaths.h>

#include "McVerter.h"
#include "OutputFactory.h"
#include "Converter.h"
#include "OutputterBase.h"
#include "StringConvert.h"
#include "version_string.h"

#include <boost/program_options.hpp>
namespace po = boost::program_options;

// Macro in place of _() since std::cout does not like utf16 or utf32.
#define _C(string) wxString(_(string)).utf8_str()

using namespace std;
using namespace jcs;

// A helper function to simplify the main part.
// Meaning, this permits cout to handle a vector. 20120111cdt
template<class T>
ostream& operator<<(ostream& os, const vector<T>& v)
{
  copy(v.begin(), v.end(), ostream_iterator<T>(std::cout, " ")); 
  return os;
}

int main(int argc, char** argv)
{
  // First make sure we can use wx stuff.
  wxInitializer initializer(0, (wxChar**) NULL);
  if ( !initializer ) {
    std::cerr << "Failed to initialize the wxWidgets library, aborting.";
    return -1;
  }

  // Localization steps
  /*
    wxLocale m_locale;
    if ( !m_locale.Init(wxLANGUAGE_DEFAULT, wxLOCALE_CONV_ENCODING) ) {
    wxLogError(_T("This language is not supported by the system."));
    }
    // Development only, distribution should install in standard location.
    wxLocale::AddCatalogLookupPathPrefix(_T("l10n"));
    // Initialize the catalogs we'll be using
    m_locale.AddCatalog(_T("mriconvert"));
  */

  // Variables to collect command-line arguments.
  std::string std_match;
  std::string std_outputDir;
  std::string std_format;
  long skip = 0;
  std::string std_fnformat;

  po::options_description additional("Additional options");
  po::options_description required("Required options");
  po::options_description hidden;
  po::options_description documented;
  po::options_description all;
  po::positional_options_description positional;
  po::variables_map vm;

  required.add_options()
    ("output,o", po::value<string>(&std_outputDir)->required(), _C("Output directory for converted files"))
    ("format,f", po::value<string>(&std_format)->required(), _C("Output format: fsl, spm, meta, nifti, analyze, or bv"))
    ;
  additional.add_options()
    ("help,h", _C("Produce help message"))
    ("version,V", _C("Version info"))
    ("split,x", _C("Save each series to a separate directory"))
    ("fourd,d", _C("Save output volumes as 4D files"))
    ("ho,a", _C("Save header only (metaimage only)"))
    ("nii,n", _C("Save files as .nii (nifti/fsl only)"))
    ("v16,b", _C("Save .v16 files (BrainVoyager only)"))
    ("rescale,r", _C("Apply rescale slope and intercept to data"))
    ("patientid,u", _C("Use patient id instead of patient name for output file"))
    ("skip,s", po::value<long>(&skip)->default_value(0), _C("Skip first 'arg' volumes"))
    ("match,m", po::value<string>(&std_match), _C("Only convert files whose series descriptions include this string"))
    ("fnformat,F", po::value<string>(&std_fnformat), _C("Use this format for name of output file: +/- PatientName, \
PatientId, \
SeriesDate, \
SeriesTime, \
StudyId, \
StudyDescription, \
SeriesNumber, \
SequenceName, \
ProtocolName, \
SeriesDescription"))
    ("verbose,v", _C("Verbose mode"))
    ;
  hidden.add_options()
    ("input-file", po::value< vector<string> >()->default_value(vector<string>(),"")->required(), "Input directory or file(s)")
    ;
  positional.add("input-file", -1);
  documented.add(required).add(additional);
  all.add(required).add(additional).add(hidden);

  try {
    po::store(po::command_line_parser(argc, argv).options(all).positional(positional).run(), vm);

    // Known bail-out cases.
    if (vm.count("version")) {
      std::cout << "mcverter " << VERSION_STRING << std::endl;
      return 0;
    }
    vector<string> v = vm["input-file"].as< vector<string> >();
    if (v.begin() == v.end()) {
      std::cout << "No files or directory specified." << std::endl;
    }
    if (vm.count("help") || (v.begin() == v.end())) {
      std::cout << "Usage: mcverter [options] <input directory or file(s)>" << std::endl;
      std::cout << documented;
      return 0;
    }
    
    po::notify(vm);
  }
  catch (po::invalid_syntax& e) {
    std::cout << "Invalid syntax." << e.kind() << " " << e.tokens() << std::endl;
    return 0;
  }
  catch (po::unknown_option& e) {
    std::cout << "Unrecognized option: " << e.get_option_name() << std::endl;
    return 0;
  }
  catch (po::ambiguous_option& e) {
    std::cout << "Ambiguous option " << e.get_option_name() << std::endl;
    return 0;
  }
  catch (po::multiple_values& e) {
    std::cout << "Multiple values invalid." << std::endl;
    return 0;
  }
  catch (po::multiple_occurrences& e) {
    std::cout << "Multiple occurrences." << std::endl;
    return 0;
  }
  catch (po::validation_error& e) {
    std::cout << "Validation error " << e.what() << std::endl;
    return 0;
  }
  catch (po::too_many_positional_options_error& e) {
    std::cout << "General error." << std::endl;
    return 0;
  }
  catch (po::invalid_command_line_style& e) {
    std::cout << "General error." << std::endl;
    return 0;
  }
  catch (po::reading_file& e) {
    std::cout << "Error reading file." << std::endl;
    return 0;
  }
  catch (po::required_option& e) {
    std::cout << "Option " << e.get_option_name() << " is required." << std::endl;
    return 0;
  }
  catch (po::error& e) {
    std::cout << "General error." << std::endl;
    return 0;
  }

  // Transfer arguments to wxString variables.
  wxString match(std_match.c_str(), wxConvUTF8);
  wxString outputDir(std_outputDir.c_str(), wxConvUTF8);
  wxString format(std_format.c_str(), wxConvUTF8);
  wxString fnformat(std_fnformat.c_str(), wxConvUTF8);

  if (vm.count("verbose")) {
    verbose = true;
  }

  // Selector for output format.
  int format_type = 0;
  int found = wxNOT_FOUND;
  while (format_type < OutputFactory::GetNumberOfTypes()) {
    found = format.Find(wxString(OutputFactory::GetShortDescription(format_type), wxConvLocal));
    if (found != wxNOT_FOUND) break;
    ++format_type;
  }
  if (found == wxNOT_FOUND) {
    // Get the list of permitted output formats.
    wxString allowed_formats_string;
    int type;
    allowed_formats_string.append(_("Available formats: "));
    for (type = 0; type < (OutputFactory::GetNumberOfTypes() - 1); type++) {
      allowed_formats_string.append(wxString(OutputFactory::GetShortDescription(type), wxConvLocal));
      allowed_formats_string.append(_T(", "));
    }
    allowed_formats_string.append(wxString(OutputFactory::GetShortDescription(type), wxConvLocal));
    std::cerr << _C("Invalid format type: ") << format.utf8_str() << std::endl;
    std::cerr << allowed_formats_string.utf8_str() << std::endl;
    return 0;
  }
  
  // Selector for output directory.
  if (!wxDir::Exists(outputDir)) {
    if (!wxFileName::Mkdir(outputDir, 0777, wxPATH_MKDIR_FULL)) {
      std::cout << _C("Unable to create directory: ") << outputDir << std::endl;
      return 0;
    }
    if (verbose)
      std::cout << _C("Created directory ") << outputDir << std::endl;
  }
  
  // We know enough to create our converter and outputter.
  Converter* converter = new Converter(format_type);
  OutputterBase* outputter = converter->GetOutputter();
  outputter->SetOption("skip", static_cast<int> (skip));
  outputter->mOutputList.rootDir = outputDir;
  outputter->SetOption("split", vm.count("split") ? true : false);
  outputter->SetOption("dim", vm.count("fourd") ? 4 : 3);
  outputter->SetOption("ho", vm.count("ho") ? true : false);
  outputter->SetOption("nii", vm.count("nii") ? true : false);
  outputter->SetOption("v16", vm.count("v16") ? true : false);
  outputter->SetOption("rescale", vm.count("rescale") ? true : false);
  
  // Handle request for using PatientId rather than PatientName.
  if (vm.count("patientid")) {
    outputter->defaultNameFields[OutputterBase::PatientName].value = false;
    outputter->defaultNameFields[OutputterBase::PatientId].value = true;
  }
  
  // Handle request for alternate output filename format.
  std::string stdfnformat = std::string (fnformat.mb_str());
  OutputterBase::FieldMap::iterator fm_it = outputter->defaultNameFields.begin();
  OutputterBase::FieldMap::iterator fm_it_end = outputter->defaultNameFields.end();
  for (; fm_it != fm_it_end; fm_it++) {
    if (stdfnformat.find("+" + fm_it->second.name) != -1)
      fm_it->second.value = true;
    if (stdfnformat.find("-" + fm_it->second.name) != -1)
      fm_it->second.value = false;
  }

  // We've processed all options and switches,
  // now add the requested files.
  if (verbose)
    std::cout << _C("Adding files...") << std::endl;
  std::vector< std::string > v = vm["input-file"].as< std::vector< std::string > >();
  if (v.begin() == v.end())
    std::cout << "No files specified." << std::endl;
  std::vector< std::string >::iterator it = v.begin();
  std::vector< std::string >::iterator it_end = v.end();
  for (; it != it_end; it++) {
    wxFileName filename(wxString(std::string(*it).c_str(), wxConvUTF8));
    if (filename.FileExists()) {
      converter->AddFile(filename.GetFullPath(), match);
    }
    else if (filename.DirExists()) {
      wxArrayString files;
      wxDir::GetAllFiles(filename.GetFullPath(), &files);
      unsigned int n_files = files.GetCount();
      for (unsigned int i = 0; i < files.GetCount(); ++i) {
	converter->AddFile(files.Item(i), match);
      }
    }
    else {
      std::cout << _C("File or directory ") << *it << _C(" not found. Add trailing directory separator and try again.") << std::endl;
    }
  }

  if (verbose) {
    MessageList::iterator ml_it = converter->messages.begin();
    MessageList::iterator ml_it_end = converter->messages.end();
    for (; ml_it != ml_it_end; ml_it++) {
      Message::iterator it = ml_it->begin();
      Message::iterator it_end = ml_it->end();
      for (; it != it_end; it++) {
	std::cout << it->mb_str(wxConvLocal) << " ";
      }
      std::cout << std::endl;
    }
  }
  converter->messages.clear();
  converter->UpdateAllOutput();

  if (verbose) std::cout << _C("Converting...") << std::endl;

  // Let the real work begin.
  converter->ConvertAll();

  if (verbose) std::cout << _C("Finished") << std::endl;

  delete converter;

  return 0;
}
