#ifndef SERIES_HANDLER_H
#define SERIES_HANDLER_H

#include <string>
#include <vector>
#include <set>
#include <map>

#include "Volume.h"
#include "DicomFile.h"

class wxFileName;

namespace jcs {

    enum AoCode {
	ASCENDING,
	DESCENDING,
	INTERLEAVED_ODD,
	INTERLEAVED_EVEN,
	UNKNOWN,
	OTHER
    };


    /** Things constant for a 3D volume.
     */
    struct VolId {
	typedef std::vector<std::string> IdType;
	IdType ids;
	int orientNumber;
	
	bool operator<  (const VolId &rhs) const
	{
	    return (ids < rhs.ids);
	}

	bool operator==  (const VolId &rhs) const
	{
	    return ((ids == rhs.ids) &&
		    (orientNumber == rhs.orientNumber));
	}
    };


    /** Stuff used for automatic generation of filenames & directories,
	save to avoid unneccessary file reads.
    */
    struct SeriesInfo {
	std::string subject_name;
	std::string subject_id;
	std::string study_number;
	std::string series_number;
	std::string StudyDate;
	std::string SeriesDate;
	std::string study_description;
	std::string series_description;
    };


    struct GradientInfo {
	std::vector<double> values;
	std::vector<double> xGrads;
	std::vector<double> yGrads;
	std::vector<double> zGrads;
    };


    void Normalize(GradientInfo& info);
    void RotateGradInfo(GradientInfo& info, std::vector<double>& r);


    ///
    /**
     */
    class SeriesHandler
    {
    public :
	SeriesHandler(const std::string& seriesUid);
	virtual ~SeriesHandler();

	const std::string& GetSeriesUid() const { return mSeriesUid; }

	///
	/**
	   \param e DicomElement, group and element of DICOM value to find.
	   \param value Variable to hold value found.
	   \return 0 if no files in mFileMap. Result of dicomFile.Find otherwise.
	*/
	template <class T> int Find(DicomElement e, T& value) const
	    {
		if (mFileMap.empty()) return 0;
		DicomFile dicomFile((*mFileMap.begin()).first.c_str());
		return dicomFile.Find(e, value);
	    }

	///
	/**
	   \param s String reference, name of DICOM element to find.
	   \param value Variable to hold value found.
	   \return 0 if no files in mFileMap. Result of dicomFile.Find otherwise.
	*/
	template <class T> int Find(const std::string& s, T& value) const
	    {
		if (mFileMap.empty()) return 0;
		DicomFile dicomFile((*mFileMap.begin()).first.c_str());
		return dicomFile.Find(s, value);
	    }

	///
	/**
	   \param s String reference, name of DICOM element to find.
	   \param value Variable to hold value found.
	   \param id Reference to VolId.
	   \return Value of DICOM element given in 's' as a string.
	*/
	template <class T> int Find(const std::string& s, T& value, const VolId& id) const
	    {
		std::string filename;
		if (!GetFirstFilename(id, filename)) return 0;
		DicomFile dicomFile(filename.c_str());
		return dicomFile.Find(s, value);
	    }

	///
	/**
	   \param s String reference, name of DICOM element to find.
	   \return Value of DICOM element given in 's' as a string.
	*/
	std::string Find(const std::string& s) const
	    {
		std::string retstring;
		if (Find(s, retstring))
		    return retstring;
		else
		    return "";
	    }

	virtual std::vector<std::string> GetStringInfo();

	virtual void GetVolumes (std::map <VolId, Volume <wxInt64> >& v) { mGetVolumes <wxInt64> (v); }
	virtual void GetVolumes (std::map <VolId, Volume <wxUint64> >& v) { mGetVolumes <wxUint64> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt32> >& v) { mGetVolumes <wxInt32> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxUint32> >& v) { mGetVolumes <wxUint32> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt16> >& v) { mGetVolumes <wxInt16> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxUint16> >& v) { mGetVolumes <wxUint16> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt8> >& v) { mGetVolumes <wxInt8> (v); }
	virtual void GetVolumes (std::map <VolId, Volume <wxUint8> >& v) { mGetVolumes <wxUint8> (v); } 

	int AddFile(const char* filename);
	int GetNumberOfFiles() const { return mFileMap.size(); }
	virtual int GetNumberOfSlices() const;
	virtual int GetNumberOfVolumes() const;
	virtual std::vector<double> GetVoxelSize();
	virtual int GetRows();
	virtual int GetColumns();
	virtual int GetFrames() const;
	virtual double GetSliceDuration() const;
	virtual double GetVolumeInterval() const;
	virtual AoCode GetAcquisitionOrder() const { return UNKNOWN; }
	virtual bool IsMosaic() const { return false; }
	virtual bool IsDti() const { return false; }
	virtual bool IsMoCo() const { return false; }
	virtual bool IsMultiEcho() const { return multiecho; }
	bool IsBigEndian() const;
	std::set<VolId> GetVolIds() const;
	std::map<VolId, std::string> GetVolIdsAndFiles() const;
	std::vector<double> GetIppForFirstSlice(const VolId& id);
	std::vector<double> GetRotationMatrix(const VolId& id);
	virtual GradientInfo GetGradientInfo() { return GradientInfo(); };
	void GetImageComments(std::vector<std::string>& comments) const;
	std::vector<std::string> GetFilenames() const;
	void GetEchoTimes();

	SeriesInfo seriesInfo;
	bool rescale;
	bool multiecho;
	/// Indexed by EchoNumber.
	std::map<int, std::string> echo_times;

	int GetRescaleSlopeAndIntercept(const VolId& id, double& slope, double& intercept) const;
	virtual int GetRescaleSlopeAndIntercept(DicomFile& dfile, double& slope, double& intercept, int frame = 0) const;

    protected :
	typedef std::vector< VolId > VolListType;
	typedef std::map < std::string, VolListType > FileMapType;
	FileMapType mFileMap;
	std::map<VolId, std::vector<double> > position; // ipp for slice 1, calculated in GetVolumes
	std::vector<std::vector<double> > orientations;

	virtual VolListType ReadVolIds(DicomFile& file);
	virtual std::string GetImagePositionPatient(DicomFile& dfile, int frame);

    private :
	// moved over from DicomSeries
	const std::string mSeriesUid;
	std::set<std::string> mInstanceUids;

	SeriesHandler& operator=(const SeriesHandler& rhs);
	bool GetFirstFilename(const VolId& id, std::string& str) const;

	// todo: see if some of this can be moved out of the template?
	template <class T> void	mGetVolumes(std::map<VolId, Volume<T> >& v);
    };


    ///
    /**
     */
    class SyngoHandler : public SeriesHandler
    {
    public:
	SyngoHandler(const std::string& seriesUid);

	virtual double GetSliceDuration() const;
	virtual AoCode GetAcquisitionOrder() const;
	virtual double GetVolumeInterval() const;
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const;
	virtual bool IsMoCo() const;

    protected:
	int ReadCSAImageHeader(const std::string& tag, std::string& value) const;
	int ReadCSASeriesHeader(const std::string& tag, std::string& value) const;
	virtual VolListType ReadVolIds(DicomFile& file);

    };


    ///
    /**
     */
    class NumarisMosaicHandler : public SeriesHandler
    {
    public :
	NumarisMosaicHandler(const std::string& seriesUid);

	virtual int GetNumberOfSlices() const;
	virtual std::vector<double> GetVoxelSize();
	virtual int GetRows();
	virtual int GetColumns();
	virtual AoCode GetAcquisitionOrder() const;
	virtual double GetVolumeInterval() const;
	virtual std::vector<std::string> GetStringInfo();
	virtual bool IsMosaic() const { return true; }

	virtual void GetVolumes (std::map <VolId, Volume <wxInt64> >& v) { mGetVolumes <wxInt64> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxUint64> >& v) { mGetVolumes <wxUint64> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt32> >& v) { mGetVolumes <wxInt32> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxUint32> >& v) { mGetVolumes <wxUint32> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt16> >& v) { mGetVolumes <wxInt16> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxUint16> >& v) { mGetVolumes <wxUint16> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt8> >& v) { mGetVolumes <wxInt8> (v); }
	virtual void GetVolumes (std::map <VolId, Volume <wxUint8> >& v) { mGetVolumes <wxUint8> (v); } 

    protected :
	virtual std::vector<double> GetSliceOrder();
	virtual VolListType ReadVolIds(DicomFile& file);

    private:
	template <class T> void
	    mGetVolumes(std::map<VolId, Volume<T> >& v);

    };


    ///
    /**
     */
    class SyngoMosaicHandler : public SyngoHandler
    {
    public :
	SyngoMosaicHandler(const std::string& seriesUid);

	virtual int GetNumberOfSlices() const;
	virtual int GetRows();
	virtual int GetColumns();
	virtual std::vector<std::string> GetStringInfo();
	virtual bool IsMosaic() const { return true; }

	virtual void GetVolumes (std::map <VolId, Volume <wxInt64> >& v) { mGetVolumes <wxInt64> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxUint64> >& v) { mGetVolumes <wxUint64> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt32> >& v) { mGetVolumes <wxInt32> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxUint32> >& v) { mGetVolumes <wxUint32> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt16> >& v) { mGetVolumes <wxInt16> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxUint16> >& v) { mGetVolumes <wxUint16> (v); } 
	virtual void GetVolumes (std::map <VolId, Volume <wxInt8> >& v) { mGetVolumes <wxInt8> (v); }
	virtual void GetVolumes (std::map <VolId, Volume <wxUint8> >& v) { mGetVolumes <wxUint8> (v); } 

    protected:
	virtual VolListType ReadVolIds(DicomFile& file);

    private :
	double GetPhaseFov() const;
	double GetRoFov() const;
	template <class T> void
	    mGetVolumes(std::map<VolId, Volume<T> >& v);

    };


    ///
    /**
     */
    class GeEpiHandler : public SeriesHandler
    {
    public:
	GeEpiHandler(const std::string& seriesUid);

    protected:
	virtual VolListType ReadVolIds(DicomFile& file);

    };


    ///
    /**
     */
    class GeDtiRbHandler : public SeriesHandler
    {
    public:
	GeDtiRbHandler(const std::string& seriesUid);
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const { return true; }

    protected:
	virtual VolListType ReadVolIds(DicomFile& file);

    };


    ///
    /**
     */
    class GeDti2Handler : public GeEpiHandler
    {
    public:
	GeDti2Handler(const std::string& seriesUid);
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const { return true; }

    protected:
	virtual VolListType ReadVolIds(DicomFile& file);

    private:
	std::vector<std::vector<double> > gradients;

    };


    ///
    /**
     */
    class EnhancedMrHandler : public SeriesHandler
    {
    public:
	EnhancedMrHandler(const std::string& seriesUid);
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const;
	virtual std::vector<std::string> GetStringInfo();
	virtual std::vector<double> GetVoxelSize();
	virtual int GetRescaleSlopeAndIntercept(DicomFile& dfile, double& slope, double& intercept, int frame = 0) const;

    protected:
	virtual VolListType ReadVolIds(DicomFile& file);
	virtual std::string GetImagePositionPatient(DicomFile& dfile, int frame);

    private:
	std::vector<std::string> ipps;
	std::vector<double> slopes;
	std::vector<double> intercepts;
	std::map<VolId, std::string> bvecMap;
	std::map<VolId, std::string> bvalMap;
	std::map<VolId, int> volFrameMap;
	std::string pixelSpacing;

    };


    ///
    /**
     */
    class AchievaDtiHandler : public SeriesHandler
    {
    public:
	AchievaDtiHandler(const std::string& seriesUid);
	virtual GradientInfo GetGradientInfo();
	virtual bool IsDti() const { return true; }

    protected:
	virtual VolListType ReadVolIds(DicomFile& file);

    private:
	std::vector<std::vector<double> > gradients;

    };

} // end namespace jcs

#include "SeriesHandler.txx"

#endif
