/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

/*---------------------------------------------------------------
	APPLICATION: Rao-Blackwellized Particle Filter SLAM
	FILE: rbpf-slam.cpp
	AUTHOR: Jose Luis Blanco Claraco <jlblanco@ctima.uma.es>

	See README.txt for instructions.
  ---------------------------------------------------------------*/

#include <mrpt/core.h>

using namespace mrpt;
using namespace mrpt::slam;
using namespace mrpt::opengl;
using namespace mrpt::gui;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;

/*****************************************************
			Config params
 *****************************************************/
std::string				INI_FILENAME;
std::string				RAWLOG_FILE;
unsigned int			rawlog_offset;
std::string				OUT_DIR_STD;
const char				*OUT_DIR;
int						LOG_FREQUENCY;
bool					GENERATE_LOG_JOINT_H;
bool					GENERATE_LOG_INFO;
bool					SAVE_POSE_LOG;
bool					SAVE_MAP_IMAGES;
bool					SAVE_3D_SCENE;
bool					SAVE_ENOSE_READINGS;
bool					CAMERA_3DSCENE_FOLLOWS_ROBOT;

bool					SHOW_PROGRESS_IN_WINDOW;
int                     SHOW_PROGRESS_IN_WINDOW_DELAY_MS;
int PROGRESS_WINDOW_WIDTH=600, PROGRESS_WINDOW_HEIGHT=500;

// Forward declaration.
void MapBuilding_RBPF();


// ------------------------------------------------------
//						MAIN
// ------------------------------------------------------
int main(int argc, char **argv)
{
	try
	{
		printf(" rbpf-slam version 0.3 - Part of the MRPT\n");
		printf(" MRPT C++ Library: %s - BUILD DATE %s\n", MRPT_getVersion().c_str(), MRPT_getCompilationDate().c_str());
		printf("-------------------------------------------------------------------\n");

		// Process arguments:
		if (argc<2)
		{
			printf("Usage: %s <config_file.ini>\n\n",argv[0]);
			mrpt::system::pause();
			return -1;
		}

		INI_FILENAME = std::string( argv[1] );
		ASSERT_(fileExists(INI_FILENAME));

		CConfigFile		iniFile( INI_FILENAME );

		// ------------------------------------------
		//			Load config from file:
		// ------------------------------------------
		RAWLOG_FILE			 = iniFile.read_string("MappingApplication","rawlog_file","",  /*Force existence:*/ true);
		rawlog_offset		 = iniFile.read_int("MappingApplication","rawlog_offset",0,  /*Force existence:*/ true);
		OUT_DIR_STD			 = iniFile.read_string("MappingApplication","logOutput_dir","log_out",  /*Force existence:*/ true);
		LOG_FREQUENCY		 = iniFile.read_int("MappingApplication","LOG_FREQUENCY",5,  /*Force existence:*/ true);
		GENERATE_LOG_JOINT_H = iniFile.read_bool("MappingApplication","GENERATE_LOG_JOINT_H", false,  /*Force existence:*/ true);
		GENERATE_LOG_INFO    = iniFile.read_bool("MappingApplication","GENERATE_LOG_INFO", false,  /*Force existence:*/ true);
		SAVE_POSE_LOG		 = iniFile.read_bool("MappingApplication","SAVE_POSE_LOG", false,  /*Force existence:*/ true);
		SAVE_MAP_IMAGES		 = iniFile.read_bool("MappingApplication","SAVE_MAP_IMAGES", false,  /*Force existence:*/ true);
		SAVE_3D_SCENE        = iniFile.read_bool("MappingApplication","SAVE_3D_SCENE", false,  /*Force existence:*/ true);
		SAVE_ENOSE_READINGS  = iniFile.read_bool("MappingApplication","SAVE_ENOSE_READINGS", false,  /*Force existence:*/ true);
		CAMERA_3DSCENE_FOLLOWS_ROBOT = iniFile.read_bool("MappingApplication","CAMERA_3DSCENE_FOLLOWS_ROBOT", true,  /*Force existence:*/ true);
		SHOW_PROGRESS_IN_WINDOW = iniFile.read_bool("MappingApplication","SHOW_PROGRESS_IN_WINDOW", false,  /*Force existence:*/ false);
		SHOW_PROGRESS_IN_WINDOW_DELAY_MS = iniFile.read_int("MappingApplication","SHOW_PROGRESS_IN_WINDOW_DELAY_MS",1, /*Force existence:*/ false);

		MRPT_LOAD_CONFIG_VAR(PROGRESS_WINDOW_WIDTH, int,  iniFile, "MappingApplication");
		MRPT_LOAD_CONFIG_VAR(PROGRESS_WINDOW_HEIGHT, int,  iniFile, "MappingApplication");


		// easier!
		OUT_DIR = OUT_DIR_STD.c_str();

		// Print params:
		printf("Running with the following parameters:\n");
		printf(" RAWLOG file:'%s'\n", RAWLOG_FILE.c_str());
		printf(" Output directory:\t\t\t'%s'\n",OUT_DIR);
		printf(" Log record freq:\t\t\t%u\n",LOG_FREQUENCY);
		printf("  GENERATE_LOG_JOINT_H:\t\t\t%c\n", GENERATE_LOG_JOINT_H ? 'Y':'N');
		printf("  GENERATE_LOG_INFO:\t\t\t%c\n", GENERATE_LOG_INFO ? 'Y':'N');
		printf("  SAVE_MAP_IMAGES:\t\t\t%c\n", SAVE_MAP_IMAGES ? 'Y':'N');
		printf("  SAVE_3D_SCENE:\t\t\t%c\n", SAVE_3D_SCENE ? 'Y':'N');
		printf("  SAVE_POSE_LOG:\t\t\t%c\n", SAVE_POSE_LOG ? 'Y':'N');
		printf("  SAVE_ENOSE_READINGS:\t\t%c\n",SAVE_ENOSE_READINGS ? 'Y':'N');
		printf("  CAMERA_3DSCENE_FOLLOWS_ROBOT:\t%c\n",CAMERA_3DSCENE_FOLLOWS_ROBOT ? 'Y':'N');
		printf("  SHOW_PROGRESS_IN_WINDOW:\t%c\n",SHOW_PROGRESS_IN_WINDOW ? 'Y':'N');
		printf("\n");

		// Checks:
		ASSERT_(RAWLOG_FILE.size()>0);
		ASSERT_(fileExists(RAWLOG_FILE));

		// Set relative path for externally-stored images in rawlogs:
		string	rawlog_images_path = extractFileDirectory( RAWLOG_FILE );
		rawlog_images_path+="/Images";
		CMRPTImage::IMAGES_PATH_BASE = rawlog_images_path;		// Set it.

		// Run:
		MapBuilding_RBPF();

		//pause();
		return 0;
	} catch (std::exception &e)
	{
		std::cerr << e.what() << std::endl << "Program finished for an exception!!" << std::endl;
		mrpt::system::pause();
		return -1;
	}
	catch (...)
	{
		std::cerr << "Untyped exception!!" << std::endl;
		mrpt::system::pause();
		return -1;
	}
}

// ------------------------------------------------------
//					MapBuilding RBPF
// ------------------------------------------------------
void MapBuilding_RBPF()
{
	MRPT_TRY_START

	CTicTac								tictac,tictacGlobal,tictac_JH;
	int									step = 0;
	FILE								*f_log,*f_info,*f_jinfo,*f_path,*f_pathOdo;
	CSensFrameProbSequence				finalMap;
	float								t_exec;
	COccupancyGridMap2D::TEntropyInfo	entropy;

	char								strFil[1000];

	size_t								rawlogEntry = 0;
	CFileGZInputStream					rawlogFile( RAWLOG_FILE );

	// ---------------------------------
	//		MapPDF opts
	// ---------------------------------
	CMetricMapBuilderRBPF::TConstructionOptions		rbpfMappingOptions;

	rbpfMappingOptions.loadFromConfigFile(CConfigFile(INI_FILENAME),"MappingApplication");
	rbpfMappingOptions.dumpToConsole();

	// ---------------------------------
	//		Constructor
	// ---------------------------------
	CMetricMapBuilderRBPF mapBuilder( rbpfMappingOptions );

	// ---------------------------------
	//   CMetricMapBuilder::TOptions
	// ---------------------------------
	mapBuilder.options.verbose					= true;
	mapBuilder.options.enableMapUpdating		= true;
    mapBuilder.options.debugForceInsertion		= false;
	mapBuilder.options.insertImagesAlways		= false;
	
	mrpt::random::Randomize();
	mrpt::registerAllClasses();

	// Prepare output directory:
	// --------------------------------
	//os::sprintf(strFil,1000,"%s/*.*",OUT_DIR);
	deleteFilesInDirectory(OUT_DIR);
	createDirectory(OUT_DIR);

	string OUT_DIR_MAPS= format("%s/maps",OUT_DIR);
	string OUT_DIR_3D= format("%s/3D",OUT_DIR);

	deleteFilesInDirectory(OUT_DIR_MAPS);
	createDirectory(OUT_DIR_MAPS);

	deleteFilesInDirectory(OUT_DIR_3D);
	createDirectory(OUT_DIR_3D);


	// Open log files:
	// ----------------------------------
	os::sprintf(strFil,1000,"%s/log_times.txt",OUT_DIR);
	f_log=os::fopen(strFil,"wt");

	os::sprintf(strFil,1000,"%s/log_info.txt",OUT_DIR);
	f_info=os::fopen(strFil,"wt");

	os::sprintf(strFil,1000,"%s/log_jinfo.txt",OUT_DIR);
	f_jinfo=os::fopen(strFil,"wt");

	os::sprintf(strFil,1000,"%s/log_estimated_path.txt",OUT_DIR);
	f_path=os::fopen(strFil,"wt");

	os::sprintf(strFil,1000,"%s/log_odometry_path.txt",OUT_DIR);
	f_pathOdo=os::fopen(strFil,"wt");

	ASSERT_(f_log!=NULL);
	ASSERT_(f_info!=NULL);
	ASSERT_(f_jinfo!=NULL);
	ASSERT_(f_path!=NULL);
	ASSERT_(f_pathOdo!=NULL);

	// ----------------------------------------------------------
	//						Map Building
	// ----------------------------------------------------------
	CActionCollectionPtr					action;
	CSensoryFramePtr						observations;
	std::deque<CObservationGasSensorsPtr>	gasObservations;
	CPose3D  odoPose(0,0,0);

	CDisplayWindow3D    *win3D = NULL;

    if (SHOW_PROGRESS_IN_WINDOW)
    {
		win3D = new CDisplayWindow3D("RBPF-SLAM @ MRPT C++ Library (C) 2004-2008", PROGRESS_WINDOW_WIDTH, PROGRESS_WINDOW_HEIGHT);
		win3D->setCameraZoom(25);
		win3D->setCameraAzimuthDeg(-50);
		win3D->setCameraElevationDeg(70);
    }


	tictacGlobal.Tic();
	for (;;)
	{
		if (os::kbhit())
		{
			char c = os::getch();
			if (c==27)
				break;
		}

		// Load action/observation pair from the rawlog:
		// --------------------------------------------------
		if (! CRawlog::readActionObservationPair( rawlogFile, action, observations, rawlogEntry) )
			break; // file EOF

		if (rawlogEntry>=rawlog_offset)
		{

			if (SAVE_ENOSE_READINGS)
			{
				CObservationGasSensorsPtr gasObs = observations->getObservationByClass<CObservationGasSensors>();
				if (gasObs)
					gasObservations.push_back( gasObs );
			}


			// Update odometry:
			{
				CActionRobotMovement2DPtr act= action->getBestMovementEstimation();
				if (act)
					odoPose = odoPose + CPose3D( act->poseChange->getEstimatedPose() );
				else
				{
					CActionRobotMovement3DPtr act3D = action->getActionByClass<CActionRobotMovement3D>();
					if (act3D)
						odoPose = odoPose + act3D->poseChange.mean;
				}
			}

			// Execute:
			// ----------------------------------------
			tictac.Tic();
				mapBuilder.processActionObservation( *action, *observations );
			t_exec = tictac.Tac();
			printf("Map building executed in %.03fms\n", 1000.0f*t_exec );


			// Info log:
			// -----------
			os::fprintf(f_log,"%f %i %i\n",
				1000.0f*t_exec,
				mapBuilder.getCurrentlyBuiltMapSize(),
				mapBuilder.m_statsLastIteration.observationsInserted ? int(1):int(0)
				);

			CPose3DPDFPtr curPDFptr = mapBuilder.getCurrentPoseEstimation();
			CPose3DPDFParticles	curPDF;

			if ( IS_CLASS( curPDFptr, CPose3DPDFParticles ) )
			{
				CPose3DPDFParticlesPtr pp= CPose3DPDFParticlesPtr(curPDFptr);
				curPDF = *pp;
			}

			if (0==(step % LOG_FREQUENCY))
			{
				CMultiMetricMap		*mostLikMap = mapBuilder.mapPDF.getCurrentMostLikelyMetricMap();

				if (GENERATE_LOG_INFO)
				{

					printf("Saving info log information...");

					tictac_JH.Tic();

					ASSERT_( mapBuilder.getCurrentlyBuiltMetricMap()->m_gridMaps.size()>0 );
					COccupancyGridMap2DPtr grid = mapBuilder.getCurrentlyBuiltMetricMap()->m_gridMaps[0];
					grid->computeEntropy( entropy );

					grid->saveAsBitmapFile(format("%s/EMMI_gridmap_%03u.bmp",OUT_DIR,step));

					os::fprintf(f_info,"%f %f %f %f %lu\n",
						entropy.I,
						entropy.H,
						entropy.mean_I,
						entropy.effectiveMappedArea,
						entropy.effectiveMappedCells);
					printf("Ok\n EMI = %.04f    EMMI=%.04f (in %.03fms)\n",entropy.I, entropy.mean_I,1000.0f*tictac_JH.Tac());
				}

				// Pose log:
				// -------------
				if (SAVE_POSE_LOG)
				{
					printf("Saving pose log information...");
					curPDF.saveToTextFile( format("%s/mapbuild_posepdf_%03u.txt",OUT_DIR,step) );
					printf("Ok\n");
				}

				// Map images:
				// -------------
				if (SAVE_MAP_IMAGES)
				{
					printf("Saving map images to files...");

					//  Most likely maps:
					// ----------------------------------------
					mostLikMap->saveMetricMapRepresentationToFile( format("%s/mapbuilt_%05u_",OUT_DIR_MAPS.c_str(),step) );

					if (mostLikMap->m_gridMaps.size()>0)
					{
						CMRPTImageFloat		imgFl;
						mapBuilder.drawCurrentEstimationToImage( &imgFl );
						imgFl.saveToFile(format("%s/mapping_%05u.png",OUT_DIR,step));
					}

					printf("Ok!\n");
				}

				// Save a 3D scene view of the mapping process:
                COpenGLScenePtr scene;
				if (SAVE_3D_SCENE || SHOW_PROGRESS_IN_WINDOW)
				{
				    scene = COpenGLScenePtr( new COpenGLScene() );

					// The ground:
					mrpt::opengl::CGridPlaneXYPtr groundPlane = mrpt::opengl::CGridPlaneXY::Create(-200,200,-200,200,0,5);
					groundPlane->m_color_R = 0.4f;
					groundPlane->m_color_G = 0.4f;
					groundPlane->m_color_B = 0.4f;
					scene->insert( groundPlane );

					// The camera pointing to the current robot pose:
					if (CAMERA_3DSCENE_FOLLOWS_ROBOT)
					{
						mrpt::opengl::CCameraPtr objCam = mrpt::opengl::CCamera::Create();
						CPose3D		robotPose = curPDF.getEstimatedPose();

						objCam->m_pointingX = robotPose.x;
						objCam->m_pointingY = robotPose.y;
						objCam->m_pointingZ = robotPose.z;
						objCam->m_azimuthDeg   = -30;
						objCam->m_elevationDeg = 30;
						scene->insert( objCam );
					}
					// Draw the map(s):
					mrpt::opengl::CSetOfObjectsPtr objs = mrpt::opengl::CSetOfObjects::Create();
					mostLikMap->getAs3DObject( objs );
					scene->insert( objs );

					// Draw the robot particles:
					size_t		M = mapBuilder.mapPDF.particlesCount();
					mrpt::opengl::CSetOfLinesPtr objLines = mrpt::opengl::CSetOfLines::Create();
					objLines->m_color_B = 1;	objLines->m_color_G = 1;	objLines->m_color_R = 0;
					for (size_t i=0;i<M;i++)
					{
						std::deque<CPose3D>		path;
						mapBuilder.mapPDF.getPath(i,path);

						float	x0=0,y0=0,z0=0;
						for (size_t k=0;k<path.size();k++)
						{
							objLines->appendLine(
								x0, y0, z0+0.001,
								path[k].x, path[k].y, path[k].z+0.001 );
							x0=path[k].x;
							y0=path[k].y;
							z0=path[k].z;
						}
					}
					scene->insert( objLines );

					// An ellipsoid:
					CPose2D			lastMeanPose;
					float			minDistBtwPoses=-1;
					std::deque<CPose3D>		dummyPath;
					mapBuilder.mapPDF.getPath(0,dummyPath);
					for (int k=(int)dummyPath.size()-1;k>=0;k--)
					{
						CPose3DPDFParticles	poseParts;
						mapBuilder.mapPDF.getEstimatedPosePDFAtTime(k,poseParts);

						CPose3D		meanPose( poseParts.getEstimatedPose() );

						if ( meanPose.distanceTo(lastMeanPose)>minDistBtwPoses )
						{
							CMatrixD COV( poseParts.getEstimatedCovariance() );

							COV.setSize(3,3);

							minDistBtwPoses = 6 * sqrt(COV(0,0)+COV(1,1));

							opengl::CEllipsoidPtr objEllip = opengl::CEllipsoid::Create();
							objEllip->m_x = meanPose.x;
							objEllip->m_y = meanPose.y;
							objEllip->m_z = meanPose.z + 0.001f;
							objEllip->m_cov = COV;
							objEllip->m_2D_segments = 30;
							objEllip->m_color_R = 0;
							objEllip->m_color_G = 0;
							objEllip->m_color_B = 1;
							objEllip->m_drawSolid3D = false;
							scene->insert( objEllip );

							lastMeanPose = meanPose;
						}
					}
				} // end if show or save 3D scene->

                if (SAVE_3D_SCENE)
                {	// Save as file:
					CFileGZOutputStream(format("%s/buildingmap_%05u.3Dscene",OUT_DIR_3D.c_str(),step)) << *scene;
				}

                if (SHOW_PROGRESS_IN_WINDOW)
                {
                    COpenGLScenePtr scenePtr = win3D->get3DSceneAndLock();
                    scenePtr = scene;
                    win3D->unlockAccess3DScene();

                    win3D->forceRepaint();
                    int add_delay = SHOW_PROGRESS_IN_WINDOW_DELAY_MS - t_exec*1000;
                    if (add_delay>0)
                        sleep(add_delay);
                }
                /*else
                {
                    // Free scene:
                    delete scene;
                    scene=NULL;
                }
				*/


			// Save the weighted entropy of each map:
			// ----------------------------------------
				if (GENERATE_LOG_JOINT_H)
				{
					printf("Saving joint H...");
					tictac_JH.Tic();

					double  H_joint = mapBuilder.getCurrentJointEntropy();
					double  H_path  = mapBuilder.mapPDF.getCurrentEntropyOfPaths();
					os::fprintf(f_jinfo,"%e %e\n",H_joint,H_path);
					printf("Ok\t joint-H=%f\t(in %.03fms)\n",H_joint,1000.0f*tictac_JH.Tac());
				}



				if (SAVE_ENOSE_READINGS) /**** OUTPUT FOR AMY'S EXPERIMENT ;-) ****/
				{
					std::deque<CPose3D>		path;
					for (unsigned int p=0;p<mapBuilder.mapPDF.particlesCount();p++)
					{
						mapBuilder.mapPDF.getPath(p, path );

						FILE	*f=os::fopen(format("%s/log_pathForGasMap_part%02u_step%03u.txt",OUT_DIR,p,step).c_str(),"wt");
						size_t i,n = gasObservations.size();
						printf("Saving odour readings... %u/%u\n", (unsigned int)path.size()-1, (unsigned int)gasObservations.size() );
						ASSERT_(n == (path.size()-1));
						for (i=0;i<n;i++)
						{
							os::fprintf(f,"%u %f %f %f",
								(unsigned int)i,
								path[i+1].x,
								path[i+1].y,
								path[i+1].yaw );

							// The reading:
							//gasObservations[i]->readings[j]
							float	raw16[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
							raw16[2]=gasObservations[i]->m_readings[0].readingsVoltage[0];
							raw16[4]=gasObservations[i]->m_readings[0].readingsVoltage[1];
							raw16[5]=gasObservations[i]->m_readings[0].readingsVoltage[2];
							raw16[6]=gasObservations[i]->m_readings[0].readingsVoltage[3];

							raw16[8]=gasObservations[i]->m_readings[1].readingsVoltage[0];
							raw16[10]=gasObservations[i]->m_readings[1].readingsVoltage[1];
							raw16[12]=gasObservations[i]->m_readings[1].readingsVoltage[2];
							raw16[14]=gasObservations[i]->m_readings[1].readingsVoltage[3];

							for (int j=0;j<16;j++)	os::fprintf(f," %f", raw16[j]);
							os::fprintf(f,"\n");
						}

						os::fclose(f);
					}
				} // end if SAVE_ENOSE_READINGS

			} // end of LOG_FREQ


			// Save the memory usage:
			// ------------------------------------------------------------------
			{
				printf("Saving memory usage...");
				unsigned long	memUsage = getMemoryUsage();
				FILE		*f=os::fopen(format("%s/log_MemoryUsage.txt",OUT_DIR).c_str(),"at");
				if (f)
				{
					os::fprintf(f,"%u\t%lu\n",step,memUsage);
					os::fclose(f);
				}
				printf("Ok! (%.04fMb)\n", ((float)memUsage)/(1024*1024) );
			}

			// Save the parts stats:
			// ------------------------------------------------------------------
			{
				printf("Saving particles stats...");
				FILE *f=os::fopen(format("%s/log_ParticlesStats.txt",OUT_DIR).c_str(),"at");
				if (f)
				{
					os::fprintf(f,"%u %u %f\n",
						(unsigned int)step,
						(unsigned int)curPDF.size(),
						curPDF.ESS()
						);
					os::fclose(f);
				}
				printf("Ok!\n");
			}

			// Save the robot estimated pose for each step:
			os::fprintf(f_path,"%i %f %f %f %f %f %f\n",
				(int)rawlogEntry,
				mapBuilder.getCurrentPoseEstimation()->getEstimatedPose().x,
				mapBuilder.getCurrentPoseEstimation()->getEstimatedPose().y,
				mapBuilder.getCurrentPoseEstimation()->getEstimatedPose().z,
				mapBuilder.getCurrentPoseEstimation()->getEstimatedPose().yaw,
				mapBuilder.getCurrentPoseEstimation()->getEstimatedPose().pitch,
				mapBuilder.getCurrentPoseEstimation()->getEstimatedPose().roll );

			os::fprintf( f_pathOdo, "%i\t%f\t%f\t%f\t%f\t%f\t%f\n", step, odoPose.x, odoPose.y, odoPose.z, odoPose.yaw, odoPose.pitch, odoPose.roll );

		} // end of if "rawlog_offset"...

		step++;
		printf("\n---------------- STEP %u | RAWLOG ENTRY %u ----------------\n",step, (unsigned)rawlogEntry);

		// Free memory:
		action.clear_unique();
		observations.clear_unique();
	}; // end while

	printf("\n---------------- END!! (total time: %.03f sec) ----------------\n",tictacGlobal.Tac());

	os::fclose(f_log);
	os::fclose(f_info);
	os::fclose(f_jinfo);
	os::fclose(f_path);
	os::fclose(f_pathOdo);

	// Save map:
	mapBuilder.getCurrentlyBuiltMap(finalMap);

	CFileOutputStream		filOut(format("%s/_finalmap_.simplemap",OUT_DIR));
	filOut << finalMap;

	// Save gridmap extend (if exists):
	CMultiMetricMap		*mostLikMap = mapBuilder.mapPDF.getCurrentMostLikelyMetricMap();
	if (mostLikMap->m_gridMaps.size()>0)
	{
		CMatrix		auxMat(1,4);
		auxMat(0,0) = mostLikMap->m_gridMaps[0]->getXMin();
		auxMat(0,1) = mostLikMap->m_gridMaps[0]->getXMax();
		auxMat(0,2) = mostLikMap->m_gridMaps[0]->getYMin();
		auxMat(0,3) = mostLikMap->m_gridMaps[0]->getYMax();
		auxMat.saveToTextFile(format("%s/finalGridmapSize.txt",OUT_DIR),1);
	}

	// Save the most likely path of the particle set
	FILE *f_pathPart;

	os::sprintf(strFil,1000,"%s/most_likely_path.txt",OUT_DIR);
	f_pathPart = os::fopen(strFil,"wt");

	ASSERT_( f_pathPart != NULL );

	std::deque<CPose3D>				outPath;
	std::deque<CPose3D>::iterator	itPath;

	mapBuilder.getCurrentMostLikelyPath( outPath );

	for( itPath = outPath.begin(); itPath != outPath.end(); itPath++ )
		os::fprintf( f_pathPart, "%.3f %.3f %.3f\n", itPath->x, itPath->y, itPath->yaw);

	os::fclose(f_pathPart);

	// Free gas readings memory (if any):
	gasObservations.clear();

	// Close 3D window, if any:
	if (win3D)
	{
	    mrpt::system::pause();
	    delete win3D;
	    win3D=NULL;
	}

	MRPT_TRY_END
}
