/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CPointPDFGaussian_H
#define CPointPDFGaussian_H

#include <mrpt/poses/CPointPDF.h>
#include <mrpt/math/CMatrix.h>

namespace mrpt
{
namespace poses
{
	DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CPointPDFGaussian, CPointPDF )

	/** A gaussian distribution for 3D points. Also a method for bayesian fusion is provided.
     *
	 * \sa CPointPDF
	 */
	class MRPTDLLIMPEXP CPointPDFGaussian : public CPointPDF
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CPointPDFGaussian )

	 public:
		/** Default constructor
		  */
		CPointPDFGaussian();

		/** Constructor
		  */
		CPointPDFGaussian( const CPoint3D &init_Mean );

		/** Constructor
		  */
		CPointPDFGaussian( const CPoint3D &init_Mean, const CMatrix &init_Cov );

		/** The mean value
		 */
		CPoint3D	mean;

		/** The 3x3 covariance matrix
		 */
		CMatrixD		cov;

		 /** Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
		  */
		CPoint3D  getEstimatedPoint() const;

		 /** Returns an estimate of the pose covariance matrix (3x3 cov.matrix  for x,y,phi variables)
		  */
		CMatrixD  getEstimatedCovariance() const;

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  */
		void  copyFrom(const CPointPDF &o);

		/** Save PDF's particles to a text file, containing the 2D pose in the first line, then the covariance matrix in next 3 lines.
		 */
		void  saveToTextFile(const std::string &file) const;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object. Both the mean value and the covariance matrix are updated correctly.
		  */
		void  changeCoordinatesReference( const CPose3D &newReferenceBase );

		/** Bayesian fusion of two points gauss. distributions, then save the result in this object.
		  *  The process is as follows:<br>
		  *		- (x1,S1): Mean and variance of the p1 distribution.
		  *		- (x2,S2): Mean and variance of the p2 distribution.
		  *		- (x,S): Mean and variance of the resulting distribution.
		  *
		  *    S = (S1<sup>-1</sup> + S2<sup>-1</sup>)<sup>-1</sup>;
		  *    x = S * ( S1<sup>-1</sup>*x1 + S2<sup>-1</sup>*x2 );
		  */
		void  bayesianFusion( CPointPDFGaussian &p1, CPointPDFGaussian &p2 );

		/** Computes the "correspondence likelihood" of this PDF with another one: This is implemented as the integral from -inf to +inf of the product of both PDF.
		  * The resulting number is >=0.
		  * \sa productIntegralNormalizedWith
		  * \exception std::exception On errors like covariance matrix with null determinant, etc...
		  */
		double  productIntegralWith( CPointPDFGaussian &p);

		/** Computes the "correspondence likelihood" of this PDF with another one: This is implemented as the integral from -inf to +inf of the product of both PDF.
		  * The resulting number is in the range [0,1]
		  * \sa productIntegralWith
		  * \exception std::exception On errors like covariance matrix with null determinant, etc...
		  */
		double  productIntegralNormalizedWith( CPointPDFGaussian *p);

		/** Computes the "correspondence likelihood" of this PDF with another one: This is implemented as the integral from -inf to +inf of the product of both PDF.
		  * The resulting number is in the range [0,1].
		  * NOTE: This version ignores the "z" coordinates!!
		  * \sa productIntegralWith
		  * \exception std::exception On errors like covariance matrix with null determinant, etc...
		  */
		double  productIntegralNormalizedWith2D( CPointPDFGaussian *p);

		/** Draw a sample from the pdf.
		  */
		void drawSingleSample(CPoint3D  &outSample) const;

		/** Bayesian fusion of two point distributions (product of two distributions->new distribution), then save the result in this object (WARNING: See implementing classes to see classes that can and cannot be mixtured!)
		  * \param p1 The first distribution to fuse
		  * \param p2 The second distribution to fuse
		  * \param minMahalanobisDistToDrop If set to different of 0, the result of very separate Gaussian modes (that will result in negligible components) in SOGs will be dropped to reduce the number of modes in the output.
		  */
		void  bayesianFusion( CPointPDF &p1, CPointPDF &p2, const double &minMahalanobisDistToDrop = 0);


	}; // End of class def.


	} // End of namespace
} // End of namespace

#endif
