/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

// Minimum dependences for this example, to avoid dependency on the entire MRPT, so this program
//  can be used to test quickly changes in matrix classes!
#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/utils.h>
#include <mrpt/utils/CTicTac.h>

using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

void RandomizeMatrix(CMatrixFloat &A);

// ------------------------------------------------------
//				TestHCH
// ------------------------------------------------------
void BenchmarkMatrix_1()
{
	CTicTac	tictac;

	CMatrixTemplateNumeric<float>		C(3,3),D;

	CMatrixTemplateNumeric<double>		Cd(3,3),Dd;

	int   i, N = 1000000;

	// TEST 1: UNIT
	// ---------------------
	cout << "TEST #1: Unit matrix..." << endl; cout.flush();
	tictac.Tic();
	for (i=0;i<N;i++)
	{
		C.unit();
	}
	cout << "  Dyn.size matrix float: " << 1e6*tictac.Tac() / ((float)N) << " us" << endl; cout.flush();
	tictac.Tic();
	for (i=0;i<N;i++)
	{
		Cd.unit();
	}
	cout << "  Dyn.size matrix double: " << 1e6*tictac.Tac() / ((float)N) << " us" << endl; cout.flush();
}

void BenchmarkMatrix_2()
{
	CTicTac	tictac;

#define SIZE1 3
#define SIZE2 4

	CMatrixTemplateNumeric<float>	Adf(SIZE1,SIZE2);
	CMatrixTemplateNumeric<float>	Bdf(SIZE2,SIZE1);
	CMatrixTemplateNumeric<float>	Rdf(SIZE1,SIZE1);

	int   i, N = 100000;
	// TEST 2: MULTIPLY
	// ---------------------
	cout << "TEST #2: Multiply 5x10 x 10x5..." << endl; cout.flush();
	tictac.Tic();
	for (i=0;i<N;i++)
	{
		Rdf.multiply( Adf, Bdf );
	}
	cout << "  Dyn.size matrix float: " << 1e6*tictac.Tac() / ((float)N) << " us" << endl;

}

void BenchmarkMatrix_3()
{
	CTicTac	tictac;

	CMatrixTemplateNumeric<float>	A(15,4),B(4,4),C,R, GOOD_R;

	// Make A a random matrix:
	RandomizeMatrix(A);
	RandomizeMatrix(B);

	B = B * (~B);
	//cout << "B:" << endl << B;


	// R = A * B * At
	C = A * B;
	A = ~A;

	GOOD_R = C*A;

//	cout << "Right result:" << endl << GOOD_R << endl;


	int   i, N = 100000;
	double At;

	// TEST 3: MULTIPLY vs MULTIPLY SYM:
	// ------------------------------------
	cout << "TEST #3: Multiply vs. multiply output is symmetrical..." << endl; cout.flush();
	tictac.Tic();
	for (i=0;i<N;i++)
	{
		R.multiply(C,A);
	}
	At = 1e6*tictac.Tac() / ((float)N);
	//R.saveToTextFile("R_normal.txt");
	cout << "  Normal multiplication (float): " << At << " us" << endl;// << R << endl;

	CMatrixTemplateNumeric<double>	Rd; Rd=R;
	CMatrixTemplateNumeric<double>	Cd; Cd=C;
	CMatrixTemplateNumeric<double>	Ad; Ad=A;
	CMatrixTemplateNumeric<double>	GOOD_Rd; GOOD_Rd = C * A;

	tictac.Tic();
	for (i=0;i<N;i++)
	{
		Rd.multiply(Cd,Ad);
	}
	At = 1e6*tictac.Tac() / ((float)N);
	cout << "  Normal multiplication (double): " << At << " us" << endl;// << R << endl;

	tictac.Tic();
	for (i=0;i<N;i++)
	{
		R.multiply_result_is_symmetric(C,A);
	}
	At = 1e6*tictac.Tac() / ((float)N);
	//R.saveToTextFile("R_fast.txt");
	cout << "  Fast multiplication method (float): " << At << " us." << endl;// << R << endl;

	tictac.Tic();
	for (i=0;i<N;i++)
	{
		Rd.multiply_result_is_symmetric(Cd,Ad);
	}
	At = 1e6*tictac.Tac() / ((float)N);
	//Rd.saveToTextFile("Rd_fast.txt");
	cout << "  Fast multiplication method (double): " << At << " us." << endl << endl;
}

void BenchmarkMatrix_4()
{
	CTicTac	tictac;

	size_t  sizes[] = { 2,3,6,20,50,100,200,400 };

	for (size_t i=0;i<sizeof(sizes)/sizeof(sizes[0]);i++)
	{
		size_t M = sizes[i];

		CMatrixTemplateNumeric<float>	A(M,M),Ainv;

		RandomizeMatrix(A);

		size_t   k, N = 10;
		double At;

		// TEST 4: MATRIX INVERSION
		// ------------------------------------
		cout << "TEST #4: " << M << "x" << M << " matrix inversion..." << endl; cout.flush();
		tictac.Tic();
		for (k=0;k<N;k++)
		{
			Ainv = A.inv();
		}
		At = 1e6*tictac.Tac() / ((float)N);
		cout << "  (float): " << At << " us" << endl;

		CMatrixTemplateNumeric<double>	Ad,Adinv;
		Ad=A;

		tictac.Tic();
		for (k=0;k<N;k++)
		{
			Adinv = Ad.inv();
		}
		At = 1e6*tictac.Tac() / ((float)N);
		cout << "  (double): " << At << " us" << endl;// << R << endl;
	}

}



void RandomizeMatrix(CMatrixFloat &A)
{
	for (size_t r=0;r<A.getRowCount();r++)
		for (size_t c=0;c<A.getColCount();c++)
			A(r,c) = double(rand()) / double(RAND_MAX);
}


// ------------------------------------------------------
//						MAIN
// ------------------------------------------------------
int main()
{
	try
	{
		BenchmarkMatrix_1();
		BenchmarkMatrix_2();
		BenchmarkMatrix_3();
		BenchmarkMatrix_4();
		return 0;
	} catch (exception &e)
	{
		cout << "MRPT exception caught: " << e.what() << endl;
		return -1;
	}
	catch (...)
	{
		printf("Untyped exception!!");
		return -1;
	}
}
