/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/math/geometry.h>
#include <mrpt/math/CPolygon.h>
#include <cmath>

using namespace mrpt;
using namespace mrpt::utils;
using namespace std;


/*---------------------------------------------------------------
				distanceBetweenPoints
 ---------------------------------------------------------------*/
double  math::distanceBetweenPoints(const double & x1,const double & y1,const double & x2,const double & y2)
{
	return sqrt( square(x1-x2)+square(y1-y2) );
}

/*---------------------------------------------------------------
	Returns the minimum distance between a point and a line.
  ---------------------------------------------------------------*/
double  math::minimumDistanceFromPointToSegment(
		const double &	Px,
		const double &	Py,
		const double &	x1,
		const double &	y1,
		const double &	x2,
		const double &	y2,
		double	&out_x,
		double	&out_y)
{
	closestFromPointToSegment(Px,Py,x1,y1,x2,y2,out_x,out_y);
	return distanceBetweenPoints(Px,Py,out_x,out_y);
}

/*---------------------------------------------------------------
	Returns the minimum distance between a point and a line.
  ---------------------------------------------------------------*/
double  math::minimumDistanceFromPointToSegment(
		const double &	Px,
		const double &	Py,
		const double &	x1,
		const double &	y1,
		const double &	x2,
		const double &	y2,
		float   &out_x,
		float   &out_y)
{
	double ox,oy;
	closestFromPointToSegment(Px,Py,x1,y1,x2,y2,ox,oy);
	out_x = ox;
	out_y = oy;
	return distanceBetweenPoints(Px,Py,out_x,out_y);
}


/*---------------------------------------------------------------
	Returns the closest point to a segment
  ---------------------------------------------------------------*/
void math::closestFromPointToSegment(
		const double &	Px,
		const double &	Py,
		const double &	x1,
		const double &	y1,
		const double &	x2,
		const double &	y2,
		double	&out_x,
		double	&out_y)
{
	double	Ratio, Dx, Dy;

	if (x1==x2 && y1==y2)
	{
		out_x = x1;
		out_y = y1;
	}
	else
	{
		Dx    = x2 - x1;
		Dy    = y2 - y1;
		Ratio = ((Px - x1) * Dx + (Py - y1) * Dy) / (Dx * Dx + Dy * Dy);
		if (Ratio<0)
		{
		 out_x = x1;
		 out_y = y1;
		}
		else
		{
			if (Ratio > 1)
			{
				out_x = x2;
				out_y = y2;
			}
			else
			{
				out_x = x1 + (Ratio * Dx);
				out_y = y1 + (Ratio * Dy);
			}
		}
	}
}

/*---------------------------------------------------------------
	Returns the closest point to a line
  ---------------------------------------------------------------*/
void math::closestFromPointToLine(
		const double &	Px,
		const double &	Py,
		const double &	x1,
		const double &	y1,
		const double &	x2,
		const double &	y2,
		double	&out_x,
		double	&out_y)
{
	double	Ratio, Dx, Dy;

	if (x1==x2 && y1==y2)
	{
		out_x = x1;
		out_y = y1;
	}
	else
	{
		Dx    = x2 - x1;
		Dy    = y2 - y1;
		Ratio = ((Px - x1) * Dx + (Py - y1) * Dy) / (Dx * Dx + Dy * Dy);

		out_x = x1 + (Ratio * Dx);
		out_y = y1 + (Ratio * Dy);
	}
}

/*---------------------------------------------------------------
	Returns the sq. distance to closest point to a line
  ---------------------------------------------------------------*/
double math::closestSquareDistanceFromPointToLine(
		const double &	Px,
		const double &	Py,
		const double &	x1,
		const double &	y1,
		const double &	x2,
		const double &	y2 )
{
	double	Ratio, Dx, Dy;

	if (x1==x2 && y1==y2)
	{
		return square( Px-x1 ) + square( Py-y1 );
	}
	else
	{
		Dx    = x2 - x1;
		Dy    = y2 - y1;
		Ratio = ((Px - x1) * Dx + (Py - y1) * Dy) / (Dx * Dx + Dy * Dy);

		return square( x1 + (Ratio * Dx) - Px ) + square( y1 + (Ratio * Dy) - Py );
	}
}



/*---------------------------------------------------------------
						Intersect
  ---------------------------------------------------------------*/
bool  math::SegmentsIntersection(
			const double &	x1,const double &	y1,
			const double &	x2,const double &	y2,
			const double &	x3,const double &	y3,
			const double &	x4,const double &	y4,
			double	&ix,double  &iy)
{
	double		UpperX,UpperY,LowerX,LowerY,Ax,Bx,Cx,Ay,By,Cy,d,f,e,Ratio;

	Ax = x2 - x1;
	Bx = x3 - x4;

	if (Ax < 0)
	{
		LowerX = x2;
		UpperX = x1;
	}
	else
	{
		UpperX = x2;
		LowerX = x1;
	}

	if (Bx > 0)
	{
			if (UpperX < x4 || x3 < LowerX) return false;
	}
	else	if (UpperX < x3 || x4 < LowerX) return false;

	Ay = y2 - y1;
	By = y3 - y4;

	if (Ay < 0)
	{
		LowerY = y2;
		UpperY = y1;
	}
	else
	{
		UpperY = y2;
		LowerY = y1;
	}

	if (By > 0)
	{
			if (UpperY < y4 || y3 < LowerY) return false;
	}
	else	if (UpperY < y3 || y4 < LowerY) return false;

	Cx = x1 - x3;
	Cy = y1 - y3;
	d  = (By * Cx) - (Bx * Cy);
	f  = (Ay * Bx) - (Ax * By);

	if (f > 0)
	{
			if (d < 0 || d > f) return false;
	}
	else	if (d > 0 || d < f) return false;

	e = (Ax * Cy) - (Ay * Cx);

	if (f > 0)
	{
			if (e < 0 || e > f) return false;
	}
	else	if (e > 0 || e < f) return false;

	Ratio = (Ax * -By) - (Ay * -Bx);

	if (Ratio!=0)
	{
		Ratio = ((Cy * -Bx) - (Cx * -By)) / Ratio;
		ix    = x1 + (Ratio * Ax);
		iy    = y1 + (Ratio * Ay);
	}
	else
	{
		if ( (Ax * -Cy)==(-Cx * Ay) )
		{
			ix = x3;
			iy = y3;
		}
		else
		{
			ix = x4;
			iy = y4;
		}
	}
	return true;
}

/*---------------------------------------------------------------
						Intersect
  ---------------------------------------------------------------*/
bool  math::SegmentsIntersection(
			const double &	x1,const double &	y1,
			const double &	x2,const double &	y2,
			const double &	x3,const double &	y3,
			const double &	x4,const double &	y4,
			float &ix,float &iy)
{
	double x,y;
	bool b = SegmentsIntersection(x1,y1,x2,y2,x3,y3,x4,y4,x,y);
	ix = x;
	iy = y;
	return b;
}


/*---------------------------------------------------------------
						Intersect
  ---------------------------------------------------------------*/
bool  math::pointIntoPolygon2D(const double & px, const double & py, unsigned int polyEdges, const double *poly_xs, const double *poly_ys )
{
	unsigned int	i,j;
	bool			res = false;

	if (polyEdges<3) return res;

	j = polyEdges - 1;

	for (i=0;i<polyEdges;i++)
	{
		if ((poly_ys[i] <= py && py < poly_ys[j]) ||     // an upward crossing
			(poly_ys[j] <= py && py < poly_ys[i]) )   // a downward crossing
		{
			// compute the edge-ray intersect @ the x-coordinate
			if (px - poly_xs[i]<((poly_xs[j] - poly_xs[i]) * (py - poly_ys[i]) / (poly_ys[j] - poly_ys[i]) ))
				res =! res;
		}
		j = i;
	}

	return res;
}

/*---------------------------------------------------------------
						Intersect
  ---------------------------------------------------------------*/
double  math::distancePointToPolygon2D(const double & px, const double & py, unsigned int polyEdges, const double *poly_xs, const double *poly_ys )
{
	unsigned int	i,j;
	double			minDist = 1e20f;

	// Is the point INTO?
	if (pointIntoPolygon2D(px,py,polyEdges,poly_xs,poly_ys))
		return 0;

	// Compute the closest distance from the point to any segment:
	j = polyEdges - 1;

	for (i=0;i<polyEdges;i++)
	{
		// segment: [j]-[i]
		// ----------------------
		double	closestX,closestY;
		double d = minimumDistanceFromPointToSegment(px,py, poly_xs[j],poly_ys[j], poly_xs[i],poly_ys[i],closestX,closestY);

		minDist = min(d,minDist);

		// For next iter:
		j = i;
	}

	return minDist;
}

/*---------------------------------------------------------------
					minDistBetweenLines
 --------------------------------------------------------------- */
bool  math::minDistBetweenLines(
					const double &	p1_x, const double &	p1_y, const double & p1_z,
					const double &	p2_x, const double &	p2_y, const double & p2_z,
					const double &	p3_x, const double & p3_y, const double & p3_z,
					const double &	p4_x, const double & p4_y, const double & p4_z,
					double	&x,   double &y,   double &z,
					double	&dist)
{
	const double EPS = 1e-30f;

	double	p13_x,p13_y,p13_z;
	double	p43_x,p43_y,p43_z;
	double	p21_x,p21_y,p21_z;

	double	d1343,d4321,d1321,d4343,d2121;
	double	numer,denom;

	p13_x = p1_x - p3_x;
	p13_y = p1_y - p3_y;
	p13_z = p1_z - p3_z;

	p43_x = p4_x - p3_x;
	p43_y = p4_y - p3_y;
	p43_z = p4_z - p3_z;

	if (fabs(p43_x)  < EPS && fabs(p43_y) < EPS && fabs(p43_z)  < EPS)	return false;

	p21_x = p2_x - p1_x;
	p21_y = p2_y - p1_y;
	p21_z = p2_z - p1_z;
	if (fabs(p21_x)  < EPS && fabs(p21_y)  < EPS && fabs(p21_z)  < EPS)	return false;

	d1343 = p13_x * p43_x + p13_y * p43_y + p13_z * p43_z;
	d4321 = p43_x * p21_x + p43_y * p21_y + p43_z * p21_z;
	d1321 = p13_x * p21_x + p13_y * p21_y + p13_z * p21_z;
	d4343 = p43_x * p43_x + p43_y * p43_y + p43_z * p43_z;
	d2121 = p21_x * p21_x + p21_y * p21_y + p21_z * p21_z;

	denom = d2121 * d4343 - d4321 * d4321;
	if (fabs(denom) < EPS)	return false;

	numer = d1343 * d4321 - d1321 * d4343;

	double mua = numer / denom;
	double mub = (d1343 + d4321 * mua) / d4343;
	double pa_x, pa_y, pa_z;
	double pb_x, pb_y, pb_z;

	pa_x = p1_x + mua * p21_x;
	pa_y = p1_y + mua * p21_y;
	pa_z = p1_z + mua * p21_z;

	pb_x = p3_x + mub * p43_x;
	pb_y = p3_y + mub * p43_y;
	pb_z = p3_z + mub * p43_z;

	dist = (double) sqrt( square( pa_x - pb_x ) + square( pa_y - pb_y ) + square( pa_z - pb_z ) );

	// the mid point:
	x = 0.5*(pa_x + pb_x);
	y = 0.5*(pa_y + pb_y);
	z = 0.5*(pa_z + pb_z);

	return true;
}


/*---------------------------------------------------------------
				Rectangles Intersect
  ---------------------------------------------------------------*/
bool  math::RectanglesIntersection(
			const double &	R1_x_min,	const double &	R1_x_max,
			const double &	R1_y_min,	const double &	R1_y_max,
			const double &	R2_x_min,	const double &	R2_x_max,
			const double &	R2_y_min,	const double &	R2_y_max,
			const double &	R2_pose_x,
			const double &	R2_pose_y,
			const double &	R2_pose_phi )
{
	// Compute the rotated R2:
	// ----------------------------------------
	vector_double	xs(4),ys(4);
	double			ccos = cos(R2_pose_phi);
	double			ssin = sin(R2_pose_phi);

	xs[0] = R2_pose_x + ccos * R2_x_min - ssin * R2_y_min;
	ys[0] = R2_pose_y + ssin * R2_x_min + ccos * R2_y_min;

	xs[1] = R2_pose_x + ccos * R2_x_max - ssin * R2_y_min;
	ys[1] = R2_pose_y + ssin * R2_x_max + ccos * R2_y_min;

	xs[2] = R2_pose_x + ccos * R2_x_max - ssin * R2_y_max;
	ys[2] = R2_pose_y + ssin * R2_x_max + ccos * R2_y_max;

	xs[3] = R2_pose_x + ccos * R2_x_min - ssin * R2_y_max;
	ys[3] = R2_pose_y + ssin * R2_x_min + ccos * R2_y_max;

	// Test for one vertice being inside the other rectangle:
	// -------------------------------------------------------
	if ( R1_x_min<=xs[0] && xs[0]<=R1_x_max && R1_y_min<=ys[0] && ys[0]<=R1_y_max) return true;
	if ( R1_x_min<=xs[1] && xs[1]<=R1_x_max && R1_y_min<=ys[1] && ys[1]<=R1_y_max) return true;
	if ( R1_x_min<=xs[2] && xs[2]<=R1_x_max && R1_y_min<=ys[2] && ys[2]<=R1_y_max) return true;
	if ( R1_x_min<=xs[3] && xs[3]<=R1_x_max && R1_y_min<=ys[3] && ys[3]<=R1_y_max) return true;

	CPolygon		poly;
	poly.AddVertex( xs[0],ys[0] );
	poly.AddVertex( xs[1],ys[1] );
	poly.AddVertex( xs[2],ys[2] );
	poly.AddVertex( xs[3],ys[3] );

	if (poly.PointIntoPolygon( R1_x_min, R1_y_min )) return true;
	if (poly.PointIntoPolygon( R1_x_max, R1_y_min )) return true;
	if (poly.PointIntoPolygon( R1_x_max, R1_y_max )) return true;
	if (poly.PointIntoPolygon( R1_x_min, R1_y_max )) return true;


	// Test for intersections:
	// ----------------------------------------
	double	ix,iy;

	for (int idx=0;idx<4;idx++)
	{
		if ( math::SegmentsIntersection( R1_x_min,R1_y_min, R1_x_max,R1_y_min, xs[idx],ys[idx], xs[(idx+1)%4],ys[(idx+1)%4], ix,iy) ) return true;
		if ( math::SegmentsIntersection( R1_x_max,R1_y_min, R1_x_max,R1_y_max, xs[idx],ys[idx], xs[(idx+1)%4],ys[(idx+1)%4], ix,iy) ) return true;
		if ( math::SegmentsIntersection( R1_x_max,R1_y_max, R1_x_min,R1_y_max, xs[idx],ys[idx], xs[(idx+1)%4],ys[(idx+1)%4], ix,iy) ) return true;
		if ( math::SegmentsIntersection( R1_x_min,R1_y_max, R1_x_min,R1_y_min, xs[idx],ys[idx], xs[(idx+1)%4],ys[(idx+1)%4], ix,iy) ) return true;
	}

	// No intersections:
	return false;
}

