/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/utils/CSimpleDatabase.h>

using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;

#undef _UNICODE			// JLBC

#include "xmlparser/xmlParser.h"

#include <iostream>

// This must be added to any CSerializable class implementation file.
IMPLEMENTS_SERIALIZABLE(CSimpleDatabase, CSerializable, mrpt::utils)
IMPLEMENTS_SERIALIZABLE(CSimpleDatabaseTable, CSerializable, mrpt::utils)

/*---------------------------------------------------------------
						writeToStream
 ---------------------------------------------------------------*/
void  CSimpleDatabase::writeToStream(CStream &out, int *out_Version) const
{
	if (out_Version)
		*out_Version = 0;
	else
	{
		uint32_t	i,n;

		// Save all tables in DB:
		n = (uint32_t)tables.size();
		out << n;

		for (i=0;i<n;i++)
		{
			out << table_names[i].c_str();
			out << *tables[i];
		}
	}
}
/*---------------------------------------------------------------
						readFromStream
 ---------------------------------------------------------------*/
void  CSimpleDatabase::readFromStream(CStream &in, int version)
{
	switch (version)
	{
	case 0:
	{
		uint32_t	i,n;
		char		aux[1000];

		// Clear existing tables:
		clear();

		// Load all tables in DB:
		in >> n;

		tables.resize(n,NULL);
		table_names.resize(n);

		for (i=0;i<n;i++)
		{
			in >> aux;
			table_names[i]=aux;

			tables[i] = new CSimpleDatabaseTable();
			in >> (*tables[i]);
		}
	}
	break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/*---------------------------------------------------------------
						writeToStream
 ---------------------------------------------------------------*/
void  CSimpleDatabaseTable::writeToStream(CStream &out, int *out_Version) const
{
	if (out_Version)
		*out_Version = 0;
	else
	{
		uint32_t	row,col,nRec = (uint32_t) getRecordCount(), nFie=(uint32_t) fieldsCount();

		out << nRec << nFie;

		for (col=0;col<nFie;col++)
		{
			out << field_names[col].c_str();
		}

		for (row=0;row<nRec;row++)
		{
			for (col=0;col<nFie;col++)
			{
				out << data[row][col].c_str();
			}
		}
	}
}
/*---------------------------------------------------------------
						readFromStream
 ---------------------------------------------------------------*/
void  CSimpleDatabaseTable::readFromStream(CStream &in, int version)
{
	switch (version)
	{
	case 0:
	{
		uint32_t	row,col,nRec,nFie;
		char		str[10000];

		in >> nRec >> nFie;

		data.resize(nRec);
		field_names.resize(nFie);

		for (col=0;col<nFie;col++)
		{
			in >> str;
			field_names[col] = str;
		}

		for (row=0;row<nRec;row++)
		{
			data[row].resize(nFie);

			for (col=0;col<nFie;col++)
			{
				in >> str;
				data[row][col]=str;
			}
		}
	}
	break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/*---------------------------------------------------------------
						Constructor
 ---------------------------------------------------------------*/
CSimpleDatabase::CSimpleDatabase( )
{

}

/*---------------------------------------------------------------
						Destructor
 ---------------------------------------------------------------*/
CSimpleDatabase::~CSimpleDatabase( )
{
	clear();
}

/*---------------------------------------------------------------
						Clear the DB
 ---------------------------------------------------------------*/
void  CSimpleDatabase::clear()
{
	for (std::vector<CSimpleDatabaseTable*>::iterator it=tables.begin();it!=tables.end();it++)
		delete (*it);

	tables.clear();
	table_names.clear();
}

/*---------------------------------------------------------------
						getTable
 ---------------------------------------------------------------*/
CSimpleDatabaseTable *  CSimpleDatabase::getTable(const char *tableName)
{
	MRPT_TRY_START

	for (size_t i=0;i<table_names.size();i++)
	{
		if ( !os::_strcmpi(tableName,table_names[i].c_str()) )
			return tables[i];
	}

	THROW_EXCEPTION("Table name does not found")

	MRPT_TRY_END
}

/*---------------------------------------------------------------
						getTable
 ---------------------------------------------------------------*/
CSimpleDatabaseTable *  CSimpleDatabase::getTable(size_t tableIndex)
{
	MRPT_TRY_START

	ASSERT_(tableIndex<tablesCount() )
	return tables[tableIndex];

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					tablesCount
 ---------------------------------------------------------------*/
size_t CSimpleDatabase::tablesCount() const
{
	return tables.size();
}

/*---------------------------------------------------------------
					tablesName
 ---------------------------------------------------------------*/
string	 CSimpleDatabase::tablesName(size_t tableIndex) const
{
	MRPT_TRY_START

	ASSERT_( tableIndex<tablesCount() )
	return table_names[tableIndex];

	MRPT_TRY_END
}

/*---------------------------------------------------------------
						createTable
 ---------------------------------------------------------------*/
CSimpleDatabaseTable *  CSimpleDatabase::createTable(const string &name)
{
	CSimpleDatabaseTable *table = new CSimpleDatabaseTable();
	tables.push_back(table);
	table_names.push_back( name );
	return table;
}

/*---------------------------------------------------------------
						Constructor
 ---------------------------------------------------------------*/
CSimpleDatabaseTable::CSimpleDatabaseTable( )
{

}

/*---------------------------------------------------------------
						Destructor
 ---------------------------------------------------------------*/
CSimpleDatabaseTable::~CSimpleDatabaseTable( )
{

}

/*---------------------------------------------------------------
					fieldsCount
 ---------------------------------------------------------------*/
size_t  CSimpleDatabaseTable::fieldsCount() const
{
	return field_names.size();
}

/*---------------------------------------------------------------
						addField
 ---------------------------------------------------------------*/
void  CSimpleDatabaseTable::addField(const char *fieldName)
{
	field_names.push_back(string(fieldName));
	data.clear();
}

/*---------------------------------------------------------------
						getFieldName
 ---------------------------------------------------------------*/
string  CSimpleDatabaseTable::getFieldName(size_t fieldIndex) const
{
	MRPT_TRY_START

	ASSERT_( fieldIndex<fieldsCount() );
	return field_names[fieldIndex];

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					fieldIndex
 ---------------------------------------------------------------*/
size_t CSimpleDatabaseTable::fieldIndex(const char *fieldName) const
{
	MRPT_TRY_START

	size_t		i,n = field_names.size();

	for (i=0;i<n;i++)
		if (!os::_strcmpi(fieldName,field_names[i].c_str()))
			return (int)i;

	THROW_EXCEPTION("fieldIndex: Field name not found");

	MRPT_TRY_END
}

/*---------------------------------------------------------------
						getRecordCount
 ---------------------------------------------------------------*/
size_t CSimpleDatabaseTable::getRecordCount() const
{
	return data.size();
}

/*---------------------------------------------------------------
						get
 ---------------------------------------------------------------*/
string  CSimpleDatabaseTable::get(
    size_t      recordIndex,
    string		field ) const
{
	MRPT_TRY_START
	ASSERT_(recordIndex>=0 && recordIndex<getRecordCount());
	return data[recordIndex][fieldIndex(field.c_str())];
	MRPT_TRY_END
}

/*---------------------------------------------------------------
						get
 ---------------------------------------------------------------*/
string  CSimpleDatabaseTable::get(
    size_t			recordIndex,
    size_t          fieldIndex ) const
{
	MRPT_TRY_START
	ASSERT_(recordIndex<getRecordCount());
	ASSERT_(fieldIndex<fieldsCount() );
	return data[recordIndex][fieldIndex];
	MRPT_TRY_END
}

/*---------------------------------------------------------------
						set
 ---------------------------------------------------------------*/
void  CSimpleDatabaseTable::set(
    size_t      recordIndex,
    string		field,
    string		value)
{
	MRPT_TRY_START

	ASSERT_(recordIndex<getRecordCount());
	data[recordIndex][fieldIndex(field.c_str())]=value;

	MRPT_TRY_END
}

/*---------------------------------------------------------------
						set
 ---------------------------------------------------------------*/
void  CSimpleDatabaseTable::set(
    size_t			recordIndex,
    size_t			fieldIndex,
    string		value)
{
	MRPT_TRY_START

	ASSERT_(recordIndex<getRecordCount());
	ASSERT_(fieldIndex<fieldsCount() );
	data[recordIndex][fieldIndex]=value;

	MRPT_TRY_END
}

/*---------------------------------------------------------------
						query
 ---------------------------------------------------------------*/
int  CSimpleDatabaseTable::query(
    string		field,
    string		value ) const
{
	int		fieldInd,i,n = (uint32_t) getRecordCount();

	try
	{
		fieldInd = (uint32_t) fieldIndex(field.c_str());
	}
	catch (...)
	{
		return -1;
	}

	for (i=0;i<n;i++)
	{
		if (!os::_strcmpi(value.c_str(),data[i][fieldInd].c_str()))
			return i;
	}

	// Do not found:
	return -1;
}

/*---------------------------------------------------------------
						appendRecord
 ---------------------------------------------------------------*/
size_t CSimpleDatabaseTable::appendRecord()
{
	vector_string	new_rec;

	new_rec.resize( fieldsCount() );
	data.push_back( new_rec );

	return data.size()-1;
}


/*---------------------------------------------------------------
						saveAsXML
 ---------------------------------------------------------------*/
bool CSimpleDatabase::saveAsXML( const string &fileName ) const
{
	try
	{
		ASSERT_(table_names.size()==tables.size());

		unsigned int    i;

		// Root node:
		XMLNode  rootXml = XMLNode::createXMLTopNode("CSimpleDatabase-MRPT-Object");

		// For each table:
		std::vector<CSimpleDatabaseTable*>::const_iterator    it;
		vector_string::const_iterator                         itName;
		for (it=tables.begin(),itName=table_names.begin();it!=tables.end();it++,itName++)
		{
			CSimpleDatabaseTable    *t = (*it);
			XMLNode tabNod = rootXml.addChild("table");
			tabNod.addAttribute( "name", itName->c_str() );

			// Add field descriptions:
			// ------------------------
			size_t  nFields = t->fieldsCount();
			size_t  nRecs   = t->getRecordCount();

			XMLNode fNod = tabNod.addChild("fields");
			for (i=0;i<nFields;i++)
				fNod.addChild( t->getFieldName(i).c_str() );

			// Add record contents:
			// ------------------------
			for (i=0;i<nRecs;i++)
			{
				XMLNode recNod = tabNod.addChild("record");
				for (size_t j=0;j<nFields;j++)
				{
					XMLNode recContent = recNod.addChild( t->getFieldName(j).c_str() );
					recContent.addText( t->get(i,j).c_str() );
				}
			}

		} // end for each table.

		rootXml.writeToFile( fileName.c_str() );

		return true; // Ok
	}
	catch (exception &e)
	{
		cerr << "[CSimpleDatabase::saveAsXML] Exception ignored:" << endl << e.what() << endl;
		return false;   // Errors found
	}
	catch (...)
	{
		return false;   // Errors found
	}
}


/*---------------------------------------------------------------
						loadFromXML
 ---------------------------------------------------------------*/
bool CSimpleDatabase::loadFromXML( const string &fileName )
{
	try
	{
		XMLResults 	results;
		XMLNode 	root = XMLNode::parseFile( fileName.c_str(), NULL, &results );

		if (results.error != eXMLErrorNone)
		{
			cerr << "[CSimpleDatabase::loadFromXML] Error loading XML file: " <<
					XMLNode::getError( results.error ) << " at line " << results.nLine << ":" << results.nColumn << endl;
			return false;
		}

		root = root.getChildNode("CSimpleDatabase-MRPT-Object");
		if (root.isEmpty())
		{
			cerr << "[CSimpleDatabase::loadFromXML] Loaded XML file does not have a 'CSimpleDatabase-MRPT-Object' tag";
			return false;
		}

		// Clear previous contents:
		clear();

		// Get tables:
		size_t i,j, nTables = root.nChildNode("table");
		for (i=0;i<nTables;i++)
		{
			XMLNode tabNod = root.getChildNode("table",(int)i);
			ASSERT_(!tabNod.isEmpty())

			// Create table:
			CSimpleDatabaseTable *t = createTable( tabNod.getAttribute("name") );

			// Create fields:
			XMLNode fNod = tabNod.getChildNode("fields");
			ASSERT_(!fNod.isEmpty())

			size_t  nFields = fNod.nChildNode();
			for (j=0;j<nFields;j++)
			{
				t->addField( fNod.getChildNode((int)j).getName() );
			} // end for each field

			// Add record data:
			size_t nRecs = tabNod.nChildNode("record");
			for (size_t k=0;k<nRecs;k++)
			{
				size_t recIdx = t->appendRecord();

				XMLNode recNod = tabNod.getChildNode("record",(int)k);
				ASSERT_(!recNod.isEmpty())

				for (j=0;j<nFields;j++)
				{
					t->set(recIdx,j, recNod.getChildNode(t->getFieldName(j).c_str() ).getText() );
				}

			} // end for each record

		} // for each table

		return true; // Ok
	}
	catch (exception &e)
	{
		cerr << "[CSimpleDatabase::loadFromXML] Exception ignored:" << endl << e.what() << endl;
		return false;   // Errors found
	}
	catch (...)
	{
		return false;   // Errors found
	}
}
