/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef opengl_CAngularObservationMesh_H
#define opengl_CAngularObservationMesh_H
#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/opengl/CSetOfTriangles.h>
#include <mrpt/math/CMatrixTemplate.h>
#include <mrpt/math/CMatrixB.h>
#include <mrpt/utils/stl_extensions.h>
#include <mrpt/slam/CObservation2DRangeScan.h>
#include <mrpt/slam/CPointsMap.h>
#include <mrpt/core.h>
namespace mrpt	{
namespace opengl	{
	using namespace mrpt::utils;
	using namespace mrpt::slam;
	using namespace mrpt::poses;
	class MRPTDLLIMPEXP CAngularObservationMesh;
	DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE(CAngularObservationMesh,CRenderizable)
	/** A mesh built from a set of 2D laser scan observations.
	  * Each element of this set is a single scan through the yaw, given a specific pitch.
	  * Each scan has a CPose3D identifying the origin of the scan, which ideally is the
	  * same for every one of them.
	  */
	class MRPTDLLIMPEXP CAngularObservationMesh:public CRenderizable	{
		DEFINE_SERIALIZABLE(CAngularObservationMesh)
	public:
		struct TFloatRange	{
		private:
			char rangeType;	//0=initial,final,increment;1=initial,final,amount;3=aperture,amount (centered in 0);4=reference to vector
			union rd	{
				struct	{
					float initial;
					float final;
					float increment;
				}	mode0;
				struct	{
					float initial;
					float final;
					size_t amount;
				}	mode1;
				struct	{
					float aperture;
					size_t amount;
					bool negToPos;
				}	mode2;
			}	rangeData;
			TFloatRange(float a,float b,float c):rangeType(0)	{
				rangeData.mode0.initial=a;
				rangeData.mode0.final=b;
				rangeData.mode0.increment=c;
			}
			TFloatRange(float a,float b,size_t c):rangeType(1)	{
				rangeData.mode1.initial=a;
				rangeData.mode1.final=b;
				rangeData.mode1.amount=c;
			}
			TFloatRange(float a,size_t b,bool c):rangeType(2)	{
				rangeData.mode2.aperture=a;
				rangeData.mode2.amount=b;
				rangeData.mode2.negToPos=c;
			}
		public:
			inline static TFloatRange CreateFromIncrement(float initial,float final,float increment)	{
				if (increment==0) throw std::logic_error("Invalid increment value.");
				return TFloatRange(initial,final,increment);
			}
			inline static TFloatRange CreateFromAmount(float initial,float final,size_t amount)	{
				return TFloatRange(initial,final,amount);
			}
			inline static TFloatRange CreateFromAperture(float aperture,size_t amount,bool negToPos=true)	{
				return TFloatRange(aperture,amount,negToPos);
			}
			inline float aperture() const	{
				switch (rangeType)	{
					case 0:return (sign(rangeData.mode0.increment)==sign(rangeData.mode0.final-rangeData.mode0.initial))?fabs(rangeData.mode0.final-rangeData.mode0.initial):0;
					case 1:return rangeData.mode1.final-rangeData.mode1.initial;
					case 2:return rangeData.mode2.aperture;
					default:throw std::logic_error("Unknown range type.");
				}
			}
			inline float initialValue() const	{
				switch (rangeType)	{
					case 0:
					case 1:return rangeData.mode0.initial;
					case 2:return rangeData.mode2.negToPos?-rangeData.mode2.aperture/2:rangeData.mode2.aperture/2;
					default:throw std::logic_error("Unknown range type.");
				}
			}
			inline float finalValue() const	{
				switch (rangeType)	{
					case 0:return (sign(rangeData.mode0.increment)==sign(rangeData.mode0.final-rangeData.mode0.initial))?rangeData.mode0.final:rangeData.mode0.initial;
					case 1:return rangeData.mode1.final;
					case 2:return rangeData.mode2.negToPos?rangeData.mode2.aperture/2:-rangeData.mode2.aperture/2;
					default:throw std::logic_error("Unknown range type.");
				}
			}
			inline float increment() const	{
				switch (rangeType)	{
					case 0:return rangeData.mode0.increment;
					case 1:return (rangeData.mode1.final-rangeData.mode1.initial)/static_cast<float>(rangeData.mode1.amount-1);
					case 2:return rangeData.mode2.negToPos?rangeData.mode2.aperture/static_cast<float>(rangeData.mode2.amount-1):-rangeData.mode2.aperture/static_cast<float>(rangeData.mode2.amount-1);
					default:throw std::logic_error("Unknown range type.");
				}
			}
			inline size_t amount() const	{
				switch (rangeType)	{
					case 0:return (sign(rangeData.mode0.increment)==sign(rangeData.mode0.final-rangeData.mode0.initial))?1+static_cast<size_t>(ceil((rangeData.mode0.final-rangeData.mode0.initial)/rangeData.mode0.increment)):1;
					case 1:return rangeData.mode1.amount;
					case 2:return rangeData.mode2.amount;
					default:throw std::logic_error("Unknown range type.");
				}
			}
			void values(vector_float &vals) const;
			inline bool negToPos() const	{
				switch (rangeType)	{
					case 0:return sign(rangeData.mode0.increment)>0;
					case 1:return sign(rangeData.mode1.final-rangeData.mode1.initial)>0;
					case 2:return rangeData.mode2.negToPos;
					default:throw std::logic_error("Unknown range type.");
				}
			}
		};
	protected:
		void updateMesh() const;
		virtual ~CAngularObservationMesh()	{}
		mutable std::vector<CSetOfTriangles::TTriangle> triangles;
		void addTriangle(const CPoint3D &p1,const CPoint3D &p2,const CPoint3D &p3) const;
		bool mWireframe;
		mutable bool meshUpToDate;
		bool mEnableTransparency;
		mutable mrpt::math::CMatrixTemplate<CPoint3D> actualMesh;
		mutable mrpt::math::CMatrixB validityMatrix;
		vector_serializable<float> pitchBounds;	//If contains exactly two elements, then they are the bounds.
		vector_serializable<CObservation2DRangeScan> scanSet;
		CAngularObservationMesh():mWireframe(true),meshUpToDate(false),mEnableTransparency(true),actualMesh(0,0),validityMatrix(0,0),pitchBounds(),scanSet()	{}
	public:
		inline bool isWireframe() const	{
			return mWireframe;
		}
		inline void setWireframe(bool enabled)	{
			mWireframe=enabled;
		}
		inline bool isTransparencyEnabled() const	{
			return mEnableTransparency;
		}
		inline void enableTransparency(bool enabled=true)	{
			mEnableTransparency=enabled;
		}
		virtual void render() const;
		virtual bool traceRay(const mrpt::poses::CPose3D &o,float &dist) const;
		void setPitchBounds(const float initial,const float final);
		void setPitchBounds(const std::vector<float> bounds);
		void getPitchBounds(float &initial,float &final) const;
		void getPitchBounds(std::vector<float> &bounds) const;
		void getScanSet(std::vector<CObservation2DRangeScan> &scans) const;
		bool setScanSet(const std::vector<CObservation2DRangeScan> scans);
		inline static CAngularObservationMeshPtr Create()	{
			return CAngularObservationMeshPtr(new CAngularObservationMesh());
		}
		void generateSetOfTriangles(CSetOfTrianglesPtr &res) const;
		void generatePointCloud(CPointsMap *out_map) const;		//!< Returns the scanned points as a 3D point cloud. The target pointmap must be passed as a pointer to allow the use of any derived class.
		void getTracedRays(CSetOfLinesPtr &res) const;
		void getUntracedRays(CSetOfLinesPtr &res,float dist) const;
	private:
		template<class T>
		class FTrace1D	{
		protected:
			const CPose3D &initial;
			const T &e;
			vector_float &values;
			std::vector<char> &valid;
		public:
			FTrace1D(const T &s,const CPose3D &p,vector_float &v,std::vector<char> &v2):initial(p),e(s),values(v),valid(v2)	{}
			void operator()(float yaw)	{
				float dist;
				const CPose3D pNew=initial+CPose3D(0.0,0.0,0.0,yaw,0.0,0.0);
				if (e->traceRay(pNew,dist))	{
					values.push_back(dist);
					valid.push_back(1);
				}	else	{
					values.push_back(0);
					valid.push_back(0);
				}
			}
		};
		template<class T>
		class FTrace2D	{
		protected:
			const T &e;
			const CPose3D &initial;
			CAngularObservationMeshPtr &caom;
			const CAngularObservationMesh::TFloatRange &yaws;
			std::vector<CObservation2DRangeScan> &vObs;
			const CPose3D &pBase;
		public:
			FTrace2D(const T &s,const CPose3D &p,CAngularObservationMeshPtr &om,const CAngularObservationMesh::TFloatRange &y,std::vector<CObservation2DRangeScan> &obs,const CPose3D &b):e(s),initial(p),caom(om),yaws(y),vObs(obs),pBase(b)	{}
			void operator()(float pitch)	{
				vector_float yValues;
				yaws.values(yValues);
				CObservation2DRangeScan o;
				const CPose3D pNew=initial+CPose3D(0,0,0,0,pitch,0);
				vector_float values;
				std::vector<char> valid;
				size_t nY=yValues.size();
				values.reserve(nY);
				valid.reserve(nY);
				for_each(yValues.begin(),yValues.end(),FTrace1D<T>(e,pNew,values,valid));
				o.aperture=yaws.aperture();
				o.rightToLeft=yaws.negToPos();
				o.maxRange=10000;
				o.sensorPose=pNew;
				o.deltaPitch=0;
				o.scan=values;
				o.validRange=valid;
				vObs.push_back(o);
			}
		};
	public:
		//This way, any class which admits traceRay (CRenderizable, children and COpenGLScene) admits trace2DSetOfRays
		template<class T>
		static void trace2DSetOfRays(const T &e,const CPose3D &initial,CAngularObservationMeshPtr &caom,const TFloatRange &pitchs,const TFloatRange &yaws)	{
			vector_float pValues;
			pitchs.values(pValues);
			std::vector<CObservation2DRangeScan> vObs;
			vObs.reserve(pValues.size());
			for_each(pValues.begin(),pValues.end(),FTrace2D<T>(e,initial,caom,yaws,vObs,initial));
			caom->mWireframe=false;
			caom->mEnableTransparency=false;
			caom->setPitchBounds(pValues);
			caom->setScanSet(vObs);
		}
		template<class T>
		static void trace1DSetOfRays(const T &e,const CPose3D &initial,CObservation2DRangeScan &obs,const TFloatRange &yaws)	{
			vector_float yValues;
			yaws.values(yValues);
			vector_float scanValues;
			std::vector<char> valid;
			size_t nV=yaws.amount();
			scanValues.reserve(nV);
			valid.reserve(nV);
			for_each(yValues.begin(),yValues.end(),FTrace1D<T>(e,initial,scanValues,valid));
			obs.aperture=yaws.aperture();
			obs.rightToLeft=yaws.negToPos();
			obs.maxRange=10000;
			obs.sensorPose=initial;
			obs.deltaPitch=0;
			obs.scan=scanValues;
			obs.validRange=valid;
		}
	};
}
}
#endif
