/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/config.h>

#include <mrpt/gui/WxUtils.h>
#include <mrpt/utils/CMRPTImage.h>



#if MRPT_HAS_WXWIDGETS


#if MRPT_HAS_OPENCV
	// OPENCV HEADERS
	#include <cv.h>
	#include <cxcore.h>
	#include <cvaux.h>
	#include <highgui.h>
#endif

using namespace mrpt;
using namespace mrpt::gui;
using namespace mrpt::utils;
using namespace std;


//------------------------------------------------------------------------
// An auxiliary function for passing MRPT images to wxWidgets images.
//   The returned object MUST be deleted by hand!
//------------------------------------------------------------------------
wxImage * mrpt::gui::MRPTImage2wxImage( const CMRPTImage &img )
{
#if MRPT_HAS_OPENCV
	IplImage* image = static_cast<IplImage *>( img.getAsIplImage() );
	bool	free_image_at_end = false;

	// If the image is GRAYSCALE, we need to convert it into RGB, so do it manually:
	if (image->nChannels==1)
	{
		IplImage* new_image = cvCreateImage( cvSize(image->width,image->height),image->depth, 3 );
		new_image->origin = image->origin;
		cvCvtColor(image,new_image, CV_GRAY2RGB);
		image = new_image; // Use this new image instead
		free_image_at_end = true;
	}

	int options = 0;
	if( image->origin == 1 ) options |= CV_CVTIMG_FLIP;
	if( image->nChannels == 3 && image->channelSeq[0] == 'B' && image->channelSeq[2]== 'R') options |= CV_CVTIMG_SWAP_RB;
	if( options )
	{
		IplImage *the_input_img = image;

		image = cvCreateImage( cvSize(the_input_img->width,the_input_img->height),the_input_img->depth, 3 );
		cvConvertImage(the_input_img, image, options); // convert image

		if (free_image_at_end) cvReleaseImage(&the_input_img);
		free_image_at_end = true; // for "image"
	}

	int row_in_bytes = image->width * image->nChannels;
	unsigned char* data = (unsigned char*)malloc( row_in_bytes * image->height );

	// Copy row by row only if necesary:
	if (row_in_bytes != image->widthStep)
	{
		unsigned char* trg = data;
		char* src = image->imageData;
		for (int y=0;y<image->height;y++, src+=image->widthStep, trg+= row_in_bytes)
			memcpy(trg,src,row_in_bytes);
	}
	else
	{
		memcpy(data,image->imageData,row_in_bytes* image->height);
	}

	int w = image->width;
	int h = image->height;

	if (free_image_at_end)
	{
		cvReleaseImage(&image);
	}

	// create and return the object
	return new wxImage(w,h, data, false );  // memory "imgData" will be freed by the object.

#else
	int   x, y, lx = img.getWidth(), ly = img.getHeight();
	unsigned char *imgData = (unsigned char *)malloc( 3*lx*ly );
	unsigned char *imgPtr = imgData;

	if (img.isColor())
	{
		// Is COLOR
		if (img.isOriginTopLeft())
		{
			for (y=0;y<ly;y++)
			{
				for (x=0;x<lx;x++)
				{
					*(imgPtr++) = *img(x,y,2);
					*(imgPtr++) = *img(x,y,1);
					*(imgPtr++) = *img(x,y,0);
				}
			}
		}
		else
		{
			for (y=ly-1;y>=0;y--)
			{
				for (x=0;x<lx;x++)
				{
					*(imgPtr++) = *img(x,y,2);
					*(imgPtr++) = *img(x,y,1);
					*(imgPtr++) = *img(x,y,0);
				}
			}
		}
	}
	else
	{
		// Is Grayscale:
		if (img.isOriginTopLeft())
		{
			for (y=0;y<ly;y++)
			{
				for (x=0;x<lx;x++)
				{
					unsigned char c = *img(x,y);
					*(imgPtr++) = c;
					*(imgPtr++) = c;
					*(imgPtr++) = c;
				}
			}
		}
		else
		{
			for (y=ly-1;y>=0;y--)
			{
				for (x=0;x<lx;x++)
				{
					unsigned char c = *img(x,y);
					*(imgPtr++) = c;
					*(imgPtr++) = c;
					*(imgPtr++) = c;
				}
			}
		}
	}
	return new wxImage( lx, ly, imgData, false );  // memory "imgData" will be freed by the object.
#endif
}

#if MRPT_HAS_OPENCV
wxImage * mrpt::gui::IplImage2wxImage( void* img )
{
	IplImage* image = static_cast<IplImage *>(img);

	ASSERT_(image);
	ASSERT_(image->nChannels == 3);

	// require conversion ?
	int options = 0;
	if( image->origin == 1 )
		options |= CV_CVTIMG_FLIP;
	if( image->channelSeq[0] == 'B' && image->channelSeq[2]== 'R')
		options |= CV_CVTIMG_SWAP_RB;
	if( options )
		cvConvertImage(image, image, options); // convert image "in place"

	int row_bytes = image->width * image->nChannels * ((image->depth & 255)>>3);

	unsigned char* imageData = (unsigned char *)malloc(row_bytes * image->height);
	ASSERT_(imageData);

	// copy row by row only if necesary
	if (row_bytes != image->widthStep)
	{
		for (int y = 0; y < image->height; y++)
			memcpy(	(imageData + y * row_bytes),
					(image->imageData + y * image->widthStep),
					row_bytes);
	}
	else
	{
		memcpy(	imageData,
				image->imageData,
				row_bytes * image->height);
	}

	// create and return the object
	return new wxImage(image->width, image->height, imageData, false );
}
#endif

//------------------------------------------------------------------------
// Convert wxImage -> MRPTImage
//------------------------------------------------------------------------
mrpt::utils::CMRPTImage * mrpt::gui::wxImage2MRPTImage( const wxImage &img )
{
	CMRPTImage	*newImg = new CMRPTImage();

	size_t lx = img.GetWidth();
	size_t ly = img.GetHeight();

	newImg->resize(lx,ly,3,true);


	for (size_t y=0;y<ly;y++)
	{
		unsigned char *row = (*newImg)(0,y);
		for (size_t x=0;x<lx;x++)
		{
			*row++ = img.GetRed(x,y);
			*row++ = img.GetGreen(x,y);
			*row++ = img.GetBlue(x,y);
		}
	}

	return newImg;
}


#endif // MRPT_HAS_WXWIDGETS
