/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/poses/CPoint3D.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/poses/CPoint2D.h>
#include <mrpt/poses/CPose2D.h>

using namespace mrpt;
using namespace mrpt::poses;
using namespace mrpt::math;
using namespace mrpt::utils;

IMPLEMENTS_SERIALIZABLE(CPoint3D, CPoint,mrpt::poses)

/*---------------------------------------------------------------
	Constructors
  ---------------------------------------------------------------*/
CPoint3D::CPoint3D(const double& x_,const double&y_,const double&z_) : aux_HM(0,0)
{
	this->x		= x_;
	this->y		= y_;
	this->z		= z_;
	is3D = true;
}

CPoint3D::CPoint3D( const CPoint2D &p) : aux_HM(0,0)
{
	x  = p.x;
	y  = p.y;
	z  = p.z;
	is3D = true;
}

CPoint3D::CPoint3D( const CPose3D &p) : aux_HM(0,0)
{
	x  = p.x;
	y  = p.y;
	z  = p.z;
	is3D = true;
}

CPoint3D::CPoint3D( const CPose2D &p) : aux_HM(0,0)
{
	x  = p.x;
	y  = p.y;
	z  = 0;
	is3D = true;
}


/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CPoint3D::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 1;
	else
	{
		// The coordinates:
		out << x << y << z;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CPoint3D::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
		{
			float f;
			in >> f; x=f;
			in >> f; y=f;
			in >> f; z=f;
		} break;
	case 1:
		{
			// The coordinates:
			in >> x >> y >> z;
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/**  Textual output stream function.
 */
std::ostream& poses::operator << (std::ostream& o, const CPoint3D& p)
{
	o << std::fixed << std::setprecision(3) << "(" << p.x << "," << p.y << "," << p.z << ")";
	return o;
}

/*---------------------------------------------------------------
	Returns the corresponding 4x4 homogeneous
	  transformation matrix for the point(translation),
	  or pose (translation+orientation).
---------------------------------------------------------------*/
const math::CMatrixDouble& CPoint3D::getHomogeneousMatrix() const
{
	aux_HM.unit(4);
	aux_HM.set_unsafe(0,3,x);
	aux_HM.set_unsafe(1,3,y);
	aux_HM.set_unsafe(2,3,z);

	return aux_HM;
}

/*---------------------------------------------------------------
	Returns the corresponding 4x4 homogeneous
	  transformation matrix for the point(translation),
	  or pose (translation+orientation).
---------------------------------------------------------------*/
void  CPoint3D::getHomogeneousMatrix(const math::CMatrixDouble *& m) const
{
	getHomogeneousMatrix();
	m = &aux_HM;
}


/*---------------------------------------------------------------
				point3D = point3D - pose3D
  ---------------------------------------------------------------*/
CPoint3D  CPoint3D::operator - (const CPose3D& b) const
{
	CPoint3D	p;

	// JLBC: 7-FEB-2008: Why computing the whole matrix multiplication?? ;-)
	//   5.7us -> 4.1us -> 3.1us (with optimization of HM matrices by reference)
	/*CMatrixDouble B_HM( b.getHomogeneousMatrix() );
	CMatrixDouble B_INV(4,4);
	B_HM.inv_fast(B_INV);*/

	CMatrixDouble B_INV(4,4);
	b.getInverseHomogeneousMatrix( B_INV );

	p.x = B_INV.get_unsafe(0,0) * x + B_INV.get_unsafe(0,1) * y + B_INV.get_unsafe(0,2) * z + B_INV.get_unsafe(0,3);
	p.y = B_INV.get_unsafe(1,0) * x + B_INV.get_unsafe(1,1) * y + B_INV.get_unsafe(1,2) * z + B_INV.get_unsafe(1,3);
	p.z = B_INV.get_unsafe(2,0) * x + B_INV.get_unsafe(2,1) * y + B_INV.get_unsafe(2,2) * z + B_INV.get_unsafe(2,3);

	return p;
}

/*---------------------------------------------------------------
				point3D = point3D - point3D
  ---------------------------------------------------------------*/
CPoint3D  CPoint3D::operator - (const CPoint3D& b) const
{
	return CPoint3D( x-b.x, y-b.y, z-b.z );
}


/*---------------------------------------------------------------
				point3D = point3D + point3D
  ---------------------------------------------------------------*/
CPoint3D  CPoint3D::operator + (const CPoint3D& b) const
{
	return CPoint3D( x+b.x, y+b.y, z+b.z );
}

/*---------------------------------------------------------------
				pose3D = point3D + pose3D
  ---------------------------------------------------------------*/
CPose3D	CPoint3D::operator + (const CPose3D& b) const
{
	return CPose3D( x+b.x, y+b.y,z+b.z, b.yaw, b.pitch, b.roll );
}
