/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/poses/CPose2D.h>
#include <mrpt/poses/CPoint2D.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/poses/CPoint3D.h>

#include <mrpt/math/utils.h>
#include <mrpt/math/CMatrix.h>

using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace mrpt::poses;


IMPLEMENTS_SERIALIZABLE(CPose3D, CPose,mrpt::poses)

/*---------------------------------------------------------------
	Constructors
  ---------------------------------------------------------------*/
CPose3D::CPose3D(const double& x,const double&y,const double&z, const double&yaw, const double&pitch, const double&roll)
	: yaw(),pitch(),roll(), HM(0,0)
{
	is3D = true;
	setFromValues(x,y,z,yaw,pitch,roll);
}

CPose3D::CPose3D( const CPose3D &o)
	: yaw(),pitch(),roll(), HM(0,0)
{
	is3D = true;
	setFromValues(o.x,o.y,o.z,o.yaw,o.pitch,o.roll);
}

CPose3D & CPose3D::operator =( const CPose3D &o)
{
	setFromValues(o.x,o.y,o.z,o.yaw,o.pitch,o.roll);
	return *this;
}

CPose3D::CPose3D(const CPose2D &p)
	: yaw(),pitch(),roll(), HM(0,0)
{
	is3D = true;
	setFromValues(p.x,p.y,0, p.phi);
}

CPose3D::CPose3D(const CPoint3D &p)
	: yaw(),pitch(),roll(), HM(0,0)
{
	is3D = true;
	setFromValues(p.x,p.y,p.z);
}

CPose3D::CPose3D(const math::CMatrixDouble & m)
	: yaw(),pitch(),roll(), HM(m)
{
	ASSERT_(m.getRowCount()==4 && m.getColCount()==4);
	is3D = true;
	x = HM(0,3);
	y = HM(1,3);
	z = HM(2,3);
	getYawPitchRoll( yaw, pitch, roll );
}


/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CPose3D::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 1;
	else
	{
		// Just for the case the user has modified by hand (x,y,z,yaw,pitch,roll) directly:
		const_cast<CPose3D*>(this)->rebuildHomogeneousMatrix();
		// The coordinates:
		out << HM;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CPose3D::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
		{
			// The coordinates:
			CMatrix  HM2;
			in >> HM2;

			HM = HM2;

			x = HM(0,3);
			y = HM(1,3);
			z = HM(2,3);
			getYawPitchRoll( yaw, pitch, roll );
		} break;
	case 1:
		{
			// The coordinates:
			in >> HM;

			x = HM(0,3);
			y = HM(1,3);
			z = HM(2,3);
			getYawPitchRoll( yaw, pitch, roll );
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/**  Textual output stream function.
 */
std::ostream& mrpt::poses::operator << (std::ostream& o, const CPose3D& p)
{
	o << "(x,y,z,yaw,pitch,roll)=(" << std::fixed << std::setprecision(4) << p.x << "," << p.y << "," << p.z <<  ","
		<< std::setprecision(2) << RAD2DEG(p.yaw) << "deg," << RAD2DEG(p.pitch) << "deg," << RAD2DEG(p.roll) << "deg)";
	return o;
}

/*---------------------------------------------------------------
	Returns the corresponding 4x4 homogeneous
	  transformation matrix for the point(translation),
	  or pose (translation+orientation).
---------------------------------------------------------------*/
const math::CMatrixDouble&  CPose3D::getHomogeneousMatrix() const
{
	const_cast<CPose3D&>(*this).rebuildHomogeneousMatrix();
	return HM;
}

void  CPose3D::getHomogeneousMatrix(const math::CMatrixDouble *& ptrHM ) const
{
	const_cast<CPose3D&>(*this).rebuildHomogeneousMatrix();
	ptrHM = &HM;
}


/*---------------------------------------------------------------
				normalizeAngles
---------------------------------------------------------------*/
void  CPose3D::normalizeAngles()
{
	setFromValues(x,y,z,yaw,pitch,roll);
}


/*---------------------------------------------------------------
 Set the pose from 3D point and yaw/pitch/roll angles, in radians.
---------------------------------------------------------------*/
void  CPose3D::setFromValues(
	const double		&x0,
	const double		&y0,
	const double		&z0,
	const double		&yaw,
	const double		&pitch,
	const double		&roll)
{
	x = x0;
	y = y0;
	z = z0;
	this->yaw = yaw;
	this->pitch = pitch;
	this->roll = roll;

	rebuildHomogeneousMatrix();
}

/*---------------------------------------------------------------
 Set the pose from 3D point and yaw/pitch/roll angles, in radians.
---------------------------------------------------------------*/
void  CPose3D::rebuildHomogeneousMatrix()
{
	HM.unit(4);	// Unit diagonal matrix

	HM.set_unsafe(0,3,x);
	HM.set_unsafe(1,3,y);
	HM.set_unsafe(2,3,z);

	const double	cy = cos(yaw);
	const double	sy = sin(yaw);
	const double	cp = cos(pitch);
	const double	sp = sin(pitch);
	const double	cr = cos(roll);
	const double	sr = sin(roll);

	HM.set_unsafe(0,0, cy*cp);	HM.set_unsafe(0,1, cy*sp*sr-sy*cr);		HM.set_unsafe(0,2, cy*sp*cr+sy*sr);
	HM.set_unsafe(1,0, sy*cp);	HM.set_unsafe(1,1, sy*sp*sr+cy*cr);		HM.set_unsafe(1,2, sy*sp*cr-cy*sr);
	HM.set_unsafe(2,0, -sp);	HM.set_unsafe(2,1, cp*sr);				HM.set_unsafe(2,2, cp*cr);
}


/*---------------------------------------------------------------
		Scalar multiplication.
---------------------------------------------------------------*/
void CPose3D::operator *=(const double & s)
{
	x*=s;
	y*=s;
	z*=s;
	yaw*=s;
	pitch*=s;
	roll*=s;
	setFromValues(x,y,z,yaw,pitch,roll);
}

/*---------------------------------------------------------------
		Scalar multiplication.
---------------------------------------------------------------*/
void  CPose3D::getYawPitchRoll( double &yaw, double &pitch, double &roll )
{
	ASSERT_(HM.getRowCount()==4 && HM.getColCount()==4);

	// Pitch is in the range [-pi/2, pi/2 ], so this calculation is enough:
	pitch = asin( - HM.get_unsafe(2,0) );

	// Roll:
	roll = atan2( HM.get_unsafe(2,1), HM.get_unsafe(2,2) );

	// Yaw:
	yaw = atan2( HM.get_unsafe(1,0), HM.get_unsafe(0,0) );
}


/*---------------------------------------------------------------
		sphericalCoordinates
---------------------------------------------------------------*/
void CPose3D::sphericalCoordinates(
    const CPoint3D &point,
    double &out_range,
    double &out_yaw,
    double &out_pitch ) const
{
    // Pass to coordinates as seen from this 6D pose:
    CPoint3D local( point - *this );

    // Range:
    out_range = local.norm();

    // Yaw:
    if (local.y!=0 || local.x!=0)
         out_yaw = atan2(local.y,local.x);
    else out_yaw = 0;

    // Pitch:
    if (out_range!=0)
         out_pitch = -asin( local.z / out_range );
    else out_pitch = 0;

}


/*---------------------------------------------------------------
		addComponents
---------------------------------------------------------------*/
void CPose3D::addComponents(const CPose3D &p)
{
	x+=p.x;
	y+=p.y;
	z+=p.z;
	yaw+=p.yaw;
	pitch+=p.pitch;
	roll+=p.roll;
}


/*---------------------------------------------------------------
		distanceEuclidean6D
---------------------------------------------------------------*/
double CPose3D::distanceEuclidean6D( const CPose3D &o ) const
{
	return sqrt(
		square( o.x - x ) +
		square( o.y - y ) +
		square( o.z - z ) +
		square( wrapToPi( o.yaw - yaw ) ) +
		square( wrapToPi( o.pitch - pitch ) ) +
		square( wrapToPi( o.roll - roll ) ) );
}


/*---------------------------------------------------------------
		composePoint
---------------------------------------------------------------*/
void CPose3D::composePoint(double lx,double ly,double lz, double &gx, double &gy, double &gz) const
{
	const double	cy = cos(yaw);
	const double	sy = sin(yaw);
	const double	cp = cos(pitch);
	const double	sp = sin(pitch);
	const double	cr = cos(roll);
	const double	sr = sin(roll);

	const double H11 = cy*cp;
	const double H12 = cy*sp*sr-sy*cr;
	const double H13 = cy*sp*cr+sy*sr;

	const double H21 = sy*cp;
	const double H22 = sy*sp*sr+cy*cr;
	const double H23 = sy*sp*cr-cy*sr;

	const double H31 = -sp;
	const double H32 = cp*sr;
	const double H33 = cp*cr;

	gx = H11*lx + H12*ly + H13*lz + x;
	gy = H21*lx + H22*ly + H23*lz + y;
	gz = H31*lx + H32*ly + H33*lz + z;
}

/*---------------------------------------------------------------
		composePoint
---------------------------------------------------------------*/
void CPose3D::composePoint(float lx,float ly,float lz, float &gx, float &gy, float &gz) const
{
	const double	cy = cos(yaw);
	const double	sy = sin(yaw);
	const double	cp = cos(pitch);
	const double	sp = sin(pitch);
	const double	cr = cos(roll);
	const double	sr = sin(roll);

	const double H11 = cy*cp;
	const double H12 = cy*sp*sr-sy*cr;
	const double H13 = cy*sp*cr+sy*sr;

	const double H21 = sy*cp;
	const double H22 = sy*sp*sr+cy*cr;
	const double H23 = sy*sp*cr-cy*sr;

	const double H31 = -sp;
	const double H32 = cp*sr;
	const double H33 = cp*cr;

	gx = static_cast<float>( H11*lx + H12*ly + H13*lz + x );
	gy = static_cast<float>( H21*lx + H22*ly + H23*lz + y );
	gz = static_cast<float>( H31*lx + H32*ly + H33*lz + z );
}

/*---------------------------------------------------------------
		getAsVector
---------------------------------------------------------------*/
vector_double CPose3D::getAsVector() const
{
	vector_double	r(6);
	r[0]=x;
	r[1]=y;
	r[2]=z;
	r[3]=yaw;
	r[4]=pitch;
	r[5]=roll;
	return r;
}
