/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 



#include <mrpt/slam/CLandmark.h>
#include <mrpt/slam/CObservation.h>
#include <mrpt/system/os.h>

using namespace mrpt::slam; using namespace mrpt::utils; using namespace mrpt::poses;

IMPLEMENTS_SERIALIZABLE(CLandmark, CSerializable,mrpt::slam)

// Initialization:
CLandmark::TLandmarkID	CLandmark::m_counterIDs= static_cast<CLandmark::TLandmarkID>(0);

/*---------------------------------------------------------------
						Default constructor
  ---------------------------------------------------------------*/
CLandmark::CLandmark( ) :
	pose_mean_x(),pose_mean_y(),pose_mean_z(),
	pose_cov_11(),pose_cov_22(),pose_cov_33(),pose_cov_12(),pose_cov_13(),pose_cov_23(),
	normal_x(),normal_y(),normal_z(),
	type( glBeacon ),
	ID( INVALID_LANDMARK_ID ),
    timestampLastSeen( INVALID_TIMESTAMP ),
	seenTimesCount(0),
	descriptor1(),
	descriptor2()
{
}

/*---------------------------------------------------------------
						Destructor
  ---------------------------------------------------------------*/
CLandmark::~CLandmark()
{

}

/*---------------------------------------------------------------
						Destructor
  ---------------------------------------------------------------*/
void 	CLandmark::getPose( CPointPDFGaussian	*pose ) const
{
	pose->mean.x = pose_mean_x;
	pose->mean.y = pose_mean_y;
	pose->mean.z = pose_mean_z;

	pose->cov(0,0) = pose_cov_11;
	pose->cov(1,1) = pose_cov_22;
	pose->cov(2,2) = pose_cov_33;

	pose->cov(0,1) = pose->cov(1,0) = pose_cov_12;
	pose->cov(0,2) = pose->cov(2,0) = pose_cov_13;

	pose->cov(1,2) = pose->cov(2,1) = pose_cov_23;
}

/*---------------------------------------------------------------
						Destructor
  ---------------------------------------------------------------*/
void 	CLandmark::setPose( CPointPDFGaussian	*pose )
{
	pose_mean_x = pose->mean.x;
	pose_mean_y = pose->mean.y;
	pose_mean_z = pose->mean.z;

	pose_cov_11 = pose->cov(0,0);
	pose_cov_22 = pose->cov(1,1);
	pose_cov_33 = pose->cov(2,2);
	pose_cov_12 = pose->cov(0,1);
	pose_cov_13 = pose->cov(0,2);
    pose_cov_23 = pose->cov(1,2);
}

/*---------------------------------------------------------------
					writeToStream
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CLandmark::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 3;
	else
	{
		out << pose_mean_x << pose_mean_y << pose_mean_z;
		out << pose_cov_11 << pose_cov_22 << pose_cov_33;
		out << pose_cov_12 << pose_cov_13 << pose_cov_23;

		out << normal_x << normal_y << normal_z;

		uint32_t	iType = type;
		out << iType;

		switch (type)
		{
		case vlColor:
			{
			uint8_t	r,g,b;
			r = descriptor1[0];
			g = descriptor1[1];
			b = descriptor1[2];
			out << r << g << b;
			}
			break;
		default:
			{
			uint32_t n = descriptor1.size();
			out << n;
			out.WriteBuffer( &descriptor1[0], sizeof(descriptor1[0]) * n );

			n = descriptor2.size();
			out << n;
			out.WriteBuffer( &descriptor2[0], sizeof(descriptor2[0]) * n );
			}
			break;
		}

		// v2:
		out << timestampLastSeen << seenTimesCount;

        // v3:
		out << ID;

	}
}

/*---------------------------------------------------------------
					readFromStream
   Implements the reading from a CStream capability of
      CSerializable objects
  ---------------------------------------------------------------*/
void  CLandmark::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
	case 1:
	case 2:
	case 3:
		{
            uint32_t	iType;

			in >> pose_mean_x >> pose_mean_y >> pose_mean_z;
			in >> pose_cov_11 >> pose_cov_22 >> pose_cov_33;
			in >> pose_cov_12 >> pose_cov_13 >> pose_cov_23;

			in >> normal_x >> normal_y >> normal_z;

			in >> iType; type= static_cast<TLandmarkType>(iType);

			switch (type)
			{
			case vlColor:
				{
                descriptor1.resize(3);
				uint8_t	r,g,b;
                in >> r >> g >> b;
				descriptor1[0] = r;
				descriptor1[1] = g;
				descriptor1[2] = b;
				}
				break;
			default:
				if (version>=1)
				{
					// Two descriptor vectors: 1 and 2:
					uint32_t n;
					in >> n;
					descriptor1.resize(n);
					in.ReadBuffer( &descriptor1[0], sizeof(descriptor1[0]) * n );
					in >> n;
					descriptor2.resize(n);
					in.ReadBuffer( &descriptor2[0], sizeof(descriptor2[0]) * n );
				}
				else
				{
					// Backward-compatibility with only 1 float descriptor:
					uint32_t n;
					vector_float	aux;
					vector_float::iterator					itSrc;
					std::vector<unsigned char>::iterator	itDest;

					in >> n;
					aux.resize(n);
					in.ReadBuffer( &aux[0], sizeof(aux[0]) * n );

					descriptor1.resize(n);
					for (itSrc=aux.begin(), itDest=descriptor1.begin(); itSrc!=aux.end(); itSrc++, itDest++)
					{
						*itDest = static_cast<unsigned char>(*itSrc);
					}
				}

				// v2:
				if (version>=2)
				{
					in >> timestampLastSeen >> seenTimesCount;
				}
				else
				{
					// Default:
					timestampLastSeen = 0;
					seenTimesCount    = 1;
				}

				// v3:
				if (version>=3)
				{
					in >> ID;
				}
				else
				{
					ID = ( ++m_counterIDs ) ;
				}

					break;

			};

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}


/*---------------------------------------------------------------
					setDescriptorFromMatrix
  ---------------------------------------------------------------*/
void  CLandmark::setDescriptorFromMatrix( CMatrix &m )
{
	size_t	r,c, nRow = m.getRowCount(), nCol = m.getColCount();

	ASSERT_(m.getRowCount()<256);
	ASSERT_(m.getColCount()<256);

	// The size:
	descriptor1.resize(2);
	descriptor1[0] = static_cast<unsigned char>( m.getRowCount() );
	descriptor1[1] = static_cast<unsigned char>( m.getColCount() );

	// The contents:
	descriptor2.resize( nRow * nCol );
	for (r=0;r<nRow;r++)
		for (c=0;c<nCol;c++)
			descriptor2[c+r*nCol] = m(r,c);
}

/*---------------------------------------------------------------
					getDescriptorAsMatrix
  ---------------------------------------------------------------*/
void  CLandmark::getDescriptorAsMatrix( CMatrix &m )
{
	ASSERT_(descriptor1.size()==2);

	unsigned int	r,c, nRow = descriptor1[0], nCol = descriptor1[1];

	// The size:
	m.setSize(nRow,nCol);

	// The contents:
	for (r=0;r<nRow;r++)
		for (c=0;c<nCol;c++)
			m(r,c) = descriptor2[c+r*nCol];

}

/*---------------------------------------------------------------
					getDescriptorAsMatrix
  ---------------------------------------------------------------*/
void  CLandmark::setDescriptorFromBeaconID( unsigned int beaconID )
{
	descriptor1.resize( sizeof(unsigned int) );

	* reinterpret_cast<unsigned int*>( &descriptor1[0] ) = beaconID;
}

/*---------------------------------------------------------------
					getDescriptorAsMatrix
  ---------------------------------------------------------------*/
unsigned int  CLandmark::getDescriptorAsBeaconID( ) const
{
	ASSERT_( descriptor1.size() == sizeof(unsigned int) );

	return * reinterpret_cast<const unsigned int*>( &descriptor1[0] );
}
