/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/slam/CObservationGPS.h>
#include <mrpt/utils/CStdOutStream.h>

using namespace std;
using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::math;

// This must be added to any CSerializable class implementation file.
IMPLEMENTS_SERIALIZABLE(CObservationGPS, CObservation,mrpt::slam)

CStdOutStream	gps_my_cout;

/** Constructor
 */
CObservationGPS::CObservationGPS( ) :
	sensorPose(),
	has_GGA_datum (false),
	has_RMC_datum (false),
	GGA_datum(),
	RMC_datum()
{
}

/*---------------------------------------------------------------
  Implements the writing to a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CObservationGPS::writeToStream(CStream &out, int *version) const
{
	MRPT_UNUSED_PARAM(out);
	if (version)
		*version = 4;
	else
	{
		out << timestamp;

		out << has_GGA_datum;
		if (has_GGA_datum)
		{
			// OLD Version 0: out.WriteBuffer( &GGA_datum, sizeof(GGA_datum) );
			// New version:
			out << GGA_datum.UTCTime.hour
			    << GGA_datum.UTCTime.minute
				<< GGA_datum.UTCTime.sec
				<< GGA_datum.latitude_degrees
				<< GGA_datum.longitude_degrees
				<< GGA_datum.fix_quality
				<< GGA_datum.altitude_meters
				<< GGA_datum.satellitesUsed
				<< GGA_datum.thereis_HDOP
				<< GGA_datum.HDOP;
		}

		out << has_RMC_datum;
		if (has_RMC_datum)
		{
			// OLD Version 0: out.WriteBuffer( &RMC_datum, sizeof(RMC_datum) );
			// New version:
			out << RMC_datum.UTCTime.hour
				<< RMC_datum.UTCTime.minute
				<< RMC_datum.UTCTime.sec
				<< RMC_datum.validity_char
				<< RMC_datum.latitude_degrees
				<< RMC_datum.longitude_degrees
				<< RMC_datum.speed_knots
				<< RMC_datum.direction_degrees;
		}

		out << sensorLabel << sensorPose;

	}
}

/*---------------------------------------------------------------
  Implements the reading from a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CObservationGPS::readFromStream(CStream &in, int version)
{
	MRPT_UNUSED_PARAM(in);
	switch(version)
	{
	case 0:
		{
			in >> has_GGA_datum;
			if (has_GGA_datum)
				in.ReadBuffer( &GGA_datum, sizeof(GGA_datum) );

			in >> has_RMC_datum;
			if (has_RMC_datum)
				in.ReadBuffer( &RMC_datum, sizeof(RMC_datum) );
		} break;
	case 1:
	case 2:
	case 3:
	case 4:
		{
			if (version>=3)
					in >> timestamp;
			else 	timestamp = INVALID_TIMESTAMP;

			in >> has_GGA_datum;
			if (has_GGA_datum)
			{
				in  >> GGA_datum.UTCTime.hour
					>> GGA_datum.UTCTime.minute
					>> GGA_datum.UTCTime.sec
					>> GGA_datum.latitude_degrees
					>> GGA_datum.longitude_degrees
					>> GGA_datum.fix_quality
					>> GGA_datum.altitude_meters
					>> GGA_datum.satellitesUsed
					>> GGA_datum.thereis_HDOP
					>> GGA_datum.HDOP;
			}

			in >> has_RMC_datum;
			if (has_RMC_datum)
			{
				in  >> RMC_datum.UTCTime.hour
					>> RMC_datum.UTCTime.minute
					>> RMC_datum.UTCTime.sec
					>> RMC_datum.validity_char
					>> RMC_datum.latitude_degrees
					>> RMC_datum.longitude_degrees
					>> RMC_datum.speed_knots
					>> RMC_datum.direction_degrees;
			}

			if (version>1)
					in >> sensorLabel;
			else 	sensorLabel = "";

			if (version>=4)
					in >> sensorPose;
			else	sensorPose.setFromValues(0,0,0,0,0,0);

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
 Inserts a pure virtual method for finding the likelihood between this
   and another observation, probably only of the same derived class. The operator
   may be asymmetric.

 \param anotherObs The other observation to compute likelihood with.
 \param anotherObsPose If known, the belief about the robot pose when the other observation was taken can be supplied here, or NULL if it is unknown.

 \return Returns a likelihood measurement, in the range [0,1].
 \exception std::exception On any error, as another observation being of an invalid class.
  ---------------------------------------------------------------*/
float  CObservationGPS::likelihoodWith(
	const CObservation		*anotherObs,
	const CPosePDF			*anotherObsPose ) const
{
	MRPT_UNUSED_PARAM(anotherObs); MRPT_UNUSED_PARAM(anotherObsPose);
	return 0;
}

/*---------------------------------------------------------------
					dumpToStream
 ---------------------------------------------------------------*/
void  CObservationGPS::dumpToStream( CStream &out )
{
	out.printf("\n--------------------- [CObservationGPS] Dump: -----------------------\n");

	out.printf("\n[GGA datum: ");
	if (has_GGA_datum)
			out.printf("YES]\n");
	else	out.printf("NO]\n");
	if (has_GGA_datum)
	{
		out.printf("  Longitude: %.09f deg  Latitude: %.09f deg  Height: %.03f m\n",
			GGA_datum.longitude_degrees,
			GGA_datum.latitude_degrees,
			GGA_datum.altitude_meters
			);
		out.printf("  UTC time-stamp: %02u:%02u:%02.03f  #sats=%2u  ",
			GGA_datum.UTCTime.hour,
			GGA_datum.UTCTime.minute,
			GGA_datum.UTCTime.sec,
			GGA_datum.satellitesUsed );

		out.printf("Fix mode: %u ",GGA_datum.fix_quality);
		switch( GGA_datum.fix_quality )
		{
			case 0: out.printf("(Invalid)\n"); break;
			case 1: out.printf("(GPS fix)\n"); break;
			case 2: out.printf("(DGPS fix)\n"); break;
			case 3: out.printf("(PPS fix)\n"); break;
			case 4: out.printf("(Real Time Kinematic/RTK Fixed)\n"); break;
			case 5: out.printf("(Float RTK)\n"); break;
			case 6: out.printf("(dead reckoning)\n"); break;
			case 7: out.printf("(Manual)\n"); break;
			case 8: out.printf("(Simulation)\n"); break;
			default: out.printf("(UNKNOWN!)\n"); break;
		};

		out.printf("  HDOP (Horizontal Dilution of Precision): ");
		if (GGA_datum.thereis_HDOP)
				out.printf(" %f\n", GGA_datum.HDOP);
		else 	out.printf(" not available\n");

	} // END GGA

	out.printf("\n[RMC datum: ");
	if (has_RMC_datum)
			out.printf("YES]\n");
	else	out.printf("NO]\n");
	if (has_RMC_datum)
	{
		out.printf("  Longitude: %.09f deg  Latitude: %.09f deg  Valid?: '%c'\n",
			RMC_datum.longitude_degrees,
			RMC_datum.latitude_degrees,
			RMC_datum.validity_char
			);
		out.printf("  UTC time-stamp: %02u:%02u:%02.03f  ",
			RMC_datum.UTCTime.hour,
			RMC_datum.UTCTime.minute,
			RMC_datum.UTCTime.sec
			);

		out.printf(" Speed: %.05f knots  Direction:%.03f deg.\n ",
			RMC_datum.speed_knots,
			RMC_datum.direction_degrees
			);

	} // END RMC

	out.printf("---------------------------------------------------------------------\n\n");
}

/*---------------------------------------------------------------
					dumpToStream
 ---------------------------------------------------------------*/
void  CObservationGPS::dumpToConsole()
{
	dumpToStream( gps_my_cout );
}


// Ctor:
CObservationGPS::TUTCTime::TUTCTime() :
	hour(0), minute(0), sec(0)
{
}

// Ctor:
CObservationGPS::TGPSDatum_RMC::TGPSDatum_RMC() :
	UTCTime(),
	validity_char('V'),
	latitude_degrees(0),
	longitude_degrees(0),
	speed_knots(0),
	direction_degrees(0)
{ }

// Ctor:
CObservationGPS::TGPSDatum_GGA::TGPSDatum_GGA() :
	UTCTime(),
	latitude_degrees(0),
	longitude_degrees(0),
	fix_quality(0),
	altitude_meters(0),
	satellitesUsed(0),
	thereis_HDOP(false),
	HDOP(0)
{ }

