/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 



#include <mrpt/config.h>

#ifdef MRPT_OS_WINDOWS
	#include <windows.h>
#else
	#if MRPT_HAS_INOTIFY
		#include <sys/inotify.h>
	#endif

//	#include <time.h>
	#include <unistd.h>
	#include <errno.h>
#endif

#include <mrpt/system/CFileSystemWatcher.h>
#include <mrpt/system/os.h>

using namespace mrpt::system;
using namespace std;

/*---------------------------------------------------------------
					Constructor
 ---------------------------------------------------------------*/
CFileSystemWatcher::CFileSystemWatcher( const std::string &path ) :
	m_watchedDirectory(path)
{
	MRPT_TRY_START

#ifdef MRPT_OS_WINDOWS
	// Windows version:


#else
#	if MRPT_HAS_INOTIFY
	// Linux version:
	m_wd = -1;

	m_fd = inotify_init();
	if (m_fd < 0)
        THROW_EXCEPTION("inotify_init returned error!");

	// Create watcher:
	m_wd = inotify_add_watch(
		m_fd,
		path.c_str(),
		IN_CLOSE_WRITE | IN_DELETE | IN_MOVED_TO | IN_MOVED_FROM | IN_CREATE
		);

	if (m_wd < 0)
        THROW_EXCEPTION("inotify_add_watch returned error!");

	if (m_watchedDirectory[m_watchedDirectory.size()-1] != '/')
		m_watchedDirectory.push_back('/');
#	endif
#endif

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					Destructor
 ---------------------------------------------------------------*/
CFileSystemWatcher::~CFileSystemWatcher( )
{
#ifdef MRPT_OS_WINDOWS
	// Windows version:


#else
#	if	MRPT_HAS_INOTIFY
	// Linux version:
	if (m_fd >= 0)
	{
		close(m_fd);
		m_fd = -1;
		if (m_wd>=0)
			inotify_rm_watch(m_fd, m_wd);
	}
#	endif
#endif
}

/*---------------------------------------------------------------
					getChanges
 ---------------------------------------------------------------*/
void CFileSystemWatcher::getChanges( TFileSystemChangeList &out_list )
{
	out_list.clear();

#ifdef MRPT_OS_WINDOWS
	// Windows version:


#else
#	if	MRPT_HAS_INOTIFY
	if (m_fd<0) return;	// Not open?

	// Linux version:
	// Refer to:
	//  http://www.linuxjournal.com/article/8478
	//  http://inotify.aiken.cz/?section=common&page=home&lang=en
	struct timeval time;
	fd_set rfds;
	int ret;

	// timeout
	time.tv_sec = 0;
	time.tv_usec = 100;

	// zero-out the fd_set
	FD_ZERO (&rfds);

	// Add inotify fd
	FD_SET (m_fd, &rfds);

	ret = select (m_fd + 1, &rfds, NULL, NULL, &time);
	if (ret < 0)
	{
		perror ("[CFileSystemWatcher::getChanges] select");
		return;
	}

	else if (!ret)
	{
		// timed out!
	}
	else if (FD_ISSET (m_fd, &rfds))
	{
		// inotify events are available! : Read them!

		/* size of the event structure, not counting name */
		#define EVENT_SIZE  (sizeof (struct inotify_event))

		/* reasonable guess as to size of 1024 events */
		#define BUF_LEN        (1024 * (EVENT_SIZE + 16))

		char buf[BUF_LEN];
		int len, i = 0;

		len = read (m_fd, buf, BUF_LEN);
		if (len < 0)
		{
			if (errno == EINTR)
			{
				/* need to reissue system call */
			}
			else
				perror ("[CFileSystemWatcher::getChanges] read");
		}
		else if (!len)
		{
				/* BUF_LEN too small? */
		}

		while (i < len)
		{
			struct inotify_event *event;
			event = (struct inotify_event *) &buf[i];
			i += EVENT_SIZE + event->len;

//			printf ("wd=%d mask=%u cookie=%u len=%u\n",event->wd, event->mask,event->cookie, event->len);

			string eventName;
			if (event->len) eventName = event->name;

			// Add event to output list:
			// ---------------------------------------------
			if ( 0==(event->mask & IN_UNMOUNT) &&
			     0==(event->mask & IN_Q_OVERFLOW) &&
			     0==(event->mask & IN_IGNORED) )
			{
				TFileSystemChange	newEntry;

				newEntry.path 				= m_watchedDirectory + eventName;
				newEntry.isDir 				= event->mask & IN_ISDIR;
				newEntry.eventModified		= event->mask & IN_MODIFY;
				newEntry.eventCloseWrite  	= event->mask & IN_CLOSE_WRITE;
				newEntry.eventDeleted  		= event->mask & IN_DELETE;
				newEntry.eventMovedTo  		= event->mask & IN_MOVED_TO;
				newEntry.eventMovedFrom  	= event->mask & IN_MOVED_FROM;
				newEntry.eventCreated		= event->mask & IN_CREATE;

				out_list.push_back( newEntry );
			}
		}

	}
#	endif
#endif

}
