/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 



#include <mrpt/vision/CStereoGrabber_Bumblebee.h>
using namespace mrpt::vision;

#if MRPT_HAS_BUMBLEBEE
	// Include the OPENCV libraries:
	#if defined(_MSC_VER)
		#pragma comment (lib,"digiclops.lib")
		#pragma comment (lib,"triclops.lib")
	#endif
#endif

/*-------------------------------------------------------------
					Constructor
 -------------------------------------------------------------*/
CStereoGrabber_Bumblebee::CStereoGrabber_Bumblebee(
	int				cameraIndex,
	unsigned int	resolutionX,
	unsigned int	resolutionY )
{
    MRPT_TRY_START;
#if MRPT_HAS_BUMBLEBEE
	m_resolutionX = resolutionX;
	m_resolutionY = resolutionY;
	m_bInitialized = false;
	m_pRectImageLeft = new unsigned char[ m_resolutionX * m_resolutionY ];
	m_pRectImageRight = new unsigned char[ m_resolutionX * m_resolutionY ];
	m_pDispImage = new unsigned short[ m_resolutionX * m_resolutionY ];

   // open the Digiclops
   if (DIGICLOPS_ok != digiclopsCreateContext( &m_digiclops ) ) return;

   // assume we are getting the first one on the bus, device 0
   if (DIGICLOPS_ok != digiclopsInitialize( m_digiclops, cameraIndex ) ) return;

   // get the camera module configuration
   if (DIGICLOPS_ok != digiclopsGetTriclopsContextFromCamera(m_digiclops, &m_triclops ) ) return;

   // set the digiclops to deliver all images
   if (DIGICLOPS_ok != digiclopsSetImageTypes( m_digiclops, ALL_IMAGES ) ) return;

   // set the Digiclops resolution
   // use 'HALF' resolution when you need faster throughput, especially for
   // color images
   // digiclopsSetImageResolution( digiclops, DIGICLOPS_HALF );
   if (DIGICLOPS_ok != digiclopsSetImageResolution(m_digiclops, DIGICLOPS_FULL ) ) return;

   // start grabbing
   if (DIGICLOPS_ok != digiclopsStart( m_digiclops ) ) return;


   //  ------------------------------------------------------
   //   CONFIGURATION
   //  ------------------------------------------------------

   // set the image size
   if (TriclopsErrorOk != triclopsSetResolution(m_triclops, m_resolutionY, m_resolutionX )) return;

   // set disparity range
   if (TriclopsErrorOk != triclopsSetDisparity( m_triclops, 1, round(0.3f*m_resolutionX)) ) return;

   // set the edge and stereo masks
   if (TriclopsErrorOk != triclopsSetStereoMask( m_triclops, 11) ) return;

   if (TriclopsErrorOk != triclopsSetEdgeCorrelation( m_triclops, 1 ) ) return;
   if (TriclopsErrorOk != triclopsSetEdgeMask( m_triclops, 11) ) return;

   triclopsSetTextureValidation( m_triclops, 1 );
   triclopsSetTextureValidationThreshold( m_triclops, 4.00f );
   triclopsSetUniquenessValidation( m_triclops, 1 );
   triclopsSetUniquenessValidationThreshold( m_triclops, 0.84f);

   // turn on sub-pixel interpolation
   triclopsSetSubpixelInterpolation( m_triclops, 1 );
   // Set up image buffers for the context
//   triclopsSetImageBuffer(m_triclops, m_pRectImageLeft, TriImg_RECTIFIED, TriCam_LEFT );
//   triclopsSetImageBuffer(m_triclops, m_pRectImageRight, TriImg_RECTIFIED, TriCam_RIGHT );

   triclopsSetImage16Buffer( m_triclops, m_pDispImage, TriImg16_DISPARITY, TriCam_REFERENCE );

   // remember that we successfully initialized everything
   m_bInitialized = true;

#else
	MRPT_UNUSED_PARAM(cameraIndex);
	MRPT_UNUSED_PARAM(resolutionX);
	MRPT_UNUSED_PARAM(resolutionY);
	THROW_EXCEPTION("This class is not available. Recompile MRPT with option 'MRPT_HAS_BUMBLEBEE' on");
#endif
    MRPT_TRY_END;
}

/*-------------------------------------------------------------
					Destructor
 -------------------------------------------------------------*/
CStereoGrabber_Bumblebee::~CStereoGrabber_Bumblebee()
{
#if MRPT_HAS_BUMBLEBEE
	MRPT_TRY_START;

	if (m_pRectImageLeft) delete[] m_pRectImageLeft;
	if (m_pRectImageRight) delete[] m_pRectImageRight;
	if (m_pDispImage) delete[] m_pDispImage;

	if ( m_bInitialized )
		digiclopsDestroyContext( m_digiclops );

	MRPT_TRY_END;
#endif
}
// ******************************** FAMD ****************************************************
/*-------------------------------------------------------------
					getObservation with ROI
 -------------------------------------------------------------*/

bool  CStereoGrabber_Bumblebee::getObservation(
		TROI ROI,
		mrpt::slam::CObservationVisualLandmarks &out_observation)
{
#if MRPT_HAS_BUMBLEBEE
	MRPT_TRY_START;

	unsigned int		x,y;
	unsigned short		disp;

	float				x_min = 0.0f, x_max = 0.0f, y_min = 0.0f, y_max = 0.0f, z_min = 0.0f, z_max = 0.0f;

	TriclopsColorImage  colorImage;
	TriclopsInput       colorData;

	// Set the timestep:
	out_observation.timestamp = system::getCurrentTime();

	// Start with an empty landmarks map:
//	out_observation.landmarks.landmarks.reserve(30000);
	out_observation.landmarks.landmarks.clear();


	if (DIGICLOPS_ok != digiclopsGrabImage( m_digiclops ) )
		return false;

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, STEREO_IMAGE, &m_triclopsInput ) )
		return false;

	// For getting the colors of the 3D points later:
	digiclopsExtractTriclopsInput( m_digiclops, RIGHT_IMAGE, &colorData );

	// Disparity will be stored in our custom buffer:
	triclopsPreprocess( m_triclops, &m_triclopsInput );
	triclopsStereo ( m_triclops );
	triclopsRectifyColorImage( m_triclops, TriCam_REFERENCE,&colorData, &colorImage );

	// Create a points-cloud representation:
	// ---------------------------------------------

	// Go across all the pixels:
	// --------------------------------------------
	for (y=0;y<m_resolutionY;y++)
	{
		for (x=0;x<m_resolutionX;x++)
		{
			int k = x+y*m_resolutionX;
			disp = m_pDispImage[k];
			if ( disp < 0xFF00 )
			{
				mrpt::slam::CLandmark		lm;
				lm.type			= mrpt::slam::CLandmark::vlColor;

				// convert the 16 bit disparity value to floating point x,y,z
				// And set the pose PDF:
				triclopsRCD16ToXYZ( m_triclops, y, x, disp, &lm.pose_mean_x, &lm.pose_mean_y, &lm.pose_mean_z ) ;

				// If the pixel is inside the ROI -> compute the rest of parameters and add to the Observation
				// Process ROI
				(ROI.xMin == 0 && ROI.xMax == 0) ? x_min = -50.0f, x_max = 50.0f : x_min = ROI.xMin, x_max = ROI.xMax;
				(ROI.yMin == 0 && ROI.zMax == 0) ? y_min = -50.0f, y_max = 50.0f : y_min = ROI.yMin, y_max = ROI.yMax;
				(ROI.zMin == 0 && ROI.zMax == 0) ? z_min = -1.0f, z_max = 100.0f : z_min = ROI.zMin, z_max = ROI.zMax;

				if ( ( lm.pose_mean_x < x_min ) || ( lm.pose_mean_x > x_max ) ||
					 ( lm.pose_mean_y < y_min ) || ( lm.pose_mean_y > y_max ) ||
					 ( lm.pose_mean_z < z_min ) || ( lm.pose_mean_z > z_max ) )
					continue;

				// Fill all required fields:
				// --------------------------------------------
				lm.pose_cov_11 = 1e-4f;
				lm.pose_cov_22 = 1e-4f;
				lm.pose_cov_33 = 1e-4f;
				lm.pose_cov_12 = lm.pose_cov_13 = lm.pose_cov_23 = 0;

				// The normal to the point (point-view direction):
				// Normalized vector:
				float		K = -1.0f / sqrt( square(lm.pose_mean_x)+square(lm.pose_mean_y)+square(lm.pose_mean_z) );
				lm.normal_x = K * lm.pose_mean_x;
				lm.normal_y = K * lm.pose_mean_y;
				lm.normal_z = K * lm.pose_mean_z;

				// Set the color:
				lm.descriptor1.resize(3);
				lm.descriptor1[0] = colorImage.red[k];
				lm.descriptor1[1] = colorImage.green[k];
				lm.descriptor1[2] = colorImage.blue[k];

				// And add new landmark:
				// --------------------------------------------
				out_observation.landmarks.landmarks.push_back( &lm );
			}

		} // end for x

	}	// end for y

	return true;
	MRPT_TRY_END;
#else
	MRPT_UNUSED_PARAM(out_observation);
	MRPT_UNUSED_PARAM(ROI);
	return false;


#endif
}
// ****************************** END FAMD ***************************************************
/*-------------------------------------------------------------
					getObservation
 -------------------------------------------------------------*/
bool  CStereoGrabber_Bumblebee::getObservation(
		mrpt::slam::CObservationVisualLandmarks &out_observation)
{
#if MRPT_HAS_BUMBLEBEE
	unsigned int		x,y;
	unsigned short		disp;
	TriclopsColorImage  colorImage;
	TriclopsInput       colorData;

	// Set the timestep:
	out_observation.timestamp = system::getCurrentTime();

	// Start with an empty landmarks map:
//	out_observation.landmarks.landmarks.reserve(30000);
	out_observation.landmarks.landmarks.clear();


	if (DIGICLOPS_ok != digiclopsGrabImage( m_digiclops ) )
		return false;

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, STEREO_IMAGE, &m_triclopsInput ) )
		return false;

	// For getting the colors of the 3D points later:
	digiclopsExtractTriclopsInput( m_digiclops, RIGHT_IMAGE, &colorData );

	// Disparity will be stored in our custom buffer:
	triclopsPreprocess( m_triclops, &m_triclopsInput );
	triclopsStereo ( m_triclops );
	triclopsRectifyColorImage( m_triclops, TriCam_REFERENCE,&colorData, &colorImage );

	// Create a points-cloud representation:
	// ---------------------------------------------

	// Go across all the pixels:
	// --------------------------------------------
	for (y=0;y<m_resolutionY;y++)
	{
		for (x=0;x<m_resolutionX;x++)
		{
			int k = x+y*m_resolutionX;
			disp = m_pDispImage[k];
			if ( disp < 0xFF00 )
			{
				mrpt::slam::CLandmark		lm;
				lm.type			= mrpt::slam::CLandmark::vlColor;

				// convert the 16 bit disparity value to floating point x,y,z
				// And set the pose PDF:
				triclopsRCD16ToXYZ( m_triclops, y, x, disp, &lm.pose_mean_x, &lm.pose_mean_y, &lm.pose_mean_z ) ;

				// Fill all required fields:
				// --------------------------------------------
				lm.pose_cov_11 = 1e-4f;
				lm.pose_cov_22 = 1e-4f;
				lm.pose_cov_33 = 1e-4f;
				lm.pose_cov_12 = lm.pose_cov_13 = lm.pose_cov_23 = 0;

				// The normal to the point (point-view direction):
				// Normalized vector:
				float		K = -1.0f / sqrt( square(lm.pose_mean_x)+square(lm.pose_mean_y)+square(lm.pose_mean_z) );
				lm.normal_x = K * lm.pose_mean_x;
				lm.normal_y = K * lm.pose_mean_y;
				lm.normal_z = K * lm.pose_mean_z;

				// Set the color:
				lm.descriptor1.resize(3);
				lm.descriptor1[0] = colorImage.red[k];
				lm.descriptor1[1] = colorImage.green[k];
				lm.descriptor1[2] = colorImage.blue[k];

				// And add new landmark:
				// --------------------------------------------
				out_observation.landmarks.landmarks.push_back( &lm );
			}

		} // End for x

	}	// end for y

	return true;
#else
	MRPT_UNUSED_PARAM(out_observation);
	return false;
#endif
}


/*-------------------------------------------------------------
					getImagesPairObservation
 -------------------------------------------------------------*/
bool  CStereoGrabber_Bumblebee::getImagesPairObservation( mrpt::slam::CObservationStereoImages &out_observation)
{
#if MRPT_HAS_BUMBLEBEE
	TriclopsColorImage  colorImage_L,colorImage_R;
	TriclopsInput       colorData;

	// Set the timestep:
	out_observation.timestamp = system::getCurrentTime();

	if (DIGICLOPS_ok != digiclopsGrabImage( m_digiclops ) )
		return false;

	// Rectify images:
	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, RIGHT_IMAGE, &colorData ) )	return false;
	triclopsRectifyColorImage( m_triclops, TriCam_RIGHT,&colorData, &colorImage_R );

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, LEFT_IMAGE, &colorData ) )	return false;
	triclopsRectifyColorImage( m_triclops, TriCam_LEFT,&colorData, &colorImage_L );

	// Set the intrinsic parameters matrix:
	out_observation.intrinsicParams = vision::vision::defaultIntrinsicParamsMatrix(
				0,				// 0 = Bumblebee camera
				m_resolutionX,
				m_resolutionY );

	// BASE LINE:
	out_observation.rightCameraPose.setFromValues(0.119415f,0,0, 0,0,0);
	out_observation.cameraPose.setFromValues(0,0,0,(float)DEG2RAD(-90),0,(float)DEG2RAD(-90));

	// Load the images:
	out_observation.imageRight.loadFromMemoryBuffer(m_resolutionX,m_resolutionY, colorImage_R.rowinc,colorImage_R.red,colorImage_R.green,colorImage_R.blue);
	out_observation.imageLeft.loadFromMemoryBuffer(m_resolutionX,m_resolutionY, colorImage_L.rowinc,colorImage_L.red,colorImage_L.green,colorImage_L.blue);

	return true;
#else
	MRPT_UNUSED_PARAM(out_observation);
	return false;
#endif
}


/*-------------------------------------------------------------
					getBothObservation
 -------------------------------------------------------------*/
bool  CStereoGrabber_Bumblebee::getBothObservation(
			mrpt::slam::CObservationVisualLandmarks	&out_observation,
			mrpt::slam::CObservationStereoImages		&out_observationStereo )
{
#if MRPT_HAS_BUMBLEBEE
	unsigned int		x,y;
	unsigned short		disp;
	TriclopsColorImage  colorImage;
	TriclopsInput       colorData;

	// Set the timestep:
	out_observation.timestamp = out_observationStereo.timestamp = system::getCurrentTime();

	// Start with an empty landmarks map:
//	out_observation.landmarks.landmarks.reserve(30000);
	out_observation.landmarks.landmarks.clear();


	if (DIGICLOPS_ok != digiclopsGrabImage( m_digiclops ) )
		return false;

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, STEREO_IMAGE, &m_triclopsInput ) )
		return false;

	// For getting the colors of the 3D points later:
	digiclopsExtractTriclopsInput( m_digiclops, RIGHT_IMAGE, &colorData );

	// Disparity will be stored in our custom buffer:
	triclopsPreprocess( m_triclops, &m_triclopsInput );
	triclopsStereo ( m_triclops );
	triclopsRectifyColorImage( m_triclops, TriCam_REFERENCE,&colorData, &colorImage );

	// Create a points-cloud representation:
	// ---------------------------------------------

	// Go across all the pixels:
	// --------------------------------------------
	for (y=0;y<m_resolutionY;y++)
	{
		for (x=0;x<m_resolutionX;x++)
		{
			int k = x+y*m_resolutionX;
			disp = m_pDispImage[k];
			if ( disp < 0xFF00 )
			{
				mrpt::slam::CLandmark		lm;
				lm.type			= mrpt::slam::CLandmark::vlColor;

				// convert the 16 bit disparity value to floating point x,y,z
				// And set the pose PDF:
				triclopsRCD16ToXYZ( m_triclops, y, x, disp, &lm.pose_mean_x, &lm.pose_mean_y, &lm.pose_mean_z ) ;

				// Fill all required fields:
				// --------------------------------------------
				lm.pose_cov_11 = 1e-4f;
				lm.pose_cov_22 = 1e-4f;
				lm.pose_cov_33 = 1e-4f;
				lm.pose_cov_12 = lm.pose_cov_13 = lm.pose_cov_23 = 0;

				// The normal to the point (point-view direction):
				// Normalized vector:
				float		K = -1.0f / sqrt( square(lm.pose_mean_x)+square(lm.pose_mean_y)+square(lm.pose_mean_z) );
				lm.normal_x = K * lm.pose_mean_x;
				lm.normal_y = K * lm.pose_mean_y;
				lm.normal_z = K * lm.pose_mean_z;

				// Set the color:
				lm.descriptor1.resize(3);
				lm.descriptor1[0] = colorImage.red[k];
				lm.descriptor1[1] = colorImage.green[k];
				lm.descriptor1[2] = colorImage.blue[k];

				// And add new landmark:
				// --------------------------------------------
				out_observation.landmarks.landmarks.push_back( &lm );
			}

		} // End for x

	}	// end for y

	// ------------------------

	TriclopsColorImage  colorImage_L,colorImage_R;

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, RIGHT_IMAGE, &colorData ) )
		return false;

	// Rectify images:
	triclopsRectifyColorImage( m_triclops, TriCam_RIGHT,&colorData, &colorImage_R );

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, LEFT_IMAGE, &colorData ) )
		return false;

	// Rectify images:
	triclopsRectifyColorImage( m_triclops, TriCam_LEFT,&colorData, &colorImage_L );

	// Set the intrinsic parameters matrix:
	out_observationStereo.intrinsicParams = vision::vision::defaultIntrinsicParamsMatrix(
				0,				// 0 = Bumblebee camera
				m_resolutionX,
				m_resolutionY );

	// BASE LINE:
	out_observationStereo.rightCameraPose.setFromValues(0.119415f,0,0, 0,0,0);

	// Load the images:
	out_observationStereo.imageRight.loadFromMemoryBuffer(m_resolutionX,m_resolutionY, colorImage_R.rowinc,colorImage_R.red,colorImage_R.green,colorImage_R.blue);
	out_observationStereo.imageLeft.loadFromMemoryBuffer(m_resolutionX,m_resolutionY, colorImage_L.rowinc,colorImage_L.red,colorImage_L.green,colorImage_L.blue);

	return true;
#else
	MRPT_UNUSED_PARAM(out_observation);
	MRPT_UNUSED_PARAM(out_observationStereo);
	return false;
#endif
}

/*-------------------------------------------------------------
					getBothObservation
 -------------------------------------------------------------*/
bool  CStereoGrabber_Bumblebee::getBothObservation(
			vector_float						&vX,
			vector_float						&vY,
			vector_float						&vZ,
			mrpt::slam::CObservationStereoImages		&out_observationStereo )
{
#if MRPT_HAS_BUMBLEBEE
	unsigned int		x,y;
	unsigned short		disp;
	TriclopsColorImage  colorImage;
	TriclopsInput       colorData;

	vX.clear(); vX.reserve( 30000 );
	vY.clear(); vY.reserve( 30000 );
	vZ.clear(); vZ.reserve( 30000 );

	// Set the timestep:
	out_observationStereo.timestamp = system::getCurrentTime();

	if (DIGICLOPS_ok != digiclopsGrabImage( m_digiclops ) )
		return false;

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, STEREO_IMAGE, &m_triclopsInput ) )
		return false;

	// For getting the colors of the 3D points later:
	digiclopsExtractTriclopsInput( m_digiclops, RIGHT_IMAGE, &colorData );

	// Disparity will be stored in our custom buffer:
	triclopsPreprocess( m_triclops, &m_triclopsInput );
	triclopsStereo ( m_triclops );

	triclopsRectifyColorImage( m_triclops, TriCam_REFERENCE, &colorData, &colorImage );

	// Create a points-cloud representation:
	// ---------------------------------------------

	// Go across all the pixels:
	// --------------------------------------------
	for (y=0;y<m_resolutionY;y++)
	{
		for (x=0;x<m_resolutionX;x++)
		{
			int k = x+y*m_resolutionX;
			disp = m_pDispImage[k];
			if ( disp < 0xFF00 )
			{
				float x3D, y3D, z3D;

				// convert the 16 bit disparity value to floating point x,y,z
				// And set the pose PDF:
				triclopsRCD16ToXYZ( m_triclops, y, x, disp, &x3D, &y3D, &z3D );

				vX.push_back( x3D );
				vY.push_back( y3D );
				vZ.push_back( z3D );
			}

		} // End for x

	}	// end for y

	// ------------------------

	TriclopsColorImage  colorImage_L,colorImage_R;

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, RIGHT_IMAGE, &colorData ) )
		return false;

	// Rectify images:
	triclopsRectifyColorImage( m_triclops, TriCam_RIGHT,&colorData, &colorImage_R );

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, LEFT_IMAGE, &colorData ) )
		return false;

	// Rectify images:
	triclopsRectifyColorImage( m_triclops, TriCam_LEFT,&colorData, &colorImage_L );

	// Set the intrinsic parameters matrix:
	out_observationStereo.intrinsicParams = vision::vision::defaultIntrinsicParamsMatrix(
				0,				// 0 = Bumblebee camera
				m_resolutionX,
				m_resolutionY );

	// BASE LINE:
	out_observationStereo.rightCameraPose.setFromValues(0.119415f,0,0, 0,0,0);

	// Load the images:
	out_observationStereo.imageRight.loadFromMemoryBuffer(m_resolutionX,m_resolutionY, colorImage_R.rowinc,colorImage_R.red,colorImage_R.green,colorImage_R.blue);
	out_observationStereo.imageLeft.loadFromMemoryBuffer(m_resolutionX,m_resolutionY, colorImage_L.rowinc,colorImage_L.red,colorImage_L.green,colorImage_L.blue);

	return true;
#else
	MRPT_UNUSED_PARAM(vX);
	MRPT_UNUSED_PARAM(vY);
	MRPT_UNUSED_PARAM(vZ);
	MRPT_UNUSED_PARAM(out_observationStereo);
	return false;
#endif
}

// ****************************** FAMD ***************************************************
/*-------------------------------------------------------------
					getBothObservation with ROI
 -------------------------------------------------------------*/
bool  CStereoGrabber_Bumblebee::getBothObservation(
			TROI								ROI,
			mrpt::slam::CObservationVisualLandmarks	&out_observation,
			mrpt::slam::CObservationStereoImages		&out_observationStereo )
{
#if MRPT_HAS_BUMBLEBEE
	unsigned int		x,y;
	unsigned short		disp;

	float				x_min = 0.0f, x_max = 0.0f, y_min = 0.0f, y_max = 0.0f, z_min = 0.0f, z_max = 0.0f;

	TriclopsColorImage  colorImage;
	TriclopsInput       colorData;

	// Set the timestep:
	out_observation.timestamp = out_observationStereo.timestamp = system::getCurrentTime();

	// Start with an empty landmarks map:
//	out_observation.landmarks.landmarks.reserve(30000);
	out_observation.landmarks.landmarks.clear();


	if (DIGICLOPS_ok != digiclopsGrabImage( m_digiclops ) )
		return false;

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, STEREO_IMAGE, &m_triclopsInput ) )
		return false;

	// For getting the colors of the 3D points later:
	digiclopsExtractTriclopsInput( m_digiclops, RIGHT_IMAGE, &colorData );

	// Disparity will be stored in our custom buffer:
	triclopsPreprocess( m_triclops, &m_triclopsInput );
	triclopsStereo ( m_triclops );
	triclopsRectifyColorImage( m_triclops, TriCam_REFERENCE,&colorData, &colorImage );

	// Create a points-cloud representation:
	// ---------------------------------------------

	// Go across all the pixels:
	// --------------------------------------------
	for (y=0;y<m_resolutionY;y++)
	{
		for (x=0;x<m_resolutionX;x++)
		{
			int k = x+y*m_resolutionX;
			disp = m_pDispImage[k];
			if ( disp < 0xFF00 )
			{
				mrpt::slam::CLandmark		lm;
				lm.type			= mrpt::slam::CLandmark::vlColor;

				// convert the 16 bit disparity value to floating point x,y,z
				// And set the pose PDF:
				triclopsRCD16ToXYZ( m_triclops, y, x, disp, &lm.pose_mean_x, &lm.pose_mean_y, &lm.pose_mean_z ) ;

				// If the pixel is inside the ROI -> compute the rest of parameters and add to the Observation
				// Process ROI
				(ROI.xMin == 0 && ROI.xMax == 0) ? x_min = -50.0f, x_max = 50.0f : x_min = ROI.xMin, x_max = ROI.xMax;
				(ROI.yMin == 0 && ROI.zMax == 0) ? y_min = -50.0f, y_max = 50.0f : x_min = ROI.yMin, x_max = ROI.yMax;
				(ROI.zMin == 0 && ROI.zMax == 0) ? z_min = -1.0f, z_max = 100.0f : x_min = ROI.zMin, x_max = ROI.zMax;

				if ( ( lm.pose_mean_x < x_min ) || ( lm.pose_mean_x > x_max ) ||
					 ( lm.pose_mean_y < y_min ) || ( lm.pose_mean_y > y_max ) ||
					 ( lm.pose_mean_z < z_min ) || ( lm.pose_mean_z > z_max ) )
					continue;

				// Fill all required fields:
				// --------------------------------------------
				lm.pose_cov_11 = 1e-4f;
				lm.pose_cov_22 = 1e-4f;
				lm.pose_cov_33 = 1e-4f;
				lm.pose_cov_12 = lm.pose_cov_13 = lm.pose_cov_23 = 0;

				// The normal to the point (point-view direction):
				// Normalized vector:
				float		K = -1.0f / sqrt( square(lm.pose_mean_x)+square(lm.pose_mean_y)+square(lm.pose_mean_z) );
				lm.normal_x = K * lm.pose_mean_x;
				lm.normal_y = K * lm.pose_mean_y;
				lm.normal_z = K * lm.pose_mean_z;

				// Set the color:
				lm.descriptor1.resize(3);
				lm.descriptor1[0] = colorImage.red[k];
				lm.descriptor1[1] = colorImage.green[k];
				lm.descriptor1[2] = colorImage.blue[k];

				// And add new landmark:
				// --------------------------------------------
				out_observation.landmarks.landmarks.push_back( &lm );
			}

		} // End for x

	}	// end for y

	// ------------------------

	TriclopsColorImage  colorImage_L,colorImage_R;

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, RIGHT_IMAGE, &colorData ) )
		return false;

	// Rectify images:
	triclopsRectifyColorImage( m_triclops, TriCam_RIGHT,&colorData, &colorImage_R );

	if (DIGICLOPS_ok != digiclopsExtractTriclopsInput(m_digiclops, LEFT_IMAGE, &colorData ) )
		return false;

	// Rectify images:
	triclopsRectifyColorImage( m_triclops, TriCam_LEFT,&colorData, &colorImage_L );

	// Set the intrinsic parameters matrix:
	out_observationStereo.intrinsicParams = vision::vision::defaultIntrinsicParamsMatrix(
				0,				// 0 = Bumblebee camera
				m_resolutionX,
				m_resolutionY );

	// BASE LINE:
	out_observationStereo.rightCameraPose.setFromValues(0.119415f,0,0, 0,0,0);

	// Load the images:
	out_observationStereo.imageRight.loadFromMemoryBuffer(m_resolutionX,m_resolutionY, colorImage_R.rowinc,colorImage_R.red,colorImage_R.green,colorImage_R.blue);
	out_observationStereo.imageLeft.loadFromMemoryBuffer(m_resolutionX,m_resolutionY, colorImage_L.rowinc,colorImage_L.red,colorImage_L.green,colorImage_L.blue);

	return true;
#else
	MRPT_UNUSED_PARAM(out_observation);
	MRPT_UNUSED_PARAM(out_observationStereo);
	MRPT_UNUSED_PARAM(ROI);
	return false;
#endif
}
// ****************************** END FAMD ***************************************************
