/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/system/os.h>
#include <mrpt/hwdrivers/C2DRangeFinderAbstract.h>

using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::hwdrivers;

/*-------------------------------------------------------------
						Constructor
-------------------------------------------------------------*/
C2DRangeFinderAbstract::C2DRangeFinderAbstract() :
	m_lastObservation		( ),
	m_lastObservationIsNew	( false ),
	m_hardwareError			( false ),
	m_nextObservation		(),
	m_stream				( NULL  )
{
}

/*-------------------------------------------------------------
						Destructor
-------------------------------------------------------------*/
C2DRangeFinderAbstract::~C2DRangeFinderAbstract()
{
}

/*-------------------------------------------------------------
						bindIO
-------------------------------------------------------------*/
void  C2DRangeFinderAbstract::bindIO( CStream	*streamIO )
{
	m_csChangeStream.enter();
	m_stream = streamIO;
	m_csChangeStream.leave();
}

/*-------------------------------------------------------------
						getObservation
-------------------------------------------------------------*/
void  C2DRangeFinderAbstract::getObservation(
	bool							&outThereIsObservation,
	mrpt::slam::CObservation2DRangeScan	&outObservation,
	bool							&hardwareError )
{
	m_csLastObservation.enter();

	hardwareError			= m_hardwareError;
	outThereIsObservation	= m_lastObservationIsNew;

	if (outThereIsObservation)
		outObservation = m_lastObservation;

	m_csLastObservation.leave();
}

/*-------------------------------------------------------------
						doProcess
-------------------------------------------------------------*/
void C2DRangeFinderAbstract::doProcess()
{
	bool	thereIs, hwError;

	if (!m_nextObservation)
		m_nextObservation =  CObservation2DRangeScanPtr( new CObservation2DRangeScan() );

	doProcessSimple( thereIs, *m_nextObservation, hwError );

	if (hwError)
	{
		m_state = ssError;
	    THROW_EXCEPTION("Couldn't communicate to the USB board!");
	}

	if (thereIs)
	{
		m_state = ssWorking;

		// Do filter:
		this->filterByExclusionAreas( *m_nextObservation );

		appendObservation( m_nextObservation );
		m_nextObservation.clear_unique(); // Create a new object in the next call
	}
}

/*-------------------------------------------------------------
						loadExclusionAreas
-------------------------------------------------------------*/
void C2DRangeFinderAbstract::loadExclusionAreas(
	const mrpt::utils::CConfigFileBase &configSource,
	const std::string	  &iniSection )
{

	unsigned int N = 1;

	for(;;)
	{
		vector_double x,y;
		configSource.read_vector( iniSection, format("exclusionZone%u_x",N), vector_double(0), x);
		configSource.read_vector( iniSection, format("exclusionZone%u_y",N++), vector_double(0), y);

		if (!x.empty() && !y.empty())
		{
			ASSERT_(x.size()==y.size())

			mrpt::math::CPolygon	p;
			p.setAllVertices(x,y);
			m_lstExclusionPolys.push_back(p);
		}
		else break;
	}
}

/*-------------------------------------------------------------
						filterByExclusionAreas
-------------------------------------------------------------*/
void C2DRangeFinderAbstract::filterByExclusionAreas( mrpt::slam::CObservation2DRangeScan &obs) const
{
	obs.filterByExclusionAreas( m_lstExclusionPolys );
}
