/*
MobileRobots Advanced Robotics Interface for Applications (ARIA)
Copyright (C) 2004, 2005 ActivMedia Robotics LLC
Copyright (C) 2006, 2007 MobileRobots Inc.

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

If you wish to redistribute ARIA under different terms, contact 
MobileRobots for information about a commercial version of ARIA at 
robots@mobilerobots.com or 
MobileRobots Inc, 19 Columbia Drive, Amherst, NH 03031; 800-639-9481
*/

#include "ArExport.h"
#include "ariaOSDef.h"
#include "ArRatioInputJoydrive.h"
#include "ArRobot.h"
#include "ariaInternal.h"

/**
   This action is for driving around the robot with a joystick, you
   must hold in a button on the joystick and then lean the joytsick
   over to have it drive.  You may need to calibrate the joystick for
   it to work right, for details about this see ArJoyHandler.  If the
   Aria static class already has a joyhandler this class will use that
   otherwise it'll make and initialize one and use that (setting it in
   the Aria class) This uses the ArActionRatioInput so that it'll work
   and play well with others


   @param stopIfNoButtonPressed if this is true and there is a
   joystick and no button is pressed, the action will have the robot
   stop... otherwise it'll do nothing (letting lower priority actions
   fire)

   @param useOSCalForJoystick if this is true we'll use the OS
   calibration, if false we'll do our own

   @see ArJoyHandler::setUseOSCal
**/

AREXPORT ArRatioInputJoydrive::ArRatioInputJoydrive(
	ArRobot *robot,
	ArActionRatioInput *input,
	int priority,
	bool stopIfNoButtonPressed,
	bool useOSCalForJoystick) :
  myFireCB(this, &ArRatioInputJoydrive::fireCallback)
{
  myRobot = robot;
  myInput = input;
  myInput->addFireCallback(priority, &myFireCB);
  myFireCB.setName("Joydrive");
  if ((myJoyHandler = Aria::getJoyHandler()) == NULL)
  {
    myJoyHandler = new ArJoyHandler;
    myJoyHandler->init();
    Aria::setJoyHandler(myJoyHandler);
  }

  myUseOSCal = useOSCalForJoystick;
  myPreviousUseOSCal = myUseOSCal;
  myStopIfNoButtonPressed = stopIfNoButtonPressed;
  myFiredLast = false;
  myPrinting = false;
}

AREXPORT ArRatioInputJoydrive::~ArRatioInputJoydrive()
{
  myInput->remFireCallback(&myFireCB);
}

AREXPORT void ArRatioInputJoydrive::setStopIfNoButtonPressed(
	bool stopIfNoButtonPressed)
{
  myStopIfNoButtonPressed = stopIfNoButtonPressed;
}

AREXPORT bool ArRatioInputJoydrive::getStopIfNoButtonPressed(void)
{
  return myStopIfNoButtonPressed;
}

AREXPORT bool ArRatioInputJoydrive::joystickInited(void)
{
  return myJoyHandler->haveJoystick();
}

/**
   @see ArJoyHandler::setUseOSCal
**/
AREXPORT void ArRatioInputJoydrive::setUseOSCal(bool useOSCal)
{
  myUseOSCal = useOSCal;
  myPreviousUseOSCal = useOSCal;
  myJoyHandler->setUseOSCal(useOSCal);
}

/**
   @see ArJoyHandler::getUseOSCal
**/
AREXPORT bool ArRatioInputJoydrive::getUseOSCal(void)
{
  return myUseOSCal;
}


void ArRatioInputJoydrive::fireCallback(void)
{
  double rot, trans, throttle;

  if (myPreviousUseOSCal != myUseOSCal)
  {
    myJoyHandler->setUseOSCal(myUseOSCal);
    myPreviousUseOSCal = myUseOSCal;
  }

  if (myJoyHandler->haveJoystick() && myJoyHandler->getButton(1))
  {
    // get the readings from the joystick
    myJoyHandler->getDoubles(&rot, &trans);
    
    if (!myJoyHandler->haveZAxis()) 
    {
      throttle = 1;
    }
    // if we are using the throttle, interpolate its position between
    // low and high throttle values
    else
    {
      throttle = myJoyHandler->getAxis(3);
      throttle += 1.0;
      throttle /= 2.0;
    }
    myInput->setRatios(trans * 100, -rot * 100, throttle * 100);
    myFiredLast = true;
    if (myPrinting)
      printf("joy %g %g %g\n", trans * 100, -rot * 100, throttle * 100);
  }
  else if (myJoyHandler->haveJoystick() && (myStopIfNoButtonPressed ||
					    myFiredLast))
  {
    if (myPrinting)
      printf("joy nothing\n");
    myFiredLast = false;
    myInput->setRatios(0, 0, myInput->getThrottleRatio());
  }
  
}
